#!/usr/bin/perl -w

use strict;
use warnings;
use Tie::IxHash;
use Getopt::Long;
use List::Util qw(sum);

###################################################################################################
#Parse input options
use vars qw($infile $filter $print $help);

$filter="all";
$print="5";

GetOptions ("c=s"  => \$infile,
            "r=s"  => \$filter,
            "p=i"  => \$print,
            "help" => \$help,
            "h"    => \$help);

usage() if($help);
$filter=".+" if($filter=~/^all$/i);

###################################################################################################
sub usage {
	print STDERR "\nProgram: compBlockStat.pl (Derive information from a blockbuster output file)\n";
	print STDERR "Author: University of Copenhagen, Denmark\n";
	print STDERR "Version: 1.0\n";
	print STDERR "Contact: sachin\@rth.dk\n";
	print STDERR "Usage: compBlockStat.pl -c <file> [OPTIONS]\n";
	print STDERR " -c <file>    [block group file]\n";
	print STDERR "[OPTIONS]\n";
	print STDERR " -r <string>  [compute only for miRNA|snoRNA|tRNA|block group Id|all (default: all)]\n";
	print STDERR " -p <int>     [1=overlap|2=block info|3=block group info|4=summary|5=annotation statistics (default: 5)]\n\n";
	exit(-1);
}

my @bg_header=(); my %bgInfo=(); my %bInfo=(); my %block_read_start=(); my %bg_read_start=(); my %bgAnnoStat=();
my @temp=(); my $id="NA"; my $key=(); my $total_bg_count=0; my %seen=(); my %read_background_dist=(); my $total_reads=0;
my @testBlockSig=(); my $is_start=0;

#####Variables for computing summary statistics
my $max_block_count=0; my $min_block_count=1000; my $mean_block_count=0; my $sum_block_count=0; my $count_block_count=0;
my $max_block_size=0; my $min_block_size=1000; my $mean_block_size=0; my $sum_block_size=0; my $count_block_size=0;
my $max_block_dist=0; my $min_block_dist=1000; my $mean_block_dist=0; my $sum_block_dist=0; my $count_block_dist=0;
my $max_block_entropy=0; my $min_block_entropy=1000; my $mean_block_entropy=0; my $sum_block_entropy=0; my $count_block_entropy=0;
my $max_bg_entropy=0; my $min_bg_entropy=1000; my $mean_bg_entropy=0; my $sum_bg_entropy=0; my $count_bg_entropy=0;
my $max_bg_relEntropy=0; my $min_bg_relEntropy=1000; my $mean_bg_relEntropy=0; my $sum_bg_relEntropy=0; my $count_bg_relEntropy=0;
my @block_count=(); my @block_size=(); my @block_dist=(); my @block_entropy=(); my @bg_entropy=(); my @bg_relEntropy=();
#####End

## open read bg file
my $INFILE=();
if(defined($infile)) {
	if($infile=~/\.gz$/) { open($INFILE, "gunzip -c $infile |") || die $!; }
	else { open($INFILE, $infile) || die $!; }
}
else { $INFILE = *STDIN; }
my @data=<$INFILE>;
close $INFILE; 

#Print header for read block statistics
print "#start_end\tblock_size\tblock_dist\tblock_entropy\tblock_reads\tblock_overlap\tncRNA_type\tncRNA_loci\tbg_id\tstart\tend\n" if($print==2);
#Print header for block group statistics
print "#start_end\tblock_count\tbg_reads\tbg_entropy\tncRNA_type\tncRNA_loci\tbg_id\tbg_relEntropy\tuniq_reads\tbg_size\n" if($print==3);

#compute background distribution of reads
my $start=(); my $end; my $strand=();
foreach(@data) {
	chomp($_);
	@temp=split(/\s+/,$_);
	if($temp[0]=~/^>.+/) {
		$strand=$temp[4];
		if($strand=~/\+/) { $start=$temp[2]; }
		elsif($strand=~/\-/) { $end=$temp[3]; }
	}
	else {
		if($strand=~/\+/) { $read_background_dist{$temp[1] - $start} += $temp[4]; }
		elsif($strand=~/\-/) { $read_background_dist{$end - $temp[2]} += $temp[4]; }
		$total_reads += $temp[4];
	}
}
#my $sum=0; foreach (sort { $a <=> $b } keys(%read_background_dist)) { $sum += $read_background_dist{$_}; print "$_\t$read_background_dist{$_}\n"; } exit; #print "\n$sum\t$total_reads\n";

#Compute block start and end coordinates.
foreach(@data) {
	chomp($_);
	@temp=split('\s+', $_);
	if($temp[0]=~/^>.+/) {
		@testBlockSig=(); $is_start=0;
		@bg_header = split(/\s+/, $_);
		if(scalar(@bg_header)<11) { print STDERR "Incomplete header information of the block group, $bg_header[0]\n"; exit(-1); }
		$bg_header[0]=~s/\>//;
		$id="$bg_header[0]_$bg_header[1]_$bg_header[2]_$bg_header[3]_$bg_header[4]";
		if(%bgInfo) {
			compBlockStat() if($print==2 || $print==4);
			compBGStat() if($print==3 || $print==4);
			compOverlapStat() if($print==1);
			%bgInfo=(); %bInfo=(); %block_read_start=(); %bg_read_start=();
		}
	}
	elsif(defined($bInfo{"$id:$temp[6]"})) {
		if($temp[1]<$bInfo{$key}{'start'}) { $bInfo{$key}{'start'}=$temp[1]; }
		if($temp[2]>$bInfo{$key}{'end'}) { $bInfo{$key}{'end'}=$temp[2]; }
		if($bgInfo{$id}{'strand'}=~/\+/ && $temp[4]>0) { $block_read_start{$key}->{$temp[1]}++; $bg_read_start{$id}->{$temp[1]} += $temp[4]; }
		elsif($bgInfo{$id}{'strand'}=~/\-/ && $temp[4]>0) { $block_read_start{$key}->{$temp[2]}++; $bg_read_start{$id}->{$temp[2]} += $temp[4]; }
		$bInfo{$key}{'tags'}++;
		$bInfo{$key}{'reads'} += $temp[4] if($temp[4]>0);
		$bgInfo{$id}{'uniqReads'}+=$temp[4] if($temp[3]=~/\_1$/);
	}
	elsif($bg_header[9]=~/$filter/ || $bg_header[0]=~/^$filter$/) {
		############ Uncomment to filter read blocks by read count ##############################
		#@testBlockSig = `grep -w "$bg_header[0]" -A $bg_header[6] $infile | grep -we "$temp[6]\$" | perl -an -F'/\\s+/' -e 'if(\$_!~/^>/) { print "\$F[4]\n"; }'`;
		@testBlockSig=(1);
		#foreach(@testBlockSig) {  if($_!~/[0-9\.]+/) { print "$_"; system("grep -w \"$bg_header[0]\" -A $bg_header[6] $infile | grep -we \"$temp[6]\$\""); exit; } }

		if(sum(@testBlockSig) > 0 && !defined($seen{$id})) {
			$total_bg_count++;
			$bgInfo{$id}{'start'}=$bg_header[2];
			$bgInfo{$id}{'end'}=$bg_header[3];
			$bgInfo{$id}{'strand'}=$bg_header[4];
			$bgInfo{$id}{'type'}=$bg_header[9];
			$bgInfo{$id}{'loci'}=$bg_header[10];
			$bgInfo{$id}{'tags'}=$bg_header[6];
			$bgInfo{$id}{'reads'}=$bg_header[5];
			$bgInfo{$id}{'blocks'}=$bg_header[7];
			$bgInfo{$id}{'size'}=($bg_header[3]-$bg_header[2])+1;
			$seen{$id}=1; $is_start=1;
			$bgInfo{$id}{'id'} = $bg_header[0];
			$bgInfo{$id}{'uniqReads'}=0;

			## compute block group annotation statistics
			if($bgInfo{$id}{'type'}=~/miRNA/) { $bgAnnoStat{'mirna'}++; }
			elsif($bgInfo{$id}{'type'}=~/snoRNA/) { $bgAnnoStat{'snorna'}++; }
			elsif($bgInfo{$id}{'type'}=~/tRNA/) { $bgAnnoStat{'trna'}++; }
			elsif($bgInfo{$id}{'type'}=~/snRNA/) { $bgAnnoStat{'snrna'}++; }
			elsif($bgInfo{$id}{'type'}=~/scRNA/) { $bgAnnoStat{'scrna'}++; }
			elsif($bgInfo{$id}{'type'}=~/Y_RNA/) { $bgAnnoStat{'yrna'}++; }
			elsif($bgInfo{$id}{'type'}=~/piRNA/) { $bgAnnoStat{'pirna'}++; }
			elsif($bgInfo{$id}{'type'}=~/n\/a/) { $bgAnnoStat{'unannotated'}++; }
			else { $bgAnnoStat{'other'}++; }
		}

		if(sum(@testBlockSig) > 0 && $is_start) {
			$key = "$id:$temp[6]";
			$bInfo{$key}{'start'} = $temp[1];
			$bInfo{$key}{'end'} = $temp[2];
			if($bgInfo{$id}{'strand'}=~/\+/ && $temp[4]>0) { $block_read_start{$key}->{$temp[1]}++; $bg_read_start{$id}->{$temp[1]} += $temp[4]; }
			elsif($bgInfo{$id}{'strand'}=~/\-/ && $temp[4]>0) { $block_read_start{$key}->{$temp[2]}++; $bg_read_start{$id}->{$temp[2]} += $temp[4]; }
			$bInfo{$key}{'tags'}++;
			$bInfo{$key}{'reads'} += $temp[4] if($temp[4]>0);
		}
		$bgInfo{$id}{'uniqReads'}+=$temp[4] if($temp[3]=~/\_1$/);
	}
}

compBGAnnoStat() if($print==5 && %bgAnnoStat);
compBlockStat() if(($print==2 || $print==4) && %bgInfo);
compBGStat() if(($print==3 || $print==4) && %bgInfo);
compOverlapStat() if(($print==1) && %bgInfo);

sub compBGAnnoStat {
	## print annotation statistics
	print "#id\tmiRNA\tsnoRNA\ttRNA\tsnRNA\tscRNA\tyRNA\tpiRNA\tothers\tunannotated\ttotal\n";
	print "$infile\t";
	if(defined($bgAnnoStat{'mirna'})) { print "$bgAnnoStat{'mirna'}\t"; } else { print "0\t"; }
	if(defined($bgAnnoStat{'snorna'})) { print "$bgAnnoStat{'snorna'}\t"; } else { print "0\t"; }
	if(defined($bgAnnoStat{'trna'})) { print "$bgAnnoStat{'trna'}\t"; } else { print "0\t"; }
	if(defined($bgAnnoStat{'snrna'})) { print "$bgAnnoStat{'snrna'}\t"; } else { print "0\t"; }
	if(defined($bgAnnoStat{'scrna'})) { print "$bgAnnoStat{'scrna'}\t"; } else { print "0\t"; }
	if(defined($bgAnnoStat{'yrna'})) { print "$bgAnnoStat{'yrna'}\t"; } else { print "0\t"; }
	if(defined($bgAnnoStat{'pirna'})) { print "$bgAnnoStat{'pirna'}\t"; } else { print "0\t"; }
	if(defined($bgAnnoStat{'other'})) { print "$bgAnnoStat{'other'}\t"; } else { print "0\t"; }
	if(defined($bgAnnoStat{'unannotated'})) { print "$bgAnnoStat{'unannotated'}\t"; } else { print "0\t"; }
	print "$total_bg_count\n";
}

sub compBGStat {

	foreach $id(keys(%bgInfo)) {
		#Compute for summary statistics
		if($bgInfo{$id}{'blocks'}>$max_block_count) { $max_block_count = $bgInfo{$id}{'blocks'}; }
		if($bgInfo{$id}{'blocks'}<$min_block_count) { $min_block_count = $bgInfo{$id}{'blocks'}; }
		$sum_block_count = $sum_block_count + $bgInfo{$id}{'blocks'};
		$count_block_count++;
		push(@block_count, $bgInfo{$id}{'blocks'});

		#Compute bg entropy
		foreach(keys(%{$bg_read_start{$id}})) {
			#$bgInfo{$id}{'entropy'} += (-1) * ($bg_read_start{$id}->{$_}/$bgInfo{$id}{'tags'})*(log($bg_read_start{$id}->{$_}/$bgInfo{$id}{'tags'})/log(2));
			$bgInfo{$id}{'entropy'} += (-1) * ($bg_read_start{$id}->{$_}/$bgInfo{$id}{'reads'})*(log($bg_read_start{$id}->{$_}/$bgInfo{$id}{'reads'})/log(2));
			if($bgInfo{$id}{'strand'}=~/\+/ && defined($read_background_dist{$_ - $bgInfo{$id}{'start'}})) {
				my $q_i = $bg_read_start{$id}->{$_}/$bgInfo{$id}{'reads'};
				my $p_i = $read_background_dist{$_ - $bgInfo{$id}{'start'}}/$total_reads;
				$bgInfo{$id}{'relEntropy'} += $q_i * log($q_i/$p_i)/log(2);
			}
			elsif($bgInfo{$id}{'strand'}=~/\-/ && defined($read_background_dist{$bgInfo{$id}{'end'} - $_})) {
				my $q_i = $bg_read_start{$id}->{$_}/$bgInfo{$id}{'reads'};
				my $p_i = $read_background_dist{$bgInfo{$id}{'end'} - $_}/$total_reads;
				$bgInfo{$id}{'relEntropy'} += $q_i * log($q_i/$p_i)/log(2);
			}
			else { print STDERR "Background distribution not found for $id\t$_\t$bgInfo{$id}{'start'}\t$bgInfo{$id}{'end'}\n"; exit; }
		}
		#$bgInfo{$id}{'entropy'} = sprintf("%0.2f", ($bgInfo{$id}{'entropy'}/$bgInfo{$id}{'blocks'}));

		#Compute for summary statistics
		if($bgInfo{$id}{'entropy'}>$max_bg_entropy) { $max_bg_entropy = $bgInfo{$id}{'entropy'}; }
		if($bgInfo{$id}{'relEntropy'}>$max_bg_relEntropy) { $max_bg_relEntropy = $bgInfo{$id}{'relEntropy'}; }
		if($bgInfo{$id}{'entropy'}<$min_bg_entropy) { $min_bg_entropy = $bgInfo{$id}{'entropy'}; }
		if($bgInfo{$id}{'relEntropy'}<$min_bg_relEntropy) { $min_bg_relEntropy = $bgInfo{$id}{'relEntropy'}; }
		$sum_bg_entropy = $sum_bg_entropy + $bgInfo{$id}{'entropy'};
		$sum_bg_relEntropy = $sum_bg_relEntropy + $bgInfo{$id}{'relEntropy'};
		$count_bg_entropy++;
		$count_bg_relEntropy++;
		push(@bg_entropy, $bgInfo{$id}{'entropy'});
		push(@bg_relEntropy, $bgInfo{$id}{'relEntropy'});

		#Print block count, total reads and block group entropy for block groups in tab limited format
		if($print==3) {
			printf("%s\t%d\t%0.2f\t%0.2f\t%s\t%s\t%s\t%0.2f\t%d\t%d\n", $id, $bgInfo{$id}{'blocks'}, $bgInfo{$id}{'reads'}, $bgInfo{$id}{'entropy'}, $bgInfo{$id}{'type'}, $bgInfo{$id}{'loci'}, $bgInfo{$id}{'id'}, $bgInfo{$id}{'relEntropy'}, $bgInfo{$id}{'uniqReads'}, $bgInfo{$id}{'size'});
		}
	}
}

sub compBlockStat {
	my $id=(); my $i=();

	#Compute block size
	foreach $key(keys(%bInfo)) {
		$id = $key; $id=~s/\:.+//;
		$bInfo{$key}{'size'} = $bInfo{$key}{'end'}-($bInfo{$key}{'start'}-1);
		#print "$key\t".$bInfo{$key}{'end'}-$bInfo{$key}{'start'}."\n";

		#Compute for summary statistics
		if($bInfo{$key}{'size'}>$max_block_size) { $max_block_size = $bInfo{$key}{'size'}; }
		if($bInfo{$key}{'size'}<$min_block_count) { $min_block_size = $bInfo{$key}{'size'}; }
		$sum_block_size = $sum_block_size + $bInfo{$key}{'size'};
		$count_block_size++;
		push(@block_size, $bInfo{$key}{'size'});
	}

	### compute block distance and overlap
	if($bgInfo{$id}{'strand'}=~/\+/) { @temp = sort { $bInfo{$a}{'start'} <=> $bInfo{$b}{'start'} } keys(%bInfo); }
	elsif($bgInfo{$id}{'strand'}=~/\-/) { @temp = reverse sort { $bInfo{$a}{'end'} <=> $bInfo{$b}{'end'} } keys(%bInfo); }

	### block distance
	for($i=0; $i<(scalar(@temp)-1); $i++) {
		#print "$temp[$i]\t$bInfo{$temp[$i]}{'start'}\t$bInfo{$temp[$i]}{'end'}\n";
		if($bgInfo{$id}{'strand'}=~/\+/) { $bInfo{$temp[$i]}{'dist'} = $bInfo{$temp[$i+1]}{'start'} - ($bInfo{$temp[$i]}{'start'}); }
		elsif($bgInfo{$id}{'strand'}=~/\-/) { $bInfo{$temp[$i]}{'dist'} = $bInfo{$temp[$i]}{'end'} - ($bInfo{$temp[$i+1]}{'end'}); }

		### compute for summary statistics
		if($bInfo{$temp[$i]}{'dist'}>$max_block_dist) { $max_block_dist = $bInfo{$temp[$i]}{'dist'}; }
		if($bInfo{$temp[$i]}{'dist'}<$min_block_dist) { $min_block_dist = $bInfo{$temp[$i]}{'dist'}; }
		$sum_block_dist = $sum_block_dist + $bInfo{$temp[$i]}{'dist'};
		$count_block_dist++;
		push(@block_dist, $bInfo{$temp[$i]}{'dist'});
	}
	#print "$temp[$i]\t$bInfo{$temp[$i]}{'start'}\t$bInfo{$temp[$i]}{'end'}\n";

	### block overlap
	my $sum=(); my $j=(); my $k=(); my $overlap=();
	for($i=0; $i<(scalar(@temp)); $i++) {
		$bInfo{$temp[$i]}{'overlap'}=0;
		for($k=0; $k<scalar(@temp); $k++) {
			if($bgInfo{$id}{'strand'}=~/\+/ && $bInfo{$temp[$k]}{'start'} > $bInfo{$temp[$i]}{'end'}) { last; }
			elsif($bgInfo{$id}{'strand'}=~/\-/ && $bInfo{$temp[$k]}{'end'} < $bInfo{$temp[$i]}{'start'}) { last; }
			if($temp[$i]=~/^\Q$temp[$k]\E$/) { next; }
			$sum=0;
			for($j=$bInfo{$temp[$i]}{'start'}; $j<=$bInfo{$temp[$i]}{'end'}; $j++) {
				if($j>=$bInfo{$temp[$k]}{'start'} && $j<=$bInfo{$temp[$k]}{'end'}) {
					$sum++;
				}
			}
			$overlap=sprintf("%0.2f", $sum/$bInfo{$temp[$i]}{'size'});
			#print "$temp[$i]\t$temp[$k]\t$bInfo{$temp[$i]}{'size'}\t".scalar(grep { $_ > 0 } @overlap)."\n";
			if($overlap > $bInfo{$temp[$i]}{'overlap'}) { $bInfo{$temp[$i]}{'overlap'} = $overlap; }
		}
		#$bInfo{$temp[$i]}{'overlap'}=0;
	}

	#Compute block entropy
	foreach $key(keys(%block_read_start)) {
		$id = $key; $id=~s/\:.+//;
		foreach(keys(%{$block_read_start{$key}})) {
			$bInfo{$key}{'entropy'} += (-1) * ($block_read_start{$key}->{$_}/$bInfo{$key}{'tags'})*(log($block_read_start{$key}->{$_}/$bInfo{$key}{'tags'})/log(2));
		}
		#$bgInfo{$id}{'entropy'} += $bInfo{$key}{'entropy'};

		#Compute for summary statistics
		if($bInfo{$key}{'entropy'}>$max_block_entropy) { $max_block_entropy = $bInfo{$key}{'entropy'}; }
		if($bInfo{$key}{'entropy'}<$min_block_entropy) { $min_block_entropy = $bInfo{$key}{'entropy'}; }
		$sum_block_entropy = $sum_block_entropy + $bInfo{$key}{'entropy'};
		$count_block_entropy++;
		push(@block_entropy, $bInfo{$key}{'entropy'});
	
	}

	#print block size, distance, entropy and reads for read blcoks in a tab limited format
	$i=();
	if($print==2) {
		foreach $key(keys(%bInfo)) {
			$id = $key; $id=~s/\:.+//;
			$key=~m/\:[0-9]+$/; $i=$&; $i=~s/\://; $i--;
			if(defined($bInfo{$key}{'dist'})) {
				printf("%s\t%d\t%d\t%0.2f\t%0.2f\t%d\t%s\t%s\t%s\t%d\t%d\n",  $key, $bInfo{$key}{'size'}, $bInfo{$key}{'dist'}, $bInfo{$key}{'entropy'}, $bInfo{$key}{'reads'}, $bInfo{$key}{'overlap'}, $bgInfo{$id}{'type'}, $bgInfo{$id}{'loci'}, $bgInfo{$id}{'id'}, $bInfo{$key}{'start'}, $bInfo{$key}{'end'}); 
			}
			else {
				printf("%s\t%d\tNA\t%0.2f\t%0.2f\t%d\t%s\t%s\t%s\t%d\t%d\n",  $key, $bInfo{$key}{'size'}, $bInfo{$key}{'entropy'}, $bInfo{$key}{'reads'}, $bInfo{$key}{'overlap'}, $bgInfo{$id}{'type'}, $bgInfo{$id}{'loci'}, $bgInfo{$id}{'id'}, $bInfo{$key}{'start'}, $bInfo{$key}{'end'}); 
			}
		}
	}
}

if($print==4) {
	use lib qw(/home/users/sachin/perl/lib/);
	use Statistics::Descriptive;
	my $stat=();
	print "\n================================================================================================================\n";
	print "Total block groups: $total_bg_count\n";
	printf("Block count (max, min, mean, stddev): %d\t%d\t%0.2f\t", $max_block_count, $min_block_count, $sum_block_count/$count_block_count);
	$stat = new Statistics::Descriptive; $stat->AddData(@block_count);
	if(scalar(@block_count)>1) { printf("%0.2f\n", $stat->StandardDeviation()); }
	else { print "NA\n"; }
	printf("Block size (max, min, mean, stddev): %d\t%d\t%0.2f\t", $max_block_size, $min_block_size, , $sum_block_size/$count_block_size);
	$stat = new Statistics::Descriptive; $stat->AddData(@block_size);
	if(scalar(@block_size)>1) { printf("%0.2f\n", $stat->StandardDeviation()); }
	else { print "NA\n"; }
	if($count_block_dist==0) {
		printf("Block dist (max, min, mean, stddev): 0\t0\t0.0\t");
	}
	else {
		printf("Block dist (max, min, mean, stddev): %d\t%d\t%0.2f\t", $max_block_dist, $min_block_dist, $sum_block_dist/$count_block_dist);
	}
	$stat = new Statistics::Descriptive; $stat->AddData(@block_dist);
	if(scalar(@block_dist)>1) { printf("%0.2f\n", $stat->StandardDeviation()); }
	else { print "NA\n"; }
	printf("Block entropy (max, min, mean, stddev): %0.2f\t%0.2f\t%0.2f\t", $max_block_entropy, $min_block_entropy, $sum_block_entropy/$count_block_entropy);
	$stat = new Statistics::Descriptive; $stat->AddData(@block_entropy);
	if(scalar(@block_entropy)>1) { printf("%0.2f\n", $stat->StandardDeviation()); }
	else { print "NA\n"; }
	printf("Block group entropy (max, min, mean, stddev): %0.2f\t%0.2f\t%0.2f\t", $max_bg_entropy, $min_bg_entropy, $sum_bg_entropy/$count_bg_entropy);
	$stat = new Statistics::Descriptive; $stat->AddData(@bg_entropy);
	if(scalar(@bg_entropy)>1) { printf("%0.2f\n", $stat->StandardDeviation()); }
	else { print "NA\n"; }
	printf("Block group relative entropy (max, min, mean, stddev): %0.2f\t%0.2f\t%0.2f\t", $max_bg_relEntropy, $min_bg_relEntropy, $sum_bg_relEntropy/$count_bg_relEntropy);
	$stat = new Statistics::Descriptive; $stat->AddData(@bg_relEntropy);
	if(scalar(@bg_relEntropy)>1) { printf("%0.2f\n", $stat->StandardDeviation()); }
	else { print "NA\n"; }
	print "================================================================================================================\n";
}

#Compute Overlapping blocks
sub compOverlapStat {
	my %overlap_block=(); my $start_coor=(); my $end_coor=(); %seen=(); my @seen=(); my $overlap_block_count=();

	#Search for Overlapping blocks
	foreach $key(keys(%bInfo)) {
		$start_coor = $bInfo{$key}{'start'};
		$end_coor = $bInfo{$key}{'end'};

		foreach(keys(%bInfo)) {
			$seen[0]="$key-$_"; $seen[1]="$_-$key";
			if((($bInfo{$_}{'start'} > $start_coor && $bInfo{$_}{'end'} < $end_coor) || ($bInfo{$_}{'start'} < $start_coor && $bInfo{$_}{'end'} > $end_coor)) && (!defined($seen{$seen[0]})) && (!defined($seen{$seen[1]}))) {
				$overlap_block{$key} = $_;
				$seen{$seen[0]}=1;  $seen{$seen[1]}=1;
			}
		}
	}

	#Print Statistics for Overlapping blocks
	$overlap_block_count=0;
	foreach $key(keys(%overlap_block)) {
		print "$key\t$overlap_block{$key}\n";
		$overlap_block_count++;
	}
}
exit;
