//===========================================================================================================================//
//                                                                                                                           //
//       Filename: deepBlockAlign.c                                                                                          //
//    Description: deepBlockAlign: A tool for aligning RNAseq profiles of read block patterns                                //
//                                                                                                                           //
//       Compiler: gcc 4.1.2 or above.                                                                                       //
//    Affiliation: Center for Non-Coding RNA in Technology and Health, IBHV, University of Copenhagen, Denmark and;          //
//                 Bioinformatics Group, Department of Computer Science and Interdisciplinary Center for Bioinformatics,     //
//                 University of Lepizig, Germany.                                                                           //
//                                                                                                                           //
//===========================================================================================================================//

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <getopt.h>

#define ACCESS(X, DIM_1, DIM_2, I, J, K, L) 	X[(((I)*(DIM_2)+(J))*DIM_1+(K))*DIM_2+(L)]
#define round(x) ((x)>0?(long)((x)+0.5):(long)((x)-0.5))

const char *VERSION="v1.2";
const char *DATE="August 15, 2013";

/************************
 * BLOCKALIGN PARAMETERS 
 ***********************/
const char *Ci="-2";
const char *Ce="-1";
const char *DELTA="1";
const char *S0="1";
const char *S1="-1";
const char *ZETA="1";

/*****************
 * PLOT ALIGNMENT
 ****************/
FILE *NPRFILE=NULL;
int count_block_aln;
int coor_block_aln[100][2];
int nprfile_line;

/***********************
 * FUNCTION PROTOTYPES
 **********************/
void replace_special_char(char*);
float** create_2D_array(int, int);
double trace_back(double*, double*, double*, double*, float**, float**, float**, float**, int, int, int, int, double);
void normalize(double*, double*, double*, int, double, double);
void fill_matrix_default(int, int, float**, float**, float**, float**);
double max2(double, double);
double max3(double, double, double);

/****************************
 * DEEPBLOCKALIGN PARAMETERS
 ***************************/
double  penalty_gap          = -1;
double  param_distance       = 1; 
double  param_block          = 1; 
int     printAlignment       = 0;
int	    printLocalAlignments = 0;
int	    plotAlignment	       = 0;
int     printNormScore       = 1;
int     onlyAlignPairs       = 0;

double s = 0;
char *query_File = NULL;
char *subject_File = NULL;
char *local_File = NULL;
char *alignment_File = NULL;

/******************
 * STRUCTURES
 *****************/

typedef struct{
	double gapPenalty;
	double score;
	int bindings;
}score_t;

/*---------------------*/

typedef struct {
	char *id;
	int start;
	int end;
	double height;
} tag_t;

/*---------------------*/

typedef struct {
	char *chrom;
	int start;
	int end;
	char *strand;
	double height;
	tag_t *tags;
	score_t *score;
	int nooftags;
} block_t;

/*---------------------*/

typedef struct {
	block_t *blocks;
	int noofblocks;
	char *id;
	char *chrom;
	int start;
	int end;
	char *strand;
	double expression;
	int tags;
	char *ncRNAid;
	char *ncRNAtype;
	char *ncRNAclass;
} cluster_t;

/*---------------------*/

typedef struct {
	cluster_t *clusters;
	int noofclusters;
} set_t;

/*---------------------*/

typedef struct{
	int i;
	int j;
	int k;
	int l;
	double score;
	int set;
}field_t;

/*---------------------*/

struct coords
{
	double max;
	int i;
	int j;
	int k;
	int l;
}Coodinate;

/******************
 * FUNCTIONS
 *****************/

void writeHeader(char *query_File, char *subject_File)
{	
	time_t t;
	t = time(NULL);
	printf("# deepBlockAlign.x %s started at %s", VERSION, ctime(&t));
	printf("# query: %s\n", query_File);
	printf("# subject: %s\n", subject_File);
	printf("# parameters (block group alignment)\n");
	printf("#  distance weight: %.1lf; block similarity weight: %.1lf; gap penalty: %.1lf\n", param_distance, param_block, penalty_gap);
	printf("# parameters (block alignment)\n");
	printf("#  match: %s; mismatch: %s; threshold: %s; shape difference penalty: %s\n", S0, S1, DELTA, ZETA);
	printf("#  gap penalty (initialization): %s; gap penalty (extension): %s\n", Ci, Ce); 
	printf("");
}

void printUsage(void)
{
	printf("\n");
	printf("==========================================================\n");
	printf("                     DeepBlockAlign\n");
	printf("a tool for aligning RNAseq profiles of read block patterns\n");
	printf("==========================================================\n\n");
	printf("usage: deepBlockAlign.x -q <file> -s <file> [ARGUMENTS]\n\n");
	printf("[INPUT]\n");
	printf(" -q  <file>   query file (blockbuster output)\n");
	printf(" -s  <file>   subject file (blockbuster output)\n");
	printf("[ARGUMENTS: Block Group Alignment]\n");
	printf(" -d  <double> distance weight (default: 1.0)\n");
	printf(" -b  <double> block similarity weight (default: 1.0)\n");
	printf(" -g  <double> gap penalty (default: -1.0)\n");
	printf(" -a  <int>    print alignment 1=on (default: 0)\n");
	printf(" -l  <file>   print local alignments of all blocks against all blocks\n");
	printf(" -f  <file>   create file necessary to plot block group alignment\n");
	printf(" -p  <int>    only align pairs of block groups e.g 1 with 1, 2 with 2 etc (default:0)\n");
	printf("[ARGUMENTS: Block Alignment]\n");
	printf(" -c  <int>    gap initialization penalty (default: -2)\n");
	printf(" -e  <int>    gap extension penalty (default: -1)\n");
	printf(" -t  <double> threshold for expression difference between two blocks (default: 1.0)\n");
	printf(" -m  <int>    match score (default: 1)\n");
	printf(" -n  <int>    mismatch score (default: -1)\n");
	printf(" -z  <int>    profile shape difference penalty (default: 1)\n");
	printf(" -h           this helpful message\n");
	printf("[VERSION]\n");
	printf(" %s (%s)\n", VERSION, DATE);
	printf("[CONTACT]\n");
	printf(" sachin@rth.dk, david@bioinf.uni-leipzig.de\n");
}

void parseopt(int argc, char *argv[]) {
	char c;
	while((c = getopt(argc, argv, "q:s:d:b:g:a:l:p:c:e:t:m:n:z:h:f:r:")) != -1) {
		switch(c) {
			case 'q':
				query_File = optarg;
				break;
			case 's':
				subject_File = optarg;
				break;
			case 'd':
				param_distance = atof(optarg);
				break;
			case 'b':
				param_block = atof(optarg);
				break;
			case 'g':
				penalty_gap = atof(optarg);
				break;
			case 'a':
				printAlignment = atoi(optarg);
				break;
			case 'l':
				local_File = optarg;
				printLocalAlignments = 1;
				break;
			case 'p':
				onlyAlignPairs = atoi(optarg);
				break;
			case 'c':
				Ci = optarg;
				break;
			case 'e':
				Ce = optarg;
				break;
			case 't':
				DELTA = optarg;
				break;
			case 'm':
				S0 = optarg;
				break;
			case 'n':
				S1 = optarg;
				break;
			case 'z':
				ZETA = optarg;
				break;
			case 'h':
				printUsage();
				break;
			case 'f':
				alignment_File = optarg;
				plotAlignment = 1;
				break;
			case 'u':
				printNormScore = atoi(optarg);
				break;
			case '?':
				printUsage();
				break;
		}
	}
}

set_t *read_cluster_file(char *filename)
/* READ FILE WITH A SET OF CLUSTERS */
{	
	// create new set of clusters
	set_t *set;
	set = calloc(1, sizeof(set_t));
	set->clusters = NULL;
	set->noofclusters = 0;
	
	FILE *f;
	f = fopen(filename, "r");
	if(!f)
	{
		printf("cannot open %s\n", filename); exit(0);
	}
	else
	{
		char c; int e = 0; 
		while((c=getc(f)) != EOF) {
			
			char cluster_id[300], cluster_chrom[50], cluster_strand[5], tag_chrom[50], tag_strand[5], tag_id[50], dum[1000], ncRNA_id[50], ncRNA_type[50] = "", ncRNA_class[100] = "";
			int  cluster_start, cluster_end, cluster_tags, cluster_blockCount, tag_start, tag_end, block_no;
			double cluster_expression, tag_height;
			
			// check for header
			int header = 0;

			if(c == '>')
				header = 1;
			ungetc (c,f);
			
			if(header == 1){
				// parse line
				e = fscanf(f, "%s %s %d %d %s %lf %d %d %s %s %s", cluster_id, cluster_chrom, &cluster_start, &cluster_end, cluster_strand, &cluster_expression, &cluster_tags, &cluster_blockCount, ncRNA_id, ncRNA_type, ncRNA_class);

				// create new cluster
				set->noofclusters++;
				set->clusters = realloc(set->clusters, (set->noofclusters+1)*sizeof(cluster_t));
				set->clusters[set->noofclusters].blocks = NULL;			
				set->clusters[set->noofclusters].noofblocks = 0;
				
				// write cluster info
				// id
				set->clusters[set->noofclusters].id = malloc(strlen(cluster_id)+1);
				strcpy(set->clusters[set->noofclusters].id, cluster_id);
				// chrom
				set->clusters[set->noofclusters].chrom = malloc(strlen(cluster_chrom)+1);
				strcpy(set->clusters[set->noofclusters].chrom, cluster_chrom);
				// start
				set->clusters[set->noofclusters].start = cluster_start;
				// end
				set->clusters[set->noofclusters].end = cluster_end;
				// strand
				set->clusters[set->noofclusters].strand = malloc(strlen(cluster_strand)+1);
				strcpy(set->clusters[set->noofclusters].strand, cluster_strand);	
				// expression
				set->clusters[set->noofclusters].expression = cluster_expression;
				// # of tags
				set->clusters[set->noofclusters].tags = cluster_tags;
				// ncRNA id
				set->clusters[set->noofclusters].ncRNAid = malloc(strlen(ncRNA_id)+1);
				strcpy(set->clusters[set->noofclusters].ncRNAid, ncRNA_id);
				// ncRNA type
				set->clusters[set->noofclusters].ncRNAtype = malloc(strlen(ncRNA_type)+1);
				strcpy(set->clusters[set->noofclusters].ncRNAtype, ncRNA_type);
				// ncRNA class
				set->clusters[set->noofclusters].ncRNAclass = malloc(strlen(ncRNA_class)+1);
				strcpy(set->clusters[set->noofclusters].ncRNAclass, ncRNA_class);
			}
			else {
				// parse line
				e = fscanf(f, "%s\t%d\t%d\t%s\t%lf\t%s\t%d \n", tag_chrom, &tag_start, &tag_end, tag_id, &tag_height, tag_strand, &block_no);

				// get relative position of the tag
				int newStart, newEnd;
				if(strcmp(tag_strand, "+"	)==0){newStart = tag_start - cluster_start; newEnd = tag_end - cluster_start;}
				else{newStart = cluster_end - tag_end; newEnd = cluster_end - tag_start;}
				
				// get memory for new block, if new block starts
				if(block_no > set->clusters[set->noofclusters].noofblocks)
				{
					set->clusters[set->noofclusters].noofblocks++;
					set->clusters[set->noofclusters].blocks = realloc(set->clusters[set->noofclusters].blocks, (set->clusters[set->noofclusters].noofblocks+1)*sizeof(block_t));
					set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].nooftags = 0;
					set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].start = newStart;
					set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].end = newEnd;
					set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].tags = NULL;
					set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].score = NULL;
					set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].height = 0;
					
					// write block locus
					// chrom
					set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].chrom = malloc(strlen(tag_chrom)+1);
					strcpy(set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].chrom, tag_chrom);
					// strand
					set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].strand = malloc(strlen(tag_strand)+1);
					strcpy(set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].strand, tag_strand);
				}
				
				// get memory for new tag
				set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].nooftags++;
				set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].tags = 
				realloc(set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].tags, 
						  (set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].nooftags+1)*sizeof(tag_t));
				
				// write tag locus
				// id
				set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].
						tags[set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].nooftags].id = malloc(strlen(tag_id)+1);
				strcpy(set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].
					   tags[set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].nooftags].id, tag_id);
				// start
				set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].
						tags[set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].nooftags].start = newStart;
				// end
				set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].
						tags[set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].nooftags].end = newEnd;
				// height
				set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].
						tags[set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].nooftags].height = tag_height;

				// block height
				set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].height += tag_height;

				// change block boundaries 
				if(set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].start > newStart)
					set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].start = newStart;
				if(set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].end < newEnd)
					set->clusters[set->noofclusters].blocks[set->clusters[set->noofclusters].noofblocks].end = newEnd;		
			
			}
		}
	}
	fclose(f);
	return set;
}
int check_file_format(char *filename) {
	/*CHECK FILE FORMAT */
	FILE *f;
	if((f = fopen(filename, "r")) == NULL) {
		fprintf(stderr, "Cannot open %s\n", filename); exit(0);
	}
	char c; int tabS=0; int tabR=0; int is_header=0;

        while((c=getc(f)) != EOF) {
		if(c=='>') {
			is_header=1;
			tabS=0; tabR=0;
		}
		else if(c=='\n') {
			if((is_header==1 && tabS!=10) || (is_header==0 && tabR!=6))
				return 1;
			tabS=0; tabR=0; is_header=0;
		}
		else if((c=='\t' || c==' ') && is_header==1) {
			tabS++;
		}
		else if((c=='\t' || c==' ') && is_header==0) {
			tabR++;
		}
	}

	fclose(f);
	return 0;
}

void printSet(set_t *set)
/* PRINT SET */
{
	int i, j, k;	
	printf("set with %d clusters:\n", set->noofclusters);
	for(i = 1; i <= set->noofclusters; i++){
		printf("cluster %d: %s\n", i, set->clusters[i].id);
		for(j = 1; j <= set->clusters[i].noofblocks; j++){
			printf("block %d: %d-%d\n", j, set->clusters[i].blocks[j].start, set->clusters[i].blocks[j].end);
			for(k = 1; k <= set->clusters[i].blocks[j].nooftags; k++){
				printf("tag %s: %d-%d\n", set->clusters[i].blocks[j].tags[k].id, set->clusters[i].blocks[j].tags[k].start, 
						set->clusters[i].blocks[j].tags[k].end);
			}
		}
	}
}

void printCluster(cluster_t *cluster)
/* PRINT CLUSTER */
{
	int i, j;	
	printf("cluster with %d blocks:\n", cluster->noofblocks);
	for(i = 1; i <= cluster->noofblocks; i++){
		printf("block: %d\t%d-%d\n", i, cluster->blocks[i].start, cluster->blocks[i].end);
		for(j = 1; j <= cluster->blocks[i].nooftags; j++){
//			printf("tag %s\t%s:%d-%d(%s)\t%lf\n", cluster->blocks[i].tags[j].id, cluster->blocks[i].chrom, 
//				   cluster->blocks[i].tags[j].start, cluster->blocks[i].tags[j].end, cluster->blocks[i].strand, 
//				   cluster->blocks[i].tags[j].height);
		}
	}
}

int struct_cmp_by_start(const void *a, const void *b) 
/* QSORT COMPARER */
{ 
	block_t *da = (block_t *) a;
	block_t *db = (block_t *) b;
	
	return ((da->start - db->start) + (da->end - db->end));
} 

cluster_t *sortCluster(cluster_t *cluster)
/* SORT BOTH CLUSTER ON START POSITION */
{	
	block_t *blocks = cluster->blocks;
	qsort(&blocks[1], cluster->noofblocks, sizeof(block_t), struct_cmp_by_start);
	cluster->blocks = blocks;
	return cluster;
}

/* Code for aligning two blocks */
double blockAlign(int a, int b, cluster_t *cluster1, cluster_t *cluster2) {
	int c, t , i, j, opt_align_count;
	double sread_count, tread_count;
	double max_sread_x_profile=0, max_tread_x_profile=0, max_sread_y_profile=0, max_tread_y_profile=0;
	int x_profile_len = 2*(cluster1->blocks[a].end - (cluster1->blocks[a].start-1));
	int y_profile_len = 2*(cluster2->blocks[b].end - (cluster2->blocks[b].start-1));
	double x_profile[x_profile_len], y_profile[y_profile_len];
	double norm_x_profile[x_profile_len/2], norm_y_profile[y_profile_len/2];
	double norm_sread_x_profile[x_profile_len/2], norm_sread_y_profile[y_profile_len/2];
	double norm_tread_x_profile[x_profile_len/2], norm_tread_y_profile[y_profile_len/2];
        float **S = create_2D_array((x_profile_len/2)+1, (y_profile_len/2)+1);
        float **M = create_2D_array((x_profile_len/2)+1, (y_profile_len/2)+1);
        float **Qx = create_2D_array((x_profile_len/2)+1, (y_profile_len/2)+1);
        float **Qy = create_2D_array((x_profile_len/2)+1, (y_profile_len/2)+1);
	int norm_x_coor, norm_y_coor;
	
	//Compute read profile of Block 'a' from Cluster '1'.
	for(c = cluster1->blocks[a].start, i=0, j=0; c <= cluster1->blocks[a].end; c++, j++)
	{
		sread_count=0; tread_count=0;
		for(t = 1; t <= cluster1->blocks[a].nooftags; t++) {
			if(c >= cluster1->blocks[a].tags[t].start && c <= cluster1->blocks[a].tags[t].end) {
				tread_count += cluster1->blocks[a].tags[t].height;
			}
			if(c == cluster1->blocks[a].tags[t].start) {
				sread_count += cluster1->blocks[a].tags[t].height;
			}
		}
		//Put start read count information.
		x_profile[i] = sread_count;
		norm_sread_x_profile[j] = sread_count;
		if(sread_count > max_sread_x_profile) { max_sread_x_profile = sread_count; }
		i++;
		//Put total read count information.
		x_profile[i] = tread_count;
		norm_tread_x_profile[j] = tread_count;
		if(tread_count > max_tread_x_profile) { max_tread_x_profile = tread_count; }
		i++;
	}
	
	//Compute read profile of Block 'b' from Cluster '2'.
	for(c = cluster2->blocks[b].start, i=0, j=0; c <= cluster2->blocks[b].end; c++, j++) 
	{
		sread_count=0; tread_count=0;
		for(t = 1; t <= cluster2->blocks[b].nooftags; t++) {
			if(c >= cluster2->blocks[b].tags[t].start && c <= cluster2->blocks[b].tags[t].end) {
				tread_count += cluster2->blocks[b].tags[t].height;
			}
			if(c == cluster2->blocks[b].tags[t].start) {
				sread_count += cluster2->blocks[b].tags[t].height;
			}
		}
		//Put start read count information.
		y_profile[i]=sread_count;
		norm_sread_y_profile[j] = sread_count;
		if(sread_count > max_sread_y_profile) { max_sread_y_profile = sread_count; }
		i++;
		//Put total read count information.
		y_profile[i]=tread_count;
		norm_tread_y_profile[j] = tread_count;
		if(tread_count > max_tread_y_profile) { max_tread_y_profile = tread_count; }
		i++;
	}
	
	//normalize(norm_x_profile, norm_sread_x_profile, norm_tread_x_profile, x_profile_len/2, cluster1->blocks[a].height, cluster1->blocks[a].height);
	normalize(norm_x_profile, norm_sread_x_profile, norm_tread_x_profile, x_profile_len/2, cluster1->expression, cluster1->expression);
	
	//normalize(norm_y_profile, norm_sread_y_profile, norm_tread_y_profile, y_profile_len/2, cluster2->blocks[b].height, cluster2->blocks[b].height);
	normalize(norm_y_profile, norm_sread_y_profile, norm_tread_y_profile, y_profile_len/2, cluster2->expression, cluster2->expression);
	
	//Fill the first row and column of the matrix with default scores.
	fill_matrix_default(x_profile_len/2, y_profile_len/2, M, Qx, Qy, S);

	//Section PFD:1
	double profile_fraction_diff;
	double etaP, etaN, currExpDiff, prevExpDiff;
	double psiP, psiN;

	//Compute percentScore;
	double percentScore, relExpBlockA, relExpBlockB, test;
	relExpBlockA = cluster1->blocks[a].height/cluster1->expression;
	relExpBlockB = cluster2->blocks[b].height/cluster2->expression;

	if(relExpBlockA > 0 && relExpBlockB > 0)
		percentScore = 1 - (fabs(relExpBlockA - relExpBlockB)/max2(relExpBlockA, relExpBlockB));
	else
		percentScore = 1 - fabs(relExpBlockA - relExpBlockB);

	//Compute score for each row and column of the matrix.
	for(norm_x_coor=0, i=1; norm_x_coor<x_profile_len/2; norm_x_coor++, i++) {
		for(norm_y_coor=0, j=1; norm_y_coor<y_profile_len/2; norm_y_coor++, j++) {

			//Compute Profile Fraction Difference (PFD)
			if(norm_x_profile[norm_x_coor]>0 && norm_y_profile[norm_y_coor]>0)
				profile_fraction_diff = fabs(norm_x_profile[norm_x_coor] - norm_y_profile[norm_y_coor])/max2(norm_x_profile[norm_x_coor], norm_y_profile[norm_y_coor]);
			else 
				profile_fraction_diff = fabs(norm_x_profile[norm_x_coor] - norm_y_profile[norm_y_coor]);

			//Compute Entropy
			if(norm_x_coor==0 || norm_y_coor==0) {
				etaP=0;
			}
			else {
				currExpDiff = fabs(norm_x_profile[norm_x_coor]-norm_y_profile[norm_y_coor]);
				prevExpDiff = fabs(norm_x_profile[norm_x_coor-1]-norm_y_profile[norm_y_coor-1]);
				if(currExpDiff>0 && prevExpDiff>0) {
					etaP = (atoi(ZETA) * percentScore) * (fabs(currExpDiff - prevExpDiff)/max2(currExpDiff, prevExpDiff));
				}
				else {
					etaP = (atoi(ZETA) * percentScore) * fabs(currExpDiff - prevExpDiff); 
				}
			}
			etaN=0;

			//Match at positions, i and j
			if(fabs(norm_x_profile[norm_x_coor]-norm_y_profile[norm_y_coor])<atof(DELTA)) {
				psiP = atoi(S0) * (1 - (profile_fraction_diff + etaP)); //Match Score (MM)
				psiN = atoi(S0) * (1 - (profile_fraction_diff + etaN)); //Match Score (GM)
			}
			//Mismatch at positions, i and j
			else {
				psiP = atoi(S1) * (profile_fraction_diff + etaP); //Mismatch Score (MN)
				psiN = atoi(S1) * (profile_fraction_diff + etaN); //Mismatch Score (GN)
			}

			//Fill Matrix 'M'
			M[i][j] = max3((M[i-1][j-1] + psiP), (Qx[i-1][j-1] + psiN), (Qy[i-1][j-1] + psiN));

			//Fill Matrix 'Qx'
			Qx[i][j] = max2(S[i][j-1] + (atoi(Ci)*percentScore), Qx[i][j-1] + (atoi(Ce)*percentScore));

			//Fill Matrix 'Qy'
			Qy[i][j] = max2(S[i-1][j] + (atoi(Ci)*percentScore), Qy[i-1][j] + (atoi(Ce)*percentScore));

			//Fill Matrix 'S'
			S[i][j] = max3(M[i][j], Qx[i][j], Qy[i][j]);
		}
	}
		
	//Compute the alignment trace back for each high scoring cell.
	double align_score=S[x_profile_len/2][y_profile_len/2];

	align_score = trace_back(x_profile, y_profile, norm_x_profile, norm_y_profile, M, Qx, Qy, S, x_profile_len/2, y_profile_len/2, \
	x_profile_len/2, y_profile_len/2, percentScore);
	
	//Free memory of 2D-Array
	for(i=0; i<(x_profile_len/2)+1; i++) {
		free(M[i]);
		free(Qx[i]);
		free(Qy[i]);
		free(S[i]);
	}
	free(M); free(Qx); free(Qy); free(S);

	return align_score;
}

/* Compute the alignment (shortest path). */
double trace_back(double *x_profile, double *y_profile, double *norm_x_profile, double *norm_y_profile, float **M, float **Qx, float **Qy, float **S, \
int row, int col, int x_profile_len, int y_profile_len, double percentScore) {
	float matrix_align_score=S[row][col];
	double align_score=0, match_score=0, mismatch_score=0, gap_score=0, norm_score=0, query_coverage=0;
	int match_count=0, mismatch_count=0, gap_open_count=0, gap_ext_count=0, align_len=0;
	int trace_back_row[500], trace_back_col[500]; int trace_back_state[500]; //State Match=0, GapX=-1, GapY=1
	int is_M=1, is_Qx=0, is_Qy=0;
	int i, j=1;

	double profile_fraction_diff;
	double etaP, etaN, currExpDiff, prevExpDiff;
	double psiP, psiN;

	while(row>0 && col>0) {
		//Compute Profile Fraction Difference (PFD)
                if(*(norm_x_profile+(row-1))>0 && *(norm_y_profile+(col-1))>0)
			profile_fraction_diff = fabs(*(norm_x_profile+(row-1)) - *(norm_y_profile+(col-1)))/max2(*(norm_x_profile+(row-1)), *(norm_y_profile+(col-1)));
		else
			profile_fraction_diff = fabs(*(norm_x_profile+(row-1)) - *(norm_y_profile+(col-1)));

		//Compute Entropy
		if((row-1)==0 || (col-1)==0) {
			etaP=0;
		}
		else {
			currExpDiff = fabs(*(norm_x_profile+(row-1))-*(norm_y_profile+(col-1)));
			prevExpDiff = fabs(*(norm_x_profile+(row-2))-*(norm_y_profile+(col-2)));
			if(currExpDiff>0 && prevExpDiff>0)
				etaP = (atoi(ZETA) * percentScore) * (fabs(currExpDiff - prevExpDiff)/max2(currExpDiff, prevExpDiff));
			else
				etaP = (atoi(ZETA) * percentScore) * fabs(currExpDiff - prevExpDiff);
		}
		etaN=0;

		if(fabs(*(norm_x_profile+(row-1))-*(norm_y_profile+(col-1)))<atof(DELTA)) {
			psiP = atoi(S0) * (1 - (profile_fraction_diff + etaP)); //Match Score
			psiN = atoi(S0) * (1 - (profile_fraction_diff + etaN)); //Match Score
		}
		else {
			psiP = atoi(S1) * (profile_fraction_diff + etaP); //Mismatch Score
			psiN = atoi(S1) * (profile_fraction_diff + etaN); //Mismatch Score
		}
		
		if(M[row][col] > Qx[row][col] && M[row][col] > Qy[row][col]) {
			norm_score += profile_fraction_diff;

			if(fabs(*(norm_x_profile+(row-1))-*(norm_y_profile+(col-1)))<atof(DELTA)) {
				match_score += S[row][col]-S[row-1][col-1];
			}
			else {
				mismatch_score += S[row][col]-S[row-1][col-1];
			}

			is_M=1; is_Qx=0; is_Qy=0;
			trace_back_row[align_len]=row; trace_back_col[align_len]=col; trace_back_state[align_len]=0;
			row--, col--;
		}
		else if((M[row][col]==Qx[row][col] || M[row][col]==Qy[row][col]) && (M[row][col]==(M[row-1][col-1] + psiP))) {
			norm_score += profile_fraction_diff;

			if(fabs(*(norm_x_profile+(row-1))-*(norm_y_profile+(col-1)))<atof(DELTA)) {
				match_score += S[row][col]-S[row-1][col-1];
			}
			else {
				mismatch_score += S[row][col]-S[row-1][col-1];
			}

			is_M=1; is_Qx=0; is_Qy=0;
			trace_back_row[align_len]=row; trace_back_col[align_len]=col; trace_back_state[align_len]=0;
			row--, col--;
		}
		else if(Qx[row][col]>=M[row][col] && Qx[row][col]>=Qy[row][col]) {
			if(is_Qx==1) { gap_ext_count++; } else { gap_open_count++; }

			is_M=0; is_Qx=1, is_Qy=0;
			trace_back_row[align_len]=row; trace_back_col[align_len]=col; trace_back_state[align_len]=-1;
			col--;
		}
		else if(Qy[row][col]>=M[row][col] && Qy[row][col]>Qx[row][col]) {
			if(is_Qy==1) { gap_ext_count++; } else { gap_open_count++; }

			is_M=0; is_Qx=0; is_Qy=1;
			trace_back_row[align_len]=row; trace_back_col[align_len]=col; trace_back_state[align_len]=1;
			row--;
		}
		align_len++;
	}

	if(plotAlignment==1 && NPRFILE!=NULL) {
		for(i=(align_len-1); i>=0; i--) {
			if(trace_back_state[i]==0) {
				fprintf(NPRFILE, "%d\t%6.2f\t%6.2f\n", j++, *(norm_x_profile+(trace_back_row[i]-1)), *(norm_y_profile+(trace_back_col[i]-1)));
				nprfile_line++;
			}
			else if(trace_back_state[i]==-1) {
				fprintf(NPRFILE, "%d\t-1\t%6.2f\n", j++, *(norm_y_profile+(trace_back_col[i]-1)));
				nprfile_line++;
			}
			else if(trace_back_state[i]==1) {
				fprintf(NPRFILE, "%d\t%6.2f\t-1\n", j++, *(norm_x_profile+(trace_back_row[i]-1)));
				nprfile_line++;
			}
		}
	}

	//Calculate the maximum possible score between two block read profiles.
	double max_possible_score;
	if(x_profile_len > y_profile_len) { max_possible_score = atoi(S0) * x_profile_len; }
	else { max_possible_score = atoi(S0) * y_profile_len; }
	matrix_align_score = (double) matrix_align_score / (double) max_possible_score;

	gap_score = (((atoi(Ci)*percentScore)*gap_open_count)+((atoi(Ce)*percentScore)*gap_ext_count));
	//printf("Align len=%d,Match=%6.2f,Mismatch=%6.2f,Gap=%6.2f,Percent score=%6.2f\n", align_len, match_score, mismatch_score, gap_score, percentScore);

	//Return performance measure entity (query_coverage|align_score|norm_score|matrix_align_score).
	return matrix_align_score;
}

/* Normalize the values of a read profile between 0 and 1. */
void normalize(double *norm_profile, double *norm_sread_profile, double *norm_tread_profile, int profile_len, double max_sread_profile, double max_tread_profile) {
	int i=0;
	while(i < profile_len) {
		*(norm_sread_profile+i) = (double) ((*(norm_sread_profile+i)/max_sread_profile) * 100); //Normalized start read count.
		*(norm_tread_profile+i) = (double) ((*(norm_tread_profile+i)/max_tread_profile) * 100); //Normalized total read count.
		*(norm_profile+i) = fabs(*(norm_sread_profile+i) - *(norm_tread_profile+i))/100;
		i++;
	}
}

/* Fill the matrix with default scores (0). */
void fill_matrix_default(int x_profile_len, int y_profile_len, float **M, float **Qx, float **Qy, float **S) {
	M[0][0]=0; Qx[0][0]=0; Qy[0][0]=0; S[0][0]=0;
	int i, j;

	for(i=1; i<=x_profile_len; i++) {
		if(i==1) {
			M[i][0] = -10000;
			Qx[i][0] = Qx[i-1][0] + atoi(Ci);
			Qy[i][0] = Qy[i-1][0] + atoi(Ci);
			S[i][0] = S[i-1][0] + atoi(Ci);
		}
		else {
			M[i][0] = -10000;
			Qx[i][0] = Qx[i-1][0] + atoi(Ce);
			Qy[i][0] = Qy[i-1][0] + atoi(Ce);
			S[i][0] = S[i-1][0] + atoi(Ce);
		}
	}

	for(j=1; j<=y_profile_len; j++) {
		if(j==1) {
			M[0][j] = -10000;
			Qx[0][j] = Qx[0][j-1] + atoi(Ci);
			Qy[0][j] = Qy[0][j-1] + atoi(Ci);
			S[0][j] = S[0][j-1] + atoi(Ci);
		}
		else {
			M[0][j] = -10000;
			Qx[0][j] = Qx[0][j-1] + atoi(Ce);
			Qy[0][j] = Qy[0][j-1] + atoi(Ce);
			S[0][j] = S[0][j-1] + atoi(Ce);
		}
	}

	for(i=1; i<=x_profile_len; i++) {
		for(j=1; j<=y_profile_len; j++) {
			M[i][j]=0;
			Qx[i][j]=0;
			Qy[i][j]=0;
			S[i][j]=0;
		}
	}
}

/* Determine maximum between two values. */
double max2(double a, double b) {
	if(a>=b) { return a; }
	else { return b;}
}

/* Determine maximum between three values. */
double max3(double a, double b, double c) {
	if(a>=b && a>=c) { return a; }
	else if(b>a && b>=c) { return b; }
	else { return c; }
}

/* Allocate memory for a 2D-Array. */
float** create_2D_array(int numrows, int numcols){
	float **data;
	data = (float **) malloc(numrows * sizeof(float *));
	if(data == NULL){
		free(data);
		printf("Memory allocation failed while allocating for data[].\n");
		exit(-1);
	}
	/* Allocate integer memory for the second dimension of a data[][]; */
	register int i;
	for(i = 0; i < numrows; i++) {
		data[i] = (float *) malloc(numcols * sizeof(float));
		if(data[i] == NULL){
			free(data[i]);
			printf("Memory allocation failed while allocating for data[x][].\n");
			exit(-1);
		}
	}
	return *&data; // Success
}

/* Replace special characters in a string with '-'. */
void replace_special_char(char* string) {
	int i;
	for(i=0; i<strlen(string); i++) {
		if(string[i]==':' || string[i]=='\\' || string[i]=='/' || string[i]=='>' || \
		string[i]=='<' || string[i]=='\"' || string[i]=='\'' || string[i]=='|' || \
		string[i]=='\?' || string[i]=='*' || string[i]=='.' || string[i]=='(' || \
		string[i]==')')
			string[i]='-';
	}
}

/* Plot block alignment */
void plot_block_aln(cluster_t *cluster1, cluster_t *cluster2) {
	int i, k;
	for(i=1; i<= cluster1->noofblocks; i++) {
		fprintf(NPRFILE, "%d:5:%d:%lf\t", cluster1->blocks[i].start, cluster1->blocks[i].end, (cluster1->blocks[i].height/cluster1->expression)*100);
	}
	fprintf(NPRFILE, "\n");
	for(k=1; k<= cluster2->noofblocks; k++) {
		fprintf(NPRFILE, "%d:15:%d:%lf\t", cluster2->blocks[k].start, cluster2->blocks[k].end, (cluster2->blocks[k].height/cluster2->expression)*100);
	}
	fprintf(NPRFILE, "\n");
	fclose(NPRFILE);
}

void precomputeBlockScores(cluster_t *cluster1, cluster_t *cluster2, FILE *file)
/* precompute all blockScores */
{	
	int i,k; count_block_aln=0;

    for(i = 1; i <= cluster1->noofblocks; i++){
        if(cluster1->blocks[i].score != NULL){
            free(cluster1->blocks[i].score);
            cluster1->blocks[i].score = NULL;
        }
    }
    for(i = 1; i <= cluster1->noofblocks; i++){
		// get memory for scores
		cluster1->blocks[i].score = realloc(cluster1->blocks[i].score, (cluster2->noofblocks + 1)*sizeof(score_t));
		for(k = 1; k <= cluster2->noofblocks; k++){
			//Plot Cluster Alignment
			coor_block_aln[count_block_aln][0]=nprfile_line;
			//Code end

/*			int iBlockLen = cluster1->blocks[i].end - cluster1->blocks[i].start;
			int kBlockLen = cluster1->blocks[k].end - cluster1->blocks[k].start;
			if(iBlockLen >= kBlockLen) { 
				int matches = kBlockLen;
				int gaps = iBlockLen - kBlockLen;
				cluster1->blocks[i].score[k].score = (matches * atoi(S0)) + (atoi(Ci) + ((gaps-1)*atoi(Ce)));
			}
			else if(iBlockLen < kBlockLen) { 
				int matches = iBlockLen;
				int gaps = kBlockLen - iBlockLen;
				cluster1->blocks[i].score[k].score = (matches * atoi(S0)) + (atoi(Ci) + ((gaps-1)*atoi(Ce)));
			}*/

			// compute score
//			if(cluster1->blocks[i].score[k].score >=0) { 
				cluster1->blocks[i].score[k].score = blockAlign(i, k, cluster1, cluster2);
//			}

			if(file != NULL) fprintf(file,"%s_block%d\t%s_block%d\t%lf\n",cluster1->id,i,cluster2->id,k,cluster1->blocks[i].score[k].score);

			//Plot Cluster Alignment
			coor_block_aln[count_block_aln][1]=nprfile_line-1;
			count_block_aln++;
			//Code end
		}
	}
}

field_t *initializeMatrix(cluster_t *cluster1, cluster_t *cluster2) {
	field_t *matrix;
	int i,j,k,l;
	int M = 1;
	
	matrix = calloc(((cluster1->noofblocks + 1)*(cluster1->noofblocks + 1))*((cluster2->noofblocks + 1)*(cluster2->noofblocks + 1)), sizeof(field_t));

	for(i = 1; i<= cluster1->noofblocks; i++){
		int k0 = i-2*M-1;
		if(k0 < 1){k0 = 1;}		
		for(k = k0; k <= cluster2->noofblocks; k++){
			if(k > i+2*M+1){break;}
			for(j = i; j <= cluster1->noofblocks; j++){
				int l0 = k + (j - i) - M - 1;
				if(l0<k){l0 = k;}
				int d = j - i + k - l0;
				for(l = l0; l<= cluster2->noofblocks; l++){	
					if(d < -M - 1){break;}
					ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).score = abs(d) * penalty_gap;
					ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).set = 1;
					d--;
				}
			}
			for(l = k; l <= cluster2->noofblocks; l++){
				if(l - k > M + 1){break;}
				ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, i-1, k, l).score = (l - k + 1) * penalty_gap;
				ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, i-1, k, l).set = 1;
			}
			for(j = i; j <= cluster1->noofblocks; j++){
				if(j - i > M + 1){break;}
				ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, k-1).score = (j - i + 1) * penalty_gap;
				ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, k-1).set = 1;
			}
			ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, i-1, k, k-1).score = 0;
			ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, i-1, k, k-1).set = 1;
		}
	}
	return matrix;
}

double getPairedScore(int i, int j, int k, int l, cluster_t *cluster1, cluster_t *cluster2, field_t *matrix)
/* GET MATCH VALUE */
{	
	// get distance difference
	double distanceDifference = (double) abs( abs((double) cluster1->blocks[i].start - (double) cluster1->blocks[j].start) - 
			            abs((double) cluster2->blocks[k].start - (double) cluster2->blocks[l].start) );
	
	// get block similarity
	double blockScore = (cluster1->blocks[i].score[k].score + cluster1->blocks[j].score[l].score);
	
	// calc score
	double	score = param_block * blockScore + param_distance * (1 - (0.025 * pow(distanceDifference,2)));	
	
	if(i==j || k==l){score = 0;}
	
	// return final score
	return score;
}

field_t fillMatrix(cluster_t *cluster1, cluster_t *cluster2, field_t *matrix)
/* FILL MATRIX */
{
	int i,j,k,l,h,q;

	for(j = 1; j <= cluster1->noofblocks; j++){
		for(l = 1; l <= cluster2->noofblocks; l++){
			for(i = j; i > 0; i--){
				for(k = l; k > 0; k--){			

					// check the 4 cases
					double score;
			
					// single block match
					double single  = ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, j, k+1, l).score + param_block * cluster1->blocks[i].score[k].score;										
					ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).score = single;
					ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).set = 1;	
					
					// block insertion
					double insertion = ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, j, k, l).score + (cluster1->blocks[i].height / cluster1->expression) * penalty_gap;				
					if(insertion > ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).score){
						ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).score = insertion;
					}
													
					// block deletion
					double deletion  = ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k+1, l).score + (cluster2->blocks[k].height / cluster2->expression) * penalty_gap;					
					if(deletion > ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).score){
						ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).score = deletion;
					}


					int h,q,hh,qq;
					double max = -10000;
					
					// recursion
					for(h = i + 1; h <= j; h++){
						for(q = k + 1; q <= l; q++){
							double value = getPairedScore(i,h,k,q,cluster1,cluster2,matrix);
							
							if(ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, h-1, k+1, q-1).set == 1){ 
								value += ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, h-1, k+1, q-1).score; 
							}
							if(ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, h+1, j, q+1, l).set == 1){ 
								value += ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, h+1, j, q+1, l).score; 
							}
							
							if(value > max){max = value;}		
						}
					}					
					if(max > ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).score){
						ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).score = max;
					}
					
				}	
			}
		}
	}
}

score_t backtrack_global(int i, int j, int k, int l, field_t *matrix, cluster_t *cluster1, cluster_t *cluster2, int print)
/* BACKTRACKING */
{	
	int h,q; 
	score_t c;
	c.bindings = 0;
	c.score = 0;
	c.gapPenalty = 0;
	replace_special_char(cluster1->ncRNAid);
	replace_special_char(cluster2->ncRNAid);
	
	int x;
	for(x = 1; x <= cluster1->noofblocks; x++){
		c.gapPenalty += (cluster1->blocks[x].height / cluster1->expression) * penalty_gap;
	}
	for(x = 1; x <= cluster2->noofblocks; x++){
		c.gapPenalty += (cluster2->blocks[x].height / cluster2->expression) * penalty_gap;
	}
	
	while(i<=j && k<=l){
		
		int found = 0;
		if(ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).set == 0){return c;}

		if(i==j || k==l){
			//Plot Cluster Alignment
			if(print==0 && plotAlignment==1 && NPRFILE!=NULL) {
			fprintf(NPRFILE, "%s_%s_%d#%s_%s_%d.pdf %d %d %d %s %s %6.2f %d %d\n", cluster1->id, cluster1->ncRNAid, i, cluster2->id, \
			cluster2->ncRNAid, k, (coor_block_aln[(((i-1)*cluster2->noofblocks)+k)-1][1]-coor_block_aln[(((i-1)*cluster2->noofblocks)+k)-1][0])+1, \
			coor_block_aln[(((i-1)*cluster2->noofblocks)+k)-1][0], coor_block_aln[(((i-1)*cluster2->noofblocks)+k)-1][1], \
			cluster1->ncRNAtype, cluster2->ncRNAtype, cluster1->blocks[i].score[k].score, i, k);
			fprintf(NPRFILE, "ALIGNMENT %d - %d (%lf)\n",i,k,cluster1->blocks[i].score[k].score); }
			//Code end

			if(print==1) printf("%d - %d (%lf)\n",i,k,cluster1->blocks[i].score[k].score);
			c.score += param_block * cluster1->blocks[i].score[k].score;
			c.gapPenalty -= (cluster1->blocks[i].height / cluster1->expression) * penalty_gap;
			c.gapPenalty -= (cluster2->blocks[k].height / cluster2->expression) * penalty_gap;
			c.bindings++;
			i++; k++; continue;
		}
		
		if(ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, j, k, l).set == 0){
			ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, j, k, l).score = (j - i + 1) * penalty_gap;
			ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, j, k, l).set = 1;
		}
		if(ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k+1, l).set == 0){
			ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k+1, l).score = (l - k + 1) * penalty_gap;
			ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k+1, l).set = 1;
		}
		
		if(ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, j, k+1, l).set == 0){
			ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, j, k+1, l).score = 0;
			ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, j, k+1, l).set = 1;
		}
		
		if(fabs(ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).score - 
		   (ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, j, k, l).score + (cluster1->blocks[i].height / cluster1->expression) * penalty_gap)) < 0.001){
			if(print==1) {printf("%d - |\n",i);}
			i++; continue;
		}
		if(fabs(ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).score - 
		   (ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k+1, l).score + (cluster2->blocks[k].height / cluster2->expression) * penalty_gap)) < 0.001){
			if(print==1) {printf("| - %d\n",k);}
			k++; continue;
		}
		if(fabs(ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).score - 
		   (ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, j, k+1, l).score + param_block * cluster1->blocks[i].score[k].score)) < 0.001){
			//Plot Cluster Alignment
			if(print==0 && plotAlignment==1 && NPRFILE!=NULL) {
			fprintf(NPRFILE, "%s_%s_%d#%s_%s_%d.pdf %d %d %d %s %s %6.2f %d %d\n", cluster1->id, cluster1->ncRNAid, i, cluster2->id, \
			cluster2->ncRNAid, k, (coor_block_aln[(((i-1)*cluster2->noofblocks)+k)-1][1]-coor_block_aln[(((i-1)*cluster2->noofblocks)+k)-1][0])+1, \
			coor_block_aln[(((i-1)*cluster2->noofblocks)+k)-1][0], coor_block_aln[(((i-1)*cluster2->noofblocks)+k)-1][1], \
			cluster1->ncRNAtype, cluster2->ncRNAtype, cluster1->blocks[i].score[k].score, i, k);
			fprintf(NPRFILE, "ALIGNMENT %d - %d (%lf)\n",i,k,cluster1->blocks[i].score[k].score); }
			//Code end

			if(print==1) {printf("%d - %d (%lf)\n",i,k,cluster1->blocks[i].score[k].score);}
			c.score += param_block * cluster1->blocks[i].score[k].score;
			c.gapPenalty -= (cluster1->blocks[i].height / cluster1->expression) * penalty_gap;
			c.gapPenalty -= (cluster2->blocks[k].height / cluster2->expression) * penalty_gap;
			c.bindings++;
			i++; k++; continue;
		}

		FOUND: for(h=i+1; h<=j; h++){
			for(q=k+1; q<=l; q++){
				if(ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, h-1, k+1, q-1).set == 0){
					ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, h-1, k+1, q-1).score = 0;
					ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, h-1, k+1, q-1).set = 1;
				}
				if(ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, h+1, j, q+1, l).set == 0){
					ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, h+1, j, q+1, l).score = 0;
					ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, h+1, j, q+1, l).set = 1;
				}
				double value = getPairedScore(i,h,k,q,cluster1,cluster2,matrix);
				value += ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i+1, h-1, k+1, q-1).score;
				value += ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, h+1, j, q+1, l).score;

				if(fabs(ACCESS(matrix, cluster1->noofblocks, cluster2->noofblocks, i, j, k, l).score - value) < 0.001){
					//Plot Cluster Alignment
					if(print==0 && plotAlignment==1 && NPRFILE!=NULL) {
					fprintf(NPRFILE, "%s_%s_%d#%s_%s_%d.pdf %d %d %d %s %s %6.2f %d %d\n", cluster1->id, cluster1->ncRNAid, i, cluster2->id, cluster2->ncRNAid, k, (coor_block_aln[(((i-1)*cluster2->noofblocks)+k)-1][1]-coor_block_aln[(((i-1)*cluster2->noofblocks)+k)-1][0])+1, coor_block_aln[(((i-1)*cluster2->noofblocks)+k)-1][0], coor_block_aln[(((i-1)*cluster2->noofblocks)+k)-1][1], \
					cluster1->ncRNAtype, cluster2->ncRNAtype, cluster1->blocks[i].score[k].score, i, k);
					fprintf(NPRFILE, "%s_%s_%d#%s_%s_%d.pdf %d %d %d %s %s %6.2f %d %d\n", cluster1->id, cluster1->ncRNAid, h, cluster2->id, cluster2->ncRNAid, q, (coor_block_aln[(((h-1)*cluster2->noofblocks)+q)-1][1]-coor_block_aln[(((h-1)*cluster2->noofblocks)+q)-1][0])+1, coor_block_aln[(((h-1)*cluster2->noofblocks)+q)-1][0], coor_block_aln[(((h-1)*cluster2->noofblocks)+q)-1][1], \
					cluster1->ncRNAtype, cluster2->ncRNAtype, cluster1->blocks[h].score[q].score, h, q);
					fprintf(NPRFILE, "ALIGNMENT %d:%d - %d:%d (%lf)\n",i,h,k,q,getPairedScore(i,h,k,q,cluster1,cluster2,matrix)); }
					//Code end

					if(print==1) {printf("%d:%d - %d:%d (%lf)\n",i,h,k,q,getPairedScore(i,h,k,q,cluster1,cluster2,matrix));} 
					c.score += getPairedScore(i,h,k,q,cluster1,cluster2,matrix);
					c.gapPenalty -= (cluster1->blocks[i].height / cluster1->expression) * penalty_gap;
					c.gapPenalty -= (cluster1->blocks[h].height / cluster1->expression) * penalty_gap;
					c.gapPenalty -= (cluster2->blocks[k].height / cluster2->expression) * penalty_gap;
					c.gapPenalty -= (cluster2->blocks[q].height / cluster2->expression) * penalty_gap;
					c.bindings += 2;
					score_t x = backtrack_global(i+1,h-1,k+1,q-1, matrix, cluster1, cluster2, print);
					c.bindings += x.bindings;
					found = 1;
					i = h+1; 
					k = q+1; 
					goto FOUND;
				}
			}
		}
		
		if(found == 0){if(print==1) printf("backtracking failed at %d %d %d %d\n",i,j,k,l); break;}
	}
	return c;
}


/* MAIN FUNCTION */
int main(int argc, char *argv[])
{  
	// check user input
	parseopt(argc, argv);
	// confirm minimum number of arguments are passed and gap penalty is negative.
	if(argc < 5) { printUsage(); exit(1); }
	if(atoi(Ci) > 0 || atoi(Ce) > 0 || penalty_gap > 0) { fprintf(stderr, "\nPositive value for Gap Penalty, please check...\n\n"); printUsage(); exit(1); }
	
	// check files
	if(!query_File || !subject_File)
	{
		printUsage(); exit(0);
	}
	
	// check file format
	if(check_file_format(query_File)==1 || check_file_format(subject_File)==1) {
		fprintf(stderr, "The input file has incorrect format. Please check.\n"); exit(0);
	}

	// open file for local alignments of all blocks against all blocks 
	FILE *localAlignmentFile = NULL;
	if(printLocalAlignments == 1)
		localAlignmentFile = fopen(local_File,"w+");
	
	// write header
	writeHeader(query_File, subject_File);	
	
	// read query set
	set_t *querySet = read_cluster_file(query_File);

	// read subject set
	set_t *subjectSet = read_cluster_file(subject_File);
	
	// compare all against all
	cluster_t *query;
	cluster_t *subject;
	int r,s;
	
	// run through the queries
	for(s = 1; s <= querySet->noofclusters; s++){
		// get cluster s of query list as query
		query = &querySet->clusters[s];	
		query = sortCluster(query);

		// calculate score of query block group with itself
		double ownScore, ownScoreQuery, ownScoreSubject, score;
		precomputeBlockScores(query, query, NULL);
		field_t *matrix = initializeMatrix(query, query);
		*matrix = fillMatrix(query, query, matrix);
		score_t c = backtrack_global(1, query->noofblocks, 1, query->noofblocks, matrix, query, query, 0);
		ownScoreQuery = c.score;
		free(matrix);

		// run through the subjects
		for(r = 1; r <= subjectSet->noofclusters; r++)
		{
			if(onlyAlignPairs==1 && s!=r) { continue; }
			// get cluster r of subject list as subject
			subject = &subjectSet->clusters[r];
			subject = sortCluster(subject);
			
			if(subject->noofblocks >= 1) {
				// calculate score of subject block group with itself
				precomputeBlockScores(subject, subject, NULL);
				field_t	*matrix = initializeMatrix(subject, subject);
				*matrix = fillMatrix(subject, subject, matrix);
				score_t c = backtrack_global(1, subject->noofblocks, 1, subject->noofblocks, matrix, subject, subject, 0);
				ownScoreSubject = c.score;
				free(matrix);
			}

			// check, if query and subject block groups are highly variable in size
			score = 0;
			if(query->noofblocks >= subject->noofblocks) {
				int pairedBlocks = subject->noofblocks % 2;
				int singleBlocks = subject->noofblocks - (2*pairedBlocks);
				int gappedBlocks = query->noofblocks - subject->noofblocks;
				score = ((singleBlocks * param_block) + (pairedBlocks * (param_block + param_distance))) + (gappedBlocks * penalty_gap);
			}
			else if(query->noofblocks < subject->noofblocks) {
				int pairedBlocks = query->noofblocks % 2;
				int singleBlocks = query->noofblocks - (2*pairedBlocks);
				int gappedBlocks = subject->noofblocks - query->noofblocks;
				score = ((singleBlocks * param_block) + (pairedBlocks * (param_block + param_distance))) + (gappedBlocks * penalty_gap);
			}
			//if(score >=0) {
				if(ownScoreQuery >= ownScoreSubject) { ownScore = ownScoreQuery; }
				else { ownScore = ownScoreSubject; }

				//Plot Cluster Alignment
				if(plotAlignment == 1 && ((NPRFILE = fopen(alignment_File, "wt")) == NULL )) {
					fprintf(stderr, "Cannot open normalized profile alignment file for writing.\n"); exit(0);
				} nprfile_line=1;
				//Code end
			
				// precompute the scores for the blocks
				precomputeBlockScores(query, subject, localAlignmentFile);

				// initialize 4-dimensional matrix for the sankoff
				field_t	*matrix = initializeMatrix(query, subject);
			
				// fill matrix
				*matrix = fillMatrix(query, subject, matrix);
			
				// normalize the scores and print result	
				score_t c = backtrack_global(1, query->noofblocks, 1, subject->noofblocks,matrix, query, subject, 0);
				if(printNormScore == 1) {
					score = (c.score + c.gapPenalty) / ownScore;
				}
				else {
					score = (c.score + c.gapPenalty);
				}

				//Plot Cluster Alignment
				if(plotAlignment == 1) plot_block_aln(query, subject);
				//Code end

				// backtrack and print alignment // last value: 1 -> show alignment; 0 -> don't show alignment
				if(printAlignment == 1){
					c = backtrack_global(1, query->noofblocks, 1, subject->noofblocks, matrix, query, subject, 1);
					printf("gaps: %d\n",((query->noofblocks + subject->noofblocks) - (c.bindings*2)));
				}

				free(matrix);
			//}

			if(score < 0){score = 0;}
			printf("%s|%s|%s|%s|%s:%d-%d(%s)\t%s|%s|%s|%s|%s:%d-%d(%s)\t%lf\n",
				querySet->clusters[s].id, querySet->clusters[s].ncRNAid, querySet->clusters[s].ncRNAtype, querySet->clusters[s].ncRNAclass,
				querySet->clusters[s].chrom,querySet->clusters[s].start,querySet->clusters[s].end,querySet->clusters[s].strand,
				subjectSet->clusters[r].id, subjectSet->clusters[r].ncRNAid, subjectSet->clusters[r].ncRNAtype, subjectSet->clusters[r].ncRNAclass,
				subjectSet->clusters[r].chrom,subjectSet->clusters[r].start,subjectSet->clusters[r].end,subjectSet->clusters[r].strand,
				score);

		}
	}
	free(querySet);
	free(subjectSet);
	
	if(printLocalAlignments == 1)
		fclose(localAlignmentFile);
	
	return(0);
}
