/*
 * file.c
 *
 * contains functions to read and write files
 *
 *    Copyright (C) 2014  Nikolai Hecker
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "datatypes.h"
#include "matrix.h"

/* ---------------------- read_rddG_mat  ----------------------------------
 * reads rddG file
 * returns pointer to NMMatrix containing rddG values
 * returns NULL if error occurs
 *
 */
NMMatrix*
read_rddG_mat(char *filename, const unsigned long max_bfsize)
{
  unsigned long n, m, lnum, uldmp1, uldmp2;
  int matches;
  double dbldmp1, dbldmp2;
  NMMatrix* mat;
  FILE *file;
  char *buffer;

  lnum = n = m = 0;
  mat = NULL;

  /* open and check file */
  file = fopen(filename,"rt");
  
  if(file == NULL)
    {
      fprintf(stderr, "Error. Can't open \'%s\'!\n", filename);
      return NULL;
    }
  
  /* check for ending newline */
  fseek(file, -1, SEEK_END);
  if(fgetc(file) != '\n')
    {
      fprintf(stderr, "Error. Missing ending newline in rddG matrix file \'%s\'!\n", filename);
      return NULL;
    }
  fseek(file, 0, SEEK_SET);
  
  if((buffer = malloc(max_bfsize * sizeof(char)) ) == NULL)
    {
      fprintf(stderr, "Error! Mem allocation for file buffer failed.\n");
      return NULL;
    }

  /* process file count entries */
  while( fgets(buffer, max_bfsize, file) != NULL)
    {
      
      if(lnum > 0 && (matches = sscanf(buffer, "%lu\t%lu\t%lf\t%lf", &uldmp1, &uldmp2, &dbldmp1, &dbldmp2) ) == 4)
	{
	  /* get max left/right*/
	  m = m > uldmp1 ? m : uldmp1;
	  n = n > uldmp2 ? n : uldmp2;
	}
      lnum++;
    }

  /* add column/row since matrix starts at zero */
  m++;
  n++;

  /* allocate and init matrix */
  if( (mat = nmm_init(m, n)) == NULL)
    {
      fprintf(stderr, "Error! Mem allocation for NMMatrix failed.\n");
      return NULL;
    }

  /* process file, fill entries */
  fseek(file, 0, SEEK_SET);
  while( fgets(buffer, max_bfsize, file) != NULL)
    {
      
      if(lnum > 0 && (matches = sscanf(buffer, "%lu\t%lu\t%lf\t%lf", &uldmp1, &uldmp2, &dbldmp1, &dbldmp2) ) == 4)
	{
	  mat->e[uldmp1][uldmp2] = dbldmp1;
	}
      lnum++;
    }

  fclose(file);
  free(buffer);
  return mat;
}
