/*  functions for N X M matrices
 *
 *    Copyright (C) 2014  Nikolai Hecker
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdlib.h>
#include "datatypes.h"

/* ------------------- nmm_init -------------------------------
 * allocates mem for N X M matrix and initializes matrix
 */
NMMatrix*
nmm_init(const size_t nrows, const size_t ncols)
{
	NMMatrix *mat;
	size_t i;
	
	if( (mat = malloc(sizeof(NMMatrix))) == NULL){
		return NULL;
	}

	mat->nrows = nrows;
	mat->ncols = ncols;

	if( (mat->e = malloc(nrows * sizeof(double*))) == NULL){
		free(mat);
		return NULL;
	}

	if( (mat->entries = malloc(nrows * ncols * sizeof(double))) == NULL ){
		free(mat->e);
		free(mat);
		return NULL;
	}

	for(i=0; i < nrows; i++)
		mat->e[i] = &mat->entries[i * ncols];

	return mat;
}


/* ---- nmm_free -----
 * frees mem from 
 * matrix
 */
void
nmm_free(NMMatrix *mat)
{
	free(mat->entries);
	free(mat->e);
	free(mat);
}

/* ------- nmm_zero --------
 * fills N X M matrix with 0s
 */
void
nmm_zero(NMMatrix *mat)
{
	size_t i, j;

	for(i=0; i < mat->nrows; i++)
		for(j=0; j < mat->ncols; j++)
			mat->e[i][j] = 0;
}


/* ------------------- nmintm_init -------------------------------
 * allocates mem for N X M matrix and initializes matrix
 */
NMINTMatrix*
nmintm_init(const size_t nrows, const size_t ncols)
{
	NMINTMatrix *mat;
	size_t i;
	
	if( (mat = malloc(sizeof(NMINTMatrix))) == NULL){
		return NULL;
	}

	mat->nrows = nrows;
	mat->ncols = ncols;

	if( (mat->e = malloc(nrows * sizeof(unsigned long*))) == NULL){
		free(mat);
		return NULL;
	}

	if( (mat->entries = malloc(nrows * ncols * sizeof(unsigned long))) == NULL ){
		free(mat->e);
		free(mat);
		return NULL;
	}

	for(i=0; i < nrows; i++)
		mat->e[i] = &mat->entries[i * ncols];

	return mat;
}


/* ---- nmintm_free -----
 * frees mem from 
 * matrix
 */
void
nmintm_free(NMINTMatrix *mat)
{
	free(mat->entries);
	free(mat->e);
	free(mat);
}


/* ------- nmintm_fill --------
 * fills N X M matrix with 0s
 */
void
nmintm_fill(NMINTMatrix *mat, const unsigned long val)
{
	size_t i, j;

	for(i=0; i < mat->nrows; i++)
		for(j=0; j < mat->ncols; j++)
			mat->e[i][j] = val;
}


/* ------- nmintm_fill_radius --------
 * fills N X M matrix with 0s
 */
void
nmintm_fill_radius(NMINTMatrix *mat, const unsigned long i, const unsigned long j, const unsigned long radius, const unsigned long val)
{
  size_t min_i, max_i,  min_j, max_j, k, l;
  
  min_i = i < radius  ? 0 : (i-radius);
  max_i = (i+radius) < mat->nrows  ? (i+radius) : (mat->nrows-1);
  min_j = j < radius  ? 0 : (j-radius);
  max_j = (j+radius) < mat->ncols  ? (j+radius) : (mat->ncols-1);
  

	for(k=min_i; k <= max_i; k++)
		for(l=min_j; l <= max_j; l++)
		  {
		    mat->e[k][l] = val;
		  }
}
