#!/usr/bin/env perl
#
# overlays dot plots of structure contraints
# where the smaller constraint is part of the larger constraint
#
# first base will be counted as 1 (to comply with dot plots)
#
#    Copyright (C) 2015  Nikolai Hecker
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.
# 
############################################################################
use strict;
use Carp;

my $usage = "Usage:\n\t$0 [CONSTRAINT-FILE1] [CONSTRAINT-FILE1] [DOTPLOT-FILE1] [DOTPLOT-FILE2]\n";


if(@ARGV < 4)
{
    print $usage;
    croak "Too few arguments.\n";
}

my $cf1=$ARGV[0];
my $cf2=$ARGV[1];
my $dpf1=$ARGV[2];
my $dpf2=$ARGV[3];

my $conscolstr1 = "1 0 0 setrgbcolor";
my $conscoloverstr1 = "1 0.6 0 setrgbcolor";
my $conscolstr2 = "1 0 0 setrgbcolor";
my $conscoloverstr2 = "1 0.6 0 setrgbcolor";
my $blackcolstr = "0 0 0 setrgbcolor";

my $head1="";
my $seq1="";
my $cons1="";

my $head2="";
my $seq2="";
my $cons2="";

my $start2 = -1; #start position of smaller constraint in larger constraint
my %h_bp = ();
my %h_bpprob2 = ();

my $fh;

##read files
open($fh, "<$cf1") || croak "Can't open '$cf1'.\n";
$head1 = readline($fh);
chomp($head1);
$seq1 = readline($fh);
chomp($seq1);
$cons1 = readline($fh);
chomp($cons1);
close($fh);

open($fh, "<$cf2") || croak "Can't open '$cf2'.\n";
$head2 = readline($fh);
chomp($head2);
$seq2 = readline($fh);
chomp($seq2);
$cons2 = readline($fh);
chomp($cons2);
close($fh);


# align
my $len1 = length($cons1);
my $len2 = length($cons2);

#if($len1 > $len2)
#{
#    croak "Structure constraint '$cf1' must be <= structure constraint '$cf2'.\n";
#}

my $start2 = index($cons2, $cons1);

if($len1 > $len2)
{
    $start2 = index($cons1, $cons2);
}

if( $start2 == -1)
{
    croak "Could not find structure constraint '$cf1' inside '$cf2'.\n";
}

#get base pairs
my @stack = ();

for(my $i=0; $i < $len1; $i++)
{
    my $c = substr($cons1, $i, 1);

    if($c eq '(')
    {
	push(@stack, ($i+1) );
    }
    elsif($c eq ')')
    {
	my $j = pop(@stack);
	$h_bp{$j} = ($i+1);
    }
}

if(@stack > 0)
{
    croak "Number of opening and closing parenthesis does not match in '$cons1'.\n";
}

#### read second box plot
open($fh, "<$dpf2") || croak "Can't open '$dpf2'.\n";

while(<$fh>) 
{
    my $line = $_;

    if($line =~ /^(\d+)\s+(\d+)\s+(\d+\.\d*)\s+ubox$/)
    {
	my $posi = $1-$start2;
	my $posj = $2-$start2;
	my $p = $3;

	if($len1 > $len2)
	{
	    $posi = $1+$start2;
	    $posj = $2+$start2;
	}
	
	if($posi >= 1 && $posj >= 1 && $posi <= $len1 && $posj <= $len1 )
	{
	    $h_bpprob2{$posi}->{$posj} = $p;
	}
    }
}
close($fh);

### modify dot plots
open($fh, "<$dpf1") || croak "Can't open '$dpf1'.\n";

my $first_lbox = 1;
my $output = "";
while(<$fh>)
{
    my $line = $_;
    
    #print consensus upper triangle
    if($line =~ /^%start of base pair probability data$/)
    {
	print $line;
	print "$conscolstr1\n";
	foreach my $k (sort {$a <=> $b} keys %h_bp)
	{
	    print "$k $h_bp{$k} 1 ubox\n";
	}
	print "$blackcolstr\n";
    }
    elsif($line =~ /^(\d+)\s+(\d+)\s+(\d+\.\d*)\s+ubox$/)
    {
	#check whether part of constrained
	my $posi = $1;
	my $posj = $2;
	my $p = $3;

	if(defined($h_bp{$posi}) && $h_bp{$posi} == $posj)
	{
	    print "$conscoloverstr1\n";
	    print $line;
	    print "$blackcolstr\n";
	}
	else
	{
	    print $line;
	}
	
    }
    elsif($line =~ /^\d+.*?\s+lbox$/)
    {
	if($first_lbox) #print consensus lower triangle
	{
	    print "$conscolstr2\n";
	    foreach my $k (sort {$a <=> $b} keys %h_bp)
	    {
		print "$k $h_bp{$k} 1 lbox\n";
	    }
	    print "$blackcolstr\n";
	    $first_lbox = 0;

	    ###probs
	    foreach my $posi (sort keys %h_bpprob2)
	    {
		foreach my $posj (sort keys %{ $h_bpprob2{$posi} })
		{
		    my $p = $h_bpprob2{$posi}->{$posj};
		    
		    if(defined($h_bp{$posi}) && $h_bp{$posi} == $posj)
		    {
			print "$conscoloverstr2\n";
			print "$posi $posj $p lbox\n";
			print "$blackcolstr\n";
		    }
		    else
		    {
			print "$posi $posj $p lbox\n";
		    }
		}
	    }
	}
    }
    else
    {
	print $line;
    }
}
