/*********************************************************************

  col2txt.c

  usage: col2txt [FILE]

  This program makes a text alignment from a col file.

  000216 Bjarne Knudsen (bk@daimi.au.dk)

  Copyright (C) 2000 Bjarne Knudsen

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

*********************************************************************/

#include "../clib/col.h"
#include "../clib/file.h"

#define MAXNAME 1000

int main(int argc, char **argv)
{
  FILE *fp;         /* For input */
  Header *header;
  Entry *entry;
  int len;          /* Sequence length */
  char field[MAXCOLW];
  int i;
  char name[MAXNAME];  /* Sequence name */
  int nuc_col;      /* Column numbers */
  int read_error;   /* For keeping track of errors in reading entries */
  CmdArg *cmdarg;   /* Command line arguments */
  char *s;          /* String for arguments */

  cmdarg = InitArgument(argc, argv);

  if ((s = GetArgument(cmdarg)) != NULL) {
    fprintf(stderr, "Usage: col2txt [FILE]\n");
    return 1; }

  if ((s = GetFilename(cmdarg)) == NULL)
    fp = stdin;
  else if (GetFilename(cmdarg) != NULL) {
    fprintf(stderr, "Usage: col2txt [FILE]\n");
    return 1; }
  else if ((fp = fopen(s, "r")) == NULL) {
    fprintf(stderr, "col2txt: Error in opening file '%s'\n", s);
    return 1; }

  header = MakeHeader();
  entry = MakeEntry();

  if (ReadHeader(fp, header) != 0)
    return 1;

  while ((read_error = ReadEntry(fp, entry)) == 0) {
    if ((nuc_col = ReadColno(entry, "residue")) == 0 &&
	(nuc_col = ReadColno(entry, "nucleotide")) == 0 &&
	(nuc_col = ReadColno(entry, "aminoacid")) == 0 &&
	(nuc_col = ReadColno(entry, "code")) == 0) {
      fprintf(stderr,
	      "col2txt: Warning: Ignoring sequence, no column info\n");
      continue; }

    if (ReadText(entry, "SEQUENCE", MAXNAME, name) != 0 &&
	ReadText(entry, "ENTRY", MAXNAME, name) != 0)
      name[0] = '\0';
    for (i = 0; name[i] != '\0'; i++)
      if (name[i] == ' ')
	name[i] = '_';

    printf("%-12.12s ", name);

    len = EntryLength(entry);
    for (i = 1; i <= len; i++) {
      GetField(field, entry, i, nuc_col);
      printf("%c", field[0]);
    }
    printf("\n");
  }

  if (fp != stdin && fclose(fp) != 0) {
    fprintf(stderr, "col2txt: Error in closing file\n");
    return 1; }

  if (read_error == 1)
    return 1;

  return 0;
}
