/*********************************************************************

  extendlist.c

  usage: extendlist [-c[CONTEXT]] [FILE]

  Thisprogram outputs a list of what should be aligned when extending
  blast hits.

  -c  Take 100 or CONTEXT positions on each side of the query sequence.

  000308 Bjarne Knudsen (bk@daimi.au.dk)

  Copyright (C) 2000 Bjarne Knudsen

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

*********************************************************************/

#include "../clib/col.h"
#include "../clib/file.h"

void usage(void);

int main(int argc, char **argv)
{
  FILE *fp;         /* For input */
  Header *header;
  Entry *entry;
  int query_seqpos_col, subject_seqpos_col;
                    /* Column numbers */
  int read_error;   /* For keeping track of errors in reading entries */
  int len;          /* Used with -C option */
  CmdArg *cmdarg;   /* Command line arguments */
  char *s;          /* String for arguments */
  char field[MAXCOLW];
  int context;
  char name[80];
  int query_len, subject_len;
  int start_pos, end_pos, temp;
  int strand;

  /* default options */
  context = 0;

  cmdarg = InitArgument(argc, argv);

  while ((s = GetArgument(cmdarg)) != NULL)
    if (strncmp(s, "c", 1) == 0) {
      if (strlen(s) == 1)
	context = 100;
      else if (sscanf(s, "c%d%n", &context, &len) != 1 ||
	       len != strlen(s)) {
	usage();
	return 1; }
    }
    else {
      usage();
      return 1; }

  if ((s = GetFilename(cmdarg)) == NULL)
    fp = stdin;
  else if (GetFilename(cmdarg) != NULL) {
      usage();
    return 1; }
  else if ((fp = fopen(s, "r")) == NULL) {
    fprintf(stderr, "extendlist: Error in opening file '%s'\n", s);
    return 1; }

  header = MakeHeader();
  entry = MakeEntry();

  if (ReadHeader(fp, header) != 0)
    return 1;

  while ((read_error = ReadEntry(fp, entry)) == 0) {
    if ((query_seqpos_col = ReadColno(entry, "query_seqpos")) == 0 ||
	(subject_seqpos_col = ReadColno(entry, "subject_seqpos")) == 0) {
      fprintf(stderr,
              "extendlist: Warning: Ignoring sequence, no column info\n");
      continue; }

    len = EntryLength(entry);

    ReadText(entry, "QUERY_LENGTH", 79, field);
    query_len = atoi(field);
    ReadText(entry, "SUBJECT_LENGTH", 79, field);
    subject_len = atoi(field);

    GetField(field, entry, 1, subject_seqpos_col);
    start_pos = atoi(field);
    GetField(field, entry, len, subject_seqpos_col);
    end_pos = atoi(field);

    if (start_pos < end_pos)   /* Plus strand */
      strand = 1;
    else {                      /* Minus strand */
      strand = -1;
    }

    GetField(field, entry, 1, query_seqpos_col);
    start_pos -= strand * (atoi(field)-1+context);
    GetField(field, entry, len, query_seqpos_col);
    end_pos += strand * (query_len-atoi(field)+context);

    if (strand == -1) {
      temp = start_pos;
      start_pos = end_pos;
      end_pos = temp;
    }

    if (start_pos < 1)
      start_pos = 1;
    if (end_pos > subject_len)
      end_pos = subject_len;

    ReadText(entry, "QUERY", 79, name);
    printf("%-12s      1 %6d ", name, query_len);
    ReadText(entry, "SUBJECT", 79, name);
    printf("vs %-12s %6d %6d %s\n", name, start_pos, end_pos,
	   strand==1?"Plus":"Minus");
  }

  if (fp != stdin && fclose(fp) != 0) {
    fprintf(stderr, "extendlist: Error in closing file\n");
    return 1; }

  if (read_error == 1)
    return 1;

  return 0;
}

void usage(void)
{
  fprintf(stderr, "usage: extendlist [-c[CONTEXT]] [FILE]\n");
}
