/*********************************************************************

  grepcol.c

  usage: grepcol
              [-s | --sorted]  [-r<range> | -range=<range> | -f |
              --file] [<rangefile>] [<file>]

  See man page for info.

  000919 Bjarne Knudsen (bk@daimi.au.dk)

  Copyright (C) 2000 Bjarne Knudsen

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

*********************************************************************/

#include "../clib/col.h"

void usage(void);
int inrange(Entry *entry, int num, char *range);
int inorderrange(Entry *entry, int num, char *range, int *ptr);

int main(int argc, char **argv)
{
  FILE *fp;
  Header *header;
  Entry *entry;
  int read_error;   /* for keeping track of errors in reading entries */
  char *range;
  int i;
  CmdArg *cmdarg;   /* Command line arguments */
  char *s;          /* String for arguments */
  int option_f, option_s;
  int order_ptr = 0;

  /* default option */
  option_f = 0;
  option_s = 0;

  cmdarg = InitArgument(argc, argv);

  while ((s = GetArgument(cmdarg)) != NULL)
    if (strncmp(s, "r", 1) == 0 && option_f == 0)
      range = &s[1];
    else if (strncmp(s, "-range=", 7) == 0 && option_f == 0)
      range = &s[7];
    else if (strncmp(s, "s", 1) == 0 && option_f == 0)
      option_s = 1;
    else if (strcmp(s, "-sorted") == 0 && option_f == 0)
      option_s = 1;
    else if (strcmp(s, "f") == 0)
      option_f = 1;
    else if (strcmp(s, "-file") == 0)
      option_f = 1;
    else {
      usage();
      return 1; }

  if (option_f == 0) {
    if ((s = GetFilename(cmdarg)) == NULL)
      fp = stdin;
    else if (GetFilename(cmdarg) != NULL) {
      usage();
      return 1; }
    else if ((fp = fopen(s, "r")) == NULL) {
      usage();
      return 1; }
  }
  else {
    if ((s = GetFilename(cmdarg)) == NULL) {
      usage();
      return 1; }
    else if ((fp = fopen(s, "r")) == NULL) {
      fprintf(stderr, "grepcol: Error in opening file '%s'\n", s);
      return 1; }
    else if ((range = GetFile(fp)) == NULL) {
      fprintf(stderr, "grepcol: Error in reading range file '%s'\n", s);
      return 1; }
    else if ((s = GetFilename(cmdarg)) == NULL)
      fp = stdin;
    else if (GetFilename(cmdarg) != NULL) {
      usage();
      return 1; }
    else if ((fp = fopen(s, "r")) == NULL) {
      fprintf(stderr, "grepcol: Error in opening file '%s'\n", s);
      return 1; }
  }

  header = MakeHeader();
  entry = MakeEntry();

  if (ReadHeader(fp, header) != 0)
    return 1;

  AddHeaderInfo(header, argc, argv);

  PrintHeader(stdout, header);

  for (i = 1; (read_error = ReadEntryText(fp, entry)) == 0; i++)
    if ((option_s == 0 && !inrange(entry, i, range)) ||
	(option_s == 1 && !inorderrange(entry, i, range, &order_ptr))) {
      while ((read_error = ReadEntryLines(fp, entry, 1)) == 0)
	;
      if (read_error == 1)
	break;  /* An error has occurred */
    }
    else {
      PrintEntryText(stdout, entry);
      while ((read_error = ReadEntryLines(fp, entry, 1)) == 0)
	PrintEntryLines(stdout, entry);
      
      if (read_error == 1)
	break;  /* An error has occurred */
      
      PrintEntryEnd(stdout, entry);
    }

  if (fp != stdin && fclose(fp) != 0) {
    fprintf(stderr, "grepcol: Error in closing file\n");
    return 1; }

  if (read_error == 1)
    return 1;

  return 0;
}

void usage(void)
{
  fprintf(stderr,
	  "usage: grepcol\n"
	  "            [-s | --sorted]  [-r<range> | -range=<range> | -f |\n"
	  "            --file] [<rangefile>] [<file>]\n");
}

/* returns true if num is in range */
int inrange(Entry *entry, int num, char *range)
{
  int ptr, ptr2;
  int low, high;
  int i;
  char s[MAXCOLW];
  char name[MAXCOLW];
  int option_name;

  ptr = 0;

  while (range[ptr] != '\0') {
    for (ptr2 = ptr; range[ptr2] != '\0' && range[ptr2] != ',' &&
	             range[ptr2] != '\n'; ptr2++)
      ;
    option_name = 0;
    if (sscanf(&range[ptr], " %d - %d", &low, &high) != 2) {
      if (sscanf(&range[ptr], " %d", &low) == 1)
	high = low;
      else {
	for (i = 0; range[ptr+i] != ',' && range[ptr+i] != '\0'
	       && range[ptr+i] != '\n'; i++)
	  s[i] = range[ptr+i];
	s[i] = '\0';
	option_name = 1;
      }
    }
    if (option_name == 1 && ReadText(entry, "ENTRY", MAXCOLW, name) == 0) {
      if (strcmp(s, name) == 0)
	return 1;
    }
    else if (low <= num && num <= high)
      return 1;
    ptr = ptr2;
    if (range[ptr] == ',' || range[ptr] == '\n')
      ptr++;
  }

  return 0;
}

/* returns true if num is in ordered range */
int inorderrange(Entry *entry, int num, char *range, int *ptr)
{
  int ptr2;
  int low, high;
  int i;
  char s[MAXCOLW];
  char name[MAXCOLW];
  int option_name;

  while (range[*ptr] != '\0') {
    for (ptr2 = *ptr; range[ptr2] != '\0' && range[ptr2] != ',' &&
	   range[ptr2] != '\n'; ptr2++)
      ;
    option_name = 0;
    if (sscanf(&range[*ptr], " %d - %d", &low, &high) != 2) {
      if (sscanf(&range[*ptr], " %d", &low) == 1)
	high = low;
      else {
	for (i = 0; range[*ptr+i] != ',' && range[*ptr+i] != '\0'
	       && range[*ptr+i] != '\n'; i++)
	  s[i] = range[*ptr+i];
	s[i] = '\0';
	option_name = 1;
      }
    }
    if (option_name == 1 && ReadText(entry, "ENTRY", MAXCOLW, name) == 0) {
      if (strcmp(s, name) == 0) {
	*ptr = ptr2;
	if (range[*ptr] == ',' || range[*ptr] == '\n')
	  (*ptr)++;
	return 1;
      }
      else
	return 0;
    }
    if (low <= num && num <= high)
      return 1;
    else if (num <= high)
      return 0;
    *ptr = ptr2;
    if (range[*ptr] == ',' || range[*ptr] == '\n')
      (*ptr)++;
  }

  return 0;
}
