/*********************************************************************

  line2col.c

  usage: line2col [FILE]

  This program transforms a line file to col format. See col2line.c.

  000125 Bjarne Knudsen (bk@daimi.au.dk)

  Copyright (C) 2000 Bjarne Knudsen

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

*********************************************************************/

#include "../clib/col.h"
#include "../clib/file.h"

int main(int argc, char **argv)
{
  FILE *fp;         /* For input */
  int sym_col, seqpos_col, alignpos_col;
  Header *header;
  Entry *entry;
  int read_error;   /* For keeping track of errors in reading entries */
  char *s;
  CmdArg *cmdarg;   /* Command line arguments */
  int numf;         /* Number of fields */
  int *width;       /* Field widths */
  char **line;
  int *ptr;
  char **fmts, **fmtd;
  char **field;
  char fld[MAXCOLW];
  int i, j;
  int len;
  int seqpos;
  int w;

  cmdarg = InitArgument(argc, argv);

  if ((s = GetArgument(cmdarg)) != NULL) {
    fprintf(stderr, "Usage: line2col [FILE]\n");
    return 1; }

  if ((s = GetFilename(cmdarg)) == NULL)
    fp = stdin;
  else if (GetFilename(cmdarg) != NULL) {
    fprintf(stderr, "Usage: line2col [FILE]\n");
    return 1; }
  else if ((fp = fopen(s, "r")) == NULL) {
    fprintf(stderr, "line2col: Error in opening file '%s'\n", s);
    return 1; }

  header = MakeHeader();
  entry = MakeEntry();

  if (ReadHeader(fp, header) != 0)
    return 1;
  PrintHeader(stdout, header);

  line = NULL;
  width = NULL;
  ptr = NULL;
  fmts = NULL;
  fmtd = NULL;
  field = NULL;

  while ((read_error = ReadEntry(fp, entry)) == 0) {
    if ((sym_col = ReadColno(entry, "residue")) != 0 ||
	(sym_col = ReadColno(entry, "nucleotide")) != 0 ||
	(sym_col = ReadColno(entry, "aminoacid")) != 0 ||
	(sym_col = ReadColno(entry, "code")) != 0)
      seqpos_col = ReadColno(entry, "seqpos");
    else
      seqpos_col = 0;
    alignpos_col = ReadColno(entry, "alignpos");

    PrintEntryText(stdout, entry);

    /* Find length from the right line */
    if (sym_col != 0)
      numf = CountField(entry, sym_col);
    else if (alignpos_col == 1)
      numf = CountField(entry, 2);
    else
      numf = CountField(entry, 1);

    len = EntryLength(entry);
    line = (char **)realloc(line, len * sizeof(char *));
    width = (int *)realloc(width, len * sizeof(int));
    ptr = (int *)realloc(ptr, len * sizeof(int));
    fmts = (char **)realloc(fmts, len * sizeof(char *));
    for (j = 1; j <= len; j++)
      fmts[j-1] = (char *)malloc(20 * sizeof(char));
    fmtd = (char **)realloc(fmtd, len * sizeof(char *));
    for (j = 1; j <= len; j++)
      fmtd[j-1] = (char *)malloc(20 * sizeof(char));

    field = (char **)realloc(field, len * sizeof(char *));
    for (j = 1; j <= len; j++)
      field[j-1] = (char *)malloc(MAXCOLW * sizeof(char));

    for (j = 1; j <= len; j++) {
      line[j-1] = LineAddress(entry, j);
      ptr[j-1] = 0;
      GetField(fld, entry, j, 1);
      width[j-1] = atoi(fld);
      if (j == alignpos_col && CountField(entry, j) != 1)
	alignpos_col = 0;  /* For wrong alignpos columns */
      else if (j == seqpos_col && CountField(entry, j) != 1)
	seqpos_col = 0;
      sprintf(fmts[j-1], " %%%ds", width[j-1]-1);
      sprintf(fmtd[j-1], " %%%dd", width[j-1]-1);
    }

    seqpos = 0;
    for (i = 1; i <= numf; i++) {
      /* move pointers */
      for (j = 1; j <= len; j++)
	if (j != seqpos_col && j != alignpos_col) {
	  sscanf(&line[j-1][ptr[j-1]], " %s%n", field[j-1], &w);
	  ptr[j-1] += w;
	}
      if (i == 1)
	continue; /* The col widths was just read */
      for (j = 1; j <= len; j++) {
	if (j == seqpos_col) {
	  if (strcmp(field[sym_col-1], "-") != 0) {
	    seqpos++;
	    printf(fmtd[j-1], seqpos);
	  }
	  else
	    printf(fmts[j-1], ".");
	}
	else if (j == alignpos_col)
	  printf(fmtd[j-1], i-1);
	else
	  printf(fmts[j-1], field[j-1]);
      }
      printf("\n");
    }
    PrintEntryEnd(stdout, entry);

    for (j = 1; j <= len; j++) {
      free(fmts[j-1]);
      free(fmtd[j-1]);
      free(field[j-1]);
    }
  }

  if (fp != stdin && fclose(fp) != 0) {
    fprintf(stderr, "line2col: Error in closing file\n");
    return 1; }

  if (read_error == 1)
    return 1;

  return 0;
}
