/*********************************************************************

  shiftchk.c

  usage: shiftchk [-nNUMBER] [FILE]

  This prgoram colours basepairs in a way that helps determining
  whether an alignment is structural.

  NUMBER is the number of colours used.

  000209 Bjarne Knudsen (bk@daimi.au.dk)

  Copyright (C) 2000 Bjarne Knudsen

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

*********************************************************************/

#include "../clib/col.h"
#include "../clib/file.h"

typedef struct tagColno {
  int align_bp, alignpos;
  int seq_bp, seqpos;
  int nuc;
  int red, green, blue;
  int red2, green2, blue2;
} Colno;

void usage(void);
void shiftchk(Entry *entry, Colno *colno, int numcolor);

int main(int argc, char **argv)
{
  FILE *fp;         /* For input */
  Header *header;
  Entry *entry;
  Colno *colno;     /* Column numbers */
  int read_error;   /* For keeping track of errors in reading entries */
  unsigned int len; /* Used with -n option */
  CmdArg *cmdarg;   /* Command line arguments */
  char *s;          /* String for arguments */
  int numcolor;     /* Number of colors used */

  colno = (Colno *) malloc(sizeof(Colno));

  cmdarg = InitArgument(argc, argv);

  numcolor = 3;

  while ((s = GetArgument(cmdarg)) != NULL)
    if (strncmp(s, "n", 1) == 0) {
      if (sscanf(&s[1], "%d%n", &numcolor, &len) != 1 ||
	  len+1 != strlen(s)) {
	usage();
	return 1; }
    }
  else if (strncmp(s, "-number=", 8) == 0) {
      if (sscanf(&s[8], "%d%n", &numcolor, &len) != 1 ||
	  len+8 != strlen(s)) {
	usage();
	return 1; }
    }
    else {
      usage();
      return 1; }

  if (numcolor > 6) {
    fprintf(stderr, "shiftchk: Warning: only using 6 colours\n");
    numcolor = 6;
  }
  
  if ((s = GetFilename(cmdarg)) == NULL)
    fp = stdin;
  else if (GetFilename(cmdarg) != NULL) {
    usage();
    return 1; }
  else if ((fp = fopen(s, "r")) == NULL) {
    fprintf(stderr, "shiftchk: Error in opening file '%s'\n", s);
    return 1; }

  header = MakeHeader();
  entry = MakeEntry();

  if (ReadHeader(fp, header) != 0)
    return 1;

  AddHeaderInfo(header, argc, argv);

  PrintHeader(stdout, header);

  while ((read_error = ReadEntry(fp, entry)) == 0) {
    if (!ReadType(entry, "RNA")) {
      PrintEntry(stdout, entry);
      continue;
    }
    if ((colno->nuc = ReadColno(entry, "residue")) == 0)
      colno->nuc = ReadColno(entry, "nucleotide");
    colno->align_bp = ReadColno(entry, "align_bp");
    colno->alignpos = ReadColno(entry, "alignpos");
    colno->seq_bp = ReadColno(entry, "seq_bp");
    colno->seqpos = ReadColno(entry, "seqpos");
    if (colno->nuc == 0 ||
	((colno->align_bp == 0 || colno->alignpos == 0) &&
	 (colno->seq_bp == 0 || colno->seqpos == 0))) {
      fprintf(stderr, "shiftchk: Warning: Ignoring sequence,"
	                " insufficient column info\n");
      PrintEntry(stdout, entry);
      continue; }

    colno->red = EnsureCol(entry, "color_r", "1.000");
    colno->green = EnsureCol(entry, "color_g", "1.000");
    colno->blue = EnsureCol(entry, "color_b", "1.000");

    shiftchk(entry, colno, numcolor);

    PrintEntry(stdout, entry);
  }

  if (fp != stdin && fclose(fp) != 0) {
    fprintf(stderr, "shiftchk: Error in closing file\n");
    return 1; }

  if (read_error == 1)
    return 1;

  return 0;
}

void usage(void)
{
  fprintf(stderr, "Usage: shiftchk [-n<number> | --number=<number>] [FILE]\n");
}

void shiftchk(Entry *entry, Colno *colno, int numcolor)
{
  int i;
  int pair;
  int len;
  double color_r[] = {1, 0, 0, 1, 1, 0};
  double color_g[] = {0, 1, 0, 0, 1, 1};
  double color_b[] = {0, 0, 1, 1, 0, 1};

  len = EntryLength(entry);

  for (i = 1; i <= len; i++) {
    pair = FindPair(entry, i, colno->align_bp, colno->alignpos,
		    colno->seq_bp, colno->seqpos);
    
    if (pair != 0) {  /* A pair is present */
      if (i < pair) {
	ChgField(entry, i, colno->red, "%5.3f", color_r[i%numcolor]);
	ChgField(entry, i, colno->green, "%5.3f", color_g[i%numcolor]);
	ChgField(entry, i, colno->blue, "%5.3f", color_b[i%numcolor]);
      }
      else {
	ChgField(entry, i, colno->red, "%5.3f", color_r[pair%numcolor]);
	ChgField(entry, i, colno->green, "%5.3f", color_g[pair%numcolor]);
	ChgField(entry, i, colno->blue, "%5.3f", color_b[pair%numcolor]);
      }
    }
  }
}
