/*********************************************************************

  colorpos.c

  usage: colorpos
              [-r<range> | -range=<range>] [-s<seqrange> |
              -seq=<seqrange>] [-c | --color |
              -C<color_r>,<color_g>,<color_b> |
              --color=<color_r>,<color_g>,<color_b>] [-f | --fgcolor |
              -F<color_r>,<color_g>,<color_b> |
              --fgcolor=<color_r>,<color_g>,<color_b>] [<file>]

  See man page for more info.

  000928 Bjarne Knudsen (bk@daimi.au.dk)

  Copyright (C) 2000 Bjarne Knudsen

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

*********************************************************************/

#include "../clib/col.h"
#include "../clib/file.h"

void usage(void);
int inrange(int num, char *range);
int inseqrange(Entry *entry, int num, char *range);

int main(int argc, char **argv)
{
  FILE *fp;         /* For input */
  int i, j;
  int len;
  Header *header;
  Entry *entry;
  int num_col;
  int red_col, green_col, blue_col;
  int red_col2, green_col2, blue_col2;
  int read_error;   /* For keeping track of errors in reading entries */
  CmdArg *cmdarg;   /* Command line arguments */
  char *s;          /* String for arguments */
  double color_r, color_g, color_b;
                    /* The colour of non-standard pairs */
  double fg_color_r, fg_color_g, fg_color_b;
                    /* The colour of non-standard pairs */
  int option_fg;
  int option_c;
  char *range, *seqrange;
  char field[MAXCOLW];
  int pos;

  /* default options */
  option_c = 0;
  color_r = 1.0;
  color_g = 0.0;
  color_b = 0.0;

  option_fg = 0;
  fg_color_r = 1.0;
  fg_color_g = 0.0;
  fg_color_b = 0.0;

  cmdarg = InitArgument(argc, argv);
  range = NULL;
  seqrange = NULL;

  while ((s = GetArgument(cmdarg)) != NULL)
    if (strncmp(s, "r", 1) == 0)
      range = &s[1];
    else if (strncmp(s, "-range=", 7) == 0)
      range = &s[7];
    else if (strncmp(s, "s", 1) == 0)
      seqrange = &s[1];
    else if (strncmp(s, "-seq=", 5) == 0)
      seqrange = &s[5];
    else if (strcmp(s, "c") == 0)
      option_c = 1;
    else if (strcmp(s, "-color") == 0)
      option_c = 1;
    else if (strcmp(s, "f") == 0)
      option_fg = 1;
    else if (strcmp(s, "-fgcolor") == 0)
      option_fg = 1;
    else if (strncmp(s, "C", 1) == 0) {
      option_c = 1;
      if (sscanf(&s[1], "%lf,%lf,%lf%n",
		 &color_r, &color_g, &color_b, &len) != 3 ||
	  len+1 != strlen(s)) {
	usage();
	return 1; }
    }
    else if (strncmp(s, "-color=", 7) == 0) {
      option_c = 1;
      if (sscanf(&s[7], "%lf,%lf,%lf%n",
		 &color_r, &color_g, &color_b, &len) != 3 ||
	  len+7 != strlen(s)) {
	usage();
	return 1; }
    }
    else if (strncmp(s, "F", 1) == 0) {
      option_fg = 1;
      if (sscanf(&s[1], "%lf,%lf,%lf%n",
		 &fg_color_r, &fg_color_g, &fg_color_b, &len) != 3 ||
	  len+1 != strlen(s)) {
	usage();
	return 1; }
    }
    else if (strncmp(s, "-fgcolor=", 9) == 0) {
      option_fg = 1;
      if (sscanf(&s[9], "%lf,%lf,%lf%n",
		 &fg_color_r, &fg_color_g, &fg_color_b, &len) != 3 ||
	  len+9 != strlen(s)) {
	usage();
	return 1; }
    }
    else {
      usage();
      return 1; }

  if (option_c == 0 && option_fg == 0)
    option_c = 1;

  if ((s = GetFilename(cmdarg)) == NULL)
    fp = stdin;
  else if (GetFilename(cmdarg) != NULL) {
    usage();
    return 1; }
  else if ((fp = fopen(s, "r")) == NULL) {
    fprintf(stderr, "colorpos: Error in opening file '%s'\n", s);
    return 1; }

  header = MakeHeader();
  entry = MakeEntry();

  if (ReadHeader(fp, header) != 0)
    return 1;

  AddHeaderInfo(header, argc, argv);

  PrintHeader(stdout, header);

  for (j = 1; (read_error = ReadEntry(fp, entry)) == 0; j++) {
    if (seqrange != NULL && !inseqrange(entry, j, seqrange)) {
      PrintEntry(stdout, entry);
      continue;
    }

    num_col = ReadColno(entry, "alignpos");
    if (num_col == 0) 
      num_col = ReadColno(entry, "seqpos");
    if (option_c == 1) {
      red_col = EnsureCol(entry, "color_r", "1.000");
      green_col = EnsureCol(entry, "color_g", "1.000");
      blue_col = EnsureCol(entry, "color_b", "1.000");
    }
    if (option_fg == 1) {
      red_col2 = EnsureCol(entry, "color2_r", "0.000");
      green_col2 = EnsureCol(entry, "color2_g", "0.000");
      blue_col2 = EnsureCol(entry, "color2_b", "0.000");
    }
    len = EntryLength(entry);

    for (i = 1; i <= len; i++) {
      if (num_col == 0)
	pos = i;
      else {
	GetField(field, entry, i, num_col);
	pos = atoi(field);
      }

      if (range == NULL || inrange(pos, range)) {
	if (option_c == 1) {
	  ChgField(entry, i, red_col, "%5.3f", color_r);
	  ChgField(entry, i, green_col, "%5.3f", color_g);
	  ChgField(entry, i, blue_col, "%5.3f", color_b);
	}
	if (option_fg == 1) {
	  ChgField(entry, i, red_col2, "%5.3f", fg_color_r);
	  ChgField(entry, i, green_col2, "%5.3f", fg_color_g);
	  ChgField(entry, i, blue_col2, "%5.3f", fg_color_b);
	}
      }
    }

    PrintEntry(stdout, entry);
  }

  if (fp != stdin && fclose(fp) != 0) {
    fprintf(stderr, "colorpos: Error in closing file\n");
    return 1; }

  if (read_error == 1)
    return 1;

  return 0;
}

/* returns true if num is in range */
int inrange(int num, char *range)
{
  int ptr, ptr2;
  int low, high;

  ptr = 0;

  while (range[ptr] != '\0') {
    for (ptr2 = ptr; range[ptr2] != '\0' && range[ptr2] != ',' &&
                     range[ptr2] != '\n'; ptr2++)
      ;
    if (sscanf(&range[ptr], " %d - %d", &low, &high) != 2) {
      if (sscanf(&range[ptr], " %d", &low) == 1)
        high = low;
      else
        continue;  /* Just continue if error in range */
    }
    if (low <= num && num <= high)
      return 1;
    ptr = ptr2;
    if (range[ptr] == ',' || range[ptr] == '\n')
      ptr++;
  }

  return 0;
}

void usage(void)
{
  fprintf(stderr, "usage: colorpos\n"
	  "            [-r<range> | -range=<range>] [-s<seqrange> |\n"
	  "            -seq=<seqrange>] [-c | --color |\n"
	  "            -C<color_r>,<color_g>,<color_b> |\n"
	  "            --color=<color_r>,<color_g>,<color_b>] [-f | --fgcolor |\n"
	  "            -F<color_r>,<color_g>,<color_b> |\n"
	  "            --fgcolor=<color_r>,<color_g>,<color_b>] [<file>]\n");
}

/* returns true if num is in range */
int inseqrange(Entry *entry, int num, char *range)
{
  int ptr, ptr2;
  int low, high;
  int i;
  char s[MAXCOLW];
  char name[MAXCOLW];
  int option_name;

  ptr = 0;

  while (range[ptr] != '\0') {
    for (ptr2 = ptr; range[ptr2] != '\0' && range[ptr2] != ',' &&
	             range[ptr2] != '\n'; ptr2++)
      ;
    option_name = 0;
    if (sscanf(&range[ptr], " %d - %d", &low, &high) != 2) {
      if (sscanf(&range[ptr], " %d", &low) == 1)
	high = low;
      else {
	for (i = 0; range[ptr+i] != ',' && range[ptr+i] != '\0'
	       && range[ptr+i] != '\n'; i++)
	  s[i] = range[ptr+i];
	s[i] = '\0';
	option_name = 1;
      }
    }
    if (option_name == 1 && ReadText(entry, "ENTRY", MAXCOLW, name) == 0)
      if (strcmp(s, name) == 0)
	return 1;
    if (low <= num && num <= high)
      return 1;
    ptr = ptr2;
    if (range[ptr] == ',' || range[ptr] == '\n')
      ptr++;
  }

  return 0;
}

