#!/usr/bin/perl -w

use strict;
use warnings;

use GD::Simple;
use Getopt::Long;
use POSIX qw(strftime);

############################################################################################################
## parse input options
use vars qw($fId $sId $fFile $sFile $progDir $outDir $tmpFilePrefix $keepAlnFile $htmlOutput $htmlImgPath $textOutput $fTitle $sTitle $help);
use vars qw($param_distance $param_block $param_gap $Ci $Ce $DELTA $S0 $S1 $ZETA);
$outDir=".";
$htmlImgPath=".";
$tmpFilePrefix="";
$fTitle="first";
$sTitle="second";
$param_distance=1;
$param_block=1;
$param_gap=-1;
$Ci=-2;
$Ce=-1;
$DELTA=1;
$S0=1;
$S1=-1;
$ZETA=1;

GetOptions ("i=s"  => \$fFile,
	          "j=s"  => \$sFile,
            "f:s"  => \$fId,
            "s:s"  => \$sId,
            "p:s"  => \$progDir,
            "o:s"  => \$outDir,
            "x:s"  => \$tmpFilePrefix,
            "l"    => \$htmlOutput,
            "a=s"  => \$htmlImgPath,
						"q"    => \$textOutput,
            "k"    => \$keepAlnFile,
            "v=s"  => \$fTitle,
            "w=s"  => \$sTitle,
            "d:f"  => \$param_distance,
            "b:f"  => \$param_block,
            "g:f"  => \$param_gap,
            "c:i"  => \$Ci,
            "e:i"  => \$Ce,
            "t:f"  => \$DELTA,
            "m:i"  => \$S0,
            "n:i"  => \$S1,
            "z:i"  => \$ZETA,
            "help" => \$help,
            "h"    => \$help);

usage() if($help || !$fFile || !$sFile);

############################################################################################################
sub usage {
	print STDERR "\nProgram: plotdeepBlockAlign.pl (align and visualize the alignment of read profiles)\n";
	print STDERR "Author: University of Copenhagen, Denmark\n";
	print STDERR "Version: 1.0\n";
	print STDERR "Contact: sachin\@rth.dk\n";
	print STDERR "Usage: plotdeepBlockAlign.pl -i <file> -j <file> [OPTIONS]\n";
	print STDERR " -i <file>     [first block group file]\n";
	print STDERR " -j <file>     [second block group file]\n";
	print STDERR " -f <id>       [first block group id]\n";
	print STDERR " -s <id>       [second block group id]\n";
	print STDERR " -p <dir>      [directory with programs (deepblockalign.x etc)]\n";
	print STDERR " -o <dir>      [directory with output files]\n";
	print STDERR " -x <string>   [prefix for temporary files, useful when running on queue]\n";
	print STDERR " -l            [output in HTML format]\n";
	print STDERR " -a <string>   [path to hyperlink alignment images from the HTML format]\n";
	print STDERR " -q            [output in plain text format]\n";
	print STDERR " -k            [keep the block and block group alignment files]\n";
	print STDERR " -v <string>   [first block group title (default: first)]\n";
	print STDERR " -w <string>   [second block group title (default: second)]\n";
	print STDERR "[OPTIONS: Block Group Alignment]\n";
	print STDERR " -d <double>   [distance weight (default: 1.0)]\n";
	print STDERR " -b <double>   [block similarity weight (default: 1.0)]\n";
	print STDERR " -g <double>   [gap penalty (default: -1.0)]\n";
	print STDERR "[OPTIONS: Block Alignment]\n";
	print STDERR " -c  <int>     [gap initialization penalty (default: -2)]\n";
	print STDERR " -e  <int>     [gap extension penalty (default: -1)]\n";
	print STDERR " -t  <double>  [threshold for expression difference between two blocks (default: 1.0)]\n";
	print STDERR " -m  <int>     [match score (default: 1)]\n";
	print STDERR " -n  <int>     [mismatch score (default: -1)]\n";
	print STDERR " -z  <int>     [profile shape difference penalty (default: 1)]\n\n";
	exit(-1);
}
############################################################################################################
## check, if output directory exist
unless( -d $outDir) { system("mkdir $outDir"); }

## grep all block group ids from first bg file, if first bg id is not predefined
my @fIds=();
if(!$fId) { @fIds=`zgrep "^>" $fFile | cut -f 1 | sed 's/\>//g'`; }
else { push(@fIds, $fId); }

## grep all block group ids from second bg file, if second bg id is not predefined
my @sIds=();
if(!$fId) { @sIds=`zgrep "^>" $sFile | cut -f 1 | sed 's/\>//g'`; }
else { push(@sIds, $sId); }

my $ALIGNMENT_COUNTER=0;
my $scale_value=();

foreach(@fIds) {
chomp($_);
$fId=$_;

foreach(@sIds) {
chomp($_);
$sId=$_;

$ALIGNMENT_COUNTER++;

my $i=(); my %fInfo=(); my %sInfo=();

print STDERR "# plotdeepBlockAlign started at ".strftime("%a, %d %b %Y %H:%M:%S %z", localtime(time())) . "\n";
print STDERR "# $fTitle block group: $fId\n";
print STDERR "# $fTitle file: $fFile\n";
print STDERR "# $sTitle block group: $sId\n";
print STDERR "# $sTitle file: $sFile\n";

## retrieve first block group information
if(my @t=split(/\s+/, `zgrep -w '$fId' $fFile`)) {
	system("zless $fFile | grep -w $fId -A $t[6] > $outDir/$tmpFilePrefix"."first.bg");
	$fInfo{'size'} = $t[3]-$t[2];
	$fInfo{'chr'} = $t[1];
	$fInfo{'start'} = $t[2];
	$fInfo{'end'} = $t[3];
	$fInfo{'strand'} = $t[4];
	if($fInfo{'strand'}=~/\+/) { $fInfo{'genomicStart'} = $t[2]; }
	else { $fInfo{'genomicStart'} = $t[3];	}
	$fInfo{'reads'} = sprintf("%0.2f", $t[5]);
	$fInfo{'tags'} = $t[6];
	$fInfo{'blocks'} = $t[7];
	$fInfo{'name'} = "$t[8]_$t[9]";
	$fInfo{'type'} = $t[9];
	$fInfo{'loci'} = $t[10];

	## compute block group statistics
	my $t="$outDir/$tmpFilePrefix"."first.bg";
	if(defined($progDir)) {
		($fInfo{'entropy'}, $fInfo{'uniqReads'}) = split(/\s+/, `$progDir/compBlockStat.pl -c $t -p 3 | grep -v \"^\#\" | cut -f 4,9`);
	}
	else {
		($fInfo{'entropy'}, $fInfo{'uniqReads'}) = split(/\s+/, `compBlockStat.pl -c $t -p 3 | grep -v \"^\#\" | cut -f 4,9`);
	}
	$fInfo{'uniqReads'} = sprintf("%0.0f", ($fInfo{'uniqReads'}/$fInfo{'reads'})*100);
}
else { print STDERR "Unable to retrieve $fId from $fFile\n\n"; exit; }

## retrieve second block group information
if(my @t=split(/\s+/, `zgrep -w '$sId' $sFile`)) {
	system("zless $sFile | grep -w $sId -A $t[6] > $outDir/$tmpFilePrefix"."second.bg");
	$sInfo{'size'} = $t[3]-$t[2];
	$sInfo{'chr'} = $t[1];
	$sInfo{'start'} = $t[2];
	$sInfo{'end'} = $t[3];
	$sInfo{'strand'} = $t[4];
	if($sInfo{'strand'}=~/\+/) { $sInfo{'genomicStart'} = $t[2]; }
	else { $sInfo{'genomicStart'} = $t[3];	}
	$sInfo{'reads'} = sprintf("%0.2f", $t[5]);
	$sInfo{'tags'} = $t[6];
	$sInfo{'blocks'} = $t[7];
	$sInfo{'name'} = "$t[8]_$t[9]";
	$sInfo{'type'} = $t[9];
	$sInfo{'loci'} = $t[10];

	## compute block group statistics
	my $t="$outDir/$tmpFilePrefix"."second.bg";
	if(defined($progDir)) {
		($sInfo{'entropy'}, $sInfo{'uniqReads'}) = split(/\s+/, `$progDir/compBlockStat.pl -c $t -p 3 | grep -v \"^\#\" | cut -f 4,9`);
	}
	else {
		($sInfo{'entropy'}, $sInfo{'uniqReads'}) = split(/\s+/, `compBlockStat.pl -c $t -p 3 | grep -v \"^\#\" | cut -f 4,9`);
	}
	$sInfo{'uniqReads'} = sprintf("%0.0f", ($sInfo{'uniqReads'}/$sInfo{'reads'})*100);
}
else { print STDERR "Unable to retrieve $sId from $sFile\n\n"; exit; }

## peform and store the alignment in an alignment file
print STDERR "computing alignment....";
my %DBA=();
$DBA{'bgFile1'}="$outDir/$tmpFilePrefix"."first.bg";
$DBA{'bgFile2'}="$outDir/$tmpFilePrefix"."second.bg";
$DBA{'alnFile'}="$outDir/$tmpFilePrefix"."DBA_alnFile";

if(defined($progDir)) {
	$DBA{'alnScore'} = `$progDir/deepBlockAlign.x -q $DBA{'bgFile1'} -s $DBA{'bgFile2'} -f $DBA{'alnFile'} -d $param_distance -b $param_block -g $param_gap -c $Ci -e $Ce -t $DELTA -m $S0 -n $S1 -z $ZETA | grep -v "^\#" | cut -f 3`;
}
else {
	$DBA{'alnScore'} = `deepBlockAlign.x -q $DBA{'bgFile1'} -s $DBA{'bgFile2'} -f $DBA{'alnFile'} -d $param_distance -b $param_block -g $param_gap -c $Ci -e $Ce -t $DELTA -m $S0 -n $S1 -z $ZETA | grep -v "^\#" | cut -f 3`;
}
$DBA{'alnScore'}=~s/\s+//g;
if($DBA{'alnScore'}=~/^$/) {
	print STDERR "\nError: no alignment computed for $DBA{'bgFile1'} and $DBA{'bgFile2'}\n";
	exit(-1);
}
else {
	$DBA{'alnScore'}=sprintf("%0.2f", $DBA{'alnScore'});
}
print STDERR "done (score: $DBA{'alnScore'})\n";

print STDERR "aligning blocks...";
## read the block group alignment file
open(INFILE, $DBA{'alnFile'}) || die $!;
my @aln=<INFILE>; close INFILE;

## determine the aligned blocks (paired and unpaired)
my %alnBlocks=();
foreach(grep { $_=~/^ALIGNMENT/ } @aln) {
	$_=~s/^ALIGNMENT//; $_=~s/\(.+//g; $_=~s/\s+//g;
	## parse paired block alignment
	if($_=~/\:/) {
		my @t=split("[\-\:]+", $_); 
		$alnBlocks{$t[0]}=$t[2];
		$alnBlocks{$t[1]}=$t[3];
	}
	## parse unpaired block alignment
	else { 
		my @t=split("[\-]", $_);
		$alnBlocks{$t[0]}=$t[1];
	}
}

#foreach(keys(%alnBlocks)) { print "$_\t$alnBlocks{$_}\n"; }

## determine shift required on x-axis for plotting the block group alignment
my $ANCHOR_BLOCK_F=(); my $ANCHOR_BLOCK_S=(); my $BG_TO_SHIFT=();
foreach(sort { $alnBlocks{$a} <=> $alnBlocks{$b} } keys(%alnBlocks)) {
	$ANCHOR_BLOCK_F=$_;
	$ANCHOR_BLOCK_S=$alnBlocks{$_};
	
	if($ANCHOR_BLOCK_F >= $ANCHOR_BLOCK_S) { $BG_TO_SHIFT="second"; }
	else { $BG_TO_SHIFT="first"; }
	last;
}

@{$fInfo{'coor'}} = split('\s+', $aln[scalar(@aln)-2]);
@{$sInfo{'coor'}} = split('\s+', $aln[scalar(@aln)-1]);

## derive anchor coordinate for plotting blocks
$fInfo{'blockShift'}=0; $sInfo{'blockShift'}=0;
if($BG_TO_SHIFT=~/second/) {
	$fInfo{'coor'}[$ANCHOR_BLOCK_F-1]=~m/^[0-9]+/;
	$fInfo{'anchorCoor'}=$&;
	$sInfo{'coor'}[$ANCHOR_BLOCK_S-1]=~m/^[0-9]+/;
	$sInfo{'anchorCoor'}=$&;
	$sInfo{'blockShift'} = $fInfo{'anchorCoor'}-$sInfo{'anchorCoor'};
}
elsif($BG_TO_SHIFT=~/first/) {
	$fInfo{'coor'}[$ANCHOR_BLOCK_F-1]=~m/^[0-9]+/;
	$fInfo{'anchorCoor'}=$&;
	$sInfo{'coor'}[$ANCHOR_BLOCK_S-1]=~m/^[0-9]+/;
	$sInfo{'anchorCoor'}=$&;
	$fInfo{'blockShift'} = $sInfo{'anchorCoor'}-$fInfo{'anchorCoor'};
}

#print "$fInfo{'blockShift'}\t$sInfo{'blockShift'}\n";

## plot the block alignments
open(OUTFILE, ">$outDir/$tmpFilePrefix"."tmp") || die $!;
for($i=0; $i<scalar(@aln); $i++) { if($aln[$i]!~/^[0-9]+/) { last; } else { print OUTFILE $aln[$i]; } }
close OUTFILE;

## output files for block and block group alignment
my @PDF_B_ALIGN=(); my $PDF_BG_ALIGN=();

$i=0;
foreach(grep { $_!~/^[0-9]+/ && $_!~/^ALIGNMENT/ } @aln) {
	## pdf file name for block group alignment
	$_=~m/^[^\s]+/;
	$PDF_BG_ALIGN = $&; $PDF_BG_ALIGN=~s/\>//g; $PDF_BG_ALIGN=~s/\_[0-9]+\#/--/;
	$PDF_BG_ALIGN=~s/\_[0-9]\.pdf//; $PDF_BG_ALIGN=~s/\//-/g;

	## pdf file name for block alignment
	$_=~s/\>//g; $_=~s/\#/--/; $_=~s/\//-/g; $_=~m/^[^\s]+/;
	$PDF_B_ALIGN[$i]=$&;

	if(defined($progDir)) {
		system("R --no-save --vanilla --slave < $progDir/plotblockAlign.r --args $outDir/$tmpFilePrefix"."tmp $outDir/$tmpFilePrefix"."$_");
	}
	else {
		system("R --no-save --vanilla --slave < plotblockAlign.r --args $outDir/$tmpFilePrefix"."tmp $outDir/$tmpFilePrefix"."$_");
	}
	$i++;
}

## remove temporary file
system("rm $outDir/$tmpFilePrefix"."tmp");
print STDERR "done ("; my $tmp=(); foreach(@PDF_B_ALIGN) { $tmp.="$_,"; } $tmp=~s/\,$//g; print STDERR "$tmp)\n";

print STDERR "aligning block groups...";
## create the tex file to plot the block group alignment
my $ALIGNMENT_DESCRIPTION=$PDF_BG_ALIGN; $ALIGNMENT_DESCRIPTION=~s/\_/-/g;

($fInfo{'name'}, $sInfo{'name'}) = split(/\-\-/, $ALIGNMENT_DESCRIPTION);
$ALIGNMENT_DESCRIPTION=~s/\-\-/\\hspace{1mm}with\\hspace{1mm}/;

$i=(); my $x1=0; my $x2=0; my $y1=(); my $y2=(); my $max_x2=0; my $default_y1=5; my $max_y2=10;
my $mod_x1=(); my $mod_x2=(); my $mod_y1=(); my $mod_y2=(); my $nBlock=(); my $start_x=1000; my $end_x=0;
my %alnBlockCoor=();

open(OUTFILE, ">$outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.tex") || die $!;

print OUTFILE <<LATEX;
\\documentclass{standalone}
\\usepackage[hmargin=0.5cm, vmargin=1cm]{geometry}
\\usepackage{graphicx}
\\usepackage{color}
\\usepackage{tikz}
\\usepackage{subfigure}
\\usepackage{adjustbox}
\\definecolor{dark-red}{RGB}{100,0,0}
\\definecolor{dark-green}{RGB}{0,100,0}
\\definecolor{dark-blue}{RGB}{0,0,100}
\\begin{document}
\\centering
\\begin{adjustbox}{width=\\textwidth,height=\\textheight,keepaspectratio}
\\begin{tikzpicture}[scale=1]
LATEX

## draw blocks for first block group
for($i=0; $i<scalar(@{$fInfo{'coor'}}); $i++) {
	my($x1, $y1, $x2, $y2) = split("\:", $fInfo{'coor'}[$i]);
	$x1 = ($x1+$fInfo{'blockShift'});
	$x2 = ($x2+$fInfo{'blockShift'});

	if($x1 < $max_x2) {
		$y1 = $default_y1+5;
		print OUTFILE "\\draw [thick,white,fill=dark-red] ($x1 mm, $y1 mm) rectangle ($x2 mm, $y1 mm+5 mm);";
		$alnBlockCoor{'first'}{$i}{'start'}="$x1 mm,$y1+5 mm";
		$alnBlockCoor{'first'}{$i}{'end'}="$x2 mm,$y1+5 mm";

		$mod_x1 = $x1; $mod_x2 = ($x1+$x2)/2;
		$mod_y1 = ($y1+($y1+5))/2; $mod_y2 = ($y1+($y1+5))/2;
		$nBlock = $i+1;
		print OUTFILE "\\draw [white] ($mod_x1 mm, $mod_y1 mm)($mod_x2 mm, $mod_y2 mm) node {Block $nBlock};\n";

		$default_y1 = $y1;
	}
	else {
		print OUTFILE "\\draw [thick,white,fill=dark-red] ($x1 mm, 5 mm) rectangle ($x2 mm, 10 mm);";

		$alnBlockCoor{'first'}{$i}{'start'}="$x1 mm,10 mm";
		$alnBlockCoor{'first'}{$i}{'end'}="$x2 mm,10 mm";

		$mod_x1 = $x1; $mod_x2 = ($x1+$x2)/2;
		$mod_y1 = (5+10)/2; $mod_y2 = (5+10)/2;
		$nBlock = $i+1;
		print OUTFILE "\\draw [white] ($mod_x1 mm, $mod_y1 mm)($mod_x2 mm, $mod_y2 mm) node {Block $nBlock};\n";

		if($i==0) {
			print OUTFILE "\\draw [dark-red] (-40 mm, $mod_y1 mm)(-20 mm, $mod_y2 mm) node {$fTitle block group};\n";
		}

		$default_y1=5;
	}

	if($x1 < $start_x) { $start_x = $x1; }
	if($x2 > $end_x) { $end_x = $x2; }
	
	$max_x2 = $x2 if($x2 > $max_x2);
	$max_y2 = $y1+5 if($y1+5 > $max_y2);
}

## draw coordinates for first block group
print OUTFILE "\\draw ($start_x mm, 4 mm) -- coordinate (x axis mid) ($end_x mm, 4 mm);\n";
print OUTFILE "\\foreach \\x in {$start_x,".($start_x+20).",...,$end_x}\n";
print OUTFILE "\\draw (\\x mm, 4 mm) -- (\\x mm, 3 mm)\n";
print OUTFILE "node[anchor=north] {};\n";
for($i=$start_x; $i<=$end_x; $i+=20) {
	if($i==$start_x) {
		print OUTFILE "\\draw [black] (".($i-4)." mm, 2 mm) (".($i-4)." mm, 1 mm) node {$fInfo{'chr'}:$fInfo{'genomicStart'}};\n";
	}
	else {
		print OUTFILE "\\draw [black] ($i mm, 2 mm) ($i mm, 1 mm) node {$fInfo{'genomicStart'}};\n";
	}
	if($fInfo{'strand'}=~/\+/) { $fInfo{'genomicStart'} += 20; }
	else { $fInfo{'genomicStart'} -= 20; }
}

## draw blocks for second block group
$max_y2 = $max_y2+5;
$max_x2=0; $default_y1=$max_y2+5; $start_x=1000; $end_x=0;

for($i=0; $i<scalar(@{$sInfo{'coor'}}); $i++) {
	($x1, $y1, $x2, $y2) = split("\:", $sInfo{'coor'}[$i]);
	$x1 = ($x1+$sInfo{'blockShift'});
	$x2 = ($x2+$sInfo{'blockShift'});

	if($x1 < $max_x2) {
		$y1 = $default_y1+5;
		print OUTFILE "\\draw [thick,white,fill=dark-blue] ($x1 mm, $y1 mm) rectangle ($x2 mm, $y1 mm + 5 mm);";

		$alnBlockCoor{'second'}{$i}{'start'}="$x1 mm,$y1 mm";
		$alnBlockCoor{'second'}{$i}{'end'}="$x2 mm,$y1 mm";

		$mod_x1 = $x1; $mod_x2 = ($x1+$x2)/2;
		$mod_y1 = ($y1+($y1+5))/2; $mod_y2 = ($y1+($y1+5))/2;
		$nBlock = $i+1;
		print OUTFILE "\\draw [white] ($mod_x1 mm, $mod_y1 mm)($mod_x2 mm, $mod_y2 mm) node {Block $nBlock};\n";

		$default_y1 = $y1;
	}
	else {
		$default_y1 = $max_y2+5;
		print OUTFILE "\\draw [thick,white,fill=dark-blue] ($x1 mm, $default_y1 mm) rectangle ($x2 mm, $default_y1 mm + 5 mm);";

		$alnBlockCoor{'second'}{$i}{'start'}="$x1 mm,$default_y1 mm";
		$alnBlockCoor{'second'}{$i}{'end'}="$x2 mm,$default_y1 mm";

		$mod_x1 = $x1; $mod_x2 = ($x1+$x2)/2;
		$mod_y1 = ($default_y1+($default_y1+5))/2; $mod_y2 = ($default_y1+($default_y1+5))/2;
		$nBlock = $i+1;
		print OUTFILE "\\draw [white] ($mod_x1 mm, $mod_y1 mm)($mod_x2 mm, $mod_y2 mm) node {Block $nBlock};\n";

		if($i==0) {
			print OUTFILE "\\draw [dark-blue] (-40 mm, $mod_y1 mm)(-20 mm, $mod_y2 mm) node {$sTitle block group};\n";
		}
	}

	if($x1 < $start_x) { $start_x = $x1; }
	if($x2 > $end_x) { $end_x = $x2; }
	
	$max_x2 = $x2 if($x2 > $max_x2);
}

$default_y1 = $max_y2+4;

## draw coordinates for second block group
print OUTFILE "\\draw ($start_x mm, $default_y1 mm) -- coordinate (x axis mid) ($end_x mm, $default_y1 mm);\n";
print OUTFILE "\\foreach \\x in {$start_x,".($start_x+20).",...,$end_x}\n";
print OUTFILE "\\draw (\\x mm,$default_y1 mm) -- (\\x mm,".($default_y1-1)." mm)\n";
print OUTFILE "node[anchor=north] {};\n";
for($i=$start_x; $i<=$end_x; $i+=20) {
	if($i==$start_x) {
		print OUTFILE "\\draw [black] (".($i-4)." mm,".($default_y1-2)." mm) (".($i-4)." mm, ".($default_y1-3)." mm) node {$sInfo{'chr'}:$sInfo{'genomicStart'}};\n";
	}
	else {
		print OUTFILE "\\draw [black] ($i mm,". ($default_y1-2) ." mm) ($i mm, ". ($default_y1-3) ." mm) node {$sInfo{'genomicStart'}};\n";
	}
	if($sInfo{'strand'}=~/\+/) { $sInfo{'genomicStart'} += 20; }
	else { $sInfo{'genomicStart'} -= 20; }
}
#print OUTFILE "\\draw [black] (-15 mm, ". ($default_y1+5) ." mm) (-10 mm, ". ($default_y1+5) ." mm) node {$subType};\n";

## connect aligned blocks with a line
foreach my $ab(keys(%alnBlocks)) {
	print OUTFILE "\\draw [line width=0.6pt, dark-green] ($alnBlockCoor{'first'}{$ab-1}{'start'}) -- ($alnBlockCoor{'second'}{$alnBlocks{$ab}-1}{'start'})[dashed];\n";
	print OUTFILE "\\draw [line width=0.6pt, dark-green] ($alnBlockCoor{'first'}{$ab-1}{'end'}) -- ($alnBlockCoor{'second'}{$alnBlocks{$ab}-1}{'end'})[dashed];\n";
	#print "$ab\t$alnBlocks{$ab}\n";
	#print "$alnBlockCoor{'first'}{$ab-1}{'start'}\t$alnBlockCoor{'first'}{$ab-1}{'end'}\n";
	#print "$alnBlockCoor{'second'}{$alnBlocks{$ab}-1}{'start'}\t$alnBlockCoor{'second'}{$alnBlocks{$ab}-1}{'end'}\n\n";
	
}

print OUTFILE <<LATEX;
\\end{tikzpicture}
\\end{adjustbox}
\\end{document}
LATEX
close OUTFILE;

## compile the tex file
system("pdflatex $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.tex > $outDir/$tmpFilePrefix"."tmp");
unless(-e "$outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.aux") {
	system("mv $tmpFilePrefix"."$PDF_BG_ALIGN.aux $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.aux");
	system("mv $tmpFilePrefix"."$PDF_BG_ALIGN.log $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.log");
	system("mv $tmpFilePrefix"."$PDF_BG_ALIGN.pdf $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.pdf");
}

## delete intermediate files.
system("rm $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.aux");
system("rm $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.log");
system("rm $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.tex");
system("rm $outDir/$tmpFilePrefix"."tmp");
print STDERR "done ($PDF_BG_ALIGN.pdf)\n";

print STDERR "creating final output file...";
## create final tex file with block group and block alignments
open(OUTFILE, ">$outDir/$tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.tex") || die $!;
print OUTFILE <<LATEX;
\\documentclass[a4paper, 10pt]{article}
\\usepackage[hmargin=0.5cm, vmargin=1cm]{geometry}
\\usepackage{graphicx}
\\usepackage{color}
\\usepackage{tikz}
\\usepackage{subfigure}
\\usepackage{adjustbox}
\\definecolor{dark-red}{RGB}{100,0,0}
\\definecolor{dark-green}{RGB}{0,100,0}
\\definecolor{dark-blue}{RGB}{0,0,100}
\\begin{document}
\\begin{center}
\\Large{\\textbf{\\underline{deepBlockAlign\\hspace{1mm}v1.0}}}
\\end{center}
\\vspace{0.3cm}
LATEX
## table with alignment parameters
print OUTFILE "\\begin{table}[h]\n";
print OUTFILE "\\normalsize \\textbf{Alignment\\hspace{1mm}parameters:} \\linebreak \n";
print OUTFILE "\\footnotesize\n";
print OUTFILE "\\begin{tabular}{ll}\n";
print OUTFILE "\\textbf{[Input]} & \\\\ \n";
print OUTFILE "date and time & ".strftime("%a, %d %b %Y %H:%M:%S %z", localtime(time()))."\\\\ \n";
$fId=~s/\_/\\_/g; $sId=~s/\_/\\_/g;
if($ALIGNMENT_COUNTER==1) { $fFile=~s/\_/\\_/g; $sFile=~s/\_/\\_/g; }
print OUTFILE "$fTitle block group & $fId \\\\ \n";
print OUTFILE "$fTitle file & $fFile \\\\ \n";
print OUTFILE "$sTitle block group & $sId \\\\ \n";
print OUTFILE "$sTitle file & $sFile \\\\ \n";
print OUTFILE "\\textbf{[Block group alignment]} & \\\\ \n";
print OUTFILE "distance weight & $param_distance \\\\ \n";
print OUTFILE "block weight & $param_block \\\\ \n";
print OUTFILE "gap penalty & $param_gap \\\\ \n";
print OUTFILE "\\textbf{[Block alignment]} & \\\\ \n";
print OUTFILE "gap initialization penalty & $Ci \\\\ \n";
print OUTFILE "gap extension penalty & $Ce \\\\ \n";
print OUTFILE "threshold for expression difference & $DELTA \\\\ \n";
print OUTFILE "match score & $S0 \\\\ \n";
print OUTFILE "mismatch score & $S1 \\\\ \n";
print OUTFILE "profile shape difference penalty & $ZETA \\\\ \n";
print OUTFILE "\\end{tabular}\n";
print OUTFILE "\\end{table}\n";
print OUTFILE "\\vspace{0.3cm}\n";
## block group alignment figure
print OUTFILE "\\normalsize \\textbf{Block\\hspace{1mm}group\\hspace{1mm}alignment:}\\hspace{1mm}\\emph{{\\color{dark-red}\\underline{$fInfo{'name'}}($fInfo{'strand'})}\\hspace{1mm}with\\hspace{1mm}{\\color{dark-blue}\\underline{$sInfo{'name'}}($sInfo{'strand'})}}\\hspace{1mm}[Score:\\hspace{1mm}$DBA{'alnScore'}]\n";
print OUTFILE "\\vspace{0.3cm}\n";
print OUTFILE "\\begin{center}\n";
#print OUTFILE "\\begin{adjustbox}{width=\\textwidth,height=\\textheight,keepaspectratio}\n";
print OUTFILE "\\begin{adjustbox}{width=20 cm,height=3 cm,keepaspectratio}\n";
print OUTFILE "\\includegraphics[scale=1]{$outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.pdf}\n";
print OUTFILE "\\end{adjustbox}\n";
print OUTFILE "\\end{center}\n";
print OUTFILE "\\vspace{0.3cm}\n";
## block alignment figure(s)
print OUTFILE "\\normalsize \\textbf{Block\\hspace{1mm}alignment:}\n";
print OUTFILE "\\vspace{0.3cm}\n";
print OUTFILE "\\begin{center}\n";
foreach(@PDF_B_ALIGN) { print OUTFILE "\\includegraphics[scale=1]{$outDir/$_}\n"; }
print OUTFILE "\\end{center}\n";
## citation
print OUTFILE "\\normalsize \\textbf{Citation:}\n";
print OUTFILE "\\footnotesize{Langenberger, D*., Pundhir, S*., Ekstr{\\o}m, C.T., Stadler, P.F., Hoffmann, S., and Gorodkin, J.(2012). deepBlockAlign: A tool for aligning RNA-seq profiles of read block patterns. \\textit{Bioinformatics}, \\textbf{28(1)},17-24. (*equal contribution)}\n";
print OUTFILE "\\end{document}";
close OUTFILE;

## compile final tex file
system("pdflatex $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.tex > $outDir/$tmpFilePrefix"."tmp");
unless(-e "$outDir/$tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.aux") {
	system("mv $tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.aux $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.aux");
	system("mv $tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.log $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.log");
	system("mv $tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.pdf $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.pdf");
}

## delete intermediate files.
system("rm $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.aux");
system("rm $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.log");
system("rm $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.tex");
system("rm $outDir/$tmpFilePrefix"."tmp");

## remove block and block group alignment files, if not asked to keep
if(!$keepAlnFile) {
	system("rm $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.pdf");
	foreach(@PDF_B_ALIGN) { system("rm $outDir/$_"); }
	system("rm $DBA{'alnFile'}");
	system("rm $DBA{'bgFile1'}");
	system("rm $DBA{'bgFile2'}");
}

print STDERR "done ($outDir/$tmpFilePrefix"."$PDF_BG_ALIGN"."_FINAL.pdf)\n\n";

## determine scale value to convert the alignments in pdf to png
if(($sInfo{'end'}-$sInfo{'start'})>($fInfo{'end'}-$fInfo{'start'})) {
	$scale_value=sprintf("%d", ($sInfo{'end'}-$sInfo{'start'})*8);
}
else {
	$scale_value=sprintf("%d", ($fInfo{'end'}-$fInfo{'start'})*8);
}

## create html output file
if(defined($htmlOutput)) {
	if($ALIGNMENT_COUNTER==1) {
		print "<h3>Alignment result</h3>";
		my %str=();
		if(scalar(@fIds)==1) { $str{'first'} = scalar(@fIds)." block group"; }
		else { $str{'first'} = scalar(@fIds). " block groups"; }
		if(scalar(@sIds)==1) { $str{'second'} = scalar(@sIds)." block group"; }
		else { $str{'second'} = scalar(@sIds). " block groups"; }
		if(scalar(@fIds)==1 && scalar(@sIds)==1) { $str{'aln'} = (scalar(@fIds)*scalar(@sIds))." alignment"; }
		else { $str{'aln'} = (scalar(@fIds)*scalar(@sIds)). " alignments"; }

		print "<p>Aligned $str{'first'} from $fTitle BED file with $str{'second'} from $sTitle BED file, leading to $str{'aln'}</p>";
		print "<table class=resultTable border=0>\n";
		print "<tr class=\"header\"><th>First block group</th><th>Second block group</th><th>Score</th><th>Description</th><th>Block group alignment</th><th>Block alignment</th><tr>\n";
	}
	create_html_file(\%fInfo, \%sInfo, $outDir, $tmpFilePrefix, $PDF_BG_ALIGN, \@PDF_B_ALIGN, \%DBA);
	if($ALIGNMENT_COUNTER==(scalar(@fIds)*scalar(@sIds))) {
		print  "<tr><td colspan=\"6\"><input type=hidden id=\"ALIGNMENT_COUNTER\" value=$ALIGNMENT_COUNTER></td></tr>\n";
		print  "</table>\n";
	}
}
## create text output file
elsif(defined($textOutput)) {
	print "#Alignment information\n";
	print "first block group:\t$fInfo{'name'}\t$fInfo{'chr'}\t$fInfo{'start'}\t$fInfo{'end'}\t$fInfo{'strand'}\t$fInfo{'reads'}\t$fInfo{'tags'}\t$fInfo{'blocks'}\t$fInfo{'entropy'}\t$fInfo{'type'}\t$fInfo{'loci'}\t$fInfo{'uniqReads'}\n";
	print "second block group:\t$sInfo{'name'}\t$sInfo{'chr'}\t$sInfo{'start'}\t$sInfo{'end'}\t$sInfo{'strand'}\t$sInfo{'reads'}\t$sInfo{'tags'}\t$sInfo{'blocks'}\t$sInfo{'entropy'}\t$sInfo{'type'}\t$sInfo{'loci'}\t$sInfo{'uniqReads'}\n";
	system("convert -geometry $scale_value\"x\"$scale_value -density 200x200 -quality 100 -trim $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.pdf $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.png");
	print "alignment:\t$DBA{'alnScore'}\t$htmlImgPath/$tmpFilePrefix$PDF_BG_ALIGN.png\t";
	for(my $i=0; $i<scalar(@PDF_B_ALIGN); $i++) {
		print "\t" if($i>0);
		$PDF_B_ALIGN[$i]=~s/\.pdf//g;
		system("convert -density 150 -trim $outDir/$tmpFilePrefix"."$PDF_B_ALIGN[$i].pdf $outDir/$tmpFilePrefix"."$PDF_B_ALIGN[$i].png");
		print "$htmlImgPath/$tmpFilePrefix$PDF_B_ALIGN[$i].png";
	} print "\n";
}
}
}

sub create_html_file {
my ($fInfo, $sInfo, $outDir, $tmpFilePrefix, $PDF_BG_ALIGN, $PDF_B_ALIGN, $DBA) = @_;

## convert block group alignment from pdf to png
#system("convert -density 600x600 -resize 1000x120 -quality 90 $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.pdf $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.png");
system("convert -geometry $scale_value\"x\"$scale_value -density 200x200 -quality 100 -trim $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.pdf $outDir/$tmpFilePrefix"."$PDF_BG_ALIGN.png");

print <<HTML;
    <tr class="result"><td>$fInfo->{'name'}</td><td>$sInfo->{'name'}</td><td>$DBA->{'alnScore'}</td>
			<td><img onClick='managePlot("description$ALIGNMENT_COUNTER");' id="img_description$ALIGNMENT_COUNTER" src="http://rth.dk/resources/dba/img/add.png"></td>
			<td><img onClick='managePlot("bg_align$ALIGNMENT_COUNTER");' id="img_bg_align$ALIGNMENT_COUNTER" src="http://rth.dk/resources/dba/img/add.png"></td>
			<td><img onClick='managePlot("b_align$ALIGNMENT_COUNTER");' id="img_b_align$ALIGNMENT_COUNTER" src="http://rth.dk/resources/dba/img/add.png"></td>
    </tr>
    <tr><td colspan="6">
      <table id="description$ALIGNMENT_COUNTER" class="detail">
        <tr class="sub_header"><th class="emptyth" rowspan="10"></th><th colspan="2">First block group</th><th colspan="2">Second block group</th></tr>
        <tr><th>name</th><td>$fInfo->{'name'}</td><th>name</th><td>$sInfo->{'name'}</td></tr>
        <tr><th>coordinate</th><td><a href="http://genome.ucsc.edu/cgi-bin/hgTracks?org=hg19&position=$fInfo->{'chr'}:$fInfo->{'start'}-$fInfo->{'end'}&hgt.customText=$htmlImgPath/DBA_f1.ucsc.gz" target="_blank">$fInfo->{'chr'}:$fInfo->{'start'}-$fInfo->{'end'}</a>($fInfo->{'strand'})</td><th>coordinate</th><td><a href="http://genome.ucsc.edu/cgi-bin/hgTracks?org=hg19&position=$sInfo->{'chr'}:$sInfo->{'start'}-$sInfo->{'end'}&hgt.customText=$htmlImgPath/DBA_f2.ucsc.gz" target="_blank">$sInfo->{'chr'}:$sInfo->{'start'}-$sInfo->{'end'}</a>($sInfo->{'strand'})</td></tr>
        <tr><th>reads</th><td>$fInfo->{'reads'}</td><th>reads</th><td>$sInfo->{'reads'}</td></tr>
        <tr><th>tags</th><td>$fInfo->{'tags'}</td><th>tags</th><td>$sInfo->{'tags'}</td></tr>
        <tr><th>blocks</th><td>$fInfo->{'blocks'}</td><th>blocks</th><td>$sInfo->{'blocks'}</td></tr>
        <tr><th>entropy</th><td>$fInfo->{'entropy'}</td><th>entropy</th><td>$sInfo->{'entropy'}</td></tr>
        <tr><th>ncRNA</th><td>$fInfo->{'type'}</td><th>ncRNA</th><td>$sInfo->{'type'}</td></tr>
        <tr><th>loci</th><td>$fInfo->{'loci'}</td><th>loci</th><td>$sInfo->{'loci'}</td></tr>
        <tr><th>unique reads (%)</th><td>$fInfo->{'uniqReads'}</td><th>unique reads (%)</th><td>$sInfo->{'uniqReads'}</td></tr>
      </table>
      <table id="bg_align$ALIGNMENT_COUNTER" class="detail">
        <tr class="sub_header"><th rowspan="2"></th><th>Block group alignment</th></tr>
        <tr><td>
          <div id="align">
            <img id="img_bg_align" src="$htmlImgPath/$tmpFilePrefix$PDF_BG_ALIGN.png" border=0>
          </div>
        </td></tr>
      </table>
      <table id="b_align$ALIGNMENT_COUNTER" class="detail">
        <tr class="sub_header"><th rowspan="10"></th><th>Block alignment</th></tr>
HTML

## convert block alignment from pdf to png
foreach(my $i=0; $i<scalar(@{$PDF_B_ALIGN}); $i++) {
	$PDF_B_ALIGN->[$i]=~s/\.pdf//g;
	if($i%2==0) {
        	print "<tr><td>\n";
		print "<div id=\"align\">\n";
	}
	#system("convert -trim -density 600x600 -resize 200x200 -quality 90 $outDir/$tmpFilePrefix"."$_.pdf $outDir/$tmpFilePrefix"."$_.png");
	system("convert -density 150 -trim $outDir/$tmpFilePrefix"."$PDF_B_ALIGN->[$i].pdf $outDir/$tmpFilePrefix"."$PDF_B_ALIGN->[$i].png");
	print "<img id=\"img_b_align\" src=\"$htmlImgPath/$tmpFilePrefix$PDF_B_ALIGN->[$i].png\" border=0>\n";
	if($i%2!=0 || $i==(scalar(@{$PDF_B_ALIGN})-1)) {
		print "</div>\n";
		print  "</td></tr>\n";
	}
}

print <<HTML;
      </table>
    </td></tr>
HTML

}
exit;
