#include "alignment.h"

Alignment::Alignment(int match, int mismatch, int gapd, int cutoff) {
  this->match = match;
  this->mismatch = mismatch;
  this->gapd = gapd;
  this->cutoff = cutoff;
}

int Alignment::alignGlobally(vector<int> x, vector<int> y) {
  int xlen = x.size();
  int ylen = y.size();
  vector<vector<int> > dp;
  vector<vector<char> > tr;
  dp.resize(xlen+1);
  tr.resize(xlen+1);

  for (int i=0; i<=xlen; i++) {
    dp[i].resize(ylen+1);
    tr[i].resize(ylen+1);
  }

  // Initialization
  dp[0][0] = 0;

  // Boundary conditions
  for (int i=1; i<=xlen; i++) {
    dp[i][0] = - i * gapd;
    tr[i][0] = 'X';
  }

  for (int j=1; j<=ylen; j++) {
    dp[0][j] = -j * gapd;
    tr[0][j] = 'Y';
  }

  // Recursion
  for (int i=1; i<=xlen; i++) {
    for (int j=1; j<=ylen; j++) {
      int d = 0;
      char t = 'M';

      // Case 1: match
      if (x[i-1] == y[j-1]) {
	if (x[i-1] == 1) d = dp[i-1][j-1] + match;
	else d = dp[i-1][j-1] + 0;
      }

      // Case2 : mismatch (cases 1 & 2 are alternative)
      else d = dp[i-1][j-1] - mismatch;

      // Case: x_i is aligned to a gap
      if (d < dp[i-1][j] - gapd) {
	d = dp[i-1][j] - gapd;
	t = 'X';
      }

      // Case: y_j is aligned to a gap
      if (d < dp[i][j-1] - gapd) {
	d = dp[i][j-1] - gapd;
	t = 'Y';
      }

      dp[i][j] = d;
      tr[i][j] = t;
    } // j
  } // i

  return dp[xlen][ylen];

  /* considering length normalized score
  else {
    // Traceback
    string xal = "", yal = "";
    int i = xlen, j = ylen;

    while (i > 0 || j > 0) {
      switch (tr[i][j]) {
      case 'M':
	xal = int2str(x[i-1]) + xal;
	yal = int2str(y[j-1]) + yal;
	--i;
	--j;
	break;

      case 'X':
	xal = int2str(x[i-1]) + xal;
	yal = "-" + yal;
	--i;
	break;

      case 'Y':
	xal = "-" + xal;
	yal = int2str(y[j-1]) + yal;
	--j;
	break;

      default:
	break;
      }
    }

    #if DEBUG
    showAlignment(xal, yal);
    #endif

    return dp[xlen][ylen] / static_cast<int>(xal.length()); // Length-normalized score
  }
  */
}

string int2str(int i) {
  string str;
  ostringstream ostr;
  ostr << i;
  str = ostr.str();
  ostr.str("");
  return str;
}

void showAlignment(string xal, string yal) {
  cout << "Alignment:\n";
  int row = (xal.length() -1) / 100 + 1;

  if (row == 1) {
    cout << xal << endl;
    cout << yal << endl;
    cout << "\n";
  }

  else if (row > 1) {
    int start;

    for (int i=1; i<=row-1; i++) {
      start = 100 * (i - 1);
      cout << xal.substr(start, 100) << endl;
      cout << yal.substr(start, 100) << endl;
      cout << "\n";
    }

    start = 100 * (row - 1);
    cout << xal.substr(start, xal.length()-100*(row-1)-1) << endl;
    cout << yal.substr(start, xal.length()-100*(row-1)-1) << endl;
    cout << "\n";
  }
}
