#include "option.h"

Option::Option() {

  // Set default values
  dna = 0;
  gzip = 0;
  entire = 0;
  region = 0;
  align = 0;
  verbose = 0;

  winsize = 120;
  pairsize = winsize;
  step = 30;
  cutoffb = 0.01;
  nu = 1;
  diagsize = 3;
  thres = 0.1;
  match = 9;
  mismatch = 2;
  gapd = 1;
  cutoff = 20;
}

int Option::parseOption(int argc, char **argv) {
  struct option long_opts[] =
    {
      {"dna", no_argument, NULL, 'd'},
      {"gzip", no_argument, NULL, 'z'},
      {"entire", no_argument, NULL, 'e'},
      {"region", no_argument, NULL, 'r'},
      {"align", no_argument, NULL, 'a'},
      {"winsize", required_argument, NULL, 'w'},
      {"step", required_argument, NULL, 's'},
      {"nu", required_argument, NULL, 'u'},
      {"thres", required_argument, NULL, 't'},
      {"match", required_argument, NULL, 'm'},
      {"mismatch", required_argument, NULL, 'n'},
      {"gapd", required_argument, NULL, 'g'},
      {"cutoff", required_argument, NULL, 'c'},
      {"verbose", no_argument, NULL, 'v'},
      {"help", no_argument, NULL, 'h'},
      {"full-help", no_argument, NULL, 'H'},
      {0, 0, 0, 0}
    };

  int opt = 0, opt_index = 0;

  while ((opt = getopt_long(argc, argv, "dzeraw:s:u:t:m:n:g:c:vhH", long_opts, &opt_index)) != -1) {
    switch(opt) {
    case 'd':
      dna = 1;
      break;

    case 'z':
      gzip = 1;
      break;

    case 'e':
      entire = 1;
      break;

    case 'r':
      region = 1;
      break;

    case 'a':
      align = 1;
      break;

    case 'w':
      winsize = atoi(optarg);
      break;

    case 's':
      step = atoi(optarg);
      break;

    case 'u':
      nu = atof(optarg);
      break;

    case 't':
      thres = atof(optarg);
      break;

    case 'm':
      match = atoi(optarg);
      break;

    case 'n':
      mismatch = atoi(optarg);
      break;

    case 'g':
      gapd = atoi(optarg);
      break;

    case 'c':
      cutoff = atoi(optarg);
      break;

    case 'v':
      verbose = 1;
      break;

    case 'h':
      Option::showHelp();
      break;

    case 'H':
      Option::showFullHelp();
      break;

    case '?':
      cout << optopt << endl;
      exit(1);
      break;
    }
  }

  argc = argc - optind;
  argv = argv + optind;

  if (argc == 0 || argc >= 3) Option::showHelp();
  if (argc != 1 && region == 1) Option::showHelp();

  filename1 = argv[0];

  if (argc == 2) filename2 = argv[1];

  if (entire == 1 && step != 0) step = 0;
  if (step == 0 && entire == 0) entire = 1;

  return argc;
}

void Option::showHelp() {
  cout << "DotcodeR ver. 1.0.0\n\n";
  cout << "Structural similarity search by DOT plot enCODEr for RNAs in genomic sequences\n\n";
  cout << "Usage: dotcoder [OPTION]... [FILE]...\n\n";
  cout << "[FILE]\n";
  cout << " At most two (gzipped) FASTA files are needed.\n\n";
  cout << "[OPTION]\n";
  cout << " -d, --dna             Read DNA sequences for input\n";
  cout << "                        (default: off, i.e. RNA sequence)\n";
  cout << " -z, --gzip            Allow gzipped files for input\n";
  cout << "                        (default: off)\n";
  cout << " -e, --entire          Compare between two entire sequences of at most 300nt length\n";
  cout << "                        (default: off)\n";
  cout << "                        (Specifying \"-e\" implies \"-s 0\" option)\n\n";
  cout << " -w, --winsize <INT>   Sliding window size in the genomic sequences\n";
  cout << "                        (default <INT> = 120)\n";
  cout << " -s, --step <INT>      Step size of the sliding window\n";
  cout << "                        (default <INT> = 30)\n";
  cout << "                        (Specifying \"-s 0\" implies \"-e\" option)\n";
  cout << " -c, --cutoff <INT>    Report only similarity scores >= cutoff\n";
  cout << "                        (default <INT> = 20)\n\n";
  cout << " -v, --verbose         Show details\n";
  cout << "                        (default: off)\n";
  cout << " -h, --help            Show this message\n";
  cout << " -H, --full-help       Show detailed help\n\n";
  exit(0);
}

void Option::showFullHelp() {
  cout << "DotcodeR ver. 1.0.0\n\n";
  cout << "Structural similarity search by DOT plot enCODEr for RNAs in genomic sequences\n\n";
  cout << "This program can be used for fast detection of potential structured RNA regions between two genomic sequences.\n\n";
  cout << "Usage: dotcoder [OPTION]... [FILE]...\n\n";
  cout << "[FILE]\n";
  cout << " At most two (gzipped) FASTA files or multi-FASTA files are needed.\n\n";
  cout << " If you enter just one FASTA sequence with \"-r\" option, potential structured RNA regions of the sequence can be found.\n";
  cout << " Handling gzipped DNA FASTA files needs \"-z\" option.\n\n";
  cout << "[OPTION]\n";
  cout << " -d, --dna             Read DNA sequences for input\n";
  cout << "                        (default: off, i.e. RNA sequence)\n";
  cout << " -z, --gzip            Allow gzipped files for input\n";
  cout << "                        (default: off)\n";
  cout << " -e, --entire          Compare between two entire sequences of at most 300nt length\n";
  cout << "                        (default: off)\n";
  cout << "                        (Specifying \"-e\" implies \"-s 0\" option)\n\n";
  cout << " -r, --region          Only detect potential structured RNA regions in one sequence\n";
  cout << "                        (default: off)\n\n";
  cout << " -a, --align           Adopt alignment calculation\n";
  cout << "                        (Run much slower than dot product calculation)\n";
  cout << "                        (default: off)\n\n";
  cout << " -w, --winsize <INT>   Sliding window size in the genomic sequences\n";
  cout << "                        (default <INT> = 120)\n";
  cout << " -s, --step <INT>      Step size of the sliding window\n";
  cout << "                        (default <INT> = 30)\n";
  cout << "                        (Specifying \"-s 0\" implies \"-e\" option)\n";
  cout << " -u, --nu <FLOAT>      Threshold for listing candidate RNA intervals\n";
  cout << "                        (default <FLOAT> = 1.0)\n";
  cout << " -t, --thres <DOUBLE>  Threshold for sum of neighboring probabilities\n";
  cout << "                        (default <DOUBLE> = 0.1)\n";
  cout << " -m, --match <INT>     Match score between two binary digits\n";
  cout << "                        (default <INT> = 9)\n";
  cout << " -n, --mismatch <INT>  Mismatch penalty between two binary digits\n";
  cout << "                        (default <INT> = 2)\n";
  cout << " -g, --gapd <INT>      Gap penalty between two binary digits\n";
  cout << "                        (default <INT> = 1)\n";
  cout << " -c, --cutoff <INT>    Report only similarity scores >= cutoff\n";
  cout << "                        (default <INT> = 20)\n\n";
  cout << " -v, --verbose         Show details\n";
  cout << "                        (default: off)\n";
  cout << " -h, --help            Show standard help\n";
  cout << " -H, --full-help       Show this message\n\n";
  exit(0);
}

string Option::getFilename(int id) {
  if (id == 1) return filename1;
  else return filename2;
}

int Option::getDNA() {
  return dna;
}

int Option::getGzip() {
  return gzip;
}

int Option::getEntire() {
  return entire;
}

int Option::getRegion() {
  return region;
}

int Option::getAlign() {
  return align;
}

int Option::getVerbose() {
  return verbose;
}

int Option::getWinsize() {
  return winsize;
}

int Option::getPairsize() {
  return pairsize;
}

int Option::getStep() {
  return step;
}

float Option::getCutoffb() {
  return cutoffb;
}

float Option::getNu() {
  return nu;
}

int Option::getDiagsize() {
  return diagsize;
}

double Option::getThres() {
  return thres;
}

int Option::getMatch() {
  return match;
}

int Option::getMismatch() {
  return mismatch;
}

int Option::getGapd() {
  return gapd;
}

int Option::getCutoff() {
  return cutoff;
}
