#ifndef FOLD
#define FOLD
#include "foldalign.h"
#include "sequence.cxx"
#include "output.cxx"
#include "helper.cxx"
#include "arguments.cxx"
#include "results.cxx"
#include "longTermMemory.cxx"
#include "shortTermMemory.cxx"
#include "exception.cxx"
#include "stack_ssl.cxx"
#include "mbllist.cxx"
#include "longCell.cxx"


// ENERGY_FUNC defines the parameters of the energy calculation functions.
// It is only used in the hope that it makes the code a little more readable
#define ENERGY_FUNC const positionType i, const positionType k, \
                    const lengthType Wi, const lengthType Wk, \
						  const positionType c_i, const positionType c_k, \
						  const positionType c_j, const positionType c_l, \
						  stmcell*& cCell

#define FOLD_TEMPLATE class stmcell, class ltmcell, class startCell, \
                      bool global, bool realigning, bool mblrealign


#define FOLD_TEMPLATE_PARAMETERS stmcell, ltmcell, startCell, \
                                 global, realigning, mblrealign

template< FOLD_TEMPLATE >
class fold {
public:
	//*************************************
	// The constructor intialize the stm matrix,
	//
	// The subsequence 
	// begin_1 --> end_1
	// in the first sequence is aligned to subsequence
	// begin_2 --> end_2
	// in the second sequence

	fold(positionType begin_1, positionType end_1, 
	     positionType begin_2, positionType end_2, 
	     results*& res,
	     const sequence* const one, const sequence* const two,
		  arguments*& argu, 
	     scorematrix& score,
	     longTermMemory< ltmcell >* const long_D,
	     longTermMemory< startCell >* const startCoordinates = 0,
	     stack_ssl< mbllist, int >* const mblMem = 0);

	inline static positionType size_k_dimension(const lengthType del) 
	                                                           {return 4*del+3;};

	inline static positionType k_offset(const positionType begin_I,
													const positionType begin_K,
													const lengthType del) 
													{return begin_I - begin_K + 2*del+1;};
	inline ~fold() {};

private:
	//*************************************
	// The global variables

	// The coordinates and score for the best alignment.
	results* r;

	// The local names for the parameters stored in the argument object
	const bool plot_score;
	const bool all_scores;
	const bool no_backtrack;
	const bool flip;
	const bool nobranch;
	const bool print_all_scores;
	const lengthType lambda;
	const lengthType delta;
	const lengthType min_loop;
	const lengthType chunk_size;
	const bool noprune;
	const bool hpstart;
	const positionType memroof;
	const bool mem_info;
	
	positionType k_offSet_i;

	// The current local coordinates
	positionType curr_begin_I;
	positionType curr_end_I;
	positionType curr_begin_K;
	positionType curr_end_K;

	// Given a Window size and a position can this position be extanded to the
	// left without violating the window size < lambda
	positionType max_bottom_I;
	positionType max_bottom_K;

	// The current minimum of i+lambda, curr_end_I. Ditto K
	positionType min_top_I;
	positionType min_top_K;

	// The output class controls most of the printing
	output* out;

	// The sequences
	const sequence* const seq_1;
	const sequence* const seq_2;

	// The argument object
	arguments*& arg;
	
	// The scorematrix
	scorematrix& s_matrix;

	// Some of score from the score matrix
	const scoreType mblNuc;
	const scoreType mblHelix;
	const scoreType affineGapCost;
	const lengthType pruneTableSize;
	
	scoreType* pruneTable;
	// During global alignmnet the pruning score is compensated for the minimum
	// number of gaps required to align the sequences. This array holds the
	// compensation as a function of Wi - Wk + delta
	scoreType* globalDeltaPrun;
	// The memory handler object
	shortTermMemory< stmcell >* stm;
	longTermMemory< ltmcell >* const ltm;

	// An object which holds start coordinates.
	longTermMemory< startCell >* const startCoord;
	
	// Stack of branch points. Not needed during local alignment
	stack_ssl< mbllist, int >* const mblm;
	stack_ssl< mbllist, int >* local_mblm;

	// The length of the two sequences
	const positionType length_I;
	const positionType length_K;
	
	// This numbers are added to i to find the range of the k coordinate during
	// global, realigning, mblrealign.
	positionType k_low;
	positionType k_high;

	// This is true if the high or low score overflow alert has been triggert
	bool highLowAlert;

	// Program flow control arrays flow_size and the index states are defined
	// in foldalign.h
	typedef void (fold< FOLD_TEMPLATE_PARAMETERS >::*flowPtr) (
	           	  const positionType i, const positionType k,
					  const lengthType Wi, const lengthType Wk,
					  const positionType c_i, const positionType c_k,
					  const positionType c_j, const positionType c_l,
					  stmcell*& cCell);
	
	flowPtr p2calc_ikWiWk[flow_size];
	flowPtr p2calc_iWi[flow_size];
	flowPtr p2calc_kWk[flow_size];
	flowPtr p2calc_ik[flow_size];
	flowPtr p2calc_WiWk[flow_size];
	flowPtr p2calc_i[flow_size];
	flowPtr p2calc_k[flow_size];
	flowPtr p2calc_Wi[flow_size];
	flowPtr p2calc_Wk[flow_size];
	flowPtr p2calc_mbl[flow_size];
	
	// The functions in this array corrects the external unpaired sequence scores
	void (fold< FOLD_TEMPLATE_PARAMETERS >::*p2end[flow_size]) (
												scoreType& score,
												const positionType i, const positionType k, 
												const lengthType Wi, const lengthType Wk,
												stmcell*& cCell);

	// True for all states which cannot be the left part
	// of a branch point
	bool right_branch[flow_size];

	// True for states which should be stored.
	// equal to the right_branch during scan but eqaul all states during
	// backtrack or global alignment
	bool right_store[flow_size];

	//*************************************
	// Functions

	// This function keeps track of the four outer loops
	inline void align(positionType begin_I, positionType end_I, positionType begin_K, positionType end_K, const bool lastRun);



	// Makes sure that the alignment is only expanded in the possible directions
	inline void expandAlignment(positionType i, positionType k, lengthType Wi, lengthType Wk, const positionType c_i, const positionType c_k, const positionType c_j, const positionType c_l, stmcell*& cCell);

	// Calls the different score functions.
	inline void expandAlignment_ikWiWk(positionType i, positionType k, lengthType Wi, lengthType Wk, const positionType c_i, const positionType c_k, const positionType c_j, const positionType c_l, const bool WkWi_m1, const bool WkWi_p1, const bool ki_m1, const bool ki_p1, stmcell*& cCell);
	inline void expandAlignment_iWi(positionType i, positionType k, lengthType Wi, lengthType Wk, const positionType c_i, const positionType c_k, const positionType c_j, const positionType c_l, const bool WkWi_m1, const bool ki_m1, const bool ki_p1, const bool WkWi_p1, stmcell*& cCell);
	inline void expandAlignment_kWk(positionType i, positionType k, lengthType Wi, lengthType Wk, const positionType c_i, const positionType c_k, const positionType c_j, const positionType c_l, const bool WkWi_m1, const bool ki_m1, const bool ki_p1, const bool WkWi_p1, stmcell*& cCell);


	// Some checks on Wimax when it is updated.
	inline void checkWimax(lengthType& Wi_max, const positionType i);


	// In which directions can an alignment be expanded.
	inline int check_coord_local(const positionType i, const positionType k,
										  const lengthType Wi, const lengthType Wk) const;

	// Is there room for one more base-pair in this sequence?
	inline bool extra_bp_coord_IK(const positionType i, const positionType k,
											const lengthType Wi, const lengthType Wk) const;
	inline bool extra_bp_coord_I(const positionType i, const positionType k,
	 									  const lengthType Wi, const lengthType Wk) const;
	inline bool extra_bp_coord_K(const positionType i, const positionType k,
										  const lengthType Wi, const lengthType Wk) const;


	//**************************************************************************
	// Energy calculation functions
	inline void hp2end(scoreType& score, const positionType i, const positionType k, const lengthType Wi, const lengthType Wk, stmcell*& cCell);
	inline void stem2end(scoreType& score, const positionType i, const positionType k, const lengthType Wi, const lengthType Wk, stmcell*& cCell);
	inline void bulge_ik2end(scoreType& score, const positionType i, const positionType k, const lengthType Wi, const lengthType Wk, stmcell*& cCell);
	inline void bulge_WiWk2end(scoreType& score, const positionType i, const positionType k, const lengthType Wi, const lengthType Wk, stmcell*& cCell);
	inline void ilIK2end(scoreType& score, const positionType i, const positionType k, const lengthType Wi, const lengthType Wk, stmcell*& cCell);

	// Hairpin functions
	template< stateType return_state, lengthType grow1, lengthType grow3 >
		inline void hp_align_one_pos(const positionType i, const positionType k,
											  const lengthType Wi, const lengthType Wk,
											  const int& nuc_I, const int& nuc_K,
											  stmcell*& cCell, const scoreType& affine = 0);
	template< stateType return_state > inline void hp_init_pot_bp_IK( ENERGY_FUNC );
	template< stateType return_state > inline void hp_align_i( ENERGY_FUNC )
		{hp_align_one_pos<return_state, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i), 0, cCell);}
	template< stateType return_state > inline void hp_align_gap_k_i( ENERGY_FUNC )
		{hp_align_one_pos<return_state, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i), 0, cCell, affineGapCost);}
	template< stateType return_state > inline void hp_align_k( ENERGY_FUNC )
		{hp_align_one_pos<return_state, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k), cCell);}
	template< stateType return_state > inline void hp_align_gap_I_k( ENERGY_FUNC )
		{hp_align_one_pos<return_state, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k), cCell, affineGapCost);}
	template< stateType return_state > inline void hp_align_Wi( ENERGY_FUNC )
		{hp_align_one_pos<return_state, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i+Wi), 0, cCell);}
	template< stateType return_state > inline void hp_align_gap_K_Wi( ENERGY_FUNC )
		{hp_align_one_pos<return_state, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i+Wi), 0, cCell, affineGapCost);}
	template< stateType return_state > inline void hp_align_Wk( ENERGY_FUNC )
		{hp_align_one_pos<return_state, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k+Wk), cCell);}
	template< stateType return_state > inline void hp_align_gap_I_Wk( ENERGY_FUNC )
		{hp_align_one_pos<return_state, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k+Wk), cCell, affineGapCost);}
	template< stateType return_state > inline void hp_align_ik( ENERGY_FUNC )
		{hp_align_one_pos<return_state, 1, 1>(i, k, Wi, Wk, seq_1->getPos(i), seq_2->getPos(k), cCell);}
	template< stateType return_state > inline void hp_align_WiWk( ENERGY_FUNC )
		{hp_align_one_pos<return_state, 1, 1>(i, k, Wi, Wk, seq_1->getPos(i+Wi), seq_2->getPos(k+Wk), cCell);}

	// Stem functions
	template< stateType return_state > inline void stemIK2ikWiWk( ENERGY_FUNC );
	template< stateType return_state > inline void stemIK2onepair(
																		const positionType i,
																		const positionType k,
																		const lengthType Wi,
																		const lengthType Wk,
																		const int& nuc_start,
																		const int& nuc_end,
																		const positionType& c_start,
																		const positionType& c_end,
																		stmcell*& cCell,
																		const scoreType& affine = 0);
	template< stateType return_state > inline void stemIK2iWi( ENERGY_FUNC )
		{stemIK2onepair<return_state>(i, k, Wi, Wk, seq_1->getPos(i), seq_1->getPos(i+Wi), c_i, c_j, cCell);}
	template< stateType return_state > inline void stemIstemgkWk2iWi( ENERGY_FUNC )
		{stemIK2onepair<return_state>(i, k, Wi, Wk, seq_1->getPos(i), seq_1->getPos(i+Wi), c_i, c_j, cCell, 2*affineGapCost);}
	template< stateType return_state > inline void stemIK2kWk( ENERGY_FUNC )
		{stemIK2onepair<return_state>(i, k, Wi, Wk, seq_2->getPos(k), seq_2->getPos(k+Wk), c_k, c_l, cCell);}
	template< stateType return_state > inline void stemgiWistemK2kWk( ENERGY_FUNC )
		{stemIK2onepair<return_state>(i, k, Wi, Wk, seq_2->getPos(k), seq_2->getPos(k+Wk), c_k, c_l, cCell, 2*affineGapCost);}

	// Bulge functions
	template< stateType return_state, lengthType grow_I, lengthType grow_K >
			inline void growBulge_ik(const positionType i, const positionType k,
											 const lengthType Wi, const lengthType Wk,
											 const int& nuc_I, const int& nuc_K,
											 stmcell*& cCell, const scoreType affine = 0);
	template< stateType return_state, lengthType grow_I, lengthType grow_K >
			inline void growBulge_WiWk(const positionType i, const positionType k,
												const lengthType Wi, const lengthType Wk,
												const int& nuc_I, const int& nuc_K,
												stmcell*& cCell, const scoreType affine = 0);
	template< stateType return_state > inline void bWiIbWkK2ikWiWkCore(
												const positionType i, const positionType k,
												const lengthType Wi, const lengthType Wk,
												const positionType c_i,
												const positionType c_k,
												const positionType c_j,
												const positionType c_l,
												stmcell*& cCell,
												const lengthType& len_I,
												const lengthType& len_K,
												const lengthType& len1,
												const lengthType& len2,
												const lengthType& len3,
												const lengthType& len4 );
	template< stateType return_state, lengthType grow_i, lengthType grow_k,
				 lengthType grow_Wi, lengthType grow_Wk >
				 	inline void bulge2intCore(const positionType i,
						const positionType k, const lengthType Wi,
						const lengthType Wk, const int& nuc_I, const int& nuc_K,
						const lengthType& len_I, const lengthType& len_K,
						stmcell*& cCell, const scoreType& affine = 0 );
	template< stateType return_state > inline void stemIK2ik( ENERGY_FUNC )
		{growBulge_ik<return_state, 1, 1>(i, k, Wi, Wk, seq_1->getPos(i), seq_2->getPos(k), cCell);}
	template< stateType return_state > inline void stemIK2i( ENERGY_FUNC )
		{growBulge_ik<return_state, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i), 0, cCell);}
	template< stateType return_state > inline void stemIKgap2i( ENERGY_FUNC )
		{growBulge_ik<return_state, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i), 0, cCell, affineGapCost);}
	template< stateType return_state > inline void stemIK2k( ENERGY_FUNC )
		{growBulge_ik<return_state, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k), cCell);}
	template< stateType return_state > inline void stemIKgap2k( ENERGY_FUNC )
		{growBulge_ik<return_state, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k), cCell, affineGapCost);}
	template< stateType return_state > inline void stemIK2WiWk( ENERGY_FUNC )
		{growBulge_WiWk<return_state, 1, 1>(i, k, Wi, Wk, seq_1->getPos(i+Wi), seq_2->getPos(k+Wk), cCell);}
	template< stateType return_state > inline void stemIK2Wi( ENERGY_FUNC )
		{growBulge_WiWk<return_state, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i+Wi), 0, cCell);}
	template< stateType return_state > inline void stemIKgap2Wi( ENERGY_FUNC )
		{growBulge_WiWk<return_state, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i+Wi), 0, cCell, affineGapCost);}
	template< stateType return_state > inline void stemIK2Wk( ENERGY_FUNC )
		{growBulge_WiWk<return_state, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k+Wk), cCell);}
	template< stateType return_state > inline void stemIKgap2Wk( ENERGY_FUNC )
		{growBulge_WiWk<return_state, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k+Wk), cCell, affineGapCost);}
	template< stateType return_state > inline void bWiIbWkK2ikWiWk( ENERGY_FUNC )
		{bWiIbWkK2ikWiWkCore<return_state>(i, k, Wi, Wk, c_i, c_k, c_j, c_l, cCell, cCell->getLength2(), cCell->getLength4(), 0, cCell->getLength2(), 0, cCell->getLength4());}
	template< stateType return_state > inline void biIbkK2ikWiWk( ENERGY_FUNC )
		{bWiIbWkK2ikWiWkCore<return_state>(i, k, Wi, Wk, c_i, c_k, c_j, c_l, cCell, cCell->getLength1(), cCell->getLength3(), cCell->getLength1(), 0, cCell->getLength3(), 0);}
	template< stateType return_state > inline void biIbkK2WiWk( ENERGY_FUNC )
		{bulge2intCore<return_state, 0, 0, 1, 1>(i, k, Wi, Wk, seq_1->getPos(i+Wi), seq_2->getPos(k+Wk), cCell->getLength1(), cCell->getLength3(), cCell);}
	template< stateType return_state > inline void biIbkK2Wk( ENERGY_FUNC )
		{bulge2intCore<return_state, 0, 0, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k+Wk), cCell->getLength1(), cCell->getLength3(), cCell);}
	template< stateType return_state > inline void biIbkKgap2Wk( ENERGY_FUNC )
		{bulge2intCore<return_state, 0, 0, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k+Wk), cCell->getLength1(), cCell->getLength3(), cCell, affineGapCost);}
	template< stateType return_state > inline void biIbkK2Wi( ENERGY_FUNC )
		{bulge2intCore<return_state, 0, 0, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i+Wi), 0, cCell->getLength1(), cCell->getLength3(), cCell);}
	template< stateType return_state > inline void biIbkKgap2Wi( ENERGY_FUNC )
		{bulge2intCore<return_state, 0, 0, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i+Wi), 0, cCell->getLength1(), cCell->getLength3(), cCell, affineGapCost);}
	template< stateType return_state > inline void bWiIbWkK2ik( ENERGY_FUNC )
		{bulge2intCore<return_state, 1, 1, 0, 0>(i, k, Wi, Wk, seq_1->getPos(i), seq_2->getPos(k), cCell->getLength2(), cCell->getLength4(), cCell);}
	template< stateType return_state > inline void bWiIbWkK2i( ENERGY_FUNC )
		{bulge2intCore<return_state, 1, 0, 0, 0>(i, k, Wi, Wk, seq_1->getPos(i), 0, cCell->getLength2(), cCell->getLength4(), cCell);}
	template< stateType return_state > inline void bWiIbWkKgap2i( ENERGY_FUNC )
		{bulge2intCore<return_state, 1, 0, 0, 0>(i, k, Wi, Wk, seq_1->getPos(i), 0, cCell->getLength2(), cCell->getLength4(), cCell, affineGapCost);}
	template< stateType return_state > inline void bWiIbWkK2k( ENERGY_FUNC )
		{bulge2intCore<return_state, 0, 1, 0, 0>(i, k, Wi, Wk, 0, seq_2->getPos(k), cCell->getLength2(), cCell->getLength4(), cCell);}
	template< stateType return_state > inline void bWiIbWkKgap2k( ENERGY_FUNC )
		{bulge2intCore<return_state, 0, 1, 0, 0>(i, k, Wi, Wk, 0, seq_2->getPos(k), cCell->getLength2(), cCell->getLength4(), cCell, affineGapCost);}

	// Internal loop functions
	template< stateType return_state > inline void ilIilK2ikWiWk( ENERGY_FUNC );
	template< stateType return_state, lengthType grow_i, lengthType grow_k, 
				 lengthType grow_Wi, lengthType grow_Wk > 
				 	inline void growInterLoop(const positionType i,
						const positionType k, const lengthType Wi,
						const lengthType Wk, const int& nuc_I, const int& nuc_K,
						stmcell*& cCell, const scoreType& affine = 0);
	template< stateType return_state > inline void ilIilK2ik( ENERGY_FUNC )
		{growInterLoop<return_state, 1, 1, 0, 0>(i, k, Wi, Wk, seq_1->getPos(i), seq_2->getPos(k), cCell);}
	template< stateType return_state > inline void ilIilK2i( ENERGY_FUNC )
		{growInterLoop<return_state, 1, 0, 0, 0>(i, k, Wi, Wk, seq_1->getPos(i), 0, cCell);}
	template< stateType return_state > inline void ilIilKgap2i( ENERGY_FUNC )
		{growInterLoop<return_state, 1, 0, 0, 0>(i, k, Wi, Wk, seq_1->getPos(i), 0, cCell, affineGapCost);}
	template< stateType return_state > inline void ilIilK2k( ENERGY_FUNC )
		{growInterLoop<return_state, 0, 1, 0, 0>(i, k, Wi, Wk, 0, seq_2->getPos(k), cCell);}
	template< stateType return_state > inline void ilIilKgap2k( ENERGY_FUNC )
		{growInterLoop<return_state, 0, 1, 0, 0>(i, k, Wi, Wk, 0, seq_2->getPos(k), cCell, affineGapCost);}
	template< stateType return_state > inline void ilIilK2WiWk( ENERGY_FUNC )
		{growInterLoop<return_state, 0, 0, 1, 1>(i, k, Wi, Wk, seq_1->getPos(i+Wi), seq_2->getPos(k+Wk), cCell);}
	template< stateType return_state > inline void ilIilK2Wi( ENERGY_FUNC )
		{growInterLoop<return_state, 0, 0, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i+Wi), 0, cCell);}
	template< stateType return_state > inline void ilIilKgap2Wi( ENERGY_FUNC )
		{growInterLoop<return_state, 0, 0, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i+Wi), 0, cCell, affineGapCost);}
	template< stateType return_state > inline void ilIilK2Wk( ENERGY_FUNC )
		{growInterLoop<return_state, 0, 0, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k+Wk), cCell);}
	template< stateType return_state > inline void ilIilKgap2Wk( ENERGY_FUNC )
		{growInterLoop<return_state, 0, 0, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k+Wk), cCell, affineGapCost);}

	// MBL functions
	template< stateType return_state > inline void mIK2ikWiWk( ENERGY_FUNC );
	template< stateType return_state, lengthType grow_i, lengthType grow_k, 
				 lengthType grow_Wi, lengthType grow_Wk >
				 	inline void mIK2unpaired(const positionType i,
						const positionType k, const lengthType Wi,
						const lengthType Wk, const int& nuc_I, const int& nuc_K,
						stmcell*& cCell, const scoreType& affine = 0);
	template< stateType return_state > inline void mIK2ik( ENERGY_FUNC )
		{mIK2unpaired<return_state, 1, 1, 0, 0>(i, k, Wi, Wk, seq_1->getPos(i), seq_2->getPos(k), cCell);}
	template< stateType return_state > inline void mIK2i( ENERGY_FUNC )
		{mIK2unpaired<return_state, 1, 0, 0, 0>(i, k, Wi, Wk, seq_1->getPos(i), 0, cCell);}
	template< stateType return_state > inline void mIKgap2i( ENERGY_FUNC )
		{mIK2unpaired<return_state, 1, 0, 0, 0>(i, k, Wi, Wk, seq_1->getPos(i), 0, cCell, affineGapCost);}
	template< stateType return_state > inline void mIK2k( ENERGY_FUNC )
		{mIK2unpaired<return_state, 0, 1, 0, 0>(i, k, Wi, Wk, 0, seq_2->getPos(k), cCell);}
	template< stateType return_state > inline void mIKgap2k( ENERGY_FUNC )
		{mIK2unpaired<return_state, 0, 1, 0, 0>(i, k, Wi, Wk, 0, seq_2->getPos(k), cCell, affineGapCost);}
	template< stateType return_state > inline void mIK2WiWk( ENERGY_FUNC )
		{mIK2unpaired<return_state, 0, 0, 1, 1>(i, k, Wi, Wk, seq_1->getPos(i+Wi), seq_2->getPos(k+Wk), cCell);}
	template< stateType return_state > inline void mIK2Wi( ENERGY_FUNC )
		{mIK2unpaired<return_state, 0, 0, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i+Wi), 0, cCell);}
	template< stateType return_state > inline void mIKgap2Wi( ENERGY_FUNC )
		{mIK2unpaired<return_state, 0, 0, 1, 0>(i, k, Wi, Wk, seq_1->getPos(i+Wi), 0, cCell, affineGapCost);}
	template< stateType return_state > inline void mIK2Wk( ENERGY_FUNC )
		{mIK2unpaired<return_state, 0, 0, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k+Wk), cCell);}
	template< stateType return_state > inline void mIKgap2Wk( ENERGY_FUNC )
		{mIK2unpaired<return_state, 0, 0, 0, 1>(i, k, Wi, Wk, 0, seq_2->getPos(k+Wk), cCell, affineGapCost);}

	template< stateType return_state > inline void bWiWkIK2mbl( ENERGY_FUNC );
	template< stateType return_state > inline void stemIK2mbl( ENERGY_FUNC );
	inline void mblCore(const scoreType left_score, const positionType i, 
							  const positionType k, const lengthType Wi,
							  const lengthType len2, const lengthType len4,
							  const lengthType Wk, const stateType return_state);

	// A method use by the constructor to initialize the flow arrays
	inline void buildFlowArrays();

	// This function should never be called
	inline void errorFunction( ENERGY_FUNC ) {
		if (highLowAlert && cCell->getState() == 0 && cCell->getScore() == big_neg) {
			std::cerr << "Score underflow error detected" << std::endl;
		}
		else {
			std::cerr << "Program error! Error function called with arguments: i = " 
						 << i << " k = " << k << " Wi = " << Wi << " Wk = " << Wk << std::flush;
			std::cerr << " Old cell score: " << cCell->getScore() << " Old cell state: " << int(cCell->getState()) << std::flush;					 
			std::cerr << " Name 1: " << seq_1->getName() << " Name 2: " << seq_2->getName() << std::endl;
		}
	};

	// The no calculation function
	inline void noCalc( ENERGY_FUNC ) {return;}
	inline void noCalc(scoreType& score, const positionType i, const positionType k, const lengthType Wi, const lengthType Wk, stmcell*& cCell) {return;}

	inline void stmstore(positionType i, positionType k,
								lengthType Wi, lengthType Wk,
								scoreType score, stateType state,
								lengthType len1, lengthType len2,
								lengthType len3, lengthType len4,
								mbllist* mblptr);

	inline void dump_all_scores(const positionType& i, const positionType& k, 
										 const lengthType& Wi, const lengthType& Wk,
										 const scoreType& score, const stateType& state,
										 stmcell*& cCell);
};

//******************
// The constructor *
//******************


template< FOLD_TEMPLATE >
inline fold< FOLD_TEMPLATE_PARAMETERS >::fold(positionType begin_1, positionType end_1, 
								  positionType begin_2, positionType end_2, 
								  results*& res,
								  const sequence* const one, const sequence* const two,
								  arguments*& argu,
								  scorematrix& score,
								  longTermMemory< ltmcell >* const long_D,
								  longTermMemory< startCell >* const startCoordinates,
								  stack_ssl< mbllist, int >* const mblMem) 
: r(res),
  plot_score(argu->boolOpt("-plot_score")),
  all_scores(argu->boolOpt("-all_scores")),
  no_backtrack(argu->boolOpt("-no_backtrack")),
  flip(argu->boolOpt("switch")),
  nobranch(argu->boolOpt("-nobranch")),
  print_all_scores(argu->boolOpt("-print_all_LS_scores")),
  lambda(argu->ltOpt("-max_length")),
  delta(argu->ltOpt("-max_diff")),
  min_loop(argu->ltOpt("-min_loop")),
  chunk_size(argu->ltOpt("-chunk_size")),
  noprune(argu->boolOpt("-no_pruning")),
  hpstart(argu->boolOpt("hpstart")),
  memroof(argu->ptOpt("memory_roof")),
  mem_info(argu->boolOpt("-memory_info")),
  seq_1(one),
  seq_2(two),
  arg(argu), 
  s_matrix(score),
  mblNuc(s_matrix.getMblNuc()),
  mblHelix(2*s_matrix.getMblAffine()),
  affineGapCost(s_matrix.getGap()),
  pruneTableSize(s_matrix.getPruneTableLength()),
  ltm(long_D),
  startCoord(startCoordinates),
  mblm(mblMem),
  local_mblm(0),
  length_I(seq_1->getLength()), 
  length_K(seq_2->getLength()),
  highLowAlert(false)
{


	//***********************************************
	//
	// Allocating memory and initalizing
	//

	std::string error = "Program error! hpstart is false and startCoord is zero. Things should not be like this.";
	if ( !hpstart && startCoord == 0) {throw exception(error, false);}
	
	try {

		// Functions in this object handles most of the printing
		error = "Could not allocate memory for out object. Most likely cause: Out of memory.";
		out = new output(arg, seq_1, seq_2, s_matrix);
		if (out == 0) {throw exception(error, false);}
	
		// Allocate the short term memory
		if ( global || realigning || mblrealign ) {

			// Global alignment and realignment
			error = "Could not allocate short term memory needed for backtrack or global alignment.";

			// Set up the short term memory

			// Dimension along the I-sequence
			const positionType i_dimension = 2;
			// Dimension along the K-sequence. Only a narrow band is needed. See
			// comment below where the range of k is defined.
			const positionType k_dimension = size_k_dimension(delta);
			// The Wi dimension. An alignment cannot be longer than this
			const lengthType Wi_dimension = end_1 - begin_1 +1;

			stm = new shortTermMemory< stmcell >(i_dimension, k_dimension,
															 Wi_dimension, arg, s_matrix,
															 k_offset(begin_1, begin_2, delta));
		}
		else {

			// The normal scan (local) alignment

			error = "Could not allocate short term memory. Most likely cause: Out of memory.";

			// Dimension along the I-sequence
			const positionType i_dimension = 2;
			// Dimension along the K-sequence. The full chunk is needed.
			const positionType k_dimension = chunk_size+3;
			// The Wi dimension. An alignment cannot be longer than this
			const lengthType Wi_dimension = lambda+1;

			stm = new shortTermMemory< stmcell >(i_dimension, k_dimension, 
															 Wi_dimension, arg, s_matrix);

		}

	}
	catch ( exception ) {throw;}
	catch ( ... ) {
		throw exception(error, false);
	}

	if (stm == 0) {
		delete out;
		throw exception(error, false);
	}

	// Set up the pruning table
	error = "Could not allocate the pruning table. Most likely cause: Out of memory.";
	
	pruneTable = new scoreType[pruneTableSize];
	if (pruneTable == 0) { delete out; delete stm; throw exception(error, false); }
	error = "Could not allocate the delta pruning table. Most likely cause: Out of memory.";
	globalDeltaPrun = new scoreType[2*delta+1];		
	if (globalDeltaPrun == 0) { delete pruneTable; delete out; delete stm; throw exception(error, false); }

	if ( !noprune ) {

		// Setup the Wi -Wk depended cost (0 in the non global case)
		if ( global ) {
			const positionType deltaLength = length_I > length_K ? length_I - length_K
			                                                     : length_K - length_I;

			for(lengthType i = 0; i < delta; i++) {
				// The maximum compensation is the length difference between the
				// two input sequences.
				scoreType pscore = delta - i < deltaLength 
				                                     ? (delta  - i) * affineGapCost
																 :  deltaLength * affineGapCost;	
				globalDeltaPrun[i] = pscore;
				globalDeltaPrun[2*delta - i] = pscore;
			}
			globalDeltaPrun[delta] = 0;
		}
		else {
			for(lengthType i = 0; i < 2*delta+1; i++) {globalDeltaPrun[i] = 0;}
		}

		// Setup the length dependent cost
		for(lengthType i = 0; i < pruneTableSize; i++) {
			pruneTable[i] = s_matrix.getPruneScore(i);
		}
	}
	else {
		for(lengthType i = 0; i < pruneTableSize; i++) {pruneTable[i] = big_neg;}
		for(lengthType i = 0; i < 2*delta+1; i++) {globalDeltaPrun[i] = 0;}
	}

	// Set up the flow arrays
	buildFlowArrays();

	// The score and coordinates of the very best alignment is stored in r
	// r (result) is initialized to the big negative number.
	r->store(big_neg, big_neg, noState, 0, 0, 0, 0);
	
	//*****************************************
	//
	// Output the headings


	// When local scores or all scores are printed an extra header section
	// is printed.
	if (plot_score || all_scores) {
			out->localscorehead();
	}
	
	
	//*******************************************
	//
	// Split sequence two into chuncks of size 2*lambda and call align.
	//
	
	// Stop is the end position for the final chunk
	positionType stop = end_2 - lambda;

	// Make sure the stop point is not placed before the starting point.
	if (stop <= begin_2) {stop = begin_2+1;}

	bool lastRun = false;
	lengthType grow = chunk_size - lambda;
	if (grow <= 0) {grow=1;}

	for(positionType begin = begin_2; begin < stop; begin+=grow) {
		positionType end = begin +chunk_size-1;
		if (end >= end_2) {end = end_2; lastRun = true;}
		try {
			align(begin_1, end_1, begin, end, lastRun);
		}
		catch ( ... ) {
		
			delete out;
			delete stm;
			delete pruneTable;
			throw;
		}
	}


	//******************************************
	//
	// Finish the output
	//

	// After print local scores or all scores print an end line. If a structure
	// is to be backtracked the print the standard header.
	if (plot_score || all_scores) {
		std::cout <<"; ******************************************************************************" << std::endl;
	}


	//*****************************************
	//
	// Clean up
	//
	
	delete out;
	delete stm;
	delete pruneTable;
	delete globalDeltaPrun;

}

template< FOLD_TEMPLATE >
void fold< FOLD_TEMPLATE_PARAMETERS >::align(positionType begin_I,
																positionType end_I,
                                 					positionType begin_K,
																positionType end_K,
															   const bool lastRun) {

	// Object for storing the best local alignment(s)
	results r_local; 
	
	// This is the maximum Wi (i,k given) for which a value has been assign.
	lengthType Wi_max;

	// Set the position of the memory matrices
	stm->setPositions(end_I, end_K);

	if (ltm != 0) {
		if ( global || realigning || mblrealign ) {
			ltm->setPositions(begin_I, begin_K);
			if (startCoord != 0) {startCoord->setPositions(begin_I, begin_K);}
		}
		else {
			ltm->setPositions(end_I, end_K);
		}
	}

	// The curr variables are global in this class the others are local to this
	// function.
	curr_begin_I = begin_I;
	curr_begin_K = begin_K;
	curr_end_I = end_I;
	curr_end_K = end_K;

	// Used to transform k position relatively to the i position to real k
	// coordinates
	k_offSet_i = begin_K - begin_I;

	// These are what must be added to i to get k during global, realign, and
	// mblrealign.
	k_low = k_offSet_i - 2*delta;
	k_high = k_offSet_i + 2*delta;


	// For all coordinates along the I-sequence
	positionType i_start = end_I;
	if (startCoord != 0) {
		if (startCoord->getIStart() > -1) {
			i_start = startCoord->getIStart();
			stm->set_I_Position(i_start);
		}
	}

	for(positionType i = i_start; i >= begin_I; i--) {

		// The j coordinate (j = i +Wi) must be less then the min_top_I if the
		// alignment is to be expanded in the j direction.
		// min_top_I must have the smallest value of i + lambda or the end of the
		// sequence.
		min_top_I = helper::min(positionType(i + lambda), curr_end_I);

		// Get the nucleotide for this position
		const positionType c_i = seq_1->getPos(i);

		// Set the k range
		const positionType k_start = global || realigning || mblrealign ?
											  i + k_low < begin_K ? begin_K :
											  i + k_low : begin_K;
		const positionType k_stop  = global || realigning || mblrealign ?
											  i + k_high > end_K ? end_K : i+k_high : end_K;
		

		// All allowed coordinates along the K-sequence.
		for(positionType k = k_stop; k >= k_start; k--) {


			// Get the nucleotide for this position
			const positionType c_k = seq_2->getPos(k);

			lengthType Wi_start = 0;

			// The i coordinate must be larger than the max_bottom_I value before
			// the alignment can be expanded in the i direction.
			max_bottom_I = curr_begin_I;

			if (hpstart) {
				//***************************************************************//
				//                                                               //
				// Store an intial alignment.                                    //
				// The algorithm will try to expand this alignment.              //
				// Currently the algortihm starts by aligning one nucleotide to  //
				// another.                                                      //
				//                                                               //
				//***************************************************************//

				// The initail state. The backtrack algorithm stop here and move to any
				// remaining parts when the hp_init value is found.
				stateType init_state = hp_init;

				// The inital score is the alignment score plus the length cost.
				scoreType startScore = s_matrix.getInit(c_i, c_k) + 
											  s_matrix.getHpLength(1,1);;
			
				// Store the alignment in short term memory.
				// Format: i position, k position, Wi length, Wk length, alignment score,
				// state, I length left, I length right, K length left, K length right.
				// Hairpins only use the left lengths, other unpaired regions need both
				// lengths.
				mbllist* empty = 0;
				stmstore(i, k, 0, 0, startScore, init_state, 1, 0, 1, 0, empty);
				
				//************************//
				//                        //
				// Initial alignment done //
				//                        //
				//************************//
			}

			if (startCoord != 0) {
			
				//****************************************************************//
				//                                                                //
				// Get the initial alignment from the startCoord object           //
				//                                                                //
				//****************************************************************//

				lengthType Wi;
				lengthType Wk;
				
				startCoord->resetCurrent(i,k);
				startCell* start = startCoord->getNextPos(i, k, Wi, Wk);
				
				if ( !hpstart ) { Wi_start = Wi; }

				while (start != 0) {
					if (start->getScore() > big_neg) {
						stmstore(i, k , Wi, Wk, start->getScore(), start->getState(),
						         0, 0, 0, 0, 0);
					}

					start = startCoord->getNextPos(i, k, Wi, Wk);
				}
			}
			

			// Get the size of the largest Wi window seen so far.
			Wi_max = stm->getWimax(i,k);


			// Make sure the top Wi size do not overflow the sequence end, or
			// lambda etc.
			checkWimax(Wi_max, i);
					
			if (mblrealign) {
				// When a mbl score is calculated its information is stored in the
				// local mblm list. When an alignment is found to be a branch point
				// then the local information is moved into the global mblm list.

				local_mblm = new stack_ssl< mbllist, int >;

				if (local_mblm == 0) {
					std::string error = "Could not allocate memory for the local mbl list. Most likely cause: Out of memory";
					throw exception(error, false);
				}
			}

			// Wi window size along the I-sequence (-1).
			// j = i + Wi. Window size 0 is a one nucleotide long alignment
			for(lengthType Wi = Wi_start; Wi <= Wi_max; Wi++) {

				// If the window size is equal to lambda then it is not possible to
				// expand in the i direction since this would make the resulting
				// alignment longer than lambda.
				if (Wi == lambda) {max_bottom_I = i;}

				// The nucleotide at the i+Wi position.
				const positionType c_j = seq_1->getPos(i+Wi);				

				// The range of the window size along the K-sequence: Wk
				const lengthType Wk_start = Wi - delta > 0 ? Wi - delta : 0;     // Plus one because the for loop
				lengthType Wk_end   = Wi + delta+1;   // runs from = to < not <=.

				if (Wk_end   > lambda  ) {Wk_end   = lambda;}
				if (Wk_end + k > end_K +1) {Wk_end = end_K - k+1;}

				// Only alignments with k larger than the max_bottom_K can be expanded
				// in the k direction.
				max_bottom_K = curr_begin_K;

				// An alignment must have an l = k + Wk which is less than the
				// min_top_K to be expanded in the l direction.
				min_top_K = (k + lambda < curr_end_K) ? k + lambda : curr_end_K;
					  
				for(lengthType Wk = Wk_start; Wk < Wk_end; Wk++) {

					// Get the current alignment stmcell. It holds score, state, and the
					// four lengths.
					stmcell* cCell = stm->getPos(i, k, Wi, Wk);

					// If the current alignment stmcell is empty (pruned away) hurry
					// to the next alignment
					if (cCell == 0) {continue;}


					// If Wk is equal to lambda then the current alignment can not
					// be expanded in the k direction. Since Wk == lambda is the last
					// Wk for a given Wi it is not nessecary to reset the max_bottom_K
					// to curr_begin_K for this Wi. For the next Wi it will happen 
					// automatically
					if (Wk == lambda) {max_bottom_K = k;}

					// The nucleotide at the k+Wk position
					const positionType c_l = seq_2->getPos(k+Wk);

					// Get the score and state
					scoreType score = cCell->getScore();
					const stateType state = cCell->getState();
					mbllist* mblpointer = 0;

					if ( (score < warn_low || score > warn_high) && !highLowAlert) {
						highLowAlert = true;
						std::cerr << "Warning. The score has a size where";
						if (score < warn_low) {std::cerr << " underflow ";}
						else {std::cerr << " overflow ";}
						std::cerr << "compared to the big_neg score is possible. ";
						std::cerr << "big_neg is defined in foldalign.h. Please change it to a lower number and recompile (or email me for help)." << std::endl;
					}

					if (mblrealign) {

						mblpointer = cCell->getPointer();
						
						if ( state == mblIK) {

							// Get the mbllist info from local_mblm
							mbllist mbl = *cCell->getPointer();
							mblpointer = mblm->push_ptr(mbl);
							cCell->setPointer(mblpointer);
						}
						else {mblpointer = cCell->getPointer();}

					}

					// If the alignment contains unpaired nucleotides which are not
					// closed by any base-pair (external unpaired nucleotides) the
					// score is recalculated to the score of mbl unpaired nucleotides.
					(this->*p2end[state])(score, i, k, Wi, Wk, cCell);

					// Store the score and state in the long term memory if
					// the structure can form the right part of a bifurcated
					// structure
					if (ltm != 0) {
					
						// Only alignments with base-pairs are stored.
						if (right_store[state]) {

							// Retrive a pointer to a long term storage cell
							ltmcell* store = ltm->putPos(i, k, Wi, Wk);

							// an store the values in it.
							store->set(score, state, mblpointer);
						}
					}


					// This prints the all_scores information. (Mainly a debug
					// option)
					if (all_scores) {dump_all_scores(i, k, Wi, Wk, score, state, cCell);}
					
					// The algorithm keeps track of the best local alignment found
					// so far and the best local alignment with coordinates i and k
					// found so far. If the current score is better than the previous
					// best local i & k alignment then store it and check if the score
					// is also better than the score of best alignment found so far.
					const double loglen = Wi > Wk ? s_matrix.getLog(Wi) : 
															  s_matrix.getLog(Wk);
					const double logScore = double(score)/loglen;
					if ((logScore > r_local.getLogScore()) && (state >= min_struc_state)) {
						r_local.store(score, logScore, state, i, k, Wi, Wk);						
						if ((score > r->getScore()) && !(global || realigning)) {
							r->store(score, logScore, state, i, k, Wi, Wk, mblpointer);
						}
					}

					// Finally it is time to expand the alignment
					expandAlignment(i, k, Wi, Wk, c_i, c_k, c_j, c_l, cCell);
				}

				// There might be a new Wi_max, check it
				Wi_max = stm->getWimax(i, k);
				max_bottom_I = curr_begin_I;
				checkWimax(Wi_max, i);
			}

			// This section handles the plot_score option.
			if (plot_score) {
				// Make sure that each line is only printed once.
				if ((k < end_K - lambda +1) || (lastRun)) {
					positionType pos_i;
					lengthType pos_j;
					positionType pos_k;
					lengthType pos_l;
					r_local.getPos(pos_i, pos_k, pos_j, pos_l);
					if (flip) {
						helper::swap(pos_i, pos_k);
						helper::swap(pos_j, pos_l);
					}

					if ((r_local.getScore() != big_neg) || print_all_scores) {

						std::cout << "LS " << pos_i << " " << pos_i+pos_j;
						std::cout << " "   << pos_k << " " << pos_k+pos_l;
						std::cout << " "   << r_local.getScore();
						std::cout << std::endl;
					}
				}

				// Reset the best score for a given i & k.
				r_local.store(big_neg, big_neg, noState, 0, 0, 0, 0);

			}
			
			if (mblrealign) {delete local_mblm;}
		}

		long stmSize = stm->getSize();
		long ltmSize = ltm != 0 ? ltm->getSize() : 0;

		if ( mem_info ) {
			std::cout << "MEM_SIZE " << i << " STM " << stmSize;
			std::cout << " mb. LTM " << ltmSize << " mb. TOTAL ";
			std::cout << stmSize + ltmSize << " mb. LEVEL " << memroof << std::endl;
		}

		if ( memroof > 0 && (stmSize + ltmSize > memroof)) {
			std::string error = "Memory roof reached.";
			throw exception(error, false);
		}

		// Slide the matrices along the I-sequence.
		stm->transfer();
		if (ltm != 0) {ltm->transfer();}

	}
}

template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::stmstore(positionType i, positionType k,
											  lengthType Wi, lengthType Wk,
											  scoreType score, stateType state,
											  lengthType len1, lengthType len2,
											  lengthType len3, lengthType len4,
											  mbllist* mblptr) {

	// Prune
	if (score < pruneTable[Wi] - globalDeltaPrun[Wi - Wk + delta]||
	    score < pruneTable[Wk] - globalDeltaPrun[Wi - Wk + delta]) {return;}
	
	// Keep the maximum score
	stmcell* cell_ptr = stm->putPos(i, k, Wi, Wk, score);
	if (score > cell_ptr->getScore()) {
		cell_ptr->set(score, state, len1, len2, len3, len4, mblptr);
	}
}

template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::checkWimax(lengthType& Wi_max, const positionType i) {

	// Do not overstep the end of the sequence.
	// This could probabaly be optimized
	if (i + Wi_max +1 > curr_end_I) {Wi_max = curr_end_I - i;}

	if (Wi_max >= lambda) {Wi_max = lambda;}

}

template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::dump_all_scores(
											const positionType& i, const positionType& k, 
											const lengthType& Wi, const lengthType& Wk,
											const scoreType& score, const stateType& state,
											stmcell*& cCell) {

	// This prints the all_scores information. (Mainly a debug
	// option)

	const int exp_score = check_coord_local(i, k, Wi, Wk);

	if (!flip) {
		std::cout << "; AS " << int(i) << " " << int(i + Wi);
	  	std::cout << " " << int(k) << " " << int(k+Wk);
   	std::cout << " LTM score " << int(score);
   	std::cout << " STM score " << int(cCell->getScore());
   	std::cout << " State " << int(state) << " Lengths: ";
   	std::cout << int(cCell->getLength1()) << " ";
   	std::cout << int(cCell->getLength2()) << " ";
   	std::cout << int(cCell->getLength3()) << " ";
   	std::cout << int(cCell->getLength4()) << " Exp_score: ";
   	std::cout << exp_score << std::endl;
	}
	else {
   	std::cout << "; AS " << int(k) << " " << int(k + Wk);
   	std::cout << " " << int(i) << " " << int(i+Wi);
   	std::cout << " LTM score " << int(score);
   	std::cout << " STM score " << int(cCell->getScore());
   	std::cout << " State " << int(state) << " Lengths: ";
		std::cout << int(cCell->getLength3()) << " ";
		std::cout << int(cCell->getLength4()) << " ";
		std::cout << int(cCell->getLength1()) << " ";
		std::cout << int(cCell->getLength2()) << " Exp_score: ";
		std::cout << exp_score << std::endl;
	}
}

template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::expandAlignment(positionType i, positionType k, lengthType Wi, lengthType Wk, const positionType c_i, const positionType c_k, const positionType c_j, const positionType c_l, stmcell*& cCell) {

	// Calls a function which expands the input alignment depended on its
	// coordinates.

	// The alignment can be expanded in four directions. These can be combined
	// into 16 different cases. The most insterresting is case 15 which expands in
	// all four direction. Case 0 is the case where expansion in any direction is
	// not possible.

	// A number maskin which direction the alignment can be expanded.
	const int exp_score = check_coord_local(i, k, Wi, Wk);

	// Checks if the alignment can be expanded to a longer alignment in only one
	// of the sequences. ie is this true when Wi or Wk is growes without the 
	// other: Wi - delta <= Wk <= Wi + delta 
	const bool WkWi_m1 = Wk - Wi -1 >= -delta ? true : false;
	const bool WkWi_p1 = Wk - Wi +1 <=  delta ? true : false;

	
	const bool ki_m1 = k - 1 < i + k_low && (global || realigning || mblrealign)
	                 ? false : true;
	const bool ki_p1 = k + 1 > i + k_high && (global || realigning || mblrealign)
	                 ? false : true;

	// The state is used to index the energy function.
	const stateType state = cCell->getState();

	switch (exp_score) {
	case 15:
		// Expand in all directions
		expandAlignment_ikWiWk(i, k, Wi, Wk, c_i, c_k, c_j, c_l, WkWi_m1, WkWi_p1, ki_m1, ki_p1, cCell);
		return;
	case 14:
		// Expand in all directions except i

		// Expand WiWk
		(this->*p2calc_WiWk[state])(i, k, Wi+1, Wk+1, c_i, c_k, c_j, c_l, cCell);

		if (WkWi_m1) {
			// Expand Wi
			(this->*p2calc_Wi[state])(i, k, Wi+1, Wk, c_i, c_k, c_j, c_l, cCell);
		}
	
		expandAlignment_kWk(i, k, Wi, Wk, c_i, c_k, c_j, c_l, WkWi_m1, WkWi_p1, ki_m1, ki_p1, cCell);

		// Branching
		(this->*p2calc_mbl[state])(i, k, Wi, Wk, c_i, c_k, c_j, c_l, cCell);
		return;
	case 13:
		// Expand in all directions except k

		// Expand WiWk
		(this->*p2calc_WiWk[state])(i, k, Wi+1, Wk+1, c_i, c_k, c_j, c_l, cCell);

		if (WkWi_p1) {
			// Expand Wk
			(this->*p2calc_Wk[state])(i, k, Wi, Wk+1, c_i, c_k, c_j, c_l, cCell);
		}
	
		expandAlignment_iWi(i, k, Wi, Wk, c_i, c_k, c_j, c_l, WkWi_m1, WkWi_p1, ki_m1, ki_p1, cCell);

		// Branching
		(this->*p2calc_mbl[state])(i, k, Wi, Wk, c_i, c_k, c_j, c_l, cCell);
		return;
	case 11:
		// Expand in all directions except Wi

		// Expand ik
		(this->*p2calc_ik[state])(i-1, k-1, Wi+1, Wk+1, c_i, c_k, c_j, c_l, cCell);

		if (WkWi_m1 && ki_m1) {
			// Expand i
			(this->*p2calc_i[state])(i-1, k, Wi+1, Wk, c_i, c_k, c_j, c_l, cCell);
		}
	
		expandAlignment_kWk(i, k, Wi, Wk, c_i, c_k, c_j, c_l, WkWi_m1, WkWi_p1, ki_m1, ki_p1, cCell);
		return;
	case 7:
		// Expand in all directions except Wk

		// Expand ik
		(this->*p2calc_ik[state])(i-1, k-1, Wi+1, Wk+1, c_i, c_k, c_j, c_l, cCell);

		if (WkWi_p1 && ki_p1) {
			// Expand k
			(this->*p2calc_k[state])(i, k-1, Wi, Wk+1, c_i, c_k, c_j, c_l, cCell);
		}
	
		expandAlignment_iWi(i, k, Wi, Wk, c_i, c_k, c_j, c_l, WkWi_m1, WkWi_p1, ki_m1, ki_p1, cCell);
		return;
	case 3:
		// Expand in the ik directions

		// Unpaired on one side
		// Expand ik
		(this->*p2calc_ik[state])(i-1, k-1, Wi+1, Wk+1, c_i, c_k, c_j, c_l, cCell);

		// One side with gap
		if (WkWi_m1 && ki_m1) {
			// Expand i
			(this->*p2calc_i[state])(i-1, k, Wi+1, Wk, c_i, c_k, c_j, c_l, cCell);
		}

		if (WkWi_p1 && ki_p1) {
			// Expand k
			(this->*p2calc_k[state])(i, k-1, Wi, Wk+1, c_i, c_k, c_j, c_l, cCell);
		}
		return;
	case 5:
		// Expand in the iWi directions
		expandAlignment_iWi(i, k, Wi, Wk, c_i, c_k, c_j, c_l, WkWi_m1, WkWi_p1, ki_m1, ki_p1, cCell);

		return;
	case 9:
		// Expand in the iWk directions

		if (WkWi_m1 && ki_m1) {
			// Expand i
			(this->*p2calc_i[state])(i-1, k, Wi+1, Wk, c_i, c_k, c_j, c_l, cCell);
		}
		if (WkWi_p1) {
			// Expand Wk
			(this->*p2calc_Wk[state])(i, k, Wi, Wk+1, c_i, c_k, c_j, c_l, cCell);
		}
		return;
	case 6:
		// Expand in the kWi directions

		if (WkWi_p1 && ki_p1) {
			// Expand k
			(this->*p2calc_k[state])(i, k-1, Wi, Wk+1, c_i, c_k, c_j, c_l, cCell);
		}
		if (WkWi_m1) {
			// Expand Wi
			(this->*p2calc_Wi[state])(i, k, Wi+1, Wk, c_i, c_k, c_j, c_l, cCell);
		}
		return;
	case 10:
		// Expand in the kWk directions
		expandAlignment_kWk(i, k, Wi, Wk, c_i, c_k, c_j, c_l, WkWi_m1, WkWi_p1, ki_m1, ki_p1, cCell);

		return;
	case 12:
		// Expand in the WiWk directions

		(this->*p2calc_WiWk[state])(i, k, Wi+1, Wk+1, c_i, c_k, c_j, c_l, cCell);
		if (WkWi_m1) {
			(this->*p2calc_Wi[state])(i, k, Wi+1, Wk, c_i, c_k, c_j, c_l, cCell);
		}
		if (WkWi_p1) {
			// Expand Wk
			(this->*p2calc_Wk[state])(i, k, Wi, Wk+1, c_i, c_k, c_j, c_l, cCell);
		}
		// Expand MBL
		(this->*p2calc_mbl[state])(i, k, Wi, Wk, c_i, c_k, c_j, c_l, cCell);
		return;
	case 1:
		// Expand i
		if (WkWi_m1 && ki_m1) {
			(this->*p2calc_i[state])(i-1, k, Wi+1, Wk, c_i, c_k, c_j, c_l, cCell);
		}
		return;
	case 2:
		// Expand k
		if (WkWi_p1 && ki_p1) {
			(this->*p2calc_k[state])(i, k-1, Wi, Wk+1, c_i, c_k, c_j, c_l, cCell);
		}
		return;
	case 4:
		// Expand Wi
		if (WkWi_m1) {
			(this->*p2calc_Wi[state])(i, k, Wi+1, Wk, c_i, c_k, c_j, c_l, cCell);
		}
		return;
	case 8:
		// Expand Wk
		if (WkWi_p1) {
			(this->*p2calc_Wk[state])(i, k, Wi, Wk+1, c_i, c_k, c_j, c_l, cCell);
		}
		return;
	case 0:
		// This is the last stmcell no expansion is possible
		if ( global || mblrealign || realigning ) {

			scoreType score = cCell->getScore();
			stateType state = cCell->getState();

			// End correction
			(this->*p2end[state])(score, i, k, Wi, Wk, cCell);

			r->store(score, big_neg, state,i, k, Wi, Wk, cCell->getPointer());
		}
		return;
	default:
		std::string error = "Program error! Illegal expand alignment score found.";
		throw exception(error, false);
	}

}


template< FOLD_TEMPLATE >
inline int fold< FOLD_TEMPLATE_PARAMETERS >::check_coord_local(const positionType i, const positionType k, const lengthType Wi, const lengthType Wk) const {

	// This function determines in which directions the alignment can be expanded
	// It should be used for local alignment (scan alignments) since there are
	// extra constrains during global alignment and realignment.

	int exp_score = 0;
	const int i_score = 1;
	const int k_score = 2;
	const int Wi_score = 4;
	const int Wk_score = 8;
	if ( i > max_bottom_I) {exp_score += i_score;}
	if ( i + Wi < min_top_I ) {exp_score += Wi_score;}
	if ( k > max_bottom_K) {exp_score += k_score;}
	if ( k + Wk < min_top_K ) {exp_score += Wk_score;}
	return exp_score;
}


template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::expandAlignment_ikWiWk(
														positionType i, positionType k,
														lengthType Wi, lengthType Wk,
														const positionType c_i,
														const positionType c_k,
														const positionType c_j,
														const positionType c_l,
														const bool WkWi_m1, const bool WkWi_p1,
														const bool ki_m1, const bool ki_p1,
														stmcell*& cCell) {
	
	// Call the functions which expand an alignment in all directions

	stateType state = cCell->getState();

	// True if the next set of nucleotides basepairs otherwise false;
	const bool base_pair_I = s_matrix.getBasepair(seq_1->getPos(i-1), seq_1->getPos(i+Wi+1));
	const bool base_pair_K = s_matrix.getBasepair(seq_2->getPos(k-1), seq_2->getPos(k+Wk+1));

	if ( base_pair_I && (Wi >= min_loop) && (Wi < lambda -1) ) {

		// The nucleotides at the new coordinates in the I-sequence base-pairs
		// the hairpin loop has the minimum length and does not exeed lambda
		
		if ( base_pair_K && (Wk >= min_loop) && (Wk < lambda -1) ) {

			// The K-sequence can also base-pair
			// Expand with base-pair in both sequences
			
			(this->*p2calc_ikWiWk[state])(i-1, k-1, Wi+2, Wk+2, c_i, c_k, c_j, c_l, cCell);

			if (Wk - Wi < delta -1) {

				// Expand with base-pair in the kWk direction
				(this->*p2calc_kWk[state])(i, k-1, Wi, Wk+2, c_i, c_k, c_j, c_l, cCell);
			}
		}

		if (Wi - Wk < delta -1) {

			// Expand with base-pair in the iWi direction
			(this->*p2calc_iWi[state])(i-1, k, Wi+2, Wk, c_i, c_k, c_j, c_l, cCell);
		}

	} else if ( base_pair_K && (Wk >= min_loop) && (Wk < lambda -1) && (Wk - Wi < delta -1) ) {

		// No I-sequence base-pair. Expand with bp in the K-sequence
		(this->*p2calc_kWk[state])(i, k-1, Wi, Wk+2, c_i, c_k, c_j, c_l, cCell);

	}

	// Unpaired on one side
	// Expand ik
	(this->*p2calc_ik[state])(i-1, k-1, Wi+1, Wk+1, c_i, c_k, c_j, c_l, cCell);


	// Expand jl
	(this->*p2calc_WiWk[state])(i, k, Wi+1, Wk+1, c_i, c_k, c_j, c_l, cCell);

	// One side with gap
	// Expand i
	if (WkWi_m1) {
		// Expand i
		if (ki_m1) {
			(this->*p2calc_i[state])(i-1, k, Wi+1, Wk, c_i, c_k, c_j, c_l, cCell);
		}
		// Expand Wi
		(this->*p2calc_Wi[state])(i, k, Wi+1, Wk, c_i, c_k, c_j, c_l, cCell);
	}

	if (WkWi_p1) {
		if (ki_p1) {
		// Expand k
			(this->*p2calc_k[state])(i, k-1, Wi, Wk+1, c_i, c_k, c_j, c_l, cCell);
		}
		// Expand Wk
		(this->*p2calc_Wk[state])(i, k, Wi, Wk+1, c_i, c_k, c_j, c_l, cCell);
	}

	// Expand with a multibranched loop
		
	(this->*p2calc_mbl[state])(i, k, Wi, Wk, c_i, c_k, c_j, c_l, cCell);
}

template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::expandAlignment_iWi(positionType i, positionType k, lengthType Wi, lengthType Wk, const positionType c_i, const positionType c_k, const positionType c_j, const positionType c_l, const bool WkWi_m1, const bool WkWi_p1, const bool ki_m1, const bool ki_p1, stmcell*& cCell) {
	// Expand in the iWi directions

	const stateType state = cCell->getState();
	const bool base_pair_I = s_matrix.getBasepair(seq_1->getPos(i-1), seq_1->getPos(i+Wi+1));

	if (base_pair_I && (Wi >= min_loop) && (Wi < lambda -1) && (Wi - Wk < delta -1) ) {
		// Potential insert bp
		(this->*p2calc_iWi[state])(i-1, k, Wi+2, Wk, c_i, c_k, c_j, c_l, cCell);
	}
	if (WkWi_m1) {
		if (ki_m1) {
			(this->*p2calc_i[state])(i-1, k, Wi+1, Wk, c_i, c_k, c_j, c_l, cCell);
		}
		(this->*p2calc_Wi[state])(i, k, Wi+1, Wk, c_i, c_k, c_j, c_l, cCell);
	}

}

template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::expandAlignment_kWk(positionType i, positionType k, lengthType Wi, lengthType Wk, const positionType c_i, const positionType c_k, const positionType c_j, const positionType c_l, const bool WkWi_m1, const bool WkWi_p1, const bool ki_m1, const bool ki_p1, stmcell*& cCell) {

	const stateType state = cCell->getState();
	const bool base_pair_K = s_matrix.getBasepair(seq_2->getPos(k-1), seq_2->getPos(k+Wk+1));

	if (base_pair_K && (Wk >= min_loop) && (Wk < lambda -1) && (Wk - Wi < delta -1) ) {
		// Expand kl
		(this->*p2calc_kWk[state])(i, k-1, Wi, Wk+2, c_i, c_k, c_j, c_l, cCell);
	}
	if (WkWi_p1) {
		if (ki_p1) {
			(this->*p2calc_k[state])(i, k-1, Wi, Wk+1, c_i, c_k, c_j, c_l, cCell);
		}
		(this->*p2calc_Wk[state])(i, k, Wi, Wk+1, c_i, c_k, c_j, c_l, cCell);
	}
}


/******************************************************************************
*
* These functions check if the alignment can expand with an extra bp
*
*/


template< FOLD_TEMPLATE >
inline bool fold< FOLD_TEMPLATE_PARAMETERS >::extra_bp_coord_IK(const positionType i, const positionType k, const lengthType Wi, const lengthType Wk) const {

	// This function returns true if an alignment with the i,k, Wi,Wk coordinates
	// can be expanded into an alignment i-1, k-1, Wi+2, Wk+2 with a basepair
	
	// Using the correct borders (max_bottom and min_top) leeds to trouble during
	// backtrack. An "easy" solution has been chosen to solve the problem.
	// Instead of using the actual borders the sequence borders are used (position
	// 1 and length). This allows the algorithm to make structures where there is
	// a single base-pair at the end of a structure. The backtrack algorithm
	// should be able to handle this, but it is still a bit messy.
	// The old "correct" code has been out-commented but is left below.

//	if ((i > max_bottom_I) && (k > max_bottom_K) && 
//	    (Wi < min_top_I) && (Wk < min_top_K) &&
//		 (Wi +2 <= lambda) && (Wk +2 <= lambda)) {
//		return s_matrix.getBasepair(seq_1->getPos(i-1), seq_1->getPos(i + Wi +1)) &&
//		       s_matrix.getBasepair(seq_2->getPos(k-1), seq_2->getPos(k + Wk +1));
//	}
	
	if ((i > 1) && (k > 1) && 
	    (i+Wi < length_I) && (k+Wk < length_K) &&
		 (Wi +2 <= lambda) && (Wk +2 <= lambda)) {
		return s_matrix.getBasepair(seq_1->getPos(i-1), seq_1->getPos(i + Wi +1)) &&
		       s_matrix.getBasepair(seq_2->getPos(k-1), seq_2->getPos(k + Wk +1));
	}
	return false;
}


template< FOLD_TEMPLATE >
inline bool fold< FOLD_TEMPLATE_PARAMETERS >::extra_bp_coord_I(const positionType i, const positionType k, const lengthType Wi, const lengthType Wk) const {

	// Can the alignment be expanded with a base-pair in the I-sequence?
	
	// See important comments in the extra_bp_coord_IK function.

//	if ((i > max_bottom_I) && (Wi < min_top_I) && (Wi +2 <= lambda)) {
	if ((i > 1) && (i+Wi < length_I) && (Wi +2 <= lambda)) {
		return s_matrix.getBasepair(seq_1->getPos(i-1), seq_1->getPos(i + Wi +1));
	}
	
	return false;
}


template< FOLD_TEMPLATE >
inline bool fold< FOLD_TEMPLATE_PARAMETERS >::extra_bp_coord_K(const positionType i, const positionType k, const lengthType Wi, const lengthType Wk) const {

	// Can the alignment be expanded with a base-pair in the K-sequence?
	
	// See important comments in the extra_bp_coord_IK function.

//	if ((k > max_bottom_K) && (Wk < min_top_K) && (Wk +2 <= lambda)) {
	if ((k > 1) && (k+Wk < length_K) && (Wk +2 <= lambda)) {
		return s_matrix.getBasepair(seq_2->getPos(k-1), seq_2->getPos(k + Wk +1));
	}
	
	return false;
}


/******************************************************************************
*
* Energy functions
*
*/

/*********************************
*
* The end functions.
* Recalculates the scores of unpaired nucleotides which are not inclosed by a
* base-pair. ie unpaired nucleotides at the ends of the alignment. The score
* is recalculated from hairpin, bulge, or internal loop to mbl unpaired state.
*/


template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::hp2end(scoreType& score, 
                                  const positionType i, const positionType k, 
											 const lengthType Wi, const lengthType Wk, 
											 stmcell*& cCell) {
	
	// Recalculate the score from hairpin state to outside state ie mbl state.
	
	lengthType len1 = cCell->getLength1();
	lengthType len3 = cCell->getLength3();
	
	// Remove the hairpin length score
	score -= s_matrix.getHpLength(len1, len3);
	
	// Add the mblNuc score
	score += (len1 + len3)*mblNuc;
	
}	

template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::stem2end(scoreType& score, 
                                  const positionType i, const positionType k, 
											 const lengthType Wi, const lengthType Wk, 
											 stmcell*& cCell) {

	// Add the nonGC end score
	score += s_matrix.getNonGCEnd(seq_1->getPos(i), 
											seq_1->getPos(i+Wi));
	score += s_matrix.getNonGCEnd(seq_2->getPos(k),
											seq_2->getPos(k+Wk));

	
}	


template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::bulge_ik2end(scoreType& score, 
                                  const positionType i, const positionType k, 
											 const lengthType Wi, const lengthType Wk, 
											 stmcell*& cCell) {
	
	// Recalculate the score from hairpin state to outside state ie mbl state.
	
	lengthType len1 = cCell->getLength1();
	lengthType len3 = cCell->getLength3();
	
	// Remove the hairpin length score
	score -= s_matrix.getBulgeLength(len1, len3);
	
	// Add the mblNuc score
	score += (len1 + len3)*mblNuc;
	
	// Add the nonGC end score
	score += s_matrix.getNonGCEnd(seq_1->getPos(i+len1), 
											seq_1->getPos(i+Wi));
	score += s_matrix.getNonGCEnd(seq_2->getPos(k+len3),
											seq_2->getPos(k+Wk));
}	


template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::bulge_WiWk2end(scoreType& score, 
                                  const positionType i, const positionType k, 
											 const lengthType Wi, const lengthType Wk, 
											 stmcell*& cCell) {
	
	// Recalculate the score from hairpin state to outside state ie mbl state.
	
	lengthType len2 = cCell->getLength2();
	lengthType len4 = cCell->getLength4();
	
	// Remove the hairpin length score
	score -= s_matrix.getBulgeLength(len2, len4);
	
	// Add the mblNuc score
	score += (len2 + len4)*mblNuc;
	
	// Add the nonGC end score
	score += s_matrix.getNonGCEnd(seq_1->getPos(i), 
											seq_1->getPos(i+Wi-len2));
	score += s_matrix.getNonGCEnd(seq_2->getPos(k),
											seq_2->getPos(k+Wk-len4));
}	


template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end(scoreType& score,  
                                  const positionType i, const positionType k, 
											 const lengthType Wi, const lengthType Wk, 
											 stmcell*& cCell) {
	
	// Recalculate an internal loop state score to an outside (mbl) state score
	
	lengthType len1 = cCell->getLength1();
	lengthType len2 = cCell->getLength2();
	lengthType len3 = cCell->getLength3();
	lengthType len4 = cCell->getLength4();
		
	score -= s_matrix.getIntLoopLength(len1, len2);
	score -= s_matrix.getIntLoopLength(len3, len4);

	score += (len1 + len2 + len3 + len4) * mblNuc;

	// Add the nonGC end score
	score += s_matrix.getNonGCEnd(seq_1->getPos(i+len1), 
											seq_1->getPos(i+Wi-len2));
	score += s_matrix.getNonGCEnd(seq_2->getPos(k+len3),
											seq_2->getPos(k+Wk-len4));
}	

//=========================================================================
// Hairpin functions


template< FOLD_TEMPLATE >
template<stateType return_state, lengthType grow1, lengthType grow3> 
inline void fold< FOLD_TEMPLATE_PARAMETERS >::hp_align_one_pos(const positionType i, const positionType k,
                                   const lengthType Wi, const lengthType Wk,
											  const int& nuc_I, const int& nuc_K,
											  stmcell*& cCell, const scoreType& affine) {

	// Hairpin to pot hp end
	scoreType score = cCell->getScore();

	score += s_matrix.getInit(nuc_I, nuc_K);

	score += affine;

	// Get the lengths
	lengthType len1 = cCell->getLength1();
	lengthType len3 = cCell->getLength3();

	// Subtract the old length cost
	score -= s_matrix.getHpLength(len1, len3);
	// Grow the lengths
	len1+=grow1; len3+=grow3;
	// Add the new length cost
	score += s_matrix.getHpLength(len1, len3);

	// Store
	stmstore(i, k, Wi, Wk, score, return_state, len1, 0, len3, 0, cCell->getPointer());

}


template< FOLD_TEMPLATE >
template<stateType return_state>
inline void fold< FOLD_TEMPLATE_PARAMETERS >::hp_init_pot_bp_IK( ENERGY_FUNC ) {

	// If the next set of nucleotides do not base-pair then it is not necessary
	// to do this calculation

	if (!extra_bp_coord_IK(i, k, Wi, Wk)) {return;}

	// Hairpin to pot hp end
	scoreType score = cCell->getScore();
	const lengthType len1 = cCell->getLength1();
	const lengthType len3 = cCell->getLength3();
	
	score += s_matrix.getScore(seq_1->getPos(i), seq_1->getPos(i+Wi), 
	                           seq_2->getPos(k), seq_2->getPos(k+Wk));
	
	// Add the closing stacking score if both loop lengths are above three
	if ((len1 > 3) && (len3 > 3)) {
		score += s_matrix.getHpClose(seq_1->getPos(i), seq_1->getPos(i+Wi), c_i, c_j);
		score += s_matrix.getHpClose(seq_2->getPos(k), seq_2->getPos(k+Wk), c_k, c_l);
	}

	stmstore(i, k, Wi, Wk, score, return_state, len1, 0, len3, 0, cCell->getPointer());

}

//=============================================================================
// Stem functions


template< FOLD_TEMPLATE >
template<stateType return_state>
inline void fold< FOLD_TEMPLATE_PARAMETERS >::stemIK2ikWiWk( ENERGY_FUNC ) {

	// Stem to stem
	
	scoreType score = cCell->getScore();

	// Add the base-pair sub scores
	score += s_matrix.getScore(seq_1->getPos(i), seq_1->getPos(i+Wi), seq_2->getPos(k), seq_2->getPos(k+Wk));
	
	// Add the stackings
	score += s_matrix.getStack(seq_1->getPos(i), seq_1->getPos(i+Wi), c_i, c_j);
	score += s_matrix.getStack(seq_2->getPos(k), seq_2->getPos(k+Wk), c_k, c_l);

	stmstore(i, k, Wi, Wk, score, return_state, 0, 0, 0, 0, cCell->getPointer());

}


template< FOLD_TEMPLATE >
template<stateType return_state>
inline void fold< FOLD_TEMPLATE_PARAMETERS >::stemIK2onepair(const positionType i, const positionType k, const lengthType Wi, const lengthType Wk, const int& nuc_start, const int& nuc_end, const positionType& c_start, const positionType& c_end, stmcell*& cCell, const scoreType& affine) {

	// Stem to stem_I_stem_gap_kWk
	
	scoreType score = cCell->getScore();

	// Current bp (init because it is bp to gap
	score += s_matrix.getInit(nuc_start,    0);
	score += s_matrix.getInit(nuc_end, 0);
	score += affine;
	
	// Add the stackings
	score += s_matrix.getStack(nuc_start, nuc_end, c_start, c_end);

	stmstore(i, k, Wi, Wk, score, return_state, 0, 0, 0, 0, cCell->getPointer());

}



//=============================================================================
// Bulge functions


template< FOLD_TEMPLATE >
template<stateType return_state, lengthType grow_I, lengthType grow_K>
inline void fold< FOLD_TEMPLATE_PARAMETERS >::growBulge_ik(const positionType i, const positionType k, const lengthType Wi, const lengthType Wk, const int& nuc_I, const int& nuc_K, stmcell*& cCell, const scoreType affine) {

	scoreType score = cCell->getScore();
	lengthType len1 = cCell->getLength1();
	lengthType len3 = cCell->getLength3();

	// Substitution score
	score += s_matrix.getInit(nuc_I, nuc_K);
	score += affine;
	
	// Subtract the old length cost
	score -= s_matrix.getBulgeLength(len1, len3);
	// Grow the lengths
	len1+=grow_I; len3+=grow_K;
	// Add the new length cost
	score += s_matrix.getBulgeLength(len1, len3);

	// Store (The ones are due to the potential unpaired nuc. in the bp)
	stmstore(i, k, Wi, Wk, score, return_state, len1, 0, len3, 0, cCell->getPointer());
}	


template< FOLD_TEMPLATE >
template<stateType return_state, lengthType grow_I, lengthType grow_K>
inline void fold< FOLD_TEMPLATE_PARAMETERS >::growBulge_WiWk(const positionType i, const positionType k, const lengthType Wi, const lengthType Wk, const int& nuc_I, const int& nuc_K, stmcell*& cCell, const scoreType affine) {

	scoreType score = cCell->getScore();
	lengthType len2 = cCell->getLength2();
	lengthType len4 = cCell->getLength4();

	// Substitution score
	score += s_matrix.getInit(nuc_I, nuc_K);
	score += affine;
	
	// Subtract the old length cost
	score -= s_matrix.getBulgeLength(len2, len4);
	// Grow the lengths
	len2+=grow_I; len4+=grow_K;
	// Add the new length cost
	score += s_matrix.getBulgeLength(len2, len4);

	// Store (The ones are due to the potential unpaired nuc. in the bp)
	stmstore(i, k, Wi, Wk, score, return_state, 0, len2, 0, len4, cCell->getPointer());
}	


template< FOLD_TEMPLATE >
template<stateType return_state>
inline void fold< FOLD_TEMPLATE_PARAMETERS >::bWiIbWkK2ikWiWkCore(const positionType i, const positionType k, const lengthType Wi, const lengthType Wk, const positionType c_i, const positionType c_k, const positionType c_j, const positionType c_l, stmcell*& cCell, const lengthType& len_I, const lengthType& len_K, const lengthType& len1, const lengthType& len2, const lengthType& len3, const lengthType& len4 ) {

	// If the next set of nucleotides do not base-pair then it is not necessary
	// to do this calculation
	if (!extra_bp_coord_IK(i, k, Wi, Wk)) {return;}

	// This alignment can only be expanded into the stem state

	// Pot hp end to stem
	scoreType score = cCell->getScore();
	
	
	// The potential end is scored as hairpin here, later it will be recalculated
	// to base-pair (if the next nucleotides base-pairs)
	score += s_matrix.getScore(seq_1->getPos(i), seq_1->getPos(i+Wi), 
	                           seq_2->getPos(k), seq_2->getPos(k+Wk));



	// The positions of the opening bp nucleotides
	const positionType po_i = i + len1 + 1; // +1 since it is one step back
	const positionType po_k = k + len3 + 1;
	const positionType po_j = i + Wi - len2 - 1;
	const positionType po_l = k + Wk - len4 - 1;
	// The nucleotides at the base-pair which opens the loop
	const positionType o_i = seq_1->getPos(po_i);
	const positionType o_k = seq_2->getPos(po_k);
	const positionType o_j = seq_1->getPos(po_j);
	const positionType o_l = seq_2->getPos(po_l);

	if ((len_I > 1) || (len_K > 1)) {
		// Either of the bulges are longer than one no stacking is allowed
		// Add non-GC cost

		// The closing bp
		score += s_matrix.getNonGCEnd(seq_1->getPos(i), seq_1->getPos(i+Wi));
		score += s_matrix.getNonGCEnd(seq_2->getPos(k), seq_2->getPos(k+Wk));
		
		// The opening bp
		score += s_matrix.getNonGCEnd(o_i, o_j);
		score += s_matrix.getNonGCEnd(o_k, o_l);

	}
	else {
		// Stacking across bulge is possible
		score += s_matrix.getStack(seq_1->getPos(i), seq_1->getPos(i+Wi), o_i, o_j);
		score += s_matrix.getStack(seq_2->getPos(k), seq_2->getPos(k+Wk), o_k, o_l);
	
	}

	stmstore(i, k, Wi, Wk, score, return_state, 0, 0, 0, 0, cCell->getPointer());


}


template< FOLD_TEMPLATE >
template<stateType return_state, lengthType grow_i, lengthType grow_k, lengthType grow_Wi, lengthType grow_Wk>
inline void fold< FOLD_TEMPLATE_PARAMETERS >::bulge2intCore(const positionType i, const positionType k, const lengthType Wi, const lengthType Wk, const int& nuc_I, const int& nuc_K, const lengthType& len_I, const lengthType& len_K, stmcell*& cCell, const scoreType& affine ) {

	scoreType score = cCell->getScore();
	lengthType len1 = cCell->getLength1();
	lengthType len2 = cCell->getLength2();
	lengthType len3 = cCell->getLength3();
	lengthType len4 = cCell->getLength4();

	// Substitution score
	score += s_matrix.getInit(nuc_I, nuc_K);
	score += affine;
	
	// Subtract the old bulge length cost
	score -= s_matrix.getBulgeLength(len_I, len_K);
	// Grow the lengths
	len1+=grow_i; len2+=grow_Wi, len3+=grow_k; len4+=grow_Wk;
	// Add the new internal loop length cost
	score += s_matrix.getIntLoopLength(len1, len2);
	score += s_matrix.getIntLoopLength(len3, len4);

	// Store (The ones are due to the potential unpaired nuc. in the bp)
	stmstore(i, k, Wi, Wk, score, return_state, len1, len2, len3, len4, cCell->getPointer());

}


//=============================================================================
// Internal loop functions



template< FOLD_TEMPLATE >
template<stateType return_state>
inline void fold< FOLD_TEMPLATE_PARAMETERS >::ilIilK2ikWiWk( ENERGY_FUNC ) {

	// If the next set of nucleotides do not base-pair then it is not necessary
	// to do this calculation
	if (!extra_bp_coord_IK(i, k, Wi, Wk)) {return;}

	// Pot il end to stem
	scoreType score = cCell->getScore();
	lengthType len1 = cCell->getLength1();
	lengthType len2 = cCell->getLength2();
	lengthType len3 = cCell->getLength3();
	lengthType len4 = cCell->getLength4();
	
	
	// Add the substitution score
	score += s_matrix.getScore(seq_1->getPos(i), seq_1->getPos(i+Wi),
	                           seq_2->getPos(k), seq_2->getPos(k+Wk));

	// Add the opening and closing stacking score
	if ((len1 > 0) && (len2 > 0)) {
		const positionType pos_i =  i + len1;
		const positionType pos_Wi = i + Wi - len2;
		score += s_matrix.getIntLoopOpen(seq_1->getPos(pos_i+1), 
		                                 seq_1->getPos(pos_Wi-1), 
													seq_1->getPos(pos_i),
													seq_1->getPos(pos_Wi));
		score += s_matrix.getIntLoopClose(seq_1->getPos(i), seq_1->getPos(i+Wi),
		                                  c_i, c_j);
	}
	if ((len3 > 0) && (len4 > 0)) {
		const positionType pos_k =  k + len3;
		const positionType pos_Wk = k + Wk - len4;
		score += s_matrix.getIntLoopOpen(seq_2->getPos(pos_k+1), 
		                                 seq_2->getPos(pos_Wk-1), 
													seq_2->getPos(pos_k), 
													seq_2->getPos(pos_Wk));
		score += s_matrix.getIntLoopClose(seq_2->getPos(k), seq_2->getPos(k+Wk), 
		                                  c_k, c_l);
	}

	// Store (The ones are due to the potential unpaired nuc. in the bp)
	stmstore(i, k, Wi, Wk, score, return_state, len1, len2, len3, len4, 
	         cCell->getPointer());

}


template< FOLD_TEMPLATE >
template<stateType return_state, lengthType grow_i, lengthType grow_k, 
         lengthType grow_Wi, lengthType grow_Wk >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::growInterLoop(
                                    const positionType i, const positionType k,
												const lengthType Wi, const lengthType Wk,
												const int& nuc_I, const int& nuc_K, 
												stmcell*& cCell, const scoreType& affine) {

	// Pot hp end to stem
	scoreType score = cCell->getScore();
	lengthType len1 = cCell->getLength1();
	lengthType len2 = cCell->getLength2();
	lengthType len3 = cCell->getLength3();
	lengthType len4 = cCell->getLength4();
	
	
	// The potential end is scored as hairpin here, later it will be recalculated
	// to base-pair (if the next nucleotides base-pairs)
	score += s_matrix.getInit(nuc_I, nuc_K);
	score += affine;
	
	// Subtract the old length cost
	score -= s_matrix.getIntLoopLength(len1, len2);
	score -= s_matrix.getIntLoopLength(len3, len4);
	// Grow the lengths
	len1+=grow_i; len2+=grow_Wi; len3+=grow_k; len4+=grow_Wk;
	// Add the new length cost
	score += s_matrix.getIntLoopLength(len1, len2);
	score += s_matrix.getIntLoopLength(len3, len4);

	// Store (The ones are due to the potential unpaired nuc. in the bp)
	stmstore(i, k, Wi, Wk, score, return_state, len1, len2, len3, len4, 
	         cCell->getPointer());

}
//=============================================================================
// These functions handle mbls


template< FOLD_TEMPLATE >
template<stateType return_state>
inline void fold< FOLD_TEMPLATE_PARAMETERS >::mIK2ikWiWk( ENERGY_FUNC ) {

	// If the next set of nucleotides do not base-pair then it is not necessary
	// to do this calculation
	if (!extra_bp_coord_IK(i, k, Wi, Wk)) {return;}

	scoreType score = cCell->getScore();
		
	// Add the substitution score
	score += s_matrix.getScore(seq_1->getPos(i), seq_1->getPos(i+Wi), 
	                           seq_2->getPos(k), seq_2->getPos(k+Wk));
	
	// Add the mbl closing cost
	score += s_matrix.getMbl();
	
	// Add the non-GC stem closing cost
	score += s_matrix.getNonGCEnd(seq_1->getPos(i), seq_1->getPos(i+Wi));
	score += s_matrix.getNonGCEnd(seq_2->getPos(k), seq_2->getPos(k+Wk));


	stmstore(i, k, Wi, Wk, score, return_state, 0, 0, 0, 0, cCell->getPointer());

}


template< FOLD_TEMPLATE >
template<stateType return_state, lengthType grow_i, lengthType grow_k, 
         lengthType grow_Wi, lengthType grow_Wk>
inline void fold< FOLD_TEMPLATE_PARAMETERS >::mIK2unpaired(
                                    const positionType i, const positionType k, 
												const lengthType Wi, const lengthType Wk, 
												const int& nuc_I, const int& nuc_K, 
												stmcell*& cCell, const scoreType& affine) {


	// Pot hp end to stem
	scoreType score = cCell->getScore();
	lengthType len1 = cCell->getLength1();
	lengthType len2 = cCell->getLength2();
	lengthType len3 = cCell->getLength3();
	lengthType len4 = cCell->getLength4();
	
	
	// The potential end is scored as hairpin here, later it will be recalculated
	// to base-pair (if the next nucleotides base-pairs)
	score += s_matrix.getInit(nuc_I, nuc_K);
	score += affine;
	
	// Grow the lengths
	len1+=grow_i; len2+=grow_Wi; len3+=grow_k; len4+=grow_Wk;

	// Add the mbl unpaired score for both nucleotides
	score += (grow_i + grow_k + grow_Wi + grow_Wk)*mblNuc;
	
	// Store (The ones are due to the potential unpaired nuc. in the bp)
	stmstore(i, k, Wi, Wk, score, return_state, len1, len2, len3, len4, 
	         cCell->getPointer());


}


template< FOLD_TEMPLATE >
template<stateType return_state>
inline void fold< FOLD_TEMPLATE_PARAMETERS >::bWiWkIK2mbl( ENERGY_FUNC ) {

	scoreType score = cCell->getScore();

	// Change the bulge score into the mbl/end score
	bulge_WiWk2end(score, i, k, Wi, Wk, cCell);
	
	mblCore(score, i, k, Wi, Wk, cCell->getLength2(), cCell->getLength4(), 
	        return_state);
}


template< FOLD_TEMPLATE >
template<stateType return_state>
inline void fold< FOLD_TEMPLATE_PARAMETERS >::stemIK2mbl( ENERGY_FUNC ) {

	scoreType score = cCell->getScore();
	const stateType state = cCell->getState();

	if (right_branch[state]) {

		// Add the non-GC stem close cost in the stem cases
		// but not in the mbl cases because the score does not apply to the mbl
		// case.

		stem2end(score, i, k, Wi, Wk, cCell);
	}

	mblCore(score, i, k, Wi, Wk, cCell->getLength2(), cCell->getLength4(), 
	        return_state);

}


template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::mblCore(const scoreType left_score,
                                  const positionType i, const positionType k,
											 const lengthType Wi, const lengthType Wk,
											 const lengthType len2, const lengthType len4,
											 const stateType return_state) {

	if ( (Wi < min_loop +4) || (Wk < min_loop +4) ) {return;}

	// Calculate the (i,k) coords of the right part
	const positionType right_i = i + Wi + 1;
	const positionType right_k = k + Wk + 1;
	
	// Calculate the maximum length of the right part
	const lengthType max_right_Wi = (i + lambda < curr_end_I ) ?
												lambda - Wi -1: curr_end_I -i - Wi;

	const lengthType max_right_Wk = (k + lambda < curr_end_K) ?
												lambda - Wk -1: curr_end_K - k - Wk;

	if (( max_right_Wi < min_loop +4) || (max_right_Wk < min_loop +4)) {return;}

	const lengthType low_left_delta  = Wi - Wk - delta;
	const lengthType high_left_delta = Wi - Wk + delta;

	if ( (global || mblrealign) && 
		  ((right_k < right_i + k_low) || (right_k > right_i + k_high))) {return;}
	
	ltm->resetCurrent(right_i, right_k);

	while (true) {
		lengthType right_Wi;
		lengthType right_Wk;

		// Get the right part information from the ltm
		ltmcell* rightCell = ltm->getNextPos(right_i, right_k, right_Wi, right_Wk);
		
		// There are not any potential right parts left at this position
		if (rightCell == 0) {return;}

		// Ensure that the right part has a branch state. This is always true
		// during local scan alignment but during global or realignmnet it is
		// not always true.
		if ( (global || mblrealign) &&
			  ( !right_branch[rightCell->getState()] )) {continue;}

		// Make sure that the length of the K-subsequence is no more than delta
		// nucleotides shorter than the I-subsequence
		// Keep getting new right parts until the minimum length is reached.
		if (low_left_delta >= right_Wk - right_Wi) {continue;}

		// Check the length of the right part
		// The right_Wk > max_right_Wk keeps the total K-subsequence length below
		// lambda
		// high_left_delta etc. Keeps the length difference between the two
		// subsequences below delta.
		if ((right_Wk > max_right_Wk) || 
		    (high_left_delta <= right_Wk - right_Wi)) {
			// The K right part is to long. Get the next right Wi

			ltm->switchWi(right_i, right_k);
			continue;
		}
		
		// Length check of the length of the right part on the I sequence
		// If it is to long then there are not any possible right parts left
		if (right_Wi > max_right_Wi) {ltm->deletePos(right_i, right_k); return;}

		const scoreType score = left_score + rightCell->getScore() + mblHelix;

		mbllist* mblpointer = 0;
		if ( (mblrealign) ) {
			// Store the extra multibranch information for later use. This is only
			// done during the mblrealignment. (normal realign should not be
			// branched)
			
			mbllist mbl(i, k, Wi - len2, Wk - len4,
			            right_i, right_k, right_Wi, right_Wk, score);
			mblpointer = local_mblm->push_ptr(mbl);
		}

		stmstore(i, k, Wi + right_Wi+1, Wk + right_Wk +1, score, return_state,
		         0,0,0,0, mblpointer);

	}
}

//=================================================================
// FLOW CONTROL ARRAYS BELOW


template< FOLD_TEMPLATE >
inline void fold< FOLD_TEMPLATE_PARAMETERS >::buildFlowArrays() {

// Diane, never drink coffee that has been anywhere near a fish. 

// These functions is used to pick the function dependent on the previous state
// they controls how a new score is calculated dependend on the previous state

	// Init all to give a big negative score
	helper::init_array(p2calc_ikWiWk, flow_size, &fold< FOLD_TEMPLATE_PARAMETERS >::errorFunction);
	helper::init_array(p2calc_iWi,   flow_size, &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc);
	helper::init_array(p2calc_kWk,   flow_size, &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc);
	helper::init_array(p2calc_ik,   flow_size, &fold< FOLD_TEMPLATE_PARAMETERS >::errorFunction);
	helper::init_array(p2calc_WiWk,   flow_size, &fold< FOLD_TEMPLATE_PARAMETERS >::errorFunction);
	helper::init_array(p2calc_i,    flow_size, &fold< FOLD_TEMPLATE_PARAMETERS >::errorFunction);
	helper::init_array(p2calc_Wi,    flow_size, &fold< FOLD_TEMPLATE_PARAMETERS >::errorFunction);
	helper::init_array(p2calc_k,    flow_size, &fold< FOLD_TEMPLATE_PARAMETERS >::errorFunction);
	helper::init_array(p2calc_Wk,    flow_size, &fold< FOLD_TEMPLATE_PARAMETERS >::errorFunction);
	helper::init_array(p2calc_mbl,    flow_size, &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc);
	helper::init_array(p2end,         flow_size, &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc);

	helper::init_array(right_branch,    flow_size, false);

	right_branch[stem_IK] 									= true;
	right_branch[stem_I_stem_gap_kWk] 					= true;
	right_branch[stem_gap_iWi_stem_K] 					= true;

	// The alignments stored in the long term memory are either the same as those
	// which can be the right part of a multiloop (during scan), or all alignments
	// (during global or realigment). This is controled by the right_store array.
	if ( realigning ) {
		// During the "real" backtrack all states are saved.
		helper::init_array(right_store, flow_size, true);
	}
	else if ( mblrealign ) {
		// During mblrealignment and global all the states which can be involved
		// in multibranch points must be stored. Otherwise it is not possible to
		// relocate the two parts of the mbl.
		helper::copyArray(right_store, right_branch, flow_size);
		right_store[mblIK] = true;
	}
	else{
		// The scan case where only the stem states needs to be stored.
		helper::copyArray(right_store, right_branch, flow_size);
	}

	p2calc_ikWiWk[hp_init]      						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_align_ik]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_align_WiWk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_I_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_I_WiWk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_K_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_K_WiWk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_Wi_ik]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_Wi_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_Wk_ik]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_Wk_Wi]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_iWi_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_iWi_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_kWk_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_kWk_Wi]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_iWk_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_iWk_Wi]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_kWi_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_init_gap_kWi_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_init_pot_bp_IK<hp_pb_IK>;
	p2calc_ikWiWk[hp_pb_IK]								= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ikWiWk<stem_IK>;				// -> stem_IK
	p2calc_ikWiWk[stem_IK]								= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ikWiWk<stem_IK>;				//	stem_IK
	p2calc_ikWiWk[stem_I_stem_gap_kWk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ikWiWk<stem_IK>;	// stem_IK
	p2calc_ikWiWk[stem_no_mbl_I_stem_gap_kWk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ikWiWk<stem_IK>;	// stem_IK
	p2calc_ikWiWk[stem_gap_iWi_stem_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ikWiWk<stem_IK>;	// stem_IK
	p2calc_ikWiWk[stem_no_mbl_gap_iWi_stem_K]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ikWiWk<stem_IK>;	// stem_IK
	p2calc_ikWiWk[bi_I_bk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2ikWiWk<bi_pb_I_bk_pb_K>;	// bi_pb_I_bk_pb_K
	p2calc_ikWiWk[bi_I_bk_gap_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2ikWiWk<bi_pb_I_bk_pb_K>;	// bi_pb_I_bk_pb_K
	p2calc_ikWiWk[bi_I_bk_gap_kWk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2ikWiWk<bi_pb_I_bk_pb_K>;	// bi_pb_I_bk_pb_K
	p2calc_ikWiWk[bi_I_bk_gap_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2ikWiWk<bi_pb_I_bk_pb_K>;	// bi_pb_I_bk_pb_K
	p2calc_ikWiWk[bi_pb_I_bk_pb_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ikWiWk<stem_IK>;		// stem_IK
	p2calc_ikWiWk[bi_gap_i_bk_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2ikWiWk<bi_pb_I_bk_pb_K>;	// bi_pb_I_bk_pb_K
	p2calc_ikWiWk[bi_gap_i_bk_gap_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2ikWiWk<bi_pb_I_bk_pb_K>;	// bi_pb_I_bk_pb_K
	p2calc_ikWiWk[bi_gap_iWi_bk_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2ikWiWk<bi_pb_I_bk_pb_K>;	// bi_pb_I_bk_pb_K
	p2calc_ikWiWk[bi_gap_Wi_bk_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2ikWiWk<bi_pb_I_bk_pb_K>;	// bi_pb_I_bk_pb_K
	p2calc_ikWiWk[bi_gap_Wi_bk_gap_k]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2ikWiWk<bi_pb_I_bk_pb_K>;	// bi_pb_I_bk_pb_K
	p2calc_ikWiWk[bWi_I_bWk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ikWiWk<bWi_pb_I_bWk_pb_K>;	// bWi_pb_I_bWk_pb_K
	p2calc_ikWiWk[bWi_I_bWk_gap_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ikWiWk<bWi_pb_I_bWk_pb_K>;	// bWi_pb_I_bWk_pb_K
	p2calc_ikWiWk[bWi_I_bWk_gap_kWk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ikWiWk<bWi_pb_I_bWk_pb_K>;	// bWi_pb_I_bWk_pb_K
	p2calc_ikWiWk[bWi_I_bWk_gap_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ikWiWk<bWi_pb_I_bWk_pb_K>;	// bWi_pb_I_bWk_pb_K
	p2calc_ikWiWk[bWi_pb_I_bWk_pb_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ikWiWk<stem_IK>;		// stem_IK
	p2calc_ikWiWk[bWi_gap_i_bWk_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ikWiWk<bWi_pb_I_bWk_pb_K>;	// bWi_pb_I_bWk_pb_K
	p2calc_ikWiWk[bWi_gap_i_bWk_gap_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ikWiWk<bWi_pb_I_bWk_pb_K>;	// bWi_pb_I_bWk_pb_K
	p2calc_ikWiWk[bWi_gap_iWi_bWk_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ikWiWk<bWi_pb_I_bWk_pb_K>;	// bWi_pb_I_bWk_pb_K
	p2calc_ikWiWk[bWi_gap_Wi_bWk_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ikWiWk<bWi_pb_I_bWk_pb_K>;	// bWi_pb_I_bWk_pb_K
	p2calc_ikWiWk[bWi_gap_Wi_bWk_gap_k]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ikWiWk<bWi_pb_I_bWk_pb_K>;	// bWi_pb_I_bWk_pb_K
	p2calc_ikWiWk[il_I_il_K_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_I_il_K_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_I_il_gap_k_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_I_il_gap_k_WiWk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_I_il_gap_Wk_ik]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_I_il_gap_Wk_Wi]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_I_il_gap_kWk_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_I_il_gap_kWk_Wi]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_pb_I_il_pb_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ikWiWk<stem_IK>;		// stemIK
	p2calc_ikWiWk[il_gap_iWi_il_K_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_gap_iWi_il_K_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_gap_i_il_K_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_gap_i_il_K_WiWk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_gap_i_il_gap_Wk_k]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_gap_i_il_gap_Wk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_gap_Wi_il_K_ik]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_gap_Wi_il_K_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_gap_Wi_il_gap_k_i]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[il_gap_Wi_il_gap_k_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ikWiWk<il_pb_I_il_pb_K>;	// il_pb_I_il_pb_K
	p2calc_ikWiWk[mblIK]									= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_bWi_I_mbl_bWk_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_bWi_I_mbl_bWk_gap_Wk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_bWi_gap_Wi_mbl_bWk_K]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_I_mbl_il_K_ik]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_I_mbl_il_K_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_I_mbl_il_gap_k_i]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_I_mbl_il_gap_k_WiWk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_I_mbl_il_gap_kWk_i]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_I_mbl_il_gap_kWk_Wi]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_I_mbl_il_gap_Wk_ik]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_I_mbl_il_gap_Wk_Wi]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_pb_I_mbl_il_pb_K]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ikWiWk<stem_IK>;		// stem_IK
	p2calc_ikWiWk[mbl_il_gap_Wi_mbl_il_K_ik]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_gap_Wi_mbl_il_K_Wk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_gap_Wi_mbl_il_gap_k_i]	= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_gap_Wi_mbl_il_gap_k_Wk]	= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_gap_iWi_mbl_il_K_k]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_gap_iWi_mbl_il_K_Wk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_gap_i_mbl_il_K_k]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_gap_i_mbl_il_K_WiWk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_gap_i_mbl_il_gap_Wk_k]	= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K
	p2calc_ikWiWk[mbl_il_gap_i_mbl_il_gap_Wk_Wi]	= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ikWiWk<mbl_il_pb_I_mbl_il_pb_K>;	// mbl_il_pb_I_mbl_il_pb_K


	p2calc_iWi[hp_pb_IK] 								= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2iWi<stem_no_mbl_I_stem_gap_kWk>;					// hp_pb_I_hp_gap_kWk
	p2calc_iWi[stem_IK] 									= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2iWi<stem_I_stem_gap_kWk>;					// stem_I_stem_gap_kWk
	p2calc_iWi[stem_I_stem_gap_kWk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIstemgkWk2iWi<stem_I_stem_gap_kWk>;				// stem_I_stem_gap_kWk
	p2calc_iWi[stem_no_mbl_I_stem_gap_kWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIstemgkWk2iWi<stem_no_mbl_I_stem_gap_kWk>;				// stem_I_stem_gap_kWk
	p2calc_iWi[stem_gap_iWi_stem_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2iWi<stem_I_stem_gap_kWk>;
	p2calc_iWi[stem_no_mbl_gap_iWi_stem_K]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2iWi<stem_I_stem_gap_kWk>;
	p2calc_iWi[bi_pb_I_bk_pb_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2iWi<stem_no_mbl_I_stem_gap_kWk>;			// stem_I_bk_pb_gap_kWk
	p2calc_iWi[bWi_pb_I_bWk_pb_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2iWi<stem_no_mbl_I_stem_gap_kWk>;			// stem_I_bWk_pb_gap_kWk
	p2calc_iWi[il_pb_I_il_pb_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2iWi<stem_no_mbl_I_stem_gap_kWk>;			// stem_I_il_pb_gap_kWk
	p2calc_iWi[mbl_il_pb_I_mbl_il_pb_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2iWi<stem_no_mbl_I_stem_gap_kWk>;			// stem_I_mbl_il_pb_gap_kWk
	

	p2calc_kWk[hp_pb_IK]									= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2kWk<stem_no_mbl_gap_iWi_stem_K>;
	p2calc_kWk[stem_IK]									= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2kWk<stem_gap_iWi_stem_K>;					// stem_gap_iWi_stem_K
	p2calc_kWk[stem_I_stem_gap_kWk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2kWk<stem_gap_iWi_stem_K>;		// stem_gap_iWi_stem_K
	p2calc_kWk[stem_no_mbl_I_stem_gap_kWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2kWk<stem_gap_iWi_stem_K>;		// stem_gap_iWi_stem_K
	p2calc_kWk[stem_gap_iWi_stem_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemgiWistemK2kWk<stem_gap_iWi_stem_K>;		// stem_gap_iWi_stem_K
	p2calc_kWk[stem_no_mbl_gap_iWi_stem_K]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemgiWistemK2kWk<stem_no_mbl_gap_iWi_stem_K>;		// stem_gap_iWi_stem_K
	p2calc_kWk[bi_pb_I_bk_pb_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2kWk<stem_no_mbl_gap_iWi_stem_K>;			// bi_pb_gap_iWi_stem_K
	p2calc_kWk[bWi_pb_I_bWk_pb_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2kWk<stem_no_mbl_gap_iWi_stem_K>;			// bWi_pb_gap_iWi_stem_K
	p2calc_kWk[il_pb_I_il_pb_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2kWk<stem_no_mbl_gap_iWi_stem_K>;			// il_pb_gap_iWi_stem_K
	p2calc_kWk[mbl_il_pb_I_mbl_il_pb_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2kWk<stem_no_mbl_gap_iWi_stem_K>;			// mbl_il_pb_gap_iWi_stem_K


	p2calc_ik[hp_init]									= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_align_ik>;		 // hp_init_align
	p2calc_ik[hp_init_align_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_align_ik>;		 // hp_init_align
	p2calc_ik[hp_init_align_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_align_ik>;		 // hp_init_align
	p2calc_ik[hp_init_gap_I_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_align_ik>;		 // hp_init_align
	p2calc_ik[hp_init_gap_I_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_align_ik>;		 // hp_init_align
	p2calc_ik[hp_init_gap_K_i]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_align_ik>;		 // hp_init_align
	p2calc_ik[hp_init_gap_K_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_align_ik>;		 // hp_init_align
	p2calc_ik[hp_init_gap_Wi_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_gap_Wi_ik>;   // hp_init_gap_Wi
	p2calc_ik[hp_init_gap_Wi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_gap_Wi_ik>;   // hp_init_gap_Wi
	p2calc_ik[hp_init_gap_Wk_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_gap_Wk_ik>;   // hp_init_gap_Wk
	p2calc_ik[hp_init_gap_Wk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_gap_Wk_ik>;   // hp_init_gap_Wk
	p2calc_ik[hp_init_gap_iWi_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_gap_Wi_ik>;  // hp_init_gap_Wi
	p2calc_ik[hp_init_gap_iWi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_gap_Wi_ik>;  // hp_init_gap_Wi
	p2calc_ik[hp_init_gap_kWk_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_gap_Wk_ik>;  // hp_init_gap_Wk
	p2calc_ik[hp_init_gap_kWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_gap_Wk_ik>;  // hp_init_gap_Wk
	p2calc_ik[hp_init_gap_iWk_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_gap_Wk_ik>;  // hp_init_gap_Wk
	p2calc_ik[hp_init_gap_iWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_gap_Wk_ik>;  // hp_init_gap_Wk
	p2calc_ik[hp_init_gap_kWi_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_gap_Wi_ik>;  // hp_init_gap_Wi
	p2calc_ik[hp_init_gap_kWi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_ik<hp_init_gap_Wi_ik>;  // hp_init_gap_Wi
	p2calc_ik[hp_pb_IK]    								= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;		 // hp_init_align
	p2calc_ik[stem_IK]									= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ik<bi_I_bk_K>;					// bi_I_bk_K
	p2calc_ik[stem_I_stem_gap_kWk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ik<bi_I_bk_gap_Wk>;			// bi_I_bk_gap_Wk
	p2calc_ik[stem_no_mbl_I_stem_gap_kWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_ik[stem_gap_iWi_stem_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ik<bi_gap_Wi_bk_K>;			// bi_gap_Wi_bk_K
	p2calc_ik[stem_no_mbl_gap_iWi_stem_K]			= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_ik[bi_I_bk_K]									= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ik<bi_I_bk_K>;					// bi_I_bk_K
	p2calc_ik[bi_I_bk_gap_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ik<bi_I_bk_K>;					// bi_I_bk_K
	p2calc_ik[bi_I_bk_gap_kWk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ik<bi_I_bk_gap_Wk>;				 // bi_I_bk_K
	p2calc_ik[bi_I_bk_gap_Wk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ik<bi_I_bk_gap_Wk>;				 // bi_I_bk_K
	p2calc_ik[bi_pb_I_bk_pb_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;  			 // il_I_il_K
	p2calc_ik[bi_gap_i_bk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ik<bi_I_bk_K>;  				 // bi_I_bk_K
	p2calc_ik[bi_gap_i_bk_gap_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ik<bi_I_bk_gap_Wk>;				 // bi_I_bk_K
	p2calc_ik[bi_gap_iWi_bk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ik<bi_gap_Wi_bk_K>;				 // bi_I_bk_K
	p2calc_ik[bi_gap_Wi_bk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ik<bi_gap_Wi_bk_K>;				 // bi_I_bk_K
	p2calc_ik[bi_gap_Wi_bk_gap_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2ik<bi_gap_Wi_bk_K>;				 // bi_I_bk_K
	p2calc_ik[bWi_I_bWk_K]								= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ik<il_I_il_K_ik>;				// il_I_il_K
	p2calc_ik[bWi_I_bWk_gap_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ik<il_I_il_K_ik>;				// il_I_il_K
	p2calc_ik[bWi_I_bWk_gap_kWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ik<il_I_il_gap_Wk_ik>;				// il_I_il_K
	p2calc_ik[bWi_I_bWk_gap_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ik<il_I_il_gap_Wk_ik>;				// il_I_il_K
	p2calc_ik[bWi_pb_I_bWk_pb_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;				// il_I_il_K
	p2calc_ik[bWi_gap_i_bWk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ik<il_I_il_K_ik>;				// il_I_il_K
	p2calc_ik[bWi_gap_i_bWk_gap_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ik<il_I_il_gap_Wk_ik>;				// il_I_il_K
	p2calc_ik[bWi_gap_iWi_bWk_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ik<il_gap_Wi_il_K_ik>;				// il_I_il_K
	p2calc_ik[bWi_gap_Wi_bWk_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ik<il_gap_Wi_il_K_ik>;				// il_I_il_K
	p2calc_ik[bWi_gap_Wi_bWk_gap_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2ik<il_gap_Wi_il_K_ik>;				// il_I_il_K
	p2calc_ik[il_I_il_K_ik]								= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_I_il_K_ik>;					// il_I_il_K
	p2calc_ik[il_I_il_K_WiWk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_I_il_K_ik>;					// il_I_il_K
	p2calc_ik[il_I_il_gap_k_i]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_I_il_K_ik>;					// il_I_il
	p2calc_ik[il_I_il_gap_k_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_I_il_K_ik>;					// il_I_il
	p2calc_ik[il_I_il_gap_Wk_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_I_il_gap_Wk_ik>;				// il_I_il_gap_Wk
	p2calc_ik[il_I_il_gap_Wk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_I_il_gap_Wk_ik>;				// il_I_il_gap_Wk
	p2calc_ik[il_I_il_gap_kWk_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_I_il_gap_Wk_ik>;				// il_I_il_gap_Wk
	p2calc_ik[il_I_il_gap_kWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_I_il_gap_Wk_ik>;				// il_I_il_gap_Wk
	p2calc_ik[il_pb_I_il_pb_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;				// il_I_il_K
	p2calc_ik[il_gap_iWi_il_K_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_gap_Wi_il_K_ik>;				// il_gap_Wi_il_K
	p2calc_ik[il_gap_iWi_il_K_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_gap_Wi_il_K_ik>;				// il_gap_Wi_il_K
	p2calc_ik[il_gap_i_il_K_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_I_il_K_ik>;					// il_I_il_K
	p2calc_ik[il_gap_i_il_K_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_I_il_K_ik>;					// il_I_il_K
	p2calc_ik[il_gap_i_il_gap_Wk_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_I_il_gap_Wk_ik>;				// il_I_il_K
	p2calc_ik[il_gap_i_il_gap_Wk_Wi]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_I_il_gap_Wk_ik>;				// il_I_il_K
	p2calc_ik[il_gap_Wi_il_K_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_gap_Wi_il_K_ik>;				// il_gap_Wi_il_K
	p2calc_ik[il_gap_Wi_il_K_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_gap_Wi_il_K_ik>;				// il_gap_Wi_il_K
	p2calc_ik[il_gap_Wi_il_gap_k_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_gap_Wi_il_K_ik>;				// il_gap_Wi_il_K
	p2calc_ik[il_gap_Wi_il_gap_k_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2ik<il_gap_Wi_il_K_ik>;				// il_gap_Wi_il_K
	p2calc_ik[mblIK]  									= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_K_ik>;						// mbl_bi_I_mbl_bk_K
	p2calc_ik[mbl_bWi_I_mbl_bWk_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_K_ik>;						// mbl_bi_I_mbl_bk_K
	p2calc_ik[mbl_bWi_I_mbl_bWk_gap_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_gap_Wk_ik>;						// mbl_bi_I_mbl_bk_K
	p2calc_ik[mbl_bWi_gap_Wi_mbl_bWk_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_gap_Wi_mbl_il_K_ik>;						// mbl_bi_I_mbl_bk_K
	p2calc_ik[mbl_il_I_mbl_il_K_ik]  				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_K_ik>;				// mbl_il_I_mbl_il_K
	p2calc_ik[mbl_il_I_mbl_il_K_WiWk]  				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_K_ik>;				// mbl_il_I_mbl_il_K
	p2calc_ik[mbl_il_I_mbl_il_gap_k_i]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_K_ik>;				// mbl_il_I_mbl_il_K
	p2calc_ik[mbl_il_I_mbl_il_gap_k_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_K_ik>;				// mbl_il_I_mbl_il_K
	p2calc_ik[mbl_il_I_mbl_il_gap_kWk_i]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_gap_Wk_ik>;			// mbl_il_I_mbl_il_gap_Wk
	p2calc_ik[mbl_il_I_mbl_il_gap_kWk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_gap_Wk_ik>;			// mbl_il_I_mbl_il_gap_Wk
	p2calc_ik[mbl_il_I_mbl_il_gap_Wk_ik]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_gap_Wk_ik>;
	p2calc_ik[mbl_il_I_mbl_il_gap_Wk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_gap_Wk_ik>;
	p2calc_ik[mbl_il_pb_I_mbl_il_pb_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_ik[mbl_il_gap_Wi_mbl_il_K_ik]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_gap_Wi_mbl_il_K_ik>;
	p2calc_ik[mbl_il_gap_Wi_mbl_il_K_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_gap_Wi_mbl_il_K_ik>;
	p2calc_ik[mbl_il_gap_Wi_mbl_il_gap_k_i]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_gap_Wi_mbl_il_K_ik>;
	p2calc_ik[mbl_il_gap_Wi_mbl_il_gap_k_Wk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_gap_Wi_mbl_il_K_ik>;
	p2calc_ik[mbl_il_gap_iWi_mbl_il_K_k]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_gap_Wi_mbl_il_K_ik>;
	p2calc_ik[mbl_il_gap_iWi_mbl_il_K_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_gap_Wi_mbl_il_K_ik>;
	p2calc_ik[mbl_il_gap_i_mbl_il_K_k]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_K_ik>;
	p2calc_ik[mbl_il_gap_i_mbl_il_K_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_K_ik>;
	p2calc_ik[mbl_il_gap_i_mbl_il_gap_Wk_k]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_gap_Wk_ik>;
	p2calc_ik[mbl_il_gap_i_mbl_il_gap_Wk_Wi]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2ik<mbl_il_I_mbl_il_gap_Wk_ik>;


	p2calc_WiWk[hp_init] 								= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_align_WiWk>;  	// hp_init_align
	p2calc_WiWk[hp_init_align_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_align_WiWk>;
	p2calc_WiWk[hp_init_align_WiWk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_align_WiWk>;
	p2calc_WiWk[hp_init_gap_I_k] 						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_gap_I_WiWk>;    // hp_init_gap_I
	p2calc_WiWk[hp_init_gap_I_WiWk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_gap_I_WiWk>;    // hp_init_gap_I
	p2calc_WiWk[hp_init_gap_K_i] 						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_gap_K_WiWk>;    // hp_init_gap_K
	p2calc_WiWk[hp_init_gap_K_WiWk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_gap_K_WiWk>;    // hp_init_gap_K
	p2calc_WiWk[hp_init_gap_Wi_ik]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_align_WiWk>;
	p2calc_WiWk[hp_init_gap_Wi_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_align_WiWk>;
	p2calc_WiWk[hp_init_gap_Wk_ik]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_align_WiWk>;
	p2calc_WiWk[hp_init_gap_Wk_Wi]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_align_WiWk>;
	p2calc_WiWk[hp_init_gap_iWi_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_gap_I_WiWk>;
	p2calc_WiWk[hp_init_gap_iWi_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_gap_I_WiWk>;
	p2calc_WiWk[hp_init_gap_kWk_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_gap_K_WiWk>;
	p2calc_WiWk[hp_init_gap_kWk_Wi]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_gap_K_WiWk>;
	p2calc_WiWk[hp_init_gap_iWk_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_gap_I_WiWk>;
	p2calc_WiWk[hp_init_gap_iWk_Wi]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_gap_I_WiWk>;
	p2calc_WiWk[hp_init_gap_kWi_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_gap_K_WiWk>;
	p2calc_WiWk[hp_init_gap_kWi_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_WiWk<hp_init_gap_K_WiWk>;
	p2calc_WiWk[hp_pb_IK]								= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_WiWk[stem_IK] 								= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2WiWk<bWi_I_bWk_K>;				// bWi_I_bWk_K
	p2calc_WiWk[stem_I_stem_gap_kWk] 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2WiWk<bWi_I_bWk_gap_k>;				// bWi_I_bWk_K
	p2calc_WiWk[stem_no_mbl_I_stem_gap_kWk]	 	= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_WiWk[stem_gap_iWi_stem_K] 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2WiWk<bWi_gap_i_bWk_K>;				// bWi_I_bWk_K
	p2calc_WiWk[stem_no_mbl_gap_iWi_stem_K] 		= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_WiWk[bi_I_bk_K]				 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2WiWk<il_I_il_K_WiWk>;
	p2calc_WiWk[bi_I_bk_gap_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2WiWk<il_I_il_gap_k_WiWk>;
	p2calc_WiWk[bi_I_bk_gap_kWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2WiWk<il_I_il_gap_k_WiWk>;
	p2calc_WiWk[bi_I_bk_gap_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2WiWk<il_I_il_K_WiWk>;
	p2calc_WiWk[bi_pb_I_bk_pb_K]  					= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_WiWk[bi_gap_i_bk_K] 		 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2WiWk<il_gap_i_il_K_WiWk>;
	p2calc_WiWk[bi_gap_i_bk_gap_Wk] 		 			= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2WiWk<il_gap_i_il_K_WiWk>;
	p2calc_WiWk[bi_gap_iWi_bk_K] 		 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2WiWk<il_gap_i_il_K_WiWk>;
	p2calc_WiWk[bi_gap_Wi_bk_K] 		 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2WiWk<il_I_il_K_WiWk>;
	p2calc_WiWk[bi_gap_Wi_bk_gap_k]	 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2WiWk<il_I_il_gap_k_WiWk>;
	p2calc_WiWk[bWi_I_bWk_K]  							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2WiWk<bWi_I_bWk_K>;
	p2calc_WiWk[bWi_I_bWk_gap_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2WiWk<bWi_I_bWk_gap_k>;
	p2calc_WiWk[bWi_I_bWk_gap_kWk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2WiWk<bWi_I_bWk_gap_k>;
	p2calc_WiWk[bWi_I_bWk_gap_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2WiWk<bWi_I_bWk_K>;
	p2calc_WiWk[bWi_pb_I_bWk_pb_K]  					= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_WiWk[bWi_gap_iWi_bWk_K] 	 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2WiWk<bWi_gap_i_bWk_K>;
	p2calc_WiWk[bWi_gap_i_bWk_K] 	 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2WiWk<bWi_gap_i_bWk_K>;
	p2calc_WiWk[bWi_gap_i_bWk_gap_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2WiWk<bWi_gap_i_bWk_K>;
	p2calc_WiWk[bWi_gap_Wi_bWk_K] 	 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2WiWk<bWi_I_bWk_K>;
	p2calc_WiWk[bWi_gap_Wi_bWk_gap_k] 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2WiWk<bWi_I_bWk_gap_k>;
	p2calc_WiWk[il_I_il_K_ik]  						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_I_il_K_WiWk>;
	p2calc_WiWk[il_I_il_K_WiWk]  						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_I_il_K_WiWk>;
	p2calc_WiWk[il_I_il_gap_k_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_I_il_gap_k_WiWk>;
	p2calc_WiWk[il_I_il_gap_k_WiWk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_I_il_gap_k_WiWk>;
	p2calc_WiWk[il_I_il_gap_Wk_ik]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_I_il_K_WiWk>;
	p2calc_WiWk[il_I_il_gap_Wk_Wi]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_I_il_K_WiWk>;
	p2calc_WiWk[il_I_il_gap_kWk_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_I_il_gap_k_WiWk>;
	p2calc_WiWk[il_I_il_gap_kWk_Wi]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_I_il_gap_k_WiWk>;
	p2calc_WiWk[il_pb_I_il_pb_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_WiWk[il_gap_iWi_il_K_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_gap_i_il_K_WiWk>;
	p2calc_WiWk[il_gap_iWi_il_K_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_gap_i_il_K_WiWk>;
	p2calc_WiWk[il_gap_i_il_K_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_gap_i_il_K_WiWk>;
	p2calc_WiWk[il_gap_i_il_K_WiWk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_gap_i_il_K_WiWk>;
	p2calc_WiWk[il_gap_i_il_gap_Wk_k]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_gap_i_il_K_WiWk>;
	p2calc_WiWk[il_gap_i_il_gap_Wk_Wi]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_gap_i_il_K_WiWk>;
	p2calc_WiWk[il_gap_Wi_il_K_ik]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_I_il_K_WiWk>;
	p2calc_WiWk[il_gap_Wi_il_K_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_I_il_K_WiWk>;
	p2calc_WiWk[il_gap_Wi_il_gap_k_i]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_I_il_gap_k_WiWk>;
	p2calc_WiWk[il_gap_Wi_il_gap_k_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2WiWk<il_I_il_gap_k_WiWk>;
	p2calc_WiWk[mblIK]									= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_bWi_I_mbl_bWk_K>;				 // mbl_bWi_I_mbl_bWk_K
	p2calc_WiWk[mbl_bWi_I_mbl_bWk_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_bWi_I_mbl_bWk_K>;
	p2calc_WiWk[mbl_bWi_I_mbl_bWk_gap_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_bWi_I_mbl_bWk_K>;
	p2calc_WiWk[mbl_bWi_gap_Wi_mbl_bWk_K]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_bWi_I_mbl_bWk_K>;
	p2calc_WiWk[mbl_il_I_mbl_il_K_ik]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_I_mbl_il_K_WiWk>;
	p2calc_WiWk[mbl_il_I_mbl_il_K_WiWk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_I_mbl_il_K_WiWk>;
	p2calc_WiWk[mbl_il_I_mbl_il_gap_k_i]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_I_mbl_il_gap_k_WiWk>;
	p2calc_WiWk[mbl_il_I_mbl_il_gap_k_WiWk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_I_mbl_il_gap_k_WiWk>;
	p2calc_WiWk[mbl_il_I_mbl_il_gap_kWk_i]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_I_mbl_il_gap_k_WiWk>;
	p2calc_WiWk[mbl_il_I_mbl_il_gap_kWk_Wi]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_I_mbl_il_gap_k_WiWk>;
	p2calc_WiWk[mbl_il_I_mbl_il_gap_Wk_ik]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_I_mbl_il_K_WiWk>;
	p2calc_WiWk[mbl_il_I_mbl_il_gap_Wk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_I_mbl_il_K_WiWk>;
	p2calc_WiWk[mbl_il_pb_I_mbl_il_pb_K]			= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_WiWk[mbl_il_gap_Wi_mbl_il_K_ik]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_I_mbl_il_K_WiWk>;
	p2calc_WiWk[mbl_il_gap_Wi_mbl_il_K_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_I_mbl_il_K_WiWk>;
	p2calc_WiWk[mbl_il_gap_Wi_mbl_il_gap_k_i]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_I_mbl_il_gap_k_WiWk>;
	p2calc_WiWk[mbl_il_gap_Wi_mbl_il_gap_k_Wk]	= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_I_mbl_il_gap_k_WiWk>;
	p2calc_WiWk[mbl_il_gap_iWi_mbl_il_K_k]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_gap_i_mbl_il_K_WiWk>;
	p2calc_WiWk[mbl_il_gap_iWi_mbl_il_K_Wk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_gap_i_mbl_il_K_WiWk>;
	p2calc_WiWk[mbl_il_gap_i_mbl_il_K_k]  			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_gap_i_mbl_il_K_WiWk>;
	p2calc_WiWk[mbl_il_gap_i_mbl_il_K_WiWk]  		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_gap_i_mbl_il_K_WiWk>;
	p2calc_WiWk[mbl_il_gap_i_mbl_il_gap_Wk_k]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_gap_i_mbl_il_K_WiWk>;
	p2calc_WiWk[mbl_il_gap_i_mbl_il_gap_Wk_Wi]	= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2WiWk<mbl_il_gap_i_mbl_il_K_WiWk>;
	

	p2calc_i[hp_init]										= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_K_i>;		 // hp_init_align
	p2calc_i[hp_init_align_ik]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_K_i>;		 // hp_init_align
	p2calc_i[hp_init_align_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_K_i>;		 // hp_init_align
	p2calc_i[hp_init_gap_I_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_K_i>;		 // hp_init_align
	p2calc_i[hp_init_gap_I_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_K_i>;		 // hp_init_align
	p2calc_i[hp_init_gap_K_i]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_k_i<hp_init_gap_K_i>;		 // hp_init_align
	p2calc_i[hp_init_gap_K_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_k_i<hp_init_gap_K_i>;		 // hp_init_align
	p2calc_i[hp_init_gap_Wi_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_kWi_i>;   // hp_init_gap_Wi
	p2calc_i[hp_init_gap_Wi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_kWi_i>;   // hp_init_gap_Wi
	p2calc_i[hp_init_gap_Wk_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_kWk_i>;   // hp_init_gap_Wk
	p2calc_i[hp_init_gap_Wk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_kWk_i>;   // hp_init_gap_Wk
	p2calc_i[hp_init_gap_iWi_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_kWi_i>;  // hp_init_gap_Wi
	p2calc_i[hp_init_gap_iWi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_kWi_i>;  // hp_init_gap_Wi
	p2calc_i[hp_init_gap_kWk_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_k_i<hp_init_gap_kWk_i>;  // hp_init_gap_Wk
	p2calc_i[hp_init_gap_kWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_k_i<hp_init_gap_kWk_i>;  // hp_init_gap_Wk
	p2calc_i[hp_init_gap_iWk_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_kWk_i>;  // hp_init_gap_Wk
	p2calc_i[hp_init_gap_iWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_i<hp_init_gap_kWk_i>;  // hp_init_gap_Wk
	p2calc_i[hp_init_gap_kWi_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_k_i<hp_init_gap_kWi_i>;  // hp_init_gap_Wi
	p2calc_i[hp_init_gap_kWi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_k_i<hp_init_gap_kWi_i>;  // hp_init_gap_Wi
	p2calc_i[hp_pb_IK]    								= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;  	  // hp_init_align
	p2calc_i[stem_IK]										= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2i<bi_I_bk_gap_k>;					// bi_I_bk_K
	p2calc_i[stem_I_stem_gap_kWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2i<bi_I_bk_gap_kWk>;
	p2calc_i[stem_no_mbl_I_stem_gap_kWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_i[stem_gap_iWi_stem_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2i<bi_gap_Wi_bk_gap_k>;			// bi_gap_Wi_bk_K
	p2calc_i[stem_no_mbl_gap_iWi_stem_K]			= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_i[bi_I_bk_K]									= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2i<bi_I_bk_gap_k>;
	p2calc_i[bi_I_bk_gap_k]								= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2i<bi_I_bk_gap_k>;  			  // bi_I_bk_K
	p2calc_i[bi_I_bk_gap_kWk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2i<bi_I_bk_gap_kWk>;			  // bi_I_bk_K
	p2calc_i[bi_I_bk_gap_Wk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2i<bi_I_bk_gap_kWk>;			  // bi_I_bk_K
	p2calc_i[bi_pb_I_bk_pb_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;			 // il_I_il_K
	p2calc_i[bi_gap_i_bk_K]								= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2i<bi_I_bk_gap_k>;  				 // bi_I_bk_K
	p2calc_i[bi_gap_i_bk_gap_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2i<bi_I_bk_gap_kWk>;				 // bi_I_bk_K
	p2calc_i[bi_gap_iWi_bk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2i<bi_gap_Wi_bk_gap_k>;				 // bi_I_bk_K
	p2calc_i[bi_gap_Wi_bk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2i<bi_gap_Wi_bk_gap_k>;				 // bi_I_bk_K
	p2calc_i[bi_gap_Wi_bk_gap_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2i<bi_gap_Wi_bk_gap_k>;				 // bi_I_bk_K
	p2calc_i[bWi_I_bWk_K]								= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2i<il_I_il_gap_k_i>;			 // il_I_il_K
	p2calc_i[bWi_I_bWk_gap_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkKgap2i<il_I_il_gap_k_i>;			 // il_I_il_K
	p2calc_i[bWi_I_bWk_gap_kWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkKgap2i<il_I_il_gap_kWk_i>; 			 // il_I_il_K
	p2calc_i[bWi_I_bWk_gap_Wk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2i<il_I_il_gap_kWk_i>; 			 // il_I_il_K
	p2calc_i[bWi_pb_I_bWk_pb_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;			 // il_I_il_K
	p2calc_i[bWi_gap_i_bWk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2i<il_I_il_gap_k_i>;			 // il_I_il_K
	p2calc_i[bWi_gap_i_bWk_gap_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2i<il_I_il_gap_kWk_i>; 			 // il_I_il_K
	p2calc_i[bWi_gap_iWi_bWk_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2i<il_gap_Wi_il_gap_k_i>; 			 // il_I_il_K
	p2calc_i[bWi_gap_Wi_bWk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2i<il_gap_Wi_il_gap_k_i>; 			 // il_I_il_K
	p2calc_i[bWi_gap_Wi_bWk_gap_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkKgap2i<il_gap_Wi_il_gap_k_i>; 			 // il_I_il_K
	p2calc_i[il_I_il_K_ik]								= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2i<il_I_il_gap_k_i>;					// il_I_il_K
	p2calc_i[il_I_il_K_WiWk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2i<il_I_il_gap_k_i>;					// il_I_il_K
	p2calc_i[il_I_il_gap_k_i]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2i<il_I_il_gap_k_i>;					// il_I_il
	p2calc_i[il_I_il_gap_k_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2i<il_I_il_gap_k_i>;					// il_I_il
	p2calc_i[il_I_il_gap_Wk_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2i<il_I_il_gap_kWk_i>;				// il_I_il_gap_Wk
	p2calc_i[il_I_il_gap_Wk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2i<il_I_il_gap_kWk_i>;				// il_I_il_gap_Wk
	p2calc_i[il_I_il_gap_kWk_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2i<il_I_il_gap_kWk_i>;				// il_I_il_gap_Wk
	p2calc_i[il_I_il_gap_kWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2i<il_I_il_gap_kWk_i>;				// il_I_il_gap_Wk
	p2calc_i[il_pb_I_il_pb_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;				// il_I_il_K
	p2calc_i[il_gap_iWi_il_K_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2i<il_gap_Wi_il_gap_k_i>;				// il_gap_Wi_il_K
	p2calc_i[il_gap_iWi_il_K_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2i<il_gap_Wi_il_gap_k_i>;				// il_gap_Wi_il_K
	p2calc_i[il_gap_i_il_K_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2i<il_I_il_gap_k_i>;					// il_I_il_K
	p2calc_i[il_gap_i_il_K_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2i<il_I_il_gap_k_i>;					// il_I_il_K
	p2calc_i[il_gap_i_il_gap_Wk_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2i<il_I_il_gap_kWk_i>;				// il_I_il_K
	p2calc_i[il_gap_i_il_gap_Wk_Wi]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2i<il_I_il_gap_kWk_i>;				// il_I_il_K
	p2calc_i[il_gap_Wi_il_K_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2i<il_gap_Wi_il_gap_k_i>;				// il_gap_Wi_il_K
	p2calc_i[il_gap_Wi_il_K_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2i<il_gap_Wi_il_gap_k_i>;				// il_gap_Wi_il_K
	p2calc_i[il_gap_Wi_il_gap_k_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2i<il_gap_Wi_il_gap_k_i>;				// il_gap_Wi_il_K
	p2calc_i[il_gap_Wi_il_gap_k_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2i<il_gap_Wi_il_gap_k_i>;				// il_gap_Wi_il_K
	p2calc_i[mblIK]  										= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_I_mbl_il_gap_k_i>;						// mbl_bi_I_mbl_bk_K
	p2calc_i[mbl_bWi_I_mbl_bWk_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_I_mbl_il_gap_k_i>;						// mbl_bi_I_mbl_bk_K
	p2calc_i[mbl_bWi_I_mbl_bWk_gap_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_I_mbl_il_gap_kWk_i>;						// mbl_bi_I_mbl_bk_K
	p2calc_i[mbl_bWi_gap_Wi_mbl_bWk_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_gap_Wi_mbl_il_gap_k_i>;						// mbl_bi_I_mbl_bk_K
	p2calc_i[mbl_il_I_mbl_il_K_ik]  					= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_I_mbl_il_gap_k_i>;				// mbl_il_I_mbl_il_K
	p2calc_i[mbl_il_I_mbl_il_K_WiWk]  				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_I_mbl_il_gap_k_i>;				// mbl_il_I_mbl_il_K
	p2calc_i[mbl_il_I_mbl_il_gap_k_i]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2i<mbl_il_I_mbl_il_gap_k_i>;				// mbl_il_I_mbl_il_K
	p2calc_i[mbl_il_I_mbl_il_gap_k_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2i<mbl_il_I_mbl_il_gap_k_i>;				// mbl_il_I_mbl_il_K
	p2calc_i[mbl_il_I_mbl_il_gap_kWk_i]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2i<mbl_il_I_mbl_il_gap_kWk_i>;			// mbl_il_I_mbl_il_gap_Wk
	p2calc_i[mbl_il_I_mbl_il_gap_kWk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2i<mbl_il_I_mbl_il_gap_kWk_i>;			// mbl_il_I_mbl_il_gap_Wk
	p2calc_i[mbl_il_I_mbl_il_gap_Wk_ik]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_I_mbl_il_gap_kWk_i>;
	p2calc_i[mbl_il_I_mbl_il_gap_Wk_Wi]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_I_mbl_il_gap_kWk_i>;
	p2calc_i[mbl_il_pb_I_mbl_il_pb_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_i[mbl_il_gap_Wi_mbl_il_K_ik]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_gap_Wi_mbl_il_gap_k_i>;
	p2calc_i[mbl_il_gap_Wi_mbl_il_K_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_gap_Wi_mbl_il_gap_k_i>;
	p2calc_i[mbl_il_gap_Wi_mbl_il_gap_k_i]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2i<mbl_il_gap_Wi_mbl_il_gap_k_i>;
	p2calc_i[mbl_il_gap_Wi_mbl_il_gap_k_Wk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2i<mbl_il_gap_Wi_mbl_il_gap_k_i>;
	p2calc_i[mbl_il_gap_iWi_mbl_il_K_k]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_gap_Wi_mbl_il_gap_k_i>;
	p2calc_i[mbl_il_gap_iWi_mbl_il_K_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_gap_Wi_mbl_il_gap_k_i>;
	p2calc_i[mbl_il_gap_i_mbl_il_K_k]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_I_mbl_il_gap_k_i>;
	p2calc_i[mbl_il_gap_i_mbl_il_K_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_I_mbl_il_gap_k_i>;
	p2calc_i[mbl_il_gap_i_mbl_il_gap_Wk_k]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_I_mbl_il_gap_kWk_i>;
	p2calc_i[mbl_il_gap_i_mbl_il_gap_Wk_Wi]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2i<mbl_il_I_mbl_il_gap_kWk_i>;

	p2calc_k[hp_init]										= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_I_k>;		 // hp_init_align
	p2calc_k[hp_init_align_ik]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_I_k>;		 // hp_init_align
	p2calc_k[hp_init_align_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_I_k>;		 // hp_init_align
	p2calc_k[hp_init_gap_I_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_I_k<hp_init_gap_I_k>;		 // hp_init_align
	p2calc_k[hp_init_gap_I_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_I_k<hp_init_gap_I_k>;		 // hp_init_align
	p2calc_k[hp_init_gap_K_i]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_I_k>;		 // hp_init_align
	p2calc_k[hp_init_gap_K_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_I_k>;		 // hp_init_align
	p2calc_k[hp_init_gap_Wi_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_iWi_k>;   // hp_init_gap_Wi
	p2calc_k[hp_init_gap_Wi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_iWi_k>;   // hp_init_gap_Wi
	p2calc_k[hp_init_gap_Wk_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_iWk_k>;   // hp_init_gap_Wk
	p2calc_k[hp_init_gap_Wk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_iWk_k>;   // hp_init_gap_Wk
	p2calc_k[hp_init_gap_iWi_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_I_k<hp_init_gap_iWi_k>;  // hp_init_gap_Wi
	p2calc_k[hp_init_gap_iWi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_I_k<hp_init_gap_iWi_k>;  // hp_init_gap_Wi
	p2calc_k[hp_init_gap_kWk_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_iWk_k>;  // hp_init_gap_Wk
	p2calc_k[hp_init_gap_kWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_iWk_k>;  // hp_init_gap_Wk
	p2calc_k[hp_init_gap_iWk_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_I_k<hp_init_gap_iWk_k>;  // hp_init_gap_Wk
	p2calc_k[hp_init_gap_iWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_I_k<hp_init_gap_iWk_k>;  // hp_init_gap_Wk
	p2calc_k[hp_init_gap_kWi_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_iWi_k>;  // hp_init_gap_Wi
	p2calc_k[hp_init_gap_kWi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_k<hp_init_gap_iWi_k>;  // hp_init_gap_Wi
	p2calc_k[hp_pb_IK]    								= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;		 // hp_init_align
	p2calc_k[stem_IK]										= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2k<bi_gap_i_bk_K>;					// bi_I_bk_K
	p2calc_k[stem_I_stem_gap_kWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2k<bi_gap_i_bk_gap_Wk>;			// bi_I_bk_gap_Wk
	p2calc_k[stem_no_mbl_I_stem_gap_kWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_k[stem_gap_iWi_stem_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2k<bi_gap_iWi_bk_K>;			// bi_gap_Wi_bk_K
	p2calc_k[stem_no_mbl_gap_iWi_stem_K]			= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_k[bi_I_bk_K]									= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2k<bi_gap_i_bk_K>;  				 // bi_I_bk_K
	p2calc_k[bi_I_bk_gap_k]								= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2k<bi_gap_i_bk_K>;  				 // bi_I_bk_K
	p2calc_k[bi_I_bk_gap_kWk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2k<bi_gap_i_bk_gap_Wk>;				 // bi_I_bk_K
	p2calc_k[bi_I_bk_gap_Wk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2k<bi_gap_i_bk_gap_Wk>;				 // bi_I_bk_K
	p2calc_k[bi_pb_I_bk_pb_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;			 // il_I_il_K
	p2calc_k[bi_gap_i_bk_K]								= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2k<bi_gap_i_bk_K>;  				 // bi_I_bk_K
	p2calc_k[bi_gap_i_bk_gap_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2k<bi_gap_i_bk_gap_Wk>;				 // bi_I_bk_K
	p2calc_k[bi_gap_iWi_bk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2k<bi_gap_iWi_bk_K>;				 // bi_I_bk_K
	p2calc_k[bi_gap_Wi_bk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2k<bi_gap_iWi_bk_K>;				 // bi_I_bk_K
	p2calc_k[bi_gap_Wi_bk_gap_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2k<bi_gap_iWi_bk_K>;				 // bi_I_bk_K
	p2calc_k[bWi_I_bWk_K]								= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2k<il_gap_i_il_K_k>;				// il_I_il_K
	p2calc_k[bWi_I_bWk_gap_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2k<il_gap_i_il_K_k>;			 // il_I_il_K
	p2calc_k[bWi_I_bWk_gap_kWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2k<il_gap_i_il_gap_Wk_k>; 			 // il_I_il_K
	p2calc_k[bWi_I_bWk_gap_Wk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2k<il_gap_i_il_gap_Wk_k>; 			 // il_I_il_K
	p2calc_k[bWi_pb_I_bWk_pb_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;			 // il_I_il_K
	p2calc_k[bWi_gap_i_bWk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkKgap2k<il_gap_i_il_K_k>;			 // il_I_il_K
	p2calc_k[bWi_gap_i_bWk_gap_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkKgap2k<il_gap_i_il_gap_Wk_k>; 			 // il_I_il_K
	p2calc_k[bWi_gap_iWi_bWk_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkKgap2k<il_gap_iWi_il_K_k>; 			 // il_I_il_K
	p2calc_k[bWi_gap_Wi_bWk_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2k<il_gap_iWi_il_K_k>; 			 // il_I_il_K
	p2calc_k[bWi_gap_Wi_bWk_gap_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiIbWkK2k<il_gap_iWi_il_K_k>; 			 // il_I_il_K
	p2calc_k[il_I_il_K_ik]								= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2k<il_gap_i_il_K_k>;					// il_I_il_K
	p2calc_k[il_I_il_K_WiWk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2k<il_gap_i_il_K_k>;					// il_I_il_K
	p2calc_k[il_I_il_gap_k_i]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2k<il_gap_i_il_K_k>;					// il_I_il
	p2calc_k[il_I_il_gap_k_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2k<il_gap_i_il_K_k>;					// il_I_il
	p2calc_k[il_I_il_gap_Wk_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2k<il_gap_i_il_gap_Wk_k>;				// il_I_il_gap_Wk
	p2calc_k[il_I_il_gap_Wk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2k<il_gap_i_il_gap_Wk_k>;				// il_I_il_gap_Wk
	p2calc_k[il_I_il_gap_kWk_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2k<il_gap_i_il_gap_Wk_k>;				// il_I_il_gap_Wk
	p2calc_k[il_I_il_gap_kWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2k<il_gap_i_il_gap_Wk_k>;				// il_I_il_gap_Wk
	p2calc_k[il_pb_I_il_pb_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;				// il_I_il_K
	p2calc_k[il_gap_iWi_il_K_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2k<il_gap_iWi_il_K_k>;				// il_gap_Wi_il_K
	p2calc_k[il_gap_iWi_il_K_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2k<il_gap_iWi_il_K_k>;				// il_gap_Wi_il_K
	p2calc_k[il_gap_i_il_K_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2k<il_gap_i_il_K_k>;					// il_I_il_K
	p2calc_k[il_gap_i_il_K_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2k<il_gap_i_il_K_k>;					// il_I_il_K
	p2calc_k[il_gap_i_il_gap_Wk_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2k<il_gap_i_il_gap_Wk_k>;				// il_I_il_K
	p2calc_k[il_gap_i_il_gap_Wk_Wi]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2k<il_gap_i_il_gap_Wk_k>;				// il_I_il_K
	p2calc_k[il_gap_Wi_il_K_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2k<il_gap_iWi_il_K_k>;				// il_gap_Wi_il_K
	p2calc_k[il_gap_Wi_il_K_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2k<il_gap_iWi_il_K_k>;				// il_gap_Wi_il_K
	p2calc_k[il_gap_Wi_il_gap_k_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2k<il_gap_iWi_il_K_k>;				// il_gap_Wi_il_K
	p2calc_k[il_gap_Wi_il_gap_k_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2k<il_gap_iWi_il_K_k>;				// il_gap_Wi_il_K
	p2calc_k[mblIK]  										= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_i_mbl_il_K_k>;						// mbl_bi_I_mbl_bk_K
	p2calc_k[mbl_bWi_I_mbl_bWk_K]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_i_mbl_il_K_k>;						// mbl_bi_I_mbl_bk_K
	p2calc_k[mbl_bWi_I_mbl_bWk_gap_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_i_mbl_il_gap_Wk_k>;						// mbl_bi_I_mbl_bk_K
	p2calc_k[mbl_bWi_gap_Wi_mbl_bWk_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_iWi_mbl_il_K_k>;						// mbl_bi_I_mbl_bk_K
	p2calc_k[mbl_il_I_mbl_il_K_ik]  					= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_i_mbl_il_K_k>;				// mbl_il_I_mbl_il_K
	p2calc_k[mbl_il_I_mbl_il_K_WiWk]  				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_i_mbl_il_K_k>;				// mbl_il_I_mbl_il_K
	p2calc_k[mbl_il_I_mbl_il_gap_k_i]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_i_mbl_il_K_k>;				// mbl_il_I_mbl_il_K
	p2calc_k[mbl_il_I_mbl_il_gap_k_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_i_mbl_il_K_k>;				// mbl_il_I_mbl_il_K
	p2calc_k[mbl_il_I_mbl_il_gap_kWk_i]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_i_mbl_il_gap_Wk_k>;			// mbl_il_I_mbl_il_gap_Wk
	p2calc_k[mbl_il_I_mbl_il_gap_kWk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_i_mbl_il_gap_Wk_k>;			// mbl_il_I_mbl_il_gap_Wk
	p2calc_k[mbl_il_I_mbl_il_gap_Wk_ik]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_i_mbl_il_gap_Wk_k>;
	p2calc_k[mbl_il_I_mbl_il_gap_Wk_Wi]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_i_mbl_il_gap_Wk_k>;
	p2calc_k[mbl_il_pb_I_mbl_il_pb_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_k[mbl_il_gap_Wi_mbl_il_K_ik]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_iWi_mbl_il_K_k>;
	p2calc_k[mbl_il_gap_Wi_mbl_il_K_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_iWi_mbl_il_K_k>;
	p2calc_k[mbl_il_gap_Wi_mbl_il_gap_k_i]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_iWi_mbl_il_K_k>;
	p2calc_k[mbl_il_gap_Wi_mbl_il_gap_k_Wk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2k<mbl_il_gap_iWi_mbl_il_K_k>;
	p2calc_k[mbl_il_gap_iWi_mbl_il_K_k]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2k<mbl_il_gap_iWi_mbl_il_K_k>;
	p2calc_k[mbl_il_gap_iWi_mbl_il_K_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2k<mbl_il_gap_iWi_mbl_il_K_k>;
	p2calc_k[mbl_il_gap_i_mbl_il_K_k]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2k<mbl_il_gap_i_mbl_il_K_k>;
	p2calc_k[mbl_il_gap_i_mbl_il_K_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2k<mbl_il_gap_i_mbl_il_K_k>;
	p2calc_k[mbl_il_gap_i_mbl_il_gap_Wk_k]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2k<mbl_il_gap_i_mbl_il_gap_Wk_k>;
	p2calc_k[mbl_il_gap_i_mbl_il_gap_Wk_Wi]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2k<mbl_il_gap_i_mbl_il_gap_Wk_k>;


	p2calc_Wi[hp_init] 									= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_Wk_Wi>;  	// hp_init_align
	p2calc_Wi[hp_init_align_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_Wk_Wi>;
	p2calc_Wi[hp_init_align_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_Wk_Wi>;
	p2calc_Wi[hp_init_gap_I_k] 						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_iWk_Wi>;    // hp_init_gap_I
	p2calc_Wi[hp_init_gap_I_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_iWk_Wi>;    // hp_init_gap_I
	p2calc_Wi[hp_init_gap_K_i] 						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_kWk_Wi>;    // hp_init_gap_K
	p2calc_Wi[hp_init_gap_K_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_kWk_Wi>;    // hp_init_gap_K
	p2calc_Wi[hp_init_gap_Wi_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_Wk_Wi>;
	p2calc_Wi[hp_init_gap_Wi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_Wk_Wi>;
	p2calc_Wi[hp_init_gap_Wk_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_K_Wi<hp_init_gap_Wk_Wi>;
	p2calc_Wi[hp_init_gap_Wk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_K_Wi<hp_init_gap_Wk_Wi>;
	p2calc_Wi[hp_init_gap_iWi_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_iWk_Wi>;
	p2calc_Wi[hp_init_gap_iWi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_iWk_Wi>;
	p2calc_Wi[hp_init_gap_kWk_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_K_Wi<hp_init_gap_kWk_Wi>;
	p2calc_Wi[hp_init_gap_kWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_K_Wi<hp_init_gap_kWk_Wi>;
	p2calc_Wi[hp_init_gap_iWk_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_K_Wi<hp_init_gap_iWk_Wi>;
	p2calc_Wi[hp_init_gap_iWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_K_Wi<hp_init_gap_iWk_Wi>;
	p2calc_Wi[hp_init_gap_kWi_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_kWk_Wi>;
	p2calc_Wi[hp_init_gap_kWi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wi<hp_init_gap_kWk_Wi>;
	p2calc_Wi[hp_pb_IK]									= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wi[stem_IK] 									= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wi<bWi_I_bWk_gap_Wk>;				// bWi_I_bWk_K
	p2calc_Wi[stem_I_stem_gap_kWk] 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2Wi<bWi_I_bWk_gap_kWk>;				// bWi_I_bWk_K
	p2calc_Wi[stem_no_mbl_I_stem_gap_kWk]		 	= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wi[stem_gap_iWi_stem_K] 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wi<bWi_gap_i_bWk_gap_Wk>;				// bWi_I_bWk_K
	p2calc_Wi[stem_no_mbl_gap_iWi_stem_K] 			= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wi[bi_I_bk_K]				 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wi<il_I_il_gap_Wk_Wi>;
	p2calc_Wi[bi_I_bk_gap_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wi<il_I_il_gap_kWk_Wi>;
	p2calc_Wi[bi_I_bk_gap_kWk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkKgap2Wi<il_I_il_gap_kWk_Wi>;
	p2calc_Wi[bi_I_bk_gap_Wk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkKgap2Wi<il_I_il_gap_Wk_Wi>;
	p2calc_Wi[bi_pb_I_bk_pb_K]  						= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wi[bi_gap_i_bk_K] 		 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wi<il_gap_i_il_gap_Wk_Wi>;
	p2calc_Wi[bi_gap_i_bk_gap_Wk] 		 			= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkKgap2Wi<il_gap_i_il_gap_Wk_Wi>;
	p2calc_Wi[bi_gap_iWi_bk_K] 		 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wi<il_gap_i_il_gap_Wk_Wi>;
	p2calc_Wi[bi_gap_Wi_bk_K] 		 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wi<il_I_il_gap_Wk_Wi>;
	p2calc_Wi[bi_gap_Wi_bk_gap_k]	 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wi<il_I_il_gap_kWk_Wi>;
	p2calc_Wi[bWi_I_bWk_K]  							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wi<bWi_I_bWk_gap_Wk>;
	p2calc_Wi[bWi_I_bWk_gap_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wi<bWi_I_bWk_gap_kWk>;
	p2calc_Wi[bWi_I_bWk_gap_kWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2Wi<bWi_I_bWk_gap_kWk>;
	p2calc_Wi[bWi_I_bWk_gap_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2Wi<bWi_I_bWk_gap_Wk>;
	p2calc_Wi[bWi_pb_I_bWk_pb_K]  					= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wi[bWi_gap_iWi_bWk_K] 	 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wi<bWi_gap_i_bWk_gap_Wk>;
	p2calc_Wi[bWi_gap_i_bWk_K] 	 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wi<bWi_gap_i_bWk_gap_Wk>;
	p2calc_Wi[bWi_gap_i_bWk_gap_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2Wi<bWi_gap_i_bWk_gap_Wk>;
	p2calc_Wi[bWi_gap_Wi_bWk_K] 	 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wi<bWi_I_bWk_gap_Wk>;
	p2calc_Wi[bWi_gap_Wi_bWk_gap_k] 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wi<bWi_I_bWk_gap_kWk>;
	p2calc_Wi[il_I_il_K_ik]  							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wi<il_I_il_gap_Wk_Wi>;
	p2calc_Wi[il_I_il_K_WiWk]  						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wi<il_I_il_gap_Wk_Wi>;
	p2calc_Wi[il_I_il_gap_k_i]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wi<il_I_il_gap_kWk_Wi>;
	p2calc_Wi[il_I_il_gap_k_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wi<il_I_il_gap_kWk_Wi>;
	p2calc_Wi[il_I_il_gap_Wk_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2Wi<il_I_il_gap_Wk_Wi>;
	p2calc_Wi[il_I_il_gap_Wk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2Wi<il_I_il_gap_Wk_Wi>;
	p2calc_Wi[il_I_il_gap_kWk_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2Wi<il_I_il_gap_kWk_Wi>;
	p2calc_Wi[il_I_il_gap_kWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2Wi<il_I_il_gap_kWk_Wi>;
	p2calc_Wi[il_pb_I_il_pb_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wi[il_gap_iWi_il_K_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wi<il_gap_i_il_gap_Wk_Wi>;
	p2calc_Wi[il_gap_iWi_il_K_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wi<il_gap_i_il_gap_Wk_Wi>;
	p2calc_Wi[il_gap_i_il_K_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wi<il_gap_i_il_gap_Wk_Wi>;
	p2calc_Wi[il_gap_i_il_K_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wi<il_gap_i_il_gap_Wk_Wi>;
	p2calc_Wi[il_gap_i_il_gap_Wk_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2Wi<il_gap_i_il_gap_Wk_Wi>;
	p2calc_Wi[il_gap_i_il_gap_Wk_Wi]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2Wi<il_gap_i_il_gap_Wk_Wi>;
	p2calc_Wi[il_gap_Wi_il_K_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wi<il_I_il_gap_Wk_Wi>;
	p2calc_Wi[il_gap_Wi_il_K_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wi<il_I_il_gap_Wk_Wi>;
	p2calc_Wi[il_gap_Wi_il_gap_k_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wi<il_I_il_gap_kWk_Wi>;
	p2calc_Wi[il_gap_Wi_il_gap_k_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wi<il_I_il_gap_kWk_Wi>;
	p2calc_Wi[mblIK]										= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_bWi_I_mbl_bWk_gap_Wk>;				 // mbl_bWi_I_mbl_bWk_K
	p2calc_Wi[mbl_bWi_I_mbl_bWk_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_bWi_I_mbl_bWk_gap_Wk>;
	p2calc_Wi[mbl_bWi_I_mbl_bWk_gap_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wi<mbl_bWi_I_mbl_bWk_gap_Wk>;
	p2calc_Wi[mbl_bWi_gap_Wi_mbl_bWk_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_bWi_I_mbl_bWk_gap_Wk>;
	p2calc_Wi[mbl_il_I_mbl_il_K_ik]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_il_I_mbl_il_gap_Wk_Wi>;
	p2calc_Wi[mbl_il_I_mbl_il_K_WiWk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_il_I_mbl_il_gap_Wk_Wi>;
	p2calc_Wi[mbl_il_I_mbl_il_gap_k_i]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_il_I_mbl_il_gap_kWk_Wi>;
	p2calc_Wi[mbl_il_I_mbl_il_gap_k_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_il_I_mbl_il_gap_kWk_Wi>;
	p2calc_Wi[mbl_il_I_mbl_il_gap_kWk_i]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wi<mbl_il_I_mbl_il_gap_kWk_Wi>;
	p2calc_Wi[mbl_il_I_mbl_il_gap_kWk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wi<mbl_il_I_mbl_il_gap_kWk_Wi>;
	p2calc_Wi[mbl_il_I_mbl_il_gap_Wk_ik]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wi<mbl_il_I_mbl_il_gap_Wk_Wi>;
	p2calc_Wi[mbl_il_I_mbl_il_gap_Wk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wi<mbl_il_I_mbl_il_gap_Wk_Wi>;
	p2calc_Wi[mbl_il_pb_I_mbl_il_pb_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wi[mbl_il_gap_Wi_mbl_il_K_ik]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_il_I_mbl_il_gap_Wk_Wi>;
	p2calc_Wi[mbl_il_gap_Wi_mbl_il_K_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_il_I_mbl_il_gap_Wk_Wi>;
	p2calc_Wi[mbl_il_gap_Wi_mbl_il_gap_k_i]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_il_I_mbl_il_gap_kWk_Wi>;
	p2calc_Wi[mbl_il_gap_Wi_mbl_il_gap_k_Wk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_il_I_mbl_il_gap_kWk_Wi>;
	p2calc_Wi[mbl_il_gap_iWi_mbl_il_K_k]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_il_gap_i_mbl_il_gap_Wk_Wi>;
	p2calc_Wi[mbl_il_gap_iWi_mbl_il_K_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_il_gap_i_mbl_il_gap_Wk_Wi>;
	p2calc_Wi[mbl_il_gap_i_mbl_il_K_k]  			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_il_gap_i_mbl_il_gap_Wk_Wi>;
	p2calc_Wi[mbl_il_gap_i_mbl_il_K_WiWk]  		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wi<mbl_il_gap_i_mbl_il_gap_Wk_Wi>;
	p2calc_Wi[mbl_il_gap_i_mbl_il_gap_Wk_k]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wi<mbl_il_gap_i_mbl_il_gap_Wk_Wi>;
	p2calc_Wi[mbl_il_gap_i_mbl_il_gap_Wk_Wi]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wi<mbl_il_gap_i_mbl_il_gap_Wk_Wi>;

	p2calc_Wk[hp_init] 									= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_Wi_Wk>;  	// hp_init_align
	p2calc_Wk[hp_init_align_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_Wi_Wk>;
	p2calc_Wk[hp_init_align_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_Wi_Wk>;
	p2calc_Wk[hp_init_gap_I_k] 						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_iWi_Wk>;    // hp_init_gap_I
	p2calc_Wk[hp_init_gap_I_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_iWi_Wk>;    // hp_init_gap_I
	p2calc_Wk[hp_init_gap_K_i] 						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_kWi_Wk>;    // hp_init_gap_K
	p2calc_Wk[hp_init_gap_K_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_kWi_Wk>;    // hp_init_gap_K
	p2calc_Wk[hp_init_gap_Wi_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_I_Wk<hp_init_gap_Wi_Wk>;
	p2calc_Wk[hp_init_gap_Wi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_I_Wk<hp_init_gap_Wi_Wk>;
	p2calc_Wk[hp_init_gap_Wk_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_Wi_Wk>;
	p2calc_Wk[hp_init_gap_Wk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_Wi_Wk>;
	p2calc_Wk[hp_init_gap_iWi_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_I_Wk<hp_init_gap_iWi_Wk>;
	p2calc_Wk[hp_init_gap_iWi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_I_Wk<hp_init_gap_iWi_Wk>;
	p2calc_Wk[hp_init_gap_kWk_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_kWi_Wk>;
	p2calc_Wk[hp_init_gap_kWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_kWi_Wk>;
	p2calc_Wk[hp_init_gap_iWk_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_iWi_Wk>;
	p2calc_Wk[hp_init_gap_iWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_Wk<hp_init_gap_iWi_Wk>;
	p2calc_Wk[hp_init_gap_kWi_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_I_Wk<hp_init_gap_kWi_Wk>;
	p2calc_Wk[hp_init_gap_kWi_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template hp_align_gap_I_Wk<hp_init_gap_kWi_Wk>;
	p2calc_Wk[hp_pb_IK]									= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wk[stem_IK] 									= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wk<bWi_gap_Wi_bWk_K>;				// bWi_I_bWk_K
	p2calc_Wk[stem_I_stem_gap_kWk] 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wk<bWi_gap_Wi_bWk_gap_k>;				// bWi_I_bWk_K
	p2calc_Wk[stem_no_mbl_I_stem_gap_kWk]		 	= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wk[stem_gap_iWi_stem_K] 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2Wk<bWi_gap_iWi_bWk_K>;				// bWi_I_bWk_K
	p2calc_Wk[stem_no_mbl_gap_iWi_stem_K]	 		= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wk[bi_I_bk_K]				 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wk<il_gap_Wi_il_K_Wk>;
	p2calc_Wk[bi_I_bk_gap_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wk<il_gap_Wi_il_gap_k_Wk>;
	p2calc_Wk[bi_I_bk_gap_kWk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wk<il_gap_Wi_il_gap_k_Wk>;
	p2calc_Wk[bi_I_bk_gap_Wk]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wk<il_gap_Wi_il_K_Wk>;
	p2calc_Wk[bi_pb_I_bk_pb_K]  						= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wk[bi_gap_i_bk_K] 		 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wk<il_gap_iWi_il_K_Wk>;
	p2calc_Wk[bi_gap_i_bk_gap_Wk] 		 			= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wk<il_gap_iWi_il_K_Wk>;
	p2calc_Wk[bi_gap_iWi_bk_K] 		 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkK2Wk<il_gap_iWi_il_K_Wk>;
	p2calc_Wk[bi_gap_Wi_bk_K] 		 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkKgap2Wk<il_gap_Wi_il_K_Wk>;
	p2calc_Wk[bi_gap_Wi_bk_gap_k]	 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template biIbkKgap2Wk<il_gap_Wi_il_gap_k_Wk>;
	p2calc_Wk[bWi_I_bWk_K]  							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wk<bWi_gap_Wi_bWk_K>;
	p2calc_Wk[bWi_I_bWk_gap_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wk<bWi_gap_Wi_bWk_gap_k>;
	p2calc_Wk[bWi_I_bWk_gap_kWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wk<bWi_gap_Wi_bWk_gap_k>;
	p2calc_Wk[bWi_I_bWk_gap_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wk<bWi_gap_Wi_bWk_K>;
	p2calc_Wk[bWi_pb_I_bWk_pb_K]  					= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wk[bWi_gap_iWi_bWk_K] 	 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wk<bWi_gap_iWi_bWk_K>;
	p2calc_Wk[bWi_gap_i_bWk_K] 	 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wk<bWi_gap_iWi_bWk_K>;
	p2calc_Wk[bWi_gap_i_bWk_gap_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2Wk<bWi_gap_iWi_bWk_K>;
	p2calc_Wk[bWi_gap_Wi_bWk_K] 	 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2Wk<bWi_gap_Wi_bWk_K>;
	p2calc_Wk[bWi_gap_Wi_bWk_gap_k] 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIKgap2Wk<bWi_gap_Wi_bWk_gap_k>;
	p2calc_Wk[il_I_il_K_ik]  							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wk<il_gap_Wi_il_K_Wk>;
	p2calc_Wk[il_I_il_K_WiWk]  						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wk<il_gap_Wi_il_K_Wk>;
	p2calc_Wk[il_I_il_gap_k_i]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wk<il_gap_Wi_il_gap_k_Wk>;
	p2calc_Wk[il_I_il_gap_k_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wk<il_gap_Wi_il_gap_k_Wk>;
	p2calc_Wk[il_I_il_gap_Wk_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wk<il_gap_Wi_il_K_Wk>;
	p2calc_Wk[il_I_il_gap_Wk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wk<il_gap_Wi_il_K_Wk>;
	p2calc_Wk[il_I_il_gap_kWk_i]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wk<il_gap_Wi_il_gap_k_Wk>;
	p2calc_Wk[il_I_il_gap_kWk_Wi]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wk<il_gap_Wi_il_gap_k_Wk>;
	p2calc_Wk[il_pb_I_il_pb_K]							= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wk[il_gap_iWi_il_K_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2Wk<il_gap_iWi_il_K_Wk>;
	p2calc_Wk[il_gap_iWi_il_K_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2Wk<il_gap_iWi_il_K_Wk>;
	p2calc_Wk[il_gap_i_il_K_k]							= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wk<il_gap_iWi_il_K_Wk>;
	p2calc_Wk[il_gap_i_il_K_WiWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wk<il_gap_iWi_il_K_Wk>;
	p2calc_Wk[il_gap_i_il_gap_Wk_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wk<il_gap_iWi_il_K_Wk>;
	p2calc_Wk[il_gap_i_il_gap_Wk_Wi]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilK2Wk<il_gap_iWi_il_K_Wk>;
	p2calc_Wk[il_gap_Wi_il_K_ik]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2Wk<il_gap_Wi_il_K_Wk>;
	p2calc_Wk[il_gap_Wi_il_K_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2Wk<il_gap_Wi_il_K_Wk>;
	p2calc_Wk[il_gap_Wi_il_gap_k_i]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2Wk<il_gap_Wi_il_gap_k_Wk>;
	p2calc_Wk[il_gap_Wi_il_gap_k_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template ilIilKgap2Wk<il_gap_Wi_il_gap_k_Wk>;
	p2calc_Wk[mblIK]										= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_bWi_gap_Wi_mbl_bWk_K>;				 // mbl_bWi_I_mbl_bWk_K
	p2calc_Wk[mbl_bWi_I_mbl_bWk_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_bWi_gap_Wi_mbl_bWk_K>;
	p2calc_Wk[mbl_bWi_I_mbl_bWk_gap_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_bWi_gap_Wi_mbl_bWk_K>;
	p2calc_Wk[mbl_bWi_gap_Wi_mbl_bWk_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wk<mbl_bWi_gap_Wi_mbl_bWk_K>;
	p2calc_Wk[mbl_il_I_mbl_il_K_ik]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_il_gap_Wi_mbl_il_K_Wk>;
	p2calc_Wk[mbl_il_I_mbl_il_K_WiWk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_il_gap_Wi_mbl_il_K_Wk>;
	p2calc_Wk[mbl_il_I_mbl_il_gap_k_i]				= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_il_gap_Wi_mbl_il_gap_k_Wk>;
	p2calc_Wk[mbl_il_I_mbl_il_gap_k_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_il_gap_Wi_mbl_il_gap_k_Wk>;
	p2calc_Wk[mbl_il_I_mbl_il_gap_kWk_i]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_il_gap_Wi_mbl_il_gap_k_Wk>;
	p2calc_Wk[mbl_il_I_mbl_il_gap_kWk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_il_gap_Wi_mbl_il_gap_k_Wk>;
	p2calc_Wk[mbl_il_I_mbl_il_gap_Wk_ik]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_il_gap_Wi_mbl_il_K_Wk>;
	p2calc_Wk[mbl_il_I_mbl_il_gap_Wk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_il_gap_Wi_mbl_il_K_Wk>;
	p2calc_Wk[mbl_il_pb_I_mbl_il_pb_K]				= &fold< FOLD_TEMPLATE_PARAMETERS >::noCalc;
	p2calc_Wk[mbl_il_gap_Wi_mbl_il_K_ik]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wk<mbl_il_gap_Wi_mbl_il_K_Wk>;
	p2calc_Wk[mbl_il_gap_Wi_mbl_il_K_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wk<mbl_il_gap_Wi_mbl_il_K_Wk>;
	p2calc_Wk[mbl_il_gap_Wi_mbl_il_gap_k_i]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wk<mbl_il_gap_Wi_mbl_il_gap_k_Wk>;
	p2calc_Wk[mbl_il_gap_Wi_mbl_il_gap_k_Wk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wk<mbl_il_gap_Wi_mbl_il_gap_k_Wk>;
	p2calc_Wk[mbl_il_gap_iWi_mbl_il_K_k]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wk<mbl_il_gap_iWi_mbl_il_K_Wk>;
	p2calc_Wk[mbl_il_gap_iWi_mbl_il_K_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIKgap2Wk<mbl_il_gap_iWi_mbl_il_K_Wk>;
	p2calc_Wk[mbl_il_gap_i_mbl_il_K_k]  			= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_il_gap_iWi_mbl_il_K_Wk>;
	p2calc_Wk[mbl_il_gap_i_mbl_il_K_WiWk]  		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_il_gap_iWi_mbl_il_K_Wk>;
	p2calc_Wk[mbl_il_gap_i_mbl_il_gap_Wk_k]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_il_gap_iWi_mbl_il_K_Wk>;
	p2calc_Wk[mbl_il_gap_i_mbl_il_gap_Wk_Wi]		= &fold< FOLD_TEMPLATE_PARAMETERS >::template mIK2Wk<mbl_il_gap_iWi_mbl_il_K_Wk>;


	if (!nobranch) {
		p2calc_mbl[stem_IK] 									= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2mbl<mblIK>;
		p2calc_mbl[stem_I_stem_gap_kWk] 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2mbl<mblIK>;
		p2calc_mbl[stem_gap_iWi_stem_K] 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2mbl<mblIK>;
		p2calc_mbl[bWi_I_bWk_K]  							= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiWkIK2mbl<mblIK>;
		p2calc_mbl[bWi_I_bWk_gap_k]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiWkIK2mbl<mblIK>;
		p2calc_mbl[bWi_I_bWk_gap_kWk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiWkIK2mbl<mblIK>;
		p2calc_mbl[bWi_I_bWk_gap_Wk]						= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiWkIK2mbl<mblIK>;
		p2calc_mbl[bWi_gap_iWi_bWk_K] 	 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiWkIK2mbl<mblIK>;
		p2calc_mbl[bWi_gap_i_bWk_K] 	 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiWkIK2mbl<mblIK>;
		p2calc_mbl[bWi_gap_i_bWk_gap_Wk]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiWkIK2mbl<mblIK>;
		p2calc_mbl[bWi_gap_Wi_bWk_K] 	 					= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiWkIK2mbl<mblIK>;
		p2calc_mbl[bWi_gap_Wi_bWk_gap_k] 				= &fold< FOLD_TEMPLATE_PARAMETERS >::template bWiWkIK2mbl<mblIK>;
		p2calc_mbl[mblIK]										= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2mbl<mblIK>;
		p2calc_mbl[mbl_bWi_I_mbl_bWk_K]					= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2mbl<mblIK>;
		p2calc_mbl[mbl_bWi_I_mbl_bWk_gap_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2mbl<mblIK>;
		p2calc_mbl[mbl_bWi_gap_Wi_mbl_bWk_K]			= &fold< FOLD_TEMPLATE_PARAMETERS >::template stemIK2mbl<mblIK>;
	}

	p2end[hp_init] 						= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_align_ik]				= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_align_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_I_k] 				= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_I_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_K_i] 				= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_K_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_Wi_ik]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_Wi_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_Wk_ik]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_Wk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_iWi_k]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_iWi_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_kWk_i]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_kWk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_iWk_k]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_iWk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_kWi_i]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[hp_init_gap_kWi_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::hp2end;
	p2end[stem_IK]							= &fold< FOLD_TEMPLATE_PARAMETERS >::stem2end;
	p2end[stem_I_stem_gap_kWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::stem2end;
	p2end[stem_no_mbl_I_stem_gap_kWk]= &fold< FOLD_TEMPLATE_PARAMETERS >::stem2end;
	p2end[stem_gap_iWi_stem_K]			= &fold< FOLD_TEMPLATE_PARAMETERS >::stem2end;
	p2end[stem_no_mbl_gap_iWi_stem_K]= &fold< FOLD_TEMPLATE_PARAMETERS >::stem2end;
	p2end[bi_I_bk_K]				 		= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_ik2end;
	p2end[bi_I_bk_gap_k]					= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_ik2end;
	p2end[bi_I_bk_gap_kWk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_ik2end;
	p2end[bi_I_bk_gap_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_ik2end;
	p2end[bi_gap_i_bk_K] 		 		= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_ik2end;
	p2end[bi_gap_i_bk_gap_Wk] 		 	= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_ik2end;
	p2end[bi_gap_iWi_bk_K] 		 		= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_ik2end;
	p2end[bi_gap_Wi_bk_K] 		 		= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_ik2end;
	p2end[bi_gap_Wi_bk_gap_k]	 		= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_ik2end;
	p2end[bWi_I_bWk_K]  					= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_WiWk2end;
	p2end[bWi_I_bWk_gap_k]				= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_WiWk2end;
	p2end[bWi_I_bWk_gap_kWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_WiWk2end;
	p2end[bWi_I_bWk_gap_Wk]				= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_WiWk2end;
	p2end[bWi_gap_iWi_bWk_K] 	 		= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_WiWk2end;
	p2end[bWi_gap_i_bWk_K] 	 			= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_WiWk2end;
	p2end[bWi_gap_i_bWk_gap_Wk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_WiWk2end;
	p2end[bWi_gap_Wi_bWk_K] 	 		= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_WiWk2end;
	p2end[bWi_gap_Wi_bWk_gap_k] 		= &fold< FOLD_TEMPLATE_PARAMETERS >::bulge_WiWk2end;
	p2end[il_I_il_K_ik]  				= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_I_il_K_WiWk]  				= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_I_il_gap_k_i]				= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_I_il_gap_k_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_I_il_gap_Wk_ik]			= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_I_il_gap_Wk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_I_il_gap_kWk_i]			= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_I_il_gap_kWk_Wi]			= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_gap_iWi_il_K_k]			= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_gap_iWi_il_K_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_gap_i_il_K_k]				= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_gap_i_il_K_WiWk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_gap_i_il_gap_Wk_k]		= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_gap_i_il_gap_Wk_Wi]		= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_gap_Wi_il_K_ik]			= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_gap_Wi_il_K_Wk]			= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_gap_Wi_il_gap_k_i]		= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;
	p2end[il_gap_Wi_il_gap_k_Wk]		= &fold< FOLD_TEMPLATE_PARAMETERS >::ilIK2end;

}


#endif /*FOLD */
