#ifndef OUTPUT
#define OUTPUT
#include "arguments.cxx"
#include "sequence.cxx"
#include "helper.cxx"
#include "stack.cxx"
#include "foldalign.h"

#include <iostream>
#include <string>
#include <iomanip>

/******************************************************************************
*                                                                             *
*   Copyright 2004 -2007 Jakob Hull Havgaard, hull@genome.ku.dk               *
*                                                                             *
*   This file is part of Foldalign                                            *
*                                                                             *
*   Foldalign is free software; you can redistribute it and/or modify         *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   Foldalign is distributed in the hope that it will be useful,              *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with Foldalign; if not, write to the Free Software                  *
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
*                                                                             *
******************************************************************************/


// Written by Jakob Hull Havgaard, 2004, hull@genome.ku.dk.
class output {
public:
	inline output(arguments*& argu, 
					  const sequence* const one, 
					  const sequence* const two, scorematrix& score) 
		: arg(argu),
		  seq_1( !arg->boolOpt("switch") ? one : two),
		  seq_2( !arg->boolOpt("switch") ? two : one),
		  s_matrix(score) {};
	
	inline void head();
	
	inline void localscorehead();
	
	inline void parameters();

	
	inline void foldout(stack<positionType>*& leftPos_1, stack<positionType>*& leftBasepair_1,stack<positionType>*& leftPos_2, stack<positionType>*& leftBasepair_2, int align_start_1, int align_len_1, int align_start_2, int align_len_2, int align_score);

	static inline void saveOutput(const bool summary = false, 
											const bool plot_score = false) {
	
		if ( plot_score ) {
			std::cout << "; ******************************************************************************" << std::endl;
		}

		std::cout << "; ==============================================================================" << std::endl;

		if ( ! summary ) {
			std::cout << "; ------------------------------------------------------------------------------" << std::endl;
			std::cout << "; ******************************************************************************" << std::endl;
			std::cout << "; ------------------------------------------------------------------------------" << std::endl;
			std::cout << "; ******************************************************************************" << std::endl;
		}
	}

	static inline void saveOutputError(const bool summary = false, 
											     const bool plot_score = false,
											     const std::string error = "") {

		std::cout << "; ERROR               ";
		if (error.compare("")) {std::cout << error << std::endl;}
		else {std::cout << "Unknown" << std::endl;}
		
		saveOutput(summary, plot_score);
	}


private:
	// These are character used to indicate base-pairs
	// There are two sets. The first one is used when the base-pair is conserved
	// between the sequences, the second is used for insert base-pairs.

	// Normal base pairs
	static const char open  = '(';
	static const char close = ')';

	// Insert base pairs
	static const char open_one_seq  = '<';
	static const char close_one_seq = '>';

	// This is the unpaired character
	static const char unpaired = '.';
	
	// The length of front text field
	static const int front_length = 22;
	
	// Length of the short names
	static const int length_name_field = 13;
	
	// Number of nucleotides pr line in summary alignment
	static const int line_length = 40;
	
	// The width of a column in the column format.
	static const int space = 10;


	inline void entry_head(const std::string name, const std::string file, const int seq_length, const int score, const int group, const int start, const int end, const int length, char seq[], int org_pos[], int org_bp[], int ali_bp[]);

	inline int makeSequences(char*& seq_1, char*& seq_2, char*& struc, 
									 int*& org_pos_1, int*& org_bp_1, int*& ali_bp_1,
									 stack<positionType>*& leftPos_1,
									 stack<positionType>*& leftBasepair_1,
									 int*& org_pos_2, int*& org_bp_2, int*& ali_bp_2,
									 stack<positionType>*& leftPos_2,
									 stack<positionType>*& leftBasepair_2);

	inline void outputAlign(std::string name_1, std::string name_2, char sequen_1[], char sequen_2[], char struc[], int length);

	inline void printLine(char line[], int start, int end, std::string head, std::string title);
	
	inline float identity(char sequence1[], char sequence2[], int len, float& similar, float& total);
	
	inline void pf(const std::string field, const bool newline = false) {
		std::cout << std::left << std::setw(front_length) << field << std::right;
		if (newline) {std::cout << std::endl;}
	}

	arguments* arg;
	const sequence* const seq_1; // Stores the sequences
	const sequence* const seq_2;
	scorematrix& s_matrix; // Contains the score matrixs
};

inline void output::head() {
	pf("; FOLDALIGN");
	std::cout << arg->stringOpt("version") << std::endl;
	helper::print_array(reference, reference_size, "; REFERENCE", front_length);
/*	pf("; REFERENCE");
	std::cout << "J.H. Havgaard, E. Torarinsson and J. Gorodkin" <<std::endl;
	pf("; REFERENCE");
	std::cout << "Fast pairwise local structural RNA alignments by pruning" << std::endl;
	pf("; REFERENCE");
	std::cout << "of the dynamical programming matrix" << std::endl;
	pf("; REFERENCE");
	std::cout << "In preparation, 2006" << std::endl;
*/
	pf("; ALIGNMENT_ID");
	std::cout << arg->stringOpt("-ID") << std::endl;
	pf("; ALIGNING");
	std::cout << seq_1->getName() << " against " << seq_2->getName() << std::endl;

}

inline void output::localscorehead() {
	head();
	pf("; SEQUENCE_1_COMMENT");
	std::cout << seq_1->getComment() << std::endl;
	pf("; SEQUENCE_2_COMMENT");
	std::cout << seq_2->getComment() << std::endl;
	pf("; LENGTH_SEQUENCE_1");
	std::cout << seq_1->getLength() << std::endl;
	pf("; LENGTH_SEQUENCE_2");
	std::cout << seq_2->getLength() << std::endl;
	if (seq_2->getFilename().compare(seq_1->getFilename())) {
		pf("; FILENAMES");
		std::cout << seq_1->getFilename() << " and ";
		std::cout << seq_2->getFilename() << std::endl;
	}
	else {
		pf("; FILENAME");
		std::cout << seq_1->getFilename() << std::endl;
	}

	parameters();

	pf("; TYPE");
	std::cout << "Foldalign_local_scores" << std::endl;
	pf("; COL 1");
	std::cout << "label" << std::endl;
	pf("; COL 2");
	std::cout << "Alignment_start_position_sequence_1" << std::endl;
	pf("; COL 3");
	std::cout << "Alignment_end_position_sequence_1" << std::endl;
	pf("; COL 4");
	std::cout << "Alignment_start_position_sequence_2" << std::endl;
	pf("; COL 5");
	std::cout << "Alignment_end_position_sequence_2" << std::endl;
	pf("; COL 6");
	std::cout << "Alignment_score" << std::endl;
	std::cout <<"; ------------------------------------------------------------------------------" << std::endl;
}

inline void output::foldout(stack<positionType>*& leftPos_1, stack<positionType>*& leftBasepair_1,stack<positionType>*& leftPos_2, stack<positionType>*& leftBasepair_2, int align_start_1, int align_len_1, int align_start_2, int align_len_2, int align_score) {
	if (arg->boolOpt("switch")) {
		// Switch back. The orginal switch was done in foldalign.cxx
		// The constructor switches seq_1 and seq_2 so they are not switched here
		helper::swap(leftBasepair_1, leftBasepair_2);
		helper::swap(leftPos_1, leftPos_2);
		helper::swap(align_start_1, align_start_2);
		helper::swap(align_len_1, align_len_2);
	}

	// Setting up
	int lambda = arg->ltOpt("-max_length");
	char* sequence_1 = new char[2*lambda]; // Stores the sequence
	char* sequence_2 = new char[2*lambda];
	char* structure = new char[2*lambda];  // Stores the structure of the first sequence
	int* org_pos_1 = new int[2*lambda];   // Position in original sequence
	int* org_pos_2 = new int[2*lambda];
	int* org_bp_1 = new int[2*lambda];    // Base-pairing in original coordinates
	int* org_bp_2 = new int[2*lambda];
	int* ali_bp_1 = new int[2*lambda];    // Base-pairing in the new coordinates
	int* ali_bp_2 = new int[2*lambda];
	std::string local_name_1 = seq_1->getName().substr(0,length_name_field);
	std::string local_name_2 = seq_2->getName().substr(0,length_name_field);
	int length_comment = 80 - front_length;
	for(int i=0; i < 2*lambda; i++) {structure[i] = '!';}

	// Make sequences and the structure
	int total_length = makeSequences(sequence_1, sequence_2, structure,
												org_pos_1, org_bp_1, ali_bp_1,
												leftPos_1, leftBasepair_1,
												org_pos_2, org_bp_2, ali_bp_2,
												leftPos_2, leftBasepair_2);

	// Print the comments
	pf("; SEQUENCE_1_COMMENT");
	std::cout << seq_1->getComment().substr(0,length_comment) << std::endl;
	pf("; SEQUENCE_2_COMMENT");
	std::cout << seq_2->getComment().substr(0,length_comment) << std::endl;

	pf("; ALIGN", true);

	// Print the score and idendity
	pf("; ALIGN");
	std::cout << "Score: " << align_score << std::endl;
	pf("; ALIGN");
	float similar;
	float total;
	float sim = 100*identity(sequence_1, sequence_2, total_length, similar, total);
	int pres = 2;
	if (sim >= 100) {pres=3;}
	std::cout << "Identity: " << std::setprecision(pres) << sim << std::setprecision(6) << " % ( " << similar << " / " << total << " )" << std::endl;

	pf("; ALIGN", true);

	// Print the start coordinates
	pf("; ALIGN");
	std::cout << std::left << std::setw(length_name_field) << local_name_1;
	std::cout << " Begin " << align_start_1 << std::endl;
	pf("; ALIGN");
	std::cout << std::left << std::setw(length_name_field) << local_name_2;
	std::cout << " Begin " << align_start_2 << std::endl;

	pf("; ALIGN", true);

	// Print the alignment
	outputAlign(local_name_1, local_name_2, sequence_1, sequence_2, structure, total_length);

	pf("; ALIGN", true);

	// Print the end positions
	pf("; ALIGN");
	std::cout << std::left << std::setw(length_name_field) << local_name_1;
	std::cout << " End " << align_start_1+align_len_1-1 << std::endl;
	pf("; ALIGN");
	std::cout << std::left << std::setw(length_name_field) << local_name_2;
	std::cout << " End " << align_start_2+align_len_2-1 << std::endl;

	std::cout <<"; ==============================================================================" << std::endl;

	// If more than just the summary is wanted print the rest.
	if(!arg->boolOpt("-summary")) {
		entry_head(seq_1->getName(), seq_1->getFilename(), seq_1->getLength(), align_score, seq_1->getGroupNumber(), align_start_1, (align_start_1+align_len_1-1), total_length, sequence_1, org_pos_1, org_bp_1, ali_bp_1);
		entry_head(seq_2->getName(), seq_2->getFilename(), seq_2->getLength(), align_score, seq_2->getGroupNumber(), align_start_2, (align_start_2+align_len_2-1), total_length, sequence_2, org_pos_2, org_bp_2, ali_bp_2);
	}

	// Clean up
	delete[] sequence_1;
	delete[] sequence_2;
	delete[] structure;
	delete[] org_pos_1;
	delete[] org_pos_2;
	delete[] org_bp_1;
	delete[] org_bp_2;
	delete[] ali_bp_1;
	delete[] ali_bp_2;
}

inline void output::parameters() {
	pf("; PARAMETER");
	std::cout <<"max_length=" << arg->ltOpt("-max_length") << std::endl;
	pf("; PARAMETER");
	std::cout <<"max_diff=" << arg->ltOpt("-max_diff") << std::endl;
	pf("; PARAMETER");
	std::cout <<"min_loop=" << arg->ltOpt("-min_loop") << std::endl;
	pf("; PARAMETER");
	std::cout <<"score_matrix=";
	if (arg->stringOpt("-score_matrix").compare("<default>")) {
		std::cout << arg->stringOpt("-score_matrix") << std::endl;
	}
	else {
		std::cout << "<" << s_matrix.getName() << ">" << std::endl;
	}
	pf("; PARAMETER");
	std::cout <<"nobranching=";
	if (arg->boolOpt("-nobranch")) {std::cout << "<true>" << std::endl;}
	else {std::cout << "<false>" << std::endl;}
	pf("; PARAMETER");
	std::cout <<"global=";
	if (arg->boolOpt("-global")) {std::cout << "<true>" << std::endl;}
	else {std::cout << "<false>" << std::endl;}
	pf("; PARAMETER");
	std::cout <<"i=" << arg->ptOpt("-i") << std::endl;
	pf("; PARAMETER");
	std::cout <<"j=" << arg->ptOpt("-j") << std::endl;
	pf("; PARAMETER");
	std::cout <<"k=" << arg->ptOpt("-k") << std::endl;
	pf("; PARAMETER");
	std::cout <<"l=" << arg->ptOpt("-l") << std::endl;

	pf("; PARAMETER");
	if (arg->boolOpt("-no_pruning")) {
		std::cout <<"no_prune=<true>" << std::endl;
	}
	else {
		std::cout <<"no_prune=<false>" << std::endl;
	}
}

inline void output::entry_head(const std::string name, const std::string file, const int seq_length, const int score, const int group, const int start, const int end, const int length, char seq[], int org_pos[], int org_bp[], int ali_bp[]) {
	// Output an alignment entry

	pf("; TYPE");
	std::cout <<"RNA" << std::endl;
	pf("; COL 1");
	std::cout <<"label" << std::endl;
	pf("; COL 2");
	std::cout <<"residue" << std::endl;
	pf("; COL 3");
	std::cout <<"seqpos" << std::endl;
	pf("; COL 4");
	std::cout <<"alignpos" << std::endl;
	pf("; COL 5");
	std::cout <<"align_bp" << std::endl;
	pf("; COL 6");
	std::cout <<"seqpos_bp" << std::endl;
	pf("; ENTRY");
	std::cout << name << std::endl;
	pf("; ALIGNMENT_ID");
	std::cout << arg->stringOpt("-ID") << std::endl;
	pf("; ALIGNMENT_LIST");
	std::cout << seq_1->getName() << " " << seq_2->getName() << std::endl;
	pf("; FOLDALIGN_SCORE");
	std::cout << score << std::endl;
	pf("; GROUP");
	std::cout << group << std::endl;
	pf("; FILENAME");
	std::cout << file << std::endl;
	pf("; START_POSITION");
	std::cout << start << std::endl;
	pf("; END_POSITION");
	std::cout << end << std::endl;
	pf("; ALIGNMENT_SIZE");
	std::cout <<"2" << std::endl;
	pf("; ALIGNMENT_LENGTH");
	std::cout << length << std::endl;
	pf("; SEQUENCE_LENGTH");
	std::cout << seq_length << std::endl;

	parameters();

	std::cout <<"; ------------------------------------------------------------------------------" << std::endl;
	for (int i=0; i<length; i++) {
		if (seq[i] != '-') {
			std::cout << "N";
			std::cout << std::setw(space) << seq[i];
			std::cout << std::setw(space) << org_pos[i];
			std::cout << std::setw(space) << (i+1);
			if (ali_bp[i] !=-1) {
				std::cout << std::setw(space) << (ali_bp[i]+1);
				std::cout << std::setw(space) << org_bp[i] << std::endl;
			}
			else {
				std::cout << std::setw(space) << ".";
				std::cout << std::setw(space) << "." << std::endl;
			}
		}
		else {
			std::cout << "G";
			std::cout << std::setw(space) << "-";
			std::cout << std::setw(space) << ".";
			std::cout << std::setw(space) << (i+1);
			std::cout << std::setw(space) << ".";
			std::cout << std::setw(space) << ".";
			std::cout << std::endl;
		}
	}
	std::cout <<"; ******************************************************************************" << std::endl;
}

inline int output::makeSequences(char*& sequence_1, char*& sequence_2, 
										char*& struc, 
               					int*& org_pos_1, int*& org_bp_1, int*& ali_bp_1, 
									   stack<positionType>*& leftPos_1,
				   					stack<positionType>*& leftBasepair_1,
               					int*& org_pos_2, int*& org_bp_2, int*& ali_bp_2,
				   					stack<positionType>*& leftPos_2,
				   					stack<positionType>*& leftBasepair_2)
{
	// Get a sequence and a structure from the backtrack stacks.
	// Returns the length of the sequence
	int counter = 0;
	lengthType length = 2*arg->ltOpt("-max_length");
	stack<positionType> bp_1(length);
	stack<positionType> bp_2(length);
	while (leftPos_1->getfifosize() > 0) {

		// Get the position
		int pos_1 = leftPos_1->fifo();
		int pos_2 = leftPos_2->fifo();

		// Build the sequences
		if (pos_1 > 0) { // Handles nucleotides (not gaps)
			sequence_1[counter] = toupper(seq_1->getLetter(pos_1));
			org_pos_1[counter] = pos_1;
		}
		else { // Handle gaps
			sequence_1[counter] = s_matrix.getLetter(0);
			org_pos_1[counter] = 0;
		}

		if (pos_2 > 0) { // Handles nucleotides (not gaps)
			sequence_2[counter] = toupper(seq_2->getLetter(pos_2));
			org_pos_2[counter] = pos_2;
		}
		else { // Handle gaps
			sequence_2[counter] = s_matrix.getLetter(0);
			org_pos_2[counter] = 0;
		}

		// Get the base-pairing information
		int basepair_1 = leftBasepair_1->fifo();
		int basepair_2 = leftBasepair_2->fifo();
		
		
		if ((basepair_1 > -1) && (basepair_2 > -1)) {
			if (pos_1 < basepair_1) {
				struc[counter] = open;
				bp_1.push(counter);
				bp_2.push(counter);
			}
			else {
				struc[counter] = close;
				int left_1 = bp_1.pop();
				int left_2 = bp_2.pop();
				ali_bp_1[counter] = left_1;
				ali_bp_2[counter] = left_2;
				ali_bp_1[left_1] = counter;
				ali_bp_2[left_2] = counter;
			}
			org_bp_1[counter] = basepair_1;
			org_bp_2[counter] = basepair_2;
		}
		else if ((basepair_1 > -1) || (basepair_2 > -1)) {
		
			if (basepair_1 > -1) {
				org_bp_1[counter] = basepair_1;
				org_bp_2[counter] = 0;
				ali_bp_2[counter] = -1;
				if (pos_1 < basepair_1) {
					struc[counter] = open_one_seq;
					bp_1.push(counter);
				}
				else {
					struc[counter] = close_one_seq;
					int left_1 = bp_1.pop();
					ali_bp_1[counter] = left_1;
					ali_bp_1[left_1] = counter;
				}
			}
			else {
				org_bp_1[counter] = 0;
				org_bp_2[counter] = basepair_2;
				ali_bp_1[counter] = -1;
				if (pos_2 < basepair_2) {
					struc[counter] = open_one_seq;
					bp_2.push(counter);
				}
				else {
					struc[counter] = close_one_seq;
					int left_2 = bp_2.pop();
					ali_bp_2[counter] = left_2;
					ali_bp_2[left_2] = counter;
				}
			}
		}		
		else {
			struc[counter] = unpaired;
			org_bp_1[counter] = 0;
			org_bp_2[counter] = 0;
			ali_bp_1[counter] = -1;
			ali_bp_2[counter] = -1;
		}
		counter++;
	}
	sequence_1[counter] = '\0';
	sequence_2[counter] = '\0';
	struc[counter] = '\0';
	return counter;
}

inline float output::identity(char sequence1[], char sequence2[], int len, float& similar, float& total) {
	total = 0; // The total number of nucleotides. Two aligned gaps are not counted
	similar = 0; // The number of similar nucleotides. Gaps do not count as similar
	for(int i=0; i < len; i++) {
		if ((sequence1[i] == '\0') || (sequence2[i] == '\0')) {
			if (sequence1[i] != sequence2[i]) {
				std::cerr << "Program error. The two sequences in the identity function do not have the same length" << std::endl;
				throw -1;
			}
			return similar/total;
		}
		if (sequence1[i] == sequence2[i]) {
			// The nucleotides are similar
			if (sequence1[i] != '-') {
				// They are not gaps
				total++;
				similar++;
			}
		}
		else {
			total++;
		}
	}
	return similar/total;
}

inline void output::outputAlign(std::string name_1, std::string name_2, char sequen_1[], char sequen_2[], char struc[], int length) {
	// Output the summary alignment
	std::string head = "; ALIGN";

	const int num_lines = length/line_length; // Calculate the number of lines (-1). Integer divison intended
	for(int i=0; i < num_lines; i++) {
		printLine(sequen_1, (i*line_length), ((i+1)*line_length), head, name_1);
		printLine(struc, (i*line_length), ((i+1)*line_length), head, "Structure");
		printLine(sequen_2, (i*line_length), ((i+1)*line_length), head, name_2);
		pf(head, true);
	}
	if (num_lines*line_length != length) {
		printLine(sequen_1, (num_lines*line_length), length, head, name_1);
		printLine(struc, (num_lines*line_length), length, head, "Structure");
		printLine(sequen_2, (num_lines*line_length), length, head, name_2);
	}
}

inline void output::printLine(char line[], int start, int end, std::string head, std::string title) {

	// Output an alignment line
	int length = end - start +1; // The length of the line
	int num_space = length/10; // The number of spaces to be added
	
	pf(head);

	std::cout << std::left << std::setw(length_name_field) << title << std::right;

	int pos = start;
	int stop = start;
	for(int i =0; i<=num_space; i++) {
		pos = stop;
		stop+=10;
		if (stop > end) {stop = end;}
		if (pos < stop) {std::cout << ' ';}
		for(int j=pos; j<stop; j++) {
			std::cout << line[j];
		}
	}
	std::cout << std::endl;
}


#endif
