#ifndef SCOREMATRIX
#define SCOREMATRIX

#include "foldalign.h"
#include "readfile.cxx"
#include "helper.cxx"
#include "stack_ssl.cxx"
#include "exception.cxx"

#include <string>
#include <iostream>
#include <math.h>

/******************************************************************************
*                                                                             *
*   Copyright 2004 - 2007 Jakob Hull Havgaard, hull@genome.ku.dk              *
*                                                                             *
*   This file is part of Foldalign                                            *
*                                                                             *
*   Foldalign is free software; you can redistribute it and/or modify         *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   Foldalign is distributed in the hope that it will be useful,              *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with Foldalign; if not, write to the Free Software                  *
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
*                                                                             *
******************************************************************************/

//*********************************************************
// This class holds the scoremtricies needed by foldalign *
// Implemented by Jakob Hull Havgaard 2004                *
// hull@genome.ku.dk                                     *
//                                                        *
// The class do not function as a template since the atoi *
// function is used to translate the values in functions  *
// setMatrix4, getValue, and setInitMatrix                *
//*********************************************************

class scorematrix {
public:
	//**************************************************************************
	// The class has 4 constructors.
	// a default which sets up the default values
	// one which updates the default values with those read from a file
	// the last two are the assignment and copy constructors

	// A default constructor
	scorematrix(const bool global = false) {score_matrix(global);};

	// One that reads all or some of the matrices from a file
	scorematrix(std::string& filename, const bool global = false);

	// and an assignment constructor
	scorematrix& operator=(const scorematrix& s){
		if (this != &s) {
			clean();
			copy(s);
		}
		return *this;
	};

	// The copy constructor
	scorematrix(const scorematrix& s) {copy(s);};

	// The destructor.
	~scorematrix() {
		clean();
		// asymTable must be deleted separately.
		delete[] asymTable;
	};

	//**************************************************************************
	// Functions for accessing the information

	//==========================================================================
	// The stacking bonus
	// last stack on first
	scoreType getStack(const int i, const int j, const int ii, const int jj,const int k, const int l, const int kk, const int ll) const {return (stack_matrix[i][j][ii][jj] + stack_matrix[k][l][kk][ll]);};
	scoreType getStack(const int i, const int j, const int ii, const int jj) const {return stack_matrix[i][j][ii][jj];};

	//==========================================================================
	// The affine gap penalty
	scoreType getGap() const {return elongation - gap_open;};
	// The gap open penalty
	scoreType getGapOpen() const {return gap_open;};
	
	//==========================================================================
	// The cost of base pair substitutions
	scoreType getScore(const int i, const int j, const int k, const int l) const {return s_matrix[i][j][k][l];};
	// The single strand substitution cost
	scoreType getInit(const int i, const int k) const {return i_matrix[i][k];};

	//==========================================================================
	// The name of the matrix: default or filename
	std::string getName() const {return matrixname;};

	//==========================================================================
	// True if the bases basepairs
	bool getBasepair(const int i, const int k) const {return basepair[i][k];};

	//==========================================================================
	// The size of the alfabet
	int alfaSize() const {return alfa_size;};
	// The letter at position pos in the alfabet
	char getLetter(const int pos) const {return alfabet[pos];};
	// The alfabet position of the letter letter
	inline int alfa(char letter) const;

	//==========================================================================
	// Returns the multibranch loop opening score
	scoreType getMbl() const {return mbl;}
	// Returns the affine multibranch loop cost
	scoreType getMblAffine() const {return mblAffine;}
	// Returns the cost for adding a nucleotide to a mbl
	scoreType getMblNuc() const {return mblNuc;}

	//==========================================================================
	// Returns the endNonGC value
	scoreType getNonGCEnd(const int i, const int j, 
	                      const int k, const int l) {
		return endNonGC[i][j] + endNonGC[k][l];
	};

	scoreType getNonGCEnd(const int i, const int j) {return endNonGC[i][j];}

	//==========================================================================
	// These functions are for handling the loop close scores
	
	scoreType getHpClose(const int i, const int j, 
	                     const int ii, const int jj,
								const int k, const int l,
								const int kk, const int ll) const {
		return hp_close_matrix[i][j][ii][jj]+hp_close_matrix[k][l][kk][ll];
	};
	scoreType getHpClose(const int i, const int j, 
	                     const int ii, const int jj) const {
		return hp_close_matrix[i][j][ii][jj];
	};
	scoreType getIntLoopOpen(const int bi, const int bj, 
	                         const int si, const int sj) const {
		return internal_loop_matrix[bi][bj][si][sj];
	};
	scoreType getIntLoopClose(const int bi, const int bj,
	                          const int si, const int sj) const {
		return internal_loop_matrix[bi][bj][si][sj];
	};
	scoreType getHpLength(const int i, const int j) const {
		return hpLength[i]+hpLength[j];
	};
	scoreType getIntLoopLength(const int i, const int j);
	scoreType getBulgeLength(const int i, const int j) const {
		return bulgeLength[i]+bulgeLength[j];
	};
	scoreType getBulgeLength(const int i) const {
		return bulgeLength[i];
	};

	//==========================================================================
	// Expand the size of the tables.

	inline void checkSize(const int size);

	//==========================================================================
	// Return the pruning minimum score given a length

	inline scoreType getPruneScore(const int i) const {return pruneTable[i];};
	inline lengthType getPruneTableLength() const {return pruneTableLength;}

	//==========================================================================
	// Return the log (base 2)

	inline double getLog(const lengthType Wi) const {return logTable[Wi];}

private:

	//======================================================================
	// The matrix's name
	std::string matrixname; // Default or filename

	//======================================================================
	// The energy matrices and tables
	int loopTableLength; // The length of the loop cost table
	scoreType**** stack_matrix;  // The stacking matrix
	scoreType**** hp_close_matrix;  // The hair-pin open matrix
	scoreType**** internal_loop_matrix;  // The internal loop open/close matrix
	scoreType** endNonGC;	// The extra cost for terminal non GC base-pair. Zero for GC-pairs.
	scoreType* hpLength;		 // The hairpin loop length cost
	scoreType* bulgeLength; // The bulge loop length cost
	scoreType* ilLength;    // The internal loop length cost
	scoreType ilLong;         // The scale factor for long internal loops
	scoreType bulgeLong;      // The scale factor for long bulge loops
	scoreType hpLong;         // The scale factor for long hairpin loops
	scoreType asym;				// The per base cost for asymetric internal loops
	scoreType masym;				// The per base cost for asymetric internal loops
	scoreType* asymTable;	   // A table of asymetry cost.
	lengthType asymTableSize;   // The size of the asymTable is 2*asymTalbeSize+1
	scoreType mbl;				// Multibranchloop opening cost
	scoreType mblAffine;	   // The affine multibranch cost
	scoreType mblNuc;         // The cost of adding a nucleotide to the mbl


	//======================================================================
	// The substitution and base pair matrices
	scoreType**** s_matrix;  // The substitution matrix
	scoreType** i_matrix;    // The scorematrix for initial alignment
	bool** basepair;   // The basepairing matrix


	//======================================================================
	// The alfabet
	char* alfabet;     // The alfabet
	int alfa_size;     // The size of the alfabet
	int tmp_alfa_size; // A tmp variable alfa_size. Used when the size of the matrices is changed
	bool alfadie;      // Used to make sure the matricies in a file is ordered correctly

	//======================================================================
	// Gaps
	scoreType gap_open;      // The gap open score
	scoreType elongation;    // The gap elongation score


	//======================================================================
	// Prunning
	scoreType* pruneTable;	// Alignments with scores below these scores are pruned
	int pruneTableLength; 	// The size of the pruning table
	int lastPruneIndex;		// From this index the linear pruning coefficient is used
	scoreType linear_prune; 	// The linear pruning coefficient

	// Used to store table values read from a file
	struct prunes {
		scoreType score;
		lengthType index;
	};

	//======================================================================
	// The log table
	double* logTable; 		// Table of logrithms. Base 2.

	//======================================================================
	// Define the defualts
	inline void score_matrix(const bool global);
	
	//======================================================================
	// Reading functions
	inline void setAlfabet(readfile*& file);     // Read a new alfabet from file
	inline void setBasepair(readfile*& file);    // Read a new basepairings matrix from file
	inline void setInitMatrix(readfile*& file); // Read a new init matrix from file
	inline void setLoopTable(readfile*&file);    // Read a new loop cost table
	inline void setMisc(readfile*& file);        // Read the misc values
	inline void setMatrix4(readfile*& file, scoreType****& matrix, scoreType init = 0); // Read a 4x4x4x4 matrix and store it

	//======================================================================
	// Helper functions
	inline void store(std::string name, scoreType value);        // A helper function to setMisc
	inline void parseLine(std::string line, int*& i_letters, int& len); // Parse a annotation line
	inline scoreType getValue(int& prev, std::string& line); // Extract a value from a line
	inline std::string findName(int& prev, std::string& line);
	inline void setGap4(scoreType****& matrix);
	inline void dumpArray(const scoreType* array, const int size);


	//======================================================================
	// General matrix functions
	template<class local_scoreType> inline void makeMatrix4(local_scoreType****& matrix, const int size); // Copy a 4-dimensional matrix
	template<class local_scoreType> inline void deleteMatrix4(local_scoreType****& matrix, const int size); // Delete a 4-dimensional matrix
	template<class local_scoreType> inline void copyMatrix4(local_scoreType****& matrix, local_scoreType**** const& old_matrix, const int size);
	template<class local_scoreType> inline void assignMatrix4(local_scoreType****& matrix, const int size, const local_scoreType value); // assigns value to all positions in the 4-dimensional matrix
	template<class local_scoreType> inline void makeMatrix2(local_scoreType**& matrix, const int size); // Copy a 2-dimensional matrix
	template<class local_scoreType> inline void deleteMatrix2(local_scoreType**& matrix, const int size); // Delete a 2-dimensional matrix
	template<class local_scoreType> inline void copyMatrix2(local_scoreType**& matrix, local_scoreType** const& old_matrix, const int size);
	template<class local_scoreType> inline void assignMatrix2(local_scoreType**& matrix, const int size, const local_scoreType value); // assigns value to all positions in the 4-dimensional matrix
	template<class local_scoreType> inline void assignArray(local_scoreType*& matrix, const local_scoreType value, const int size);


	//======================================================================
	//Misc calculations
	inline scoreType calcLongScore(const int factor, const lengthType i, const lengthType length, const scoreType zero);
	inline void calcScore(const int factor, const int size, scoreType*& array);
	inline void calcAsym(const int size);
	inline void makeLogTable(lengthType end);

	//======================================================================
	// Prunings functions
	inline scoreType pruneScore(const scoreType co, const lengthType len_new, const lengthType len_old, const scoreType zero);
	inline void calcPrune(const lengthType size);
	inline void setPruneTable(readfile*& file);
	inline void fillPrune(const lengthType x1, const scoreType y1, const lengthType x2, stack_ssl<prunes, lengthType>& lineStack);
	inline void assignFromStack(scoreType*& store, stack_ssl<prunes, lengthType>& lineStack);

	
	//======================================================================
	// Helper functions for coping and destruction
	// Deletes every thing except the asymetry table
	inline void clean();
	// Copies everything.
	inline void copy(const scorematrix& s);

};

//********************************************
// Read a scorematrix file

inline scorematrix::scorematrix(std::string& filename, const bool global) {
	score_matrix(global);
	matrixname = filename;
	std::string line;

	// Opening the new file
	readfile* file;
	try {
		file = new readfile(filename);
	}
	catch ( ... ) {
		std::string error = "It is not possible to open the scorematrix file: ";
		error += filename;
		throw exception(error, false);
	}

	while (file->get_line(line)) {
		if (!line.compare("Alfabet:")) {setAlfabet(file);}
		else if (!line.compare("Base-pair:")) {setBasepair(file);}
		else if (!line.compare("Base-pair substitution:")) {setMatrix4(file, s_matrix);}
		else if (!line.compare("Single strand substitution:")) {setInitMatrix(file);}
		else if (!line.compare("Stacking:")) {setMatrix4(file, stack_matrix);}
		else if (!line.compare("Hairpin Close:")) {setMatrix4(file, hp_close_matrix);}
		else if (!line.compare("Internal loop:")) {setMatrix4(file, internal_loop_matrix);}
		else if (!line.compare("5' Dangle:")) {
			std::cerr << "Note! The 5' dangle matrix is no longer used. Any values will be ignored." << std::endl;
			std::cerr << "Skipping to the next empty line" << std::endl;
			file->skip_to_empty();
		}
		else if (!line.compare("3' Dangle:")) {
			std::cerr << "Note! The 3' dangle matrix is no longer used. Any values will be ignored." << std::endl;
			std::cerr << "Skipping to the next empty line" << std::endl;
			file->skip_to_empty();
		}
		else if (!line.compare("Loop length costs:")) {setLoopTable(file);}
		else if (!line.compare("Miscellaneous:")) {setMisc(file);}
		else if (!line.compare("Pruning:")) {setPruneTable(file);}
		else if (!line.compare("")) {}
		else {
			std::string error = "The score matrix do not have the right format\n";
			error += line;
			throw exception(error, false);
		}
	}

	delete file;
	
	alfa_size = tmp_alfa_size;

	// Setup the gap opening cost
	for(int i=0; i<alfa_size; i++) {
		i_matrix[i][0] = gap_open;
		i_matrix[0][i] = gap_open;
	}
	i_matrix[0][0] = gap_open;
	setGap4(s_matrix);
	calcAsym(asymTableSize);
	calcPrune(pruneTableLength-1);
}

//*********************************************
// Assignment operator for the scorematrix object

inline int scorematrix::alfa(char letter) const {
	char upper = toupper(letter);
	for(int i =0; i<alfa_size; i++) {
		if (alfabet[i]==upper) {return i;}
	}
	if(upper=='T') {return 4;}
	std::cerr << "Unknown nucleotide: " << letter << " found." << std::endl;
	return alfa_size -1;
}

inline scoreType scorematrix::getIntLoopLength(const int i, const int j) {

	scoreType res = ilLength[i+j];

	// The Ninno equation (in the case where the cost is independent of the length)
	// A value of 5 is ten times the values set in the miscloop file of dynalign
	res += asymTable[i-j+asymTableSize];

	return res;
}

//******************************************************
// The private functions

inline void scorematrix::clean() {

	// Note assymTable is not deleted by this function.
	
	int i_size=alfaSize();
	deleteMatrix4(s_matrix, i_size);
	deleteMatrix4(stack_matrix, i_size);
	deleteMatrix4(hp_close_matrix, i_size);
	deleteMatrix4(internal_loop_matrix, i_size);
	deleteMatrix2(i_matrix, i_size);
	deleteMatrix2(basepair, i_size);
	deleteMatrix2(endNonGC, i_size);
	delete[] alfabet;
	delete[] hpLength;
	delete[] bulgeLength;
	delete[] ilLength;
	delete[] logTable;
	delete[] pruneTable;
}

inline void scorematrix::copy(const scorematrix& s) {
	matrixname = s.matrixname;

	alfa_size = s.alfaSize();
	alfabet = new char[alfa_size];
	helper::copyArray(alfabet, s.alfabet, alfa_size);

	gap_open = s.gap_open;
	elongation = s.elongation;
	asym = s.asym;
	masym = s.masym;
	mbl = s.mbl;
	mblAffine = s.mblAffine;
	mblNuc = s.mblNuc;

	makeMatrix4(s_matrix, alfa_size);
	copyMatrix4(s_matrix, s.s_matrix, alfa_size);
	makeMatrix4(stack_matrix, alfa_size);
	copyMatrix4(stack_matrix, s.stack_matrix, alfa_size);
	makeMatrix4(hp_close_matrix, alfa_size);
	copyMatrix4(hp_close_matrix, s.hp_close_matrix, alfa_size);
	makeMatrix4(internal_loop_matrix, alfa_size);
	copyMatrix4(internal_loop_matrix, s.internal_loop_matrix, alfa_size);
	makeMatrix2(i_matrix, alfa_size);
	copyMatrix2(i_matrix, s.i_matrix, alfa_size);
	makeMatrix2(basepair, alfa_size);
	copyMatrix2(basepair, s.basepair, alfa_size);
	makeMatrix2(endNonGC, alfa_size);
	copyMatrix2(endNonGC, s.endNonGC, alfa_size);

	loopTableLength = s.loopTableLength;

	hpLong = s.hpLong;
	hpLength = new scoreType[loopTableLength];
	helper::copyArray(hpLength, s.hpLength, loopTableLength);

	bulgeLong = s.bulgeLong;
	bulgeLength = new scoreType[loopTableLength];
	helper::copyArray(bulgeLength, s.bulgeLength, loopTableLength);

	ilLong = s.ilLong;
	ilLength = new scoreType[loopTableLength];
	helper::copyArray(ilLength, s.ilLength, loopTableLength);

	// A dummy pointer. It will be deleted by the calcAsym function
	asymTableSize = s.asymTableSize;
	asymTable = new scoreType[1];
	calcAsym(asymTableSize);

	asym = s.asym;
	masym = s.masym;
	mbl = s.mbl;
	mblAffine = s.mblAffine;
	mblNuc = s.mblNuc;

	logTable = new double[loopTableLength];
	helper::copyArray(logTable, s.logTable, loopTableLength);

	linear_prune = s.linear_prune;
	pruneTableLength = s.pruneTableLength;
	pruneTable = new scoreType[pruneTableLength];
	helper::copyArray(pruneTable, s.pruneTable, pruneTableLength);

}

inline void scorematrix::setGap4(scoreType****& matrix) {
	for(int i=0; i<alfa_size; i++) {
		for(int j=0; j<alfa_size; j++) {
			for(int k=0; k<alfa_size; k++) {
				for(int l=0; l<alfa_size; l++) {
					if ((i==0) || (j==0) || (k==0) || (l==0)) {
						matrix[i][j][k][l] = i_matrix[i][j] + i_matrix[k][l];
					}
				}
			}
		}
	}
}

inline scoreType scorematrix::calcLongScore(const int factor, const lengthType i, const lengthType length, const scoreType zero) {
	return static_cast<scoreType>(log((static_cast<float>(factor*i)/static_cast<float>(factor*(length))))) + zero;
}

// A simple way of calculate the cost of very long loops.
// A better function would be based on the log of i scaled with the factor and the top value
inline void scorematrix::calcScore(const int factor, const int size, scoreType*& array) {
	scoreType* tmp = new scoreType[size+1];
	helper::copyArray(tmp, array, loopTableLength);
	if (factor != 0) {
		for(int i=loopTableLength; i<=size; i++) {
			tmp[i] = calcLongScore(factor, i, loopTableLength-1, array[loopTableLength-1]);
		}
	}
	else {
		for(int i=loopTableLength; i<=size; i++) {
			tmp[i] = 0;
		}
	}
	delete[] array;
	array = tmp;
}

inline void scorematrix::dumpArray(const scoreType* array, const int size) {
	std::cout << "Array size: " << size << std::endl;
	for(int i=0; i<size; i++) {std::cout << i << " " << array[i] << std::endl;}
}

inline void scorematrix::calcAsym(const int size) {

	asymTableSize = size;
	const int real_size = 2*size +1;

	delete[] asymTable;

	asymTable = new scoreType[real_size];

	for (int i = 0; i < real_size; i++) {
		asymTable[i] = asym*(i - size);

		// The final cost has to be negative
		if (asymTable[i] > 0) {asymTable[i] *= -1;}

		// Limits the maximum asymmetric cost
		if (asymTable[i] < masym) {asymTable[i] = masym;}

	}
}


inline scoreType scorematrix::pruneScore(const scoreType co, const lengthType len_new, const lengthType len_old, const scoreType zero) {
	return co*(len_new - len_old) + zero;
}

inline void scorematrix::calcPrune(const lengthType size) {

	// The old table is stored temporarily
	scoreType* old_table = pruneTable;
	lengthType old_size = pruneTableLength;
	
	// The new table is made
	pruneTableLength = size +1;
	pruneTable = new scoreType[pruneTableLength];
	
	// Copy the old table to the new.
	lengthType min_size = old_size < pruneTableLength ? old_size : pruneTableLength;
	helper::copyArray(pruneTable, old_table, min_size);
	
	// If there is no pruning (ie the prune score is big_neg for the last entry)
	// keep it that way
	scoreType co = linear_prune;
	if (pruneTable[lastPruneIndex] == big_neg) {co = 0;}
	
	// Calculate the pruning scores for the rest of the table
	for(lengthType i = lastPruneIndex+1; i < pruneTableLength; i++) {
		pruneTable[i] = pruneScore(co, i, lastPruneIndex, pruneTable[lastPruneIndex]);
	}

	delete[] old_table;
	
}

inline void scorematrix::makeLogTable(lengthType end) {
	// Make sure the zero index is not 0 or very small
	if (end > 0) {logTable[0] = 1;}

	// The length of a sequence with window size Wi is Wi +1 hence the i+1
	for(lengthType i = 1; i < end; i++) {
		logTable[i] = log(i+1)/log(2);
	}
}

inline void scorematrix::checkSize(const int size) {
	if (size > loopTableLength) {
		calcScore(hpLong, size, hpLength);
		calcScore(bulgeLong, size, bulgeLength);
		calcScore(ilLong, size, ilLength);
		calcAsym(size);
		helper::expandArray(logTable, loopTableLength, size);
		makeLogTable(size);
		loopTableLength = size+1;
		calcPrune(size);
	}
}

//********************
// Functions for reading matrixes

inline void scorematrix::setAlfabet(readfile*& file) {
	if (alfadie) {
		std::string error = "The alfabet has to be defined before anything else in the score matrix file.";
		throw exception(error, false);
	}
	delete[] alfabet;
	int prev = 0;
	int pos, end_pos;
	std::string line;
	file->get_line(line);
	pos = line.find_last_not_of(" ");
	end_pos = line.find_last_of(" ", pos);
	
	tmp_alfa_size = atoi(line.substr(pos, (end_pos - pos)).c_str());
	alfabet = new char[tmp_alfa_size];
	for(int i=0; i<tmp_alfa_size; i++) {
		pos = line.find_first_not_of(" ", prev);
		alfabet[i] = line[pos];
		prev=pos+1;
	}
}

inline void scorematrix::setBasepair(readfile*& file) {
	alfadie=true;
	int as = alfaSize();
	deleteMatrix2(basepair, as);
	as = tmp_alfa_size;
	makeMatrix2(basepair, as);
	assignMatrix2(basepair, as, false);
	std::string line;
	int len = as-1;
	int prev = 0;
	int pos, end_pos;
	int* i_letters = new int[len];
	int k_letter;
	char letter;
	file->get_line(line);
	parseLine(line, i_letters, len);
	for(int i=0; i<len; i++) {
		file->get_line(line);
		pos = line.find_last_not_of(" ");
		letter = line[pos];
		k_letter = alfa(letter);
		prev = 0;
		for(int j=0; j<len; j++) {
			pos = line.find_first_not_of(" ", prev);
			end_pos = line.find(" ", pos);
			if (!line.substr(pos, (end_pos - pos)).compare("1")) {
				basepair[i_letters[j]][k_letter] = true;
			}
			else {
				basepair[i_letters[j]][k_letter] = false;
			}
			prev = end_pos+1;
		}
	}
	delete[] i_letters;
}
	
inline void scorematrix::setMatrix4(readfile*& file, scoreType****& matrix, scoreType init) {
	alfadie=true;
	int as = alfaSize();
	deleteMatrix4(matrix, as);
	as = tmp_alfa_size;
	makeMatrix4(matrix, as);
	assignMatrix4(matrix, as, init);
	std::string line;
	int len = (as-2)*(as-2); // -1 for gaps and -1 for ambiguous nucleotides
	int prev = 0;
	int pos;
	int* i_letters = new int[len];
	int* j_letters = new int[len];
	int k_letter;
	int l_letter;
	char letter;
	file->get_line(line);
	parseLine(line, i_letters, len);
	file->get_line(line);
	parseLine(line, j_letters, len);
	for(int i=0; i<len; i++) {
		prev=0;
		file->get_line(line);
		pos = line.find_last_not_of(" ");
		letter = line[pos];
		l_letter = alfa(letter);
		prev = pos-1;
		pos = line.find_last_not_of(" ", prev);
		letter = line[pos];
		k_letter = alfa(letter);
		prev = 0;
		for(int j=0; j<len; j++) {
			matrix[i_letters[j]][j_letters[j]][k_letter][l_letter] = getValue(prev, line);
		}
	}
	delete[] i_letters;
	delete[] j_letters;	
	
}
	
inline void scorematrix::setInitMatrix(readfile*& file) {
	alfadie=true;
	int as = alfaSize();
	deleteMatrix2(i_matrix, as);
	as = tmp_alfa_size;
	makeMatrix2(i_matrix, as);
	assignMatrix2(i_matrix, as, scoreType(0));
	std::string line;
	int len = as-2; // -1 for gaps and -1 for ambiguous nucleotides
	int prev = 0;
	int pos; //, end_pos;
	int* i_letters = new int[len];
	int k_letter;
	char letter;
	file->get_line(line);
	parseLine(line, i_letters, len);
	for(int i=0; i<len; i++) {
		file->get_line(line);
		pos = line.find_last_not_of(" ");
		letter = line[pos];
		k_letter = alfa(letter);
		prev = 0;
		for(int j=0; j<len; j++) {
			i_matrix[i_letters[j]][k_letter] = getValue(prev, line);
		}
	}
	delete[] i_letters;
}


inline void scorematrix::setLoopTable(readfile*& file) {

	alfadie=true; // It is no longer allowed to change the alfabet.
	std::string line;

	stack_ssl<prunes, lengthType> hpStack = stack_ssl<prunes, lengthType>();
	stack_ssl<prunes, lengthType> blStack = stack_ssl<prunes, lengthType>();
	stack_ssl<prunes, lengthType> ilStack = stack_ssl<prunes, lengthType>();

	// Add the zero values
	prunes zero = {0, 0};
	hpStack.push(zero);
	blStack.push(zero);
	ilStack.push(zero);
	
	lengthType last_index = 0;
	scoreType last_hp = 0;
	scoreType last_bl = 0;
	scoreType last_il = 0;

	while(file->get_line_failEmpty(line)) {
		// Read the values stored in the file
		int prev = 0;
		lengthType index = getValue(prev, line);
		if (index != last_index+1) {std::cerr << "Warning: Length " << index << " in the Loop length costs table is shorter than the privious length " << last_index << ". Parts of the table may be corrupt or missing. Ignoring this value." << std::endl; continue;}

		prunes hp = {getValue(prev, line), index};
		prunes bl = {getValue(prev, line), index};
		prunes il = {getValue(prev, line), index};

		hpStack.push(hp);
		blStack.push(bl);
		ilStack.push(il);

		last_index = index;
		last_hp = hp.score;
		last_bl = bl.score;
		last_il = il.score;
	}

	// Shut down the old arrays an make new ones.
	delete[] hpLength;
	delete[] bulgeLength;
	delete[] ilLength;
	loopTableLength = last_index+1;
	hpLength = new scoreType[loopTableLength];
	bulgeLength = new scoreType[loopTableLength];
	ilLength = new scoreType[loopTableLength];

	assignFromStack(hpLength, hpStack);
	assignFromStack(bulgeLength, blStack);
	assignFromStack(ilLength, ilStack);

}
			
inline void scorematrix::setPruneTable(readfile*& file) {
	alfadie=true; // It is no longer allowed to change the alfabet.
	std::string line;

	pruneTable[0] = big_neg;

	stack_ssl<prunes, lengthType> lineStack = stack_ssl<prunes, lengthType>();
	scoreType last_score = big_neg;
	lengthType last_index = 0;
	while (file->get_line_failEmpty(line)) {
		int prev = 0;
		lengthType index = getValue(prev, line);
		scoreType score = getValue(prev, line);
		if (index < last_index) {std::cerr << "Warning: Length " << index << " in the Pruning: table has a smaller length than the previous entry: " << last_index  << ". This table must be sorted by length value. The value is ignored." << std::endl; continue;}

		if (index > last_index +1) {fillPrune(last_index, last_score, index, lineStack);}

		prunes prune = {score, index};
		last_score = score;
		last_index = index;
		lineStack.push(prune);
	}

	if (last_index > pruneTableLength) {
		helper::expandArray(pruneTable, pruneTableLength, last_index+1);
		pruneTableLength = last_index+1;
	}

	assignFromStack(pruneTable, lineStack);
		
	lastPruneIndex = last_index;

}


inline void scorematrix::fillPrune(const lengthType x1, const scoreType y1, const lengthType x2, stack_ssl<prunes, lengthType>& lineStack) {

	for(lengthType x = x1+1; x < x2; x++) {

		prunes prune = {y1, x};
		lineStack.push(prune);
		
	}
}

inline void scorematrix::assignFromStack(scoreType*& store, stack_ssl<prunes, lengthType>& lineStack) {

	const lengthType size = lineStack.size();
	for(lengthType i = 0; i < size; i++) {
		prunes prune = lineStack.pop();
		store[prune.index] = prune.score;
	}
}

inline void scorematrix::setMisc(readfile*& file) {
	std::string line;
	std::string name;
	scoreType value;
	while (file->get_line_failEmpty(line)) {
		int prev = 0;
		name = findName(prev, line);
		value = getValue(prev,line);
		store(name, value);
	}
}

inline void scorematrix::store(std::string name, scoreType value) {
	     if (!name.compare("Gap_open:")) {gap_open=value;}
	else if (!name.compare("Elongation_bonus:")) {elongation=value;}
	else if (!name.compare("Multibranchloop:")) {mbl=value;}
	else if (!name.compare("Multibranchloop_helix:")) {mblAffine=value;}
	else if (!name.compare("Multibranchloop_nucleotide:")) {mblNuc=value;}
	else if (!name.compare("Multibranchloop_non_GC_stem_end:")) {assignMatrix2(endNonGC, alfaSize(), value); endNonGC[2][3] = 0; endNonGC[3][2] = 0;}
	else if (!name.compare("Asymmetric_cost:")) {asym=value;}
	else if (!name.compare("Asymmetric_cost_limit:")) {masym=value;}
	else if (!name.compare("Long_hairpin_loop_factor:")) {ilLong=value;}
	else if (!name.compare("Long_bulge_loop_factor:")) {bulgeLong=value;}
	else if (!name.compare("Long_Internal_loop_factor:")) {hpLong=value;}
	else if (!name.compare("Linear_prunings_coefficient:")) {linear_prune=value;}
	else {std::cerr << "Warning: Unknown parameter " << name << " with value " << value << " found" << std::endl;}
}
	
inline scoreType scorematrix::getValue(int& prev, std::string& line) {
	std::string substr = findName(prev, line);
	if (substr.compare("none")) {
		return atoi(substr.c_str());
	}
	else {
		return big_neg;
	}
}
	
inline std::string scorematrix::findName(int& prev, std::string& line) {
	int pos;
	int end_pos = -1;
	int len = line.length();
	for(pos=prev; pos < len; pos++) {
		if ((line[pos] != ' ') && (line[pos] != '\t')) {
			end_pos = pos+1;
			while ((line[end_pos] != ' ') && (line[end_pos] != '\t') && (end_pos < len)) {end_pos++;}
			break;
		}
	}
	if (end_pos == -1) {
		std::string error = "Could not read score matrix. Somethings wrong with this line\n";
		error +=  line;
		throw exception(error, false);
	}
	prev = end_pos+1;
	return line.substr(pos, (end_pos - pos));
}

//*****************************************************
// These function reads and parses an input line

inline void scorematrix::parseLine(std::string line, int*& i_letters, int& len) {
	char letter;
	int pos=0;
	int length = line.length();
	for(int i=0; i<len; i++) {
		while ((line[pos] == ' ') || (line[pos] == '\t')) {pos++;}
		if (pos > length) {
			std::string error = "Could not read score matrix. Somethings wrong with this line\n";
			error += line;
			throw exception(error, false);
		}
		letter = line[pos];
		i_letters[i] = alfa(letter);
		pos++;
	}
}

inline void scorematrix::score_matrix(const bool global) {
	//*****************************************************
	// This function sets the default values.

	// The alphabet can stil be changed.
	alfadie=false;
	alfa_size = tmp_alfa_size = 6;
	alfabet = new char[alfa_size];
	alfabet[0] = '-';
	alfabet[1] = 'A';
	alfabet[2] = 'C';
	alfabet[3] = 'G';
	alfabet[4] = 'U';
	alfabet[5] = 'N';

	// Make the matrices. They will be filled below
	loopTableLength = 31;
	int i_size=alfaSize();
	makeMatrix4(s_matrix, i_size);
	makeMatrix4(stack_matrix, i_size);
	makeMatrix4(hp_close_matrix, i_size);
	makeMatrix4(internal_loop_matrix, i_size);
	makeMatrix2(i_matrix, i_size);
	makeMatrix2(basepair, i_size);
	makeMatrix2(endNonGC, i_size);
	hpLength = new scoreType[loopTableLength];
	bulgeLength = new scoreType[loopTableLength];
	ilLength = new scoreType[loopTableLength];
	
	// Build log table
	logTable = new double[loopTableLength];
	makeLogTable(loopTableLength);

	if (global) {
		matrixname ="default_global";

		gap_open = -50;
		elongation = -25;

		// Setting default pruning parameters:
		linear_prune = 1;
		pruneTableLength = loopTableLength;
		pruneTable = new scoreType[pruneTableLength];
		for (lengthType i = 0; i< pruneTableLength; i++) {
			pruneTable[i] = -200 +i;
		}
		lastPruneIndex = pruneTableLength -1;
	
		
		// The gaps will be set after the i_matrix has been initilized
		assignMatrix4(s_matrix, alfa_size, scoreType(0));
		s_matrix[1][4][1][4] = 21;
		s_matrix[2][3][1][4] = 5;
		s_matrix[3][2][1][4] = 9;
		s_matrix[4][1][1][4] = 4;
		s_matrix[4][3][1][4] = -6;
		s_matrix[1][4][2][3] = 5;
		s_matrix[2][3][2][3] = 26;
		s_matrix[3][2][2][3] = 10;
		s_matrix[3][4][2][3] = -2;
		s_matrix[4][1][2][3] = 10;
		s_matrix[4][3][2][3] = 2;
		s_matrix[1][4][3][2] = 9;
		s_matrix[2][3][3][2] = 10;
		s_matrix[3][2][3][2] = 28;
		s_matrix[3][4][3][2] = 4;
		s_matrix[4][1][3][2] = 4;
		s_matrix[4][3][3][2] = -2;
		s_matrix[2][3][3][4] = -2;
		s_matrix[3][2][3][4] = 4;
		s_matrix[3][4][3][4] = 17;
		s_matrix[4][1][3][4] = -2;
		s_matrix[4][3][3][4] = -9;
		s_matrix[1][4][4][1] = 4;
		s_matrix[2][3][4][1] = 10;
		s_matrix[3][2][4][1] = 4;
		s_matrix[3][4][4][1] = -2;
		s_matrix[4][1][4][1] = 24;
		s_matrix[4][3][4][1] = 1;
		s_matrix[1][4][4][3] = -6;
		s_matrix[2][3][4][3] = 2;
		s_matrix[3][2][4][3] = -2;
		s_matrix[3][4][4][3] = -9;
		s_matrix[4][1][4][3] = 1;
		s_matrix[4][3][4][3] = 16;

		assignMatrix2(i_matrix, alfa_size, scoreType(0));		
		i_matrix[1][1] = 10;
		i_matrix[2][1] = -10;
		i_matrix[3][1] = -8;
		i_matrix[4][1] = -9;
		i_matrix[1][2] = -10;
		i_matrix[2][2] = 6;
		i_matrix[3][2] = -12;
		i_matrix[4][2] = -7;
		i_matrix[1][3] = -8;
		i_matrix[2][3] = -12;
		i_matrix[3][3] = 5;
		i_matrix[4][3] = -10;
		i_matrix[1][4] = -9;
		i_matrix[2][4] = -7;
		i_matrix[3][4] = -10;
		i_matrix[4][4] = 6;
		i_matrix[1][0] = gap_open;
		i_matrix[2][0] = gap_open;
		i_matrix[3][0] = gap_open;
		i_matrix[4][0] = gap_open;
		i_matrix[0][0] = gap_open;
		i_matrix[0][1] = gap_open;
		i_matrix[0][2] = gap_open;
		i_matrix[0][3] = gap_open;
		i_matrix[0][4] = gap_open;

		// Setting the gaps
		setGap4(s_matrix);
	}
	else {
		matrixname ="default_local";

		gap_open = -110;
		elongation = -55;

		// Setting default pruning parameters:
		linear_prune = 1;
		pruneTableLength = loopTableLength;
		pruneTable = new scoreType[pruneTableLength];
		for (lengthType i = 0; i< pruneTableLength; i++) {
			pruneTable[i] = -400 +i;
		}
		lastPruneIndex = pruneTableLength -1;

		
		// The gaps will be set after the i_matrix has been initilized
		assignMatrix4(s_matrix, alfa_size, scoreType(0));
		s_matrix[1][4][1][4] = 42;
		s_matrix[2][3][1][4] = 11;
		s_matrix[3][2][1][4] = 19;
		s_matrix[4][1][1][4] = 8;
		s_matrix[4][3][1][4] = -14;
		s_matrix[1][4][2][3] = 11;
		s_matrix[2][3][2][3] = 53;
		s_matrix[3][2][2][3] = 20;
		s_matrix[3][4][2][3] = -4;
		s_matrix[4][1][2][3] = 19;
		s_matrix[4][3][2][3] = 4;
		s_matrix[1][4][3][2] = 19;
		s_matrix[2][3][3][2] = 20;
		s_matrix[3][2][3][2] = 56;
		s_matrix[3][4][3][2] = 7;
		s_matrix[4][1][3][2] = 9;
		s_matrix[4][3][3][2] = -6;
		s_matrix[2][3][3][4] = -4;
		s_matrix[3][2][3][4] = 7;
		s_matrix[3][4][3][4] = 34;
		s_matrix[4][1][3][4] = -6;
		s_matrix[4][3][3][4] = -19;
		s_matrix[1][4][4][1] = 8;
		s_matrix[2][3][4][1] = 19;
		s_matrix[3][2][4][1] = 9;
		s_matrix[3][4][4][1] = -6;
		s_matrix[4][1][4][1] = 48;
		s_matrix[4][3][4][1] = 2;
		s_matrix[1][4][4][3] = -14;
		s_matrix[2][3][4][3] = 4;
		s_matrix[3][2][4][3] = -6;
		s_matrix[3][4][4][3] = -19;
		s_matrix[4][1][4][3] = 2;
		s_matrix[4][3][4][3] = 32;
		
		assignMatrix2(i_matrix, alfa_size, scoreType(0));
		i_matrix[1][1] = 19;
		i_matrix[2][1] = -22;
		i_matrix[3][1] = -18;
		i_matrix[4][1] = -19;
		i_matrix[1][2] = -22;
		i_matrix[2][2] = 11;
		i_matrix[3][2] = -25;
		i_matrix[4][2] = -15;
		i_matrix[1][3] = -18;
		i_matrix[2][3] = -25;
		i_matrix[3][3] = 9;
		i_matrix[4][3] = -20;
		i_matrix[1][4] = -19;
		i_matrix[2][4] = -15;
		i_matrix[3][4] = -20;
		i_matrix[4][4] = 13;
		i_matrix[1][0] = gap_open;
		i_matrix[2][0] = gap_open;
		i_matrix[3][0] = gap_open;
		i_matrix[4][0] = gap_open;
		i_matrix[0][0] = gap_open;
		i_matrix[0][1] = gap_open;
		i_matrix[0][2] = gap_open;
		i_matrix[0][3] = gap_open;
		i_matrix[0][4] = gap_open;
		// Setting the gaps
		setGap4(s_matrix);
	}

	// The non GC ends set. Ambigouos treated as non GC
	scoreType default_nonGC_score = -5;
	assignMatrix2(endNonGC, alfa_size, default_nonGC_score); 
	endNonGC[2][3] = 0; 
	endNonGC[3][2] = 0;

	// Setting the misc values
	ilLong = -11;
	bulgeLong = -11;
	hpLong = -11;
	asym = -5;
	masym = -30;
	mbl=0;
	mblAffine=-4;
	mblNuc=-1;
		
	// A dummy memory cell. It will be deleted and resized calcAsym
	asymTable = new scoreType[1];
	calcAsym(loopTableLength);
	
	// Fill the matrices
	assignMatrix2(basepair, alfa_size, false); 
	basepair[1][4]=true;
	basepair[2][3]=true;
	basepair[3][2]=true;
	basepair[3][4]=true;
	basepair[4][1]=true;
	basepair[4][3]=true;
	
	assignMatrix4(stack_matrix, alfa_size, scoreType(0));
	stack_matrix[1][4][1][4] = 9;
	stack_matrix[2][3][1][4] = 21;
	stack_matrix[3][2][1][4] = 24;
	stack_matrix[3][4][1][4] = 13;
	stack_matrix[4][1][1][4] = 13;
	stack_matrix[4][3][1][4] = 10;
	stack_matrix[1][4][2][3] = 22;
	stack_matrix[2][3][2][3] = 33;
	stack_matrix[3][2][2][3] = 34;
	stack_matrix[3][4][2][3] = 25;
	stack_matrix[4][1][2][3] = 24;
	stack_matrix[4][3][2][3] = 15;
	stack_matrix[1][4][3][2] = 21;
	stack_matrix[2][3][3][2] = 24;
	stack_matrix[3][2][3][2] = 33;
	stack_matrix[3][4][3][2] = 21;
	stack_matrix[4][1][3][2] = 21;
	stack_matrix[4][3][3][2] = 14;
	stack_matrix[1][4][3][4] = 6;
	stack_matrix[2][3][3][4] = 14;
	stack_matrix[3][2][3][4] = 15;
	stack_matrix[3][4][3][4] = 5;
	stack_matrix[4][1][3][4] = 10;
	stack_matrix[4][3][3][4] = -3;
	stack_matrix[1][4][4][1] = 11;
	stack_matrix[2][3][4][1] = 21;
	stack_matrix[3][2][4][1] = 22;
	stack_matrix[3][4][4][1] = 14;
	stack_matrix[4][1][4][1] = 9;
	stack_matrix[4][3][4][1] = 6;
	stack_matrix[1][4][4][3] = 14;
	stack_matrix[2][3][4][3] = 21;
	stack_matrix[3][2][4][3] = 25;
	stack_matrix[3][4][4][3] = -13;
	stack_matrix[4][1][4][3] = 13;
	stack_matrix[4][3][4][3] = 5;

	assignMatrix4(hp_close_matrix, alfa_size, scoreType(0));
	hp_close_matrix[1][4][1][1] = 3;
	hp_close_matrix[2][3][1][1] = 15;
	hp_close_matrix[3][2][1][1] = 11;
	hp_close_matrix[3][4][1][1] = -2;
	hp_close_matrix[4][1][1][1] = 5;
	hp_close_matrix[4][3][1][1] = 5;
	hp_close_matrix[1][4][1][2] = 5;
	hp_close_matrix[2][3][1][2] = 15;
	hp_close_matrix[3][2][1][2] = 15;
	hp_close_matrix[3][4][1][2] = 5;
	hp_close_matrix[4][1][1][2] = 3;
	hp_close_matrix[4][3][1][2] = 3;
	hp_close_matrix[1][4][1][3] = 3;
	hp_close_matrix[2][3][1][3] = 14;
	hp_close_matrix[3][2][1][3] = 13;
	hp_close_matrix[3][4][1][3] = 3;
	hp_close_matrix[4][1][1][3] = 6;
	hp_close_matrix[4][3][1][3] = 6;
	hp_close_matrix[1][4][1][4] = 3;
	hp_close_matrix[2][3][1][4] = 18;
	hp_close_matrix[3][2][1][4] = 21;
	hp_close_matrix[3][4][1][4] = 3;
	hp_close_matrix[4][1][1][4] = 5;
	hp_close_matrix[4][3][1][4] = 5;
	hp_close_matrix[1][4][2][1] = 1;
	hp_close_matrix[2][3][2][1] = 10;
	hp_close_matrix[3][2][2][1] = 11;
	hp_close_matrix[3][4][2][1] = 1;
	hp_close_matrix[4][1][2][1] = 2;
	hp_close_matrix[4][3][2][1] = 2;
	hp_close_matrix[1][4][2][2] = 2;
	hp_close_matrix[2][3][2][2] = 9;
	hp_close_matrix[3][2][2][2] = 7;
	hp_close_matrix[3][4][2][2] = 2;
	hp_close_matrix[4][1][2][2] = 1;
	hp_close_matrix[4][3][2][2] = 1;
	hp_close_matrix[1][4][2][3] = 15;
	hp_close_matrix[2][3][2][3] = 29;
	hp_close_matrix[3][2][2][3] = 24;
	hp_close_matrix[3][4][2][3] = 15;
	hp_close_matrix[4][1][2][3] = 12;
	hp_close_matrix[4][3][2][3] = 17;
	hp_close_matrix[1][4][2][4] = 2;
	hp_close_matrix[2][3][2][4] = 8;
	hp_close_matrix[3][2][2][4] = 5;
	hp_close_matrix[3][4][2][4] = 2;
	hp_close_matrix[1][4][3][1] = 11;
	hp_close_matrix[2][3][3][1] = 22;
	hp_close_matrix[3][2][3][1] = 24;
	hp_close_matrix[3][4][3][1] = 9;
	hp_close_matrix[4][1][3][1] = 14;
	hp_close_matrix[4][3][3][1] = 8;
	hp_close_matrix[1][4][3][2] = 12;
	hp_close_matrix[2][3][3][2] = 20;
	hp_close_matrix[3][2][3][2] = 29;
	hp_close_matrix[3][4][3][2] = 11;
	hp_close_matrix[4][1][3][2] = 12;
	hp_close_matrix[4][3][3][2] = 12;
	hp_close_matrix[1][4][3][3] = 2;
	hp_close_matrix[2][3][3][3] = 16;
	hp_close_matrix[3][2][3][3] = 14;
	hp_close_matrix[3][4][3][3] = 3;
	hp_close_matrix[4][1][3][3] = 7;
	hp_close_matrix[4][3][3][3] = 3;
	hp_close_matrix[1][4][3][4] = -2;
	hp_close_matrix[2][3][3][4] = 11;
	hp_close_matrix[3][2][3][4] = 12;
	hp_close_matrix[4][1][3][4] = 2;
	hp_close_matrix[4][3][3][4] = 7;
	hp_close_matrix[1][4][4][1] = 3;
	hp_close_matrix[2][3][4][1] = 17;
	hp_close_matrix[3][2][4][1] = 19;
	hp_close_matrix[3][4][4][1] = 3;
	hp_close_matrix[4][1][4][1] = 3;
	hp_close_matrix[4][3][4][1] = 6;
	hp_close_matrix[1][4][4][2] = 3;
	hp_close_matrix[2][3][4][2] = 14;
	hp_close_matrix[3][2][4][2] = 10;
	hp_close_matrix[3][4][4][2] = 3;
	hp_close_matrix[4][1][4][2] = 1;
	hp_close_matrix[4][3][4][2] = 1;
	hp_close_matrix[1][4][4][3] = 6;
	hp_close_matrix[2][3][4][3] = 18;
	hp_close_matrix[3][2][4][3] = 22;
	hp_close_matrix[3][4][4][3] = 4;
	hp_close_matrix[4][1][4][3] = 5;
	hp_close_matrix[4][3][4][3] = 6;
	hp_close_matrix[1][4][4][4] = 11;
	hp_close_matrix[2][3][4][4] = 20;
	hp_close_matrix[3][2][4][4] = 15;
	hp_close_matrix[3][4][4][4] = 11;
	hp_close_matrix[4][1][4][4] = 8;
	hp_close_matrix[4][3][4][4] = 8;
	
	
	assignMatrix4(internal_loop_matrix, alfa_size, scoreType(0));
	internal_loop_matrix[1][4][1][1] = -7;
	internal_loop_matrix[3][4][1][1] = -7;
	internal_loop_matrix[4][1][1][1] = -7;
	internal_loop_matrix[4][3][1][1] = -7;
	internal_loop_matrix[1][4][1][2] = -7;
	internal_loop_matrix[3][4][1][2] = -7;
	internal_loop_matrix[4][1][1][2] = -7;
	internal_loop_matrix[4][3][1][2] = -7;
	internal_loop_matrix[1][4][1][3] = 4;
	internal_loop_matrix[2][3][1][3] = 11;
	internal_loop_matrix[3][2][1][3] = 11;
	internal_loop_matrix[3][4][1][3] = 4;
	internal_loop_matrix[4][1][1][3] = 4;
	internal_loop_matrix[4][3][1][3] = 4;
	internal_loop_matrix[1][4][1][4] = -7;
	internal_loop_matrix[3][4][1][4] = -7;
	internal_loop_matrix[4][1][1][4] = -7;
	internal_loop_matrix[4][3][1][4] = -7;
	internal_loop_matrix[1][4][2][1] = -7;
	internal_loop_matrix[3][4][2][1] = -7;
	internal_loop_matrix[4][1][2][1] = -7;
	internal_loop_matrix[4][3][2][1] = -7;
	internal_loop_matrix[1][4][2][2] = -7;
	internal_loop_matrix[3][4][2][2] = -7;
	internal_loop_matrix[4][1][2][2] = -7;
	internal_loop_matrix[4][3][2][2] = -7;
	internal_loop_matrix[1][4][2][3] = -7;
	internal_loop_matrix[3][4][2][3] = -7;
	internal_loop_matrix[4][1][2][3] = -7;
	internal_loop_matrix[4][3][2][3] = -7;
	internal_loop_matrix[1][4][2][4] = -7;
	internal_loop_matrix[3][4][2][4] = -7;
	internal_loop_matrix[4][1][2][4] = -7;
	internal_loop_matrix[4][3][2][4] = -7;
	internal_loop_matrix[1][4][3][1] = 4;
	internal_loop_matrix[2][3][3][1] = 11;
	internal_loop_matrix[3][2][3][1] = 11;
	internal_loop_matrix[3][4][3][1] = 4;
	internal_loop_matrix[4][1][3][1] = 4;
	internal_loop_matrix[4][3][3][1] = 4;
	internal_loop_matrix[1][4][3][2] = -7;
	internal_loop_matrix[3][4][3][2] = -7;
	internal_loop_matrix[4][1][3][2] = -7;
	internal_loop_matrix[4][3][3][2] = -7;
	internal_loop_matrix[1][4][3][3] = -7;
	internal_loop_matrix[3][4][3][3] = -7;
	internal_loop_matrix[4][1][3][3] = -7;
	internal_loop_matrix[4][3][3][3] = -7;
	internal_loop_matrix[1][4][3][4] = -7;
	internal_loop_matrix[3][4][3][4] = -7;
	internal_loop_matrix[4][1][3][4] = -7;
	internal_loop_matrix[4][3][3][4] = -7;
	internal_loop_matrix[1][4][4][1] = -7;
	internal_loop_matrix[3][4][4][1] = -7;
	internal_loop_matrix[4][1][4][1] = -7;
	internal_loop_matrix[4][3][4][1] = -7;
	internal_loop_matrix[1][4][4][2] = -7;
	internal_loop_matrix[3][4][4][2] = -7;
	internal_loop_matrix[4][1][4][2] = -7;
	internal_loop_matrix[4][3][4][2] = -7;
	internal_loop_matrix[1][4][4][3] = -7;
	internal_loop_matrix[3][4][4][3] = -7;
	internal_loop_matrix[4][1][4][3] = -7;
	internal_loop_matrix[4][3][4][3] = -7;
	internal_loop_matrix[2][3][4][4] = 7;
	internal_loop_matrix[3][2][4][4] = 7;
	

	// Setting up the loop length cost tables
	int index=0;
	index =  0; hpLength[index] =   0; bulgeLength[index] =   0; ilLength[index] =   0;
	index =  1; hpLength[index] = -57; bulgeLength[index] = -38; ilLength[index] = -17;
	index =  2; hpLength[index] = -57; bulgeLength[index] = -28; ilLength[index] = -17;
	index =  3; hpLength[index] = -57; bulgeLength[index] = -32; ilLength[index] = -17;
	index =  4; hpLength[index] = -56; bulgeLength[index] = -36; ilLength[index] = -17;
	index =  5; hpLength[index] = -56; bulgeLength[index] = -40; ilLength[index] = -18;
	index =  6; hpLength[index] = -54; bulgeLength[index] = -44; ilLength[index] = -20;
	index =  7; hpLength[index] = -59; bulgeLength[index] = -46; ilLength[index] = -22;
	index =  8; hpLength[index] = -56; bulgeLength[index] = -47; ilLength[index] = -23;
	index =  9; hpLength[index] = -64; bulgeLength[index] = -48; ilLength[index] = -24;
	index = 10; hpLength[index] = -65; bulgeLength[index] = -49; ilLength[index] = -25;
	index = 11; hpLength[index] = -66; bulgeLength[index] = -50; ilLength[index] = -26;
	index = 12; hpLength[index] = -67; bulgeLength[index] = -51; ilLength[index] = -27;
	index = 13; hpLength[index] = -68; bulgeLength[index] = -52; ilLength[index] = -28;
	index = 14; hpLength[index] = -69; bulgeLength[index] = -53; ilLength[index] = -29;
	index = 15; hpLength[index] = -69; bulgeLength[index] = -54; ilLength[index] = -30;
	index = 16; hpLength[index] = -70; bulgeLength[index] = -54; ilLength[index] = -30;
	index = 17; hpLength[index] = -71; bulgeLength[index] = -55; ilLength[index] = -31;
	index = 18; hpLength[index] = -71; bulgeLength[index] = -55; ilLength[index] = -31;
	index = 19; hpLength[index] = -72; bulgeLength[index] = -56; ilLength[index] = -32;
	index = 20; hpLength[index] = -72; bulgeLength[index] = -57; ilLength[index] = -33;
	index = 21; hpLength[index] = -73; bulgeLength[index] = -57; ilLength[index] = -33;
	index = 22; hpLength[index] = -73; bulgeLength[index] = -58; ilLength[index] = -34;
	index = 23; hpLength[index] = -74; bulgeLength[index] = -58; ilLength[index] = -34;
	index = 24; hpLength[index] = -74; bulgeLength[index] = -58; ilLength[index] = -34;
	index = 25; hpLength[index] = -75; bulgeLength[index] = -59; ilLength[index] = -35;
	index = 26; hpLength[index] = -75; bulgeLength[index] = -59; ilLength[index] = -35;
	index = 27; hpLength[index] = -75; bulgeLength[index] = -60; ilLength[index] = -36;
	index = 28; hpLength[index] = -76; bulgeLength[index] = -60; ilLength[index] = -36;
	index = 29; hpLength[index] = -76; bulgeLength[index] = -60; ilLength[index] = -36;
	index = 30; hpLength[index] = -77; bulgeLength[index] = -61; ilLength[index] = -37;

}

//******************************************************
// These function templates makes or deletes
// 2 or 4 dimensional matrixes
// These should be changed to use stdlib or/and moved to their own class

template<class local_scoreType> inline void scorematrix::makeMatrix2(local_scoreType**& matrix, const int size) {
	matrix = new local_scoreType*[size];
	for(int i=0; i<size; i++) {
		matrix[i] = new local_scoreType[size];
	}
}

template<class local_scoreType> inline void scorematrix::deleteMatrix2(local_scoreType**& matrix, const int size) {
	for(int i=0; i<size; i++) {
		delete[] matrix[i];
	}
	delete[] matrix;
}

template<class local_scoreType> inline void scorematrix::copyMatrix2(local_scoreType**& matrix, local_scoreType** const& old_matrix, const int size) {
	for(int i=0; i<size; i++) {
		for(int j=0; j < size; j++) {
			matrix[i][j] = old_matrix[i][j];
		}
	}
}

template<class local_scoreType> 
inline void scorematrix::assignMatrix2(local_scoreType**& matrix, 
                                       const int size, 
													const local_scoreType value) {
	for(int i=0; i<size; i++) {
		for(int j=0; j < size; j++) {
			matrix[i][j] = value;
		}
	}
}

template<class local_scoreType>
inline void scorematrix::assignArray(local_scoreType*& matrix, 
												 const local_scoreType value, const int size) {
	for(int i=0; i<size; i++) {matrix[i] = value;}
}

template<class local_scoreType> inline void scorematrix::makeMatrix4(local_scoreType****& matrix, const int size) {
	matrix = new local_scoreType***[size];
	for(int i=0; i<size; i++) {
		matrix[i] = new local_scoreType**[size];
		for(int j=0; j < size; j++) {
			matrix[i][j] = new local_scoreType*[size];
			for(int k=0; k<size; k++) {
				matrix[i][j][k] = new local_scoreType[size];
			}
		}
	}
}

template<class local_scoreType>
inline void scorematrix::assignMatrix4(local_scoreType****& matrix, 
                                       const int size, const local_scoreType value) {
	for(int i=0; i<size; i++) {
		for(int j=0; j < size; j++) {
			for(int k=0; k<size; k++) {
				for(int l=0; l<size; l++) {
					matrix[i][j][k][l] = value;
				}
			}
		}
	}
}

template<class local_scoreType> inline void scorematrix::copyMatrix4(local_scoreType****& matrix, local_scoreType**** const& old_matrix, const int size) {
	for(int i=0; i<size; i++) {
		for(int j=0; j < size; j++) {
			for(int k=0; k<size; k++) {
				for(int l=0; l<size; l++) {
					matrix[i][j][k][l] = old_matrix[i][j][k][l];
				}
			}
		}
	}
}


template<class local_scoreType> inline void scorematrix::deleteMatrix4(local_scoreType****& matrix, const int size) {
	for(int i=0; i<size; i++) {
		for(int j=0; j < size; j++) {
			for(int k=0; k<size; k++) {
				delete[] matrix[i][j][k];
			}
			delete[] matrix[i][j];
		}
		delete[] matrix[i];
	}
	delete[] matrix;
}


#endif /* SCOREMATRIX */
