#ifndef SHORTTEMMEMORY
#define SHORTTEMMEMORY

#include "arguments.cxx"
#include "helper.cxx"
#include "scorematrix.cxx"
#include "exception.cxx"

#include "foldalign.h"

/******************************************************************************
*                                                                             *
*   Copyright 2004 - 2007 Jakob Hull Havgaard, hull@genome.ku.dk              *
*                                                                             *
*   This file is part of Foldalign                                            *
*                                                                             *
*   Foldalign is free software; you can redistribute it and/or modify         *
*   it under the terms of the GNU General Public License as published by      *
*   the Free Software Foundation; either version 2 of the License, or         *
*   (at your option) any later version.                                       *
*                                                                             *
*   Foldalign is distributed in the hope that it will be useful,              *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*   GNU General Public License for more details.                              *
*                                                                             *
*   You should have received a copy of the GNU General Public License         *
*   along with Foldalign; if not, write to the Free Software                  *
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA *
*                                                                             *
******************************************************************************/

#define TEMP_DEF_SHORTTERMMEMORY template< class cell >
#define TEMP_SPES_SHORTTERMMEMORY cell

TEMP_DEF_SHORTTERMMEMORY
class shortTermMemory {
public:

	inline shortTermMemory(const positionType& i_dim, 
	                       const positionType& k_dim,
								  const lengthType& Wi_dim,
	                       arguments*& argu,
								  scorematrix& sm, 
								  positionType k_off = 0, 
								  const lengthType Wi_minimum = 0);
	
	inline cell* getPos(positionType i, positionType k,
	                    lengthType Wi, lengthType Wk) {
		calcpos2(i, k, Wi, Wk);
		if (matrix[i][k][Wi] == 0) {return 0;}
		return matrix[i][k][Wi][Wk];
	}
	
	inline cell* putPos(positionType i, positionType k, 
	                   lengthType Wi, lengthType Wk,
							 const scoreType sc);

	inline lengthType getWimax(positionType i, positionType k)
		{(this->*p2calcpos)(i, k); return wimax[i][k];};

	inline void transfer();

	inline void setPositions(const positionType i_pos, 
	                         const positionType k_pos) {
		i_position = i_pos;
		k_position = k_pos;
	};
	inline void set_I_Position(const positionType pos) {i_position = pos;};
	inline void set_K_Position(const positionType pos) {k_position = pos;};

	inline long getSize() const {
		return char_size*(cell_count * cell_size + mat3d_size +
								matWk_cell_count * ptr_size)/mem_scale;
	}
	
	inline ~shortTermMemory();

private:

	const positionType i_dimension;

	// The dimension along the k-chunk
	const positionType k_dimension;
	
	// The window size along sequence I dimension
	const lengthType Wi_dimension;

	// The window size along sequence K dimension
	// This is not const since it is depended on the Wi. In the extremes of Wi
	// it is not const.
	lengthType Wk_dimension;
	lengthType Wk_zero;

	arguments* arg;

	// The matrixs position on the I-sequence
	positionType i_position;
	
	positionType k_position;

	// The data matrix
	cell***** matrix;
	
	// The max size matrix. The max_length along the 3rd dimension of matrix.
	lengthType** wimax;

	// The length of the sequences
	const positionType seq_length1;
	const positionType seq_length2;

	lengthType wimax_init;
	

	inline void calcPos_local(positionType& i, positionType& k) const {
		i = i_position - i;
		k = k_position - k;
	};

	inline void calcPos_global(positionType& i, positionType& k) const {
		k = k - i + k_offset;
		i = i_position - i;
	};

	inline void calcpos2(positionType& i, positionType& k,
	                     lengthType& Wi, lengthType& Wk) const {
		(this->*p2calcpos)(i, k);
		Wk = Wk - Wi + Wk_zero;
		Wi -= Wi_min;
	};
	
	const positionType k_offset;

	const lengthType Wi_min;
	
	void (shortTermMemory::* const p2calcpos)(positionType&, positionType&) const;

	const long cell_size;
	const long ptr_size;
	const long mat3d_size;
	long cell_count;
	long matWk_cell_count;

	inline void newCell (const positionType i, const positionType k,
								const lengthType Wi, const lengthType Wk) {
		matrix[i][k][Wi][Wk] = new cell();
		if (Wi > wimax[i][k]) {wimax[i][k] = Wi;}
		cell_count++;
	}


};

TEMP_DEF_SHORTTERMMEMORY
inline shortTermMemory< TEMP_SPES_SHORTTERMMEMORY >::shortTermMemory
	(const positionType& i_dim,
	 const positionType& k_dim,
	 const lengthType& Wi_dim,
	 arguments*& argu, 
	 scorematrix& sm,
	 positionType k_off,
	 const lengthType Wi_minimum) : 
i_dimension(i_dim),
k_dimension(k_dim),
Wi_dimension(Wi_dim - Wi_minimum),
Wk_dimension(lengthType(2*argu->ltOpt("-max_diff")+3)),
Wk_zero(lengthType(argu->ltOpt("-max_diff")+1)), 
arg(argu), 
i_position(arg->ptOpt("-j")),
k_position(arg->ptOpt("-l")),
seq_length1(arg->ptOpt("lenSeq1")), 
seq_length2(arg->ptOpt("lenSeq2")), 
wimax_init(lengthType(1)),
k_offset(k_off),
Wi_min(Wi_minimum),
p2calcpos( arg->boolOpt("-global") || arg->boolOpt("realigning") || arg->boolOpt("mblrealign") ? &shortTermMemory::calcPos_global : &shortTermMemory::calcPos_local),
cell_size(sizeof(cell)),
ptr_size(sizeof(cell*)),
mat3d_size( i_dimension * k_dimension * ((Wi_dimension +1)* ptr_size + sizeof(lengthType)))
{

	if (wimax_init >= Wi_dimension) {wimax_init = Wi_dimension -1;}

	if (Wk_dimension > seq_length1 + seq_length2 +3) {
		Wk_dimension = seq_length1 + seq_length2 +3;
		Wk_zero = lengthType(seq_length1 +1);
	}

	// The out of memory error handling could probally be improved.
	std::string error = "Could not allocate the shortTermMemory memory stack. Most likely cause: Out of memory.";
	try {

		// The first dimension of the matrix is only two. The current start and
		// the next

		error = "Could not allocate shortTerm memory. Most likely cause: Out of memory";

		matrix = new cell****[i_dimension];
		if (matrix == 0) {throw exception(error, false);}

		wimax = new lengthType*[i_dimension];
		if (wimax == 0) {delete[] matrix; throw exception(error, false);}

		for(positionType i = 0; i < i_dimension; i++) {

			// The second dimension is the chunk_size
			matrix[i] = new cell***[k_dimension];
			wimax[i] = new lengthType[k_dimension];
			if (matrix[i] == 0 || wimax[i] == 0) {throw exception(error, false);}

			for( positionType k = 0; k < k_dimension; k++ ) {

				// There has to be room for lambda positions in the third dimension
				// plus the 0 value.
				matrix[i][k] = new cell**[Wi_dimension];
				if (matrix[i][k] == 0) {throw exception(error, false);}

				// Set the value of the highest Wi seen so far to the init value.
				wimax[i][k] = wimax_init;

				for(lengthType Wi = 0; Wi < Wi_dimension; Wi++ ) {

					matrix[i][k][Wi] = 0;
				}

			}
		}
	}
	catch ( exception ) {throw;}
	catch ( ... ) {
		throw exception(error, false);
	}

	cell_count = 0;
	matWk_cell_count = 0;
	// I have seen some slip-shod backwater burgs, but this place takes the cake.
}

TEMP_DEF_SHORTTERMMEMORY
inline cell* shortTermMemory< TEMP_SPES_SHORTTERMMEMORY >::putPos(
positionType i, positionType k, lengthType Wi, 
lengthType Wk, const scoreType sc)
{

	calcpos2(i, k, Wi, Wk);

	if (matrix[i][k][Wi] == 0) {

		matrix[i][k][Wi] = new cell*[Wk_dimension];

		for ( lengthType tWk = 0; tWk < Wk_dimension; tWk++ ) {
			// Initially every slot is empty
			matrix[i][k][Wi][tWk] = 0;
		}
		matWk_cell_count += Wk_dimension;
		newCell(i, k, Wi, Wk);
	}
	else if (matrix[i][k][Wi][Wk] == 0) { newCell(i, k, Wi, Wk); }

	return matrix[i][k][Wi][Wk];
}

TEMP_DEF_SHORTTERMMEMORY
inline void shortTermMemory< TEMP_SPES_SHORTTERMMEMORY >::transfer() {

	// Switch the two I-positions
	helper::swap(matrix[0], matrix[1]);
	helper::swap(wimax[0], wimax[1]);

	// Delete all the old i==1 cells
	const positionType i = 1;
	for (positionType k=0; k < k_dimension; k++) {
		lengthType max = wimax[i][k];
		if (max == 0) {continue;}
		for(lengthType Wi = 0; Wi <= max; Wi++) {
			if (matrix[i][k][Wi] != 0) {
				for(lengthType Wk = 0; Wk < Wk_dimension; Wk++) {
					if (matrix[i][k][Wi][Wk] != 0) {
						delete matrix[i][k][Wi][Wk];
						matrix[i][k][Wi][Wk] = 0;
						cell_count--;
					}
				}
				delete[] matrix[i][k][Wi];
				matrix[i][k][Wi] = 0;
				matWk_cell_count -= Wk_dimension;
			}
		}
		wimax[i][k] = wimax_init;
	}
				
	i_position--;
	
}

TEMP_DEF_SHORTTERMMEMORY
inline shortTermMemory< TEMP_SPES_SHORTTERMMEMORY >::~shortTermMemory() {

	for(positionType i = 0; i < i_dimension; i++) {
		for( positionType k = 0; k < k_dimension; k++ ) {
			for( lengthType Wi = 0; Wi < Wi_dimension; Wi++ ) {
				if (matrix[i][k][Wi] != 0) {
					for ( lengthType Wk = 0; Wk < Wk_dimension; Wk++ ) {
						if ( matrix[i][k][Wi][Wk] != 0 ) {delete matrix[i][k][Wi][Wk];}
					}
					delete[] matrix[i][k][Wi];
				}
			}
			delete[] matrix[i][k];
		}
		delete[] matrix[i];
		delete[] wimax[i];
	}
	delete[] matrix;
	delete[] wimax;
}
#endif /* SHORTTEMMEMORY */
