#ifndef TESTREADFILE
#define TESTREADFILE

#include "test.cxx"
#include "../../src/readfile.cxx"
#include "../../src/exception.cxx"

class testReadfile {
public:
	testReadfile(int& passed, int& ran, int& expected, std::string& messages) 
		: tester(21, "readfile test"), filename("test/auxfiles/readfile.cxx.aux") {
	
		testStdin();
				
		testNamedFile();
				
		testGet_line();
		testGet_line_failEmpty();
		testSkip_to_empty();
		
		tester.printResult();

		passed += tester.getTestsPassed();
		ran += tester.getTestsRan();
		expected += tester.getTestsExpected();
	}
private:

	test tester;
	const std::string filename;

	//! \brief Check that the name is set correct when stdin is used
	void testStdin();

	//! \brief Test that the named files can be open and that an error is given
	//! if the file can not be opened.
	void testNamedFile();

	//! \brief Test the get_line member function
	void testGet_line();

	//! \brief Test the get_line_failEmpty member function
	void testGet_line_failEmpty();

	//! \brief Test the skip_to_empty member function
	void testSkip_to_empty();

	//! \brief Opens the named file (filename) and returns a pointer to the
	//! readfile object. The pointer must be destroyed with delete
	readfile* openFile();
};

void testReadfile::testStdin() {

	readfile stdin;
	
	tester.equal(stdin.name(), std::string("<STDIN>"));
	
}

void testReadfile::testNamedFile() {

	tester.constructorThrowsException<readfile, exception, std::string>(
		std::string("hopefullydoesnotexists"));
	
	readfile* file = openFile();
	tester.equal(filename, file->name());
	delete file;
}

void testReadfile::testGet_line() {

	readfile* file = openFile();

	std::string line;
	tester.isTrue(file->get_line(line));
	tester.equal(line, std::string("1"));
	tester.isTrue(file->get_line(line));
	tester.equal(line, std::string("2"));
	tester.isFalse(file->get_line(line));
	tester.equal(line, std::string(""));

	delete file;
}	

void testReadfile::testGet_line_failEmpty() {

	readfile* file = openFile();

	std::string line;
	tester.isTrue(file->get_line_failEmpty(line));
	tester.equal(line, std::string("1"));

	tester.isFalse(file->get_line_failEmpty(line));
	tester.equal(line, std::string(""));

	tester.isTrue(file->get_line_failEmpty(line));
	tester.equal(line, std::string("2"));

	tester.isFalse(file->get_line_failEmpty(line));
	tester.equal(line, std::string(""));

	delete file;
}	

void testReadfile::testSkip_to_empty() {

	readfile* file = openFile();

	std::string line;
	
	file->skip_to_empty();
	tester.isTrue(file->get_line(line));
	tester.equal(line, std::string("2"));

	file->skip_to_empty();
	tester.isFalse(file->get_line(line));
	tester.equal(line, std::string(""));

	delete file;
}	

readfile* testReadfile::openFile() {

	readfile* file;

	try {
		file = new readfile(filename);
	}
	catch (...) {
		std::string error = "Could not open file: " + filename;
		tester.failedTest(error);
		throw;
	}

	return file;
}
#endif
