package EST2ncRNA::MysqlInterface;

use strict;
use warnings;
use DBI;

use EST2ncRNA::SequenceInterface qw(get_subject_subsequences_of_one_chromosome);
require Exporter;

our @ISA = qw(Exporter);  # inherits from Exporter

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

our @EXPORT_OK = qw(prepare_db
		    );

our @EXPORT = qw();

our $VERSION = '0.01';


# constructor
sub new {
    my $class = shift @_;
    my ($datasource, $username, $password) = @_;

    my $self = {
	_datasource => $datasource,
	_username   => $username,
	_password   => $password,
	_dbh        => undef
    };
    bless $self, $class;

    return $self;
}


# accessor method for MysqlInterface datasource
sub datasource {
    my ($self, $datasource) = @_;
    $self->{_datasource} = $datasource if defined($datasource);
    return $self->{_datasource};
}


# accessor method for MysqlInterface username
sub username {
    my ($self, $username) = @_;
    $self->{_username} = $username if defined($username);
    return $self->{_username};
}


# accessor method for MysqlInterface password
sub password {
    my ($self, $password) = @_;
    $self->{_password} = $password if defined($password);
    return $self->{_password};
}


# accessor method for MysqlInterface dbh
sub dbh {
    my ($self) = @_;
    # establish database connection
    if( !defined($self->{_dbh}) || !$self->{_dbh}->ping==1 ) {
	$self->connect_database;
    }
    return $self->{_dbh};
}


# connect the database and set the MysqlInterface dbh
sub connect_database {
    my ($self) = @_;

    $self->{_dbh} = DBI->connect($self->datasource, $self->username, $self->password) or die("Can't connect to $self->datasource: $DBI::errstr");
    $self->{_dbh}->{RaiseError} = 1;   # DBI will automatically die if any DBI method call fails
    $self->{_dbh}->{AutoCommit} = 0;   # enable transaction, if possible

    print "Connection with the database ", $self->datasource, " is established.\n";
}


# delete all entries of all existing tables in the database
# and creates the following tables if not already exist
#   assest        ...  saves the assembled EST data (pipieline input)
#   candncrna     ...  holds during the whole pipeline the actual candidates of ncRNA
#   codingrna     ...  saves protein coding RNAs (including ORFs)
#   knownncrna    ...  saves known ncRNAs
#   origspecest   ...  saves ESTs being not conserved in the closed related organism
#   blastn        ...  saves blastn results
#   conservedest  ...  saves ESTs being conserved in the closed related organism
#   estcoverage   ...  saves all aligned sequences of other mammalians to the ESTs
#   rnazwindows   ...  saves all windows with locally conserved secondary structures of the candncrna's
#   microrna      ...  saves EST locations predicted as microRNAs by RNAmicro
#
# input is the database handler
sub prepare_db {
    my ($self) = @_;
    my ($newtable, $droptable);

    # create new tables if not already exists
    $newtable = q{create table if not exists assest (ass_id int not null primary key, ass_name varchar(100), ass_sequence longtext)};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists alignversion (align_id int not null, align_organism varchar(20), primary key(align_id, align_organism))};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists candncrna (candnc_id int not null, candnc_start int, candnc_end int, candnc_pvalue float, candnc_zscore float, candnc_sci float, candnc_locus int, candnc_annotation varchar(200), candnc_version int, candnc_blastnid int, primary key(candnc_id, candnc_start), foreign key(candnc_version) references alignversion(align_id) on delete set null)};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists candcisreg (candcis_id int not null, candcis_start int, candcis_end int, candcis_pvalue float, candcis_zscore float, candcis_sci float, candcis_locus int, candcis_annotation varchar(200), candcis_utr varchar(50), candcis_version int, candcis_blastnid int, primary key(candcis_id, candcis_start), foreign key(candcis_version) references alignversion(align_id) on delete set null)};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists codingrna (coding_id int not null primary key, coding_start int, coding_end int, coding_mlevel int, coding_protein varchar(100))};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists annotatedncrna (anno_id int not null, anno_start int, anno_end int, anno_subject_id varchar(100), anno_evalue float, anno_identity float, anno_subject_coverage float, anno_description varchar(100), primary key(anno_id, anno_subject_id))};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists knownncrna (known_id int not null, known_start int not null, known_end int not null, known_score float, known_family varchar(7), known_basepairs int, known_status varchar(5), known_annotation varchar(200), primary key(known_id, known_start, known_end))};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists origspecest (orig_id int not null primary key)};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists blastnannotation (blastnanno_organism varchar(50) not null, blastnanno_assembly varchar(100), blastnanno_parameter varchar(20), blastnanno_blastnid_from int, blastnanno_blastnid_to int, primary key(blastnanno_organism, blastnanno_assembly, blastnanno_parameter))};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists blastn (blastn_blastnid int not null primary key, blastn_query_id int not null, blastn_subject_id varchar(50) not null, blastn_bit_score float, blastn_score int, blastn_align_length int, blastn_identity int, blastn_evalue float, blastn_query_start int, blastn_query_stop int, blastn_subject_start int, blastn_subject_stop int, blastn_query_gaps int, blastn_subject_gaps int)};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists candidate (cand_id int not null, cand_start int, cand_end int, cand_blastnid int, primary key(cand_id, cand_start))};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists estcoverage (estcov_source_id int not null, estcov_source_start int, estcov_subject_name varchar(50), estcov_subject_chr varchar(10), estcov_subject_start int, estcov_subject_end int, estcov_subject_sequence longtext, estcov_align_type varchar(20), estcov_blastnid int, primary key(estcov_blastnid, estcov_subject_name, estcov_subject_chr, estcov_align_type))};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists rnazwindows (rnaz_locus int, rnaz_window int not null primary key, rnaz_start int, rnaz_end int, rnaz_strand varchar(1), rnaz_pvalue float, rnaz_zscore float, rnaz_sci float, rnaz_combPerPair float, rnaz_align_identity float, rnaz_sequence longtext, rnaz_cons_structure longtext, rnaz_align_organisms int, rnaz_align_organisms_list longtext, foreign key(rnaz_align_organisms) references alignversion(align_id) on delete set null)};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists microrna (micro_id int, micro_start int, micro_end int, micro_strand varchar(1), micro_pvalue float, primary key(micro_id, micro_start, micro_strand))};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists snorna (sno_id int not null, sno_strand varchar(1), sno_pvalue float, primary key(sno_id, sno_strand))};
    $self->dbh->do($newtable);
    $newtable = q{create table if not exists ncbiannotation (ncbi_id int not null, ncbi_start int, ncbi_query_start int, ncbi_query_end int, ncbi_subject_id varchar(100), ncbi_evalue float, ncbi_identity float, ncbi_est_coverage float, ncbi_description varchar(100), primary key(ncbi_id, ncbi_start, ncbi_subject_id))};
    $self->dbh->do($newtable);

    # delete all entries of all existing tables
    $droptable = q{delete from assest};
    $self->dbh->do($droptable);
    $droptable = q{delete from alignversion};
    $self->dbh->do($droptable);
    $droptable = q{delete from candncrna};
    $self->dbh->do($droptable);
    $droptable = q{delete from candcisreg};
    $self->dbh->do($droptable);
    $droptable = q{delete from codingrna};
    $self->dbh->do($droptable);
    $droptable = q{delete from annotatedncrna};
    $self->dbh->do($droptable);
    $droptable = q{delete from knownncrna};
    $self->dbh->do($droptable);
    $droptable = q{delete from origspecest};
    $self->dbh->do($droptable);
    $droptable = q{delete from blastnannotation};
    $self->dbh->do($droptable);
    $droptable = q{delete from blastn};
    $self->dbh->do($droptable);
    $droptable = q{delete from candidate};
    $self->dbh->do($droptable);
    $droptable = q{delete from estcoverage};
    $self->dbh->do($droptable);
    $droptable = q{delete from rnazwindows};
    $self->dbh->do($droptable);
    $droptable = q{delete from microrna};
    $self->dbh->do($droptable);
    $droptable = q{delete from snorna};
    $self->dbh->do($droptable);
    $droptable = q{delete from ncbiannotation};
    $self->dbh->do($droptable);

    $self->dbh->commit;
}


# stores the input assembled EST data in a table names 'assest'
# and their indices in the 'candncrna'-table
#
# input is the assembled EST data in FASTA format
# assisted are two formats: '><unique id> <name> ...\n<sequence>' and '><name> ...\n<sequence>'; in the second format a new unique id is created automatically
# returns the number of inserted ass ests
sub insert_assest {
    my $self = shift;
    my $file = shift;
    my $tmpfile = $file;
    my ($id, $name, $seq, $newassest, $newcandncrna, $sth2, $sth3, $line, $form);
    my ($searchid, $sth1, @count);
    my $nr = 0;

    # extracts the data from the fasta-file and writes them in the 'assest'-table
    if ($file =~ /\.gz$/) {
        open FH, "gunzip -c $file |" || die("Can not open the file!\n");
    }
    else {
        open FH, $file || die("Can not open the file!\n");
    }

    $newassest = q{insert into assest values (?,?,?)};
    $sth2 = $self->dbh->prepare($newassest);
    $newcandncrna = q{insert into candidate (cand_id, cand_start, cand_end) values (?, 1, ?)};
    $sth3 = $self->dbh->prepare($newcandncrna);


    while( $line = <FH> ) {
	
        if( $line =~ /^>/ ) {
	    $nr++;

	    if( defined $seq ) {
                $sth2->execute($id,$name,$seq);
                $sth3->execute($id,length($seq));
                $seq = "";
            }

	    if( $line=~/^>(\d+)\s+(\S+)/ ) {
		$id = $1;
		$name = $2;
	    }
	    else {
                $id = $nr+1000000000;
	    	$line=~/^>(\S+)/;
		$name = $1;
	    }
        }
        else {
            $seq = $seq.$1 if $line =~ /^(\D+)\s/;
        }
    }
    # last entry
    $sth2->execute($id,$name,$seq);
    $sth3->execute($id,length($seq));

    close FH;

    $self->dbh->commit;

    return $nr;
}


# deletes from the candncrna-table the small ests with a length shorter as the input (recommended is 60nts)
#
# input is the unusable length
# returns the number of small ests
sub delete_small_ests_from_candncrna {
    my $self = shift;
    my $length = shift;
    my ($query, $sth1, $sth2, $id);
    my $nr = 0;

    # searchs the ids of ests being shorter as $length and binding them to $id
    $query = q{select cand_id from candidate inner join assest where cand_id=ass_id and length(ass_sequence)<?};
    $sth1 = $self->dbh->prepare($query);
    $sth1->execute($length);
    $sth1->bind_columns(\$id);

    # deletes the small ests from the candidate table
    $query = q{delete from candidate where cand_id=?};
    $sth2 = $self->dbh->prepare($query);
    while( $sth1->fetch ) {
        $sth2->execute($id);
        $nr++;
    }
    $self->dbh->commit;

    return $nr;
}


# stores all blastx results in the codingrna-table with their M-level
# and predicts EST hits with M-level 0 to 4 as protein coding RNAs (deletes them from the candncrna-table)
#
# input are a reference to an array with all available blastx-result files
# returns the number of coding RNAs
sub insert_codingrna {
    my $self = shift;
    my $blastxres_ref = shift;

    my (@line, @el, @count, $level);
    my ($getassest, $getcodingrna, $newcodingrna, $deletecand, $deletecoding, $sth0, $sth1, $sth2, $sth3, $sth4);

    # prepare db statements
    $getassest = q{select * from assest where ass_id=?};
    $sth0 = $self->dbh->prepare($getassest);
    $getcodingrna = q{select * from codingrna where coding_id=?};
    $sth1 = $self->dbh->prepare($getcodingrna);
    $newcodingrna = q{insert into codingrna values (?,NULL,NULL,?,?)};
    $sth2 = $self->dbh->prepare($newcodingrna);
    $deletecand = q{delete from candidate where cand_id=?};
    $sth3 = $self->dbh->prepare($deletecand);
    $deletecoding = q{delete from codingrna where coding_id=?};
    $sth4 = $self->dbh->prepare($deletecoding);

    # read BLASTX results and write them in the db
    foreach( @$blastxres_ref ) {
        @line = `zcat $_ | grep -w M[0-6]`;

        foreach( @line ) {
            @el = split " ", $_;
	    ( $level ) = $el[1] =~ /M(\d)/;
	    
	    # control if coding RNA is part of input EST data
	    $sth0->execute($el[2]);
	    @count = $sth0->fetchrow_array;
	    next unless $count[0];

	    # if coding RNA is already stored then delete it if the actual M-level is lower
	    $sth1->execute($el[2]);
	    @count = $sth1->fetchrow_array;
	    if( $count[0] ) {
		next if $count[1]<=$level;
		$sth4->execute($el[2]);
	    }
	    
	    # store new coding RNA and delete it from 'candncrna'-table
	    $sth2->execute($el[2], $level, $el[3]);
	    $sth3->execute($el[2]) if $level<5;
        }
    }

    # get number of protein coding RNAs (return value)
    $getcodingrna = q{select count(*) from codingrna where coding_mlevel<5};
    $sth1 = $self->dbh->prepare($getcodingrna);
    $sth1->execute();
    @count = $sth1->fetchrow_array;

    $self->dbh->commit;

    return $count[0];
}


# creates a gzipped sequence fasta file of the sequences of the input statement
#
# input are the mysql-statement (returns four arguments: id, start, end, sequence),
#           how many sequences in one file
#           a boolean if subsequences should be extracted ("1", otherwise "0") and poly-A tails should be cutted ("2")
#           the output fasta-file (basic name + index)
#	    a boolean if sequence name should be sequence id and start seperated by ":" (optional)
# returns the array including all fasta file names
sub get_statement_fasta {
    my $self = shift;
    my $std = shift;
    my $length = shift;
    my $subseq = shift;
    my $file = shift;
    my $seqname = shift;
    my ($sth, $id, $start, $stop, $seq, $s, $e, $rate, $revseq, @fastafiles);
    my $nr = 0;

    $sth = $self->dbh->prepare($std);
    $sth->execute();
    $sth->bind_columns(\$id, \$start, \$stop, \$seq);

    # create sequence fasta file
    print "Create sequence fasta file.\n";
    my $i = 0;
    my $j = 0;
    push @fastafiles, "$file$j.fa.gz";
    open OUT, "| gzip > $file$j.fa.gz" || die("Can not open the file!\n");

    while( $sth->fetch ) {
	if( $i==$length ) {
		close OUT;
		$i = 0;
		$j++;
		push @fastafiles, "$file$j.fa.gz";
		open OUT, "| gzip > $file$j.fa.gz" || die("Can not open the file!\n");
	}
	$i++;

	#get subsequence
	$seq = $self->get_assest_subsequence($id, $start, $stop) if $subseq==1;

	#remove poly-A tail (as a sequence of As or Ts in the beginning or ending of the sequence if the length greater 9nts)
	if( $subseq == 2 ) {
		#remove at 5'-end poly-As or poly-Ts with length>10nt and (A|T)-rate>80%
		$seq =~ /([Aa]{10,}|[Tt]{10,})/;
		if( defined $1 ) {
			$s=index($seq,$1);
			$e=$s+length($1);
			$rate=length($1)*100/$e;
			$seq=substr($seq, $e, length($seq)) if $rate>80;
		}
		#remove at 3'-end poly-As or poly-Ts with length>10nt and (A|T)-rate>80%
		$revseq = reverse($seq);
		$revseq =~ /([Aa]{10,}|[Tt]{10,})/;
		if( defined $1 ) {
			$s=index($revseq,$1);
			$e=$s+length($1);
			$rate=length($1)*100/$e;
			$seq=substr($seq, 0, length($seq)-$e) if $rate>80;
		}
	}

        #create fasta file
	$id = ( defined $seqname ) ? "$id:$start" : $id;
	print OUT ">$id\n$seq\n";
        $nr++;
    }

    close OUT;

    return \@fastafiles;
}


# create sequence fasta file (.gz) including all candidate ncRNAs from the mySQL-db
#
# returns the array including all fasta file names
sub get_candncrna_fasta {
    my $self = shift;
    my $file = shift;
    my $length = shift;
    my ($query, $rfastafiles);

    # fetch the candidate ncRNAs
    $query = q{select assest.ass_id, candidate.cand_start, candidate.cand_end, assest.ass_sequence from assest inner join candidate where cand_id=ass_id};

    # creates the fasta-file of the total sequences
    $rfastafiles = $self->get_statement_fasta($query, $length, 0, $file);
    
    return $rfastafiles;
}


# create sequence fasta file (.gz) including all candidate ncRNAs from the mySQL-db whereby the poly-A tails of the ESTs are deleted
#
# returns the number of candidate ncRNAs
sub get_candncrna_fasta_without_polyA {
    my $self = shift;
    my $file = shift;
    my ($query, $nr);

    my $LENGTH = 1000000000;

    # fetch the candidate ncRNAs
    $query = q{select assest.ass_id, candidate.cand_start, candidate.cand_end, assest.ass_sequence from assest inner join candidate where cand_id=ass_id};

    # creates the fasta-file of the total sequences
    $nr = $self->get_statement_fasta($query, $LENGTH, 2, $file);

    return $nr;
}


# for all closed organism chromosomes in %chr creates a sequence fasta file with candidate ncRNAs which hit it
#
# returns a reference to an array with all query files
sub get_candncrna_fasta_from_blasthits {
    my $self = shift;
    my $refchr = shift;
    my $blasttable = shift;
    my $SUBDIR = shift;
    my $UID = shift;
    my ($chrfile, $chr, $chrnr, $QUERYFILE, @advquery, @line, %idhash, $query, $sth, $id, $seq);

    foreach $chr ( keys %$refchr ) {
	# create file names
#	$chrfile =~ /chr(\w+)\./;
#	$chr = "chr$1";
	$chrfile = $$refchr{$chr};
	$chr =~ /(chr|scaffold)(\w+)/;
	$chrnr = $2;
        $QUERYFILE = "advblastquery$UID.".$chr.".fa.gz";
	push @advquery, $QUERYFILE;

	# get all chromosome related ESTs (use a hash to get a distinct list)
        open IN, "zcat $blasttable |" || die("Can not open the file!\n");
	<IN>;
        foreach( <IN> ) {
            @line = split " ";
            $idhash{$line[0]} = 1 if $line[1] =~ /(chr|scaffold)?$chrnr/; #index($line[1],"$chr")>=0;
        }
        close IN;

	# create advanced blast query file in fasta format
	print "Create sequence fasta file of all standard aligned candidate ESTs on $chr.\n";
        $query = qq{select ass_id, ass_sequence from assest, candidate where ass_id=cand_id group by ass_id};
	#$query = qq{select ass_id, ass_sequence from assest};
        $sth = $self->dbh->prepare($query);
        $sth->execute();
        $sth->bind_columns(\$id, \$seq);

        open OUT, "| gzip > $SUBDIR/$QUERYFILE" || die("Can not open the file!\n");
        while( $sth->fetch ) {
            #create fasta file entry for chromosome related ESTs
            print OUT ">$id\n$seq\n" if defined $idhash{$id};
        }
        close OUT;
    }

    return \@advquery;
}

sub insert_knownncrna {
	my $self = shift;
	my $knownncrna = shift;

	my ($searchid, $newknownncrna, $deletecand, $sth1, $sth2, $sth3);
	my ($hit, @count);
	my $nr = 0;

	# prepare the input-statement of the 'knownncrna'-table
    	$searchid = q{select count(*) from knownncrna where known_id=?};
    	$sth1 = $self->dbh->prepare($searchid);
    	$newknownncrna = q{insert into knownncrna (known_id,known_start,known_end,known_score,known_family,known_basepairs,known_status) values (?,?,?,?,?,?,?)};
    	$sth2 = $self->dbh->prepare($newknownncrna);

    	# prepare the delete-statement of the 'candidate'-table
    	$deletecand = q{delete from candidate where cand_id=?};
    	$sth3 = $self->dbh->prepare($deletecand);

	foreach $hit ( @$knownncrna ) {
		# removes a known ncrna from the 'candidate'-table if it wasn't already removed and its status is a 'HIT'
                if( ${$hit}[6] eq "HIT" ) {
                    $sth1->execute(${$hit}[0]);
                    @count = $sth1->fetchrow_array;
                    while( $sth1->fetchrow_array ) {}
                    unless( $count[0] ) {
                        $sth3->execute(${$hit}[0]);
                        $nr++;
                    }
                }

                # add every hits to the covariance models in the 'knownncrna'-table
                $sth2->execute(${$hit}[0], ${$hit}[1], ${$hit}[2], ${$hit}[3], ${$hit}[4], ${$hit}[5], ${$hit}[6]);
	}
	$self->dbh->commit;

	return $nr;
}


sub insert_knownncrna2 {
    my $self = shift;
    my $subdir = shift;
    my $refbp = shift;
    my $length = shift;

    my ($searchid, $newknownncrna, $deletecand, $sth1, $sth2, $sth3);
    my ($cvsfiles, $family, @par, $status, $subseq, $sshit, $line, $id, $bp, $modbpr, $index, @count);
    my $nr = 0;

    # prepare the input-statement of the 'knownncrna'-table
    $searchid = q{select count(*) from knownncrna where known_id=?};
    $sth1 = $self->dbh->prepare($searchid);
    $newknownncrna = q{insert into knownncrna (known_id,known_start,known_end,known_score,known_family,known_basepairs,known_status) values (?,?,?,?,?,?,?)};
    $sth2 = $self->dbh->prepare($newknownncrna);

    # prepare the delete-statement of the 'candidate'-table
    $deletecand = q{delete from candidate where cand_id=?};
    $sth3 = $self->dbh->prepare($deletecand);

    # analyse the results and update the mySQL-db
    $index = 0;
    while( $cvsfiles = <$subdir/*.cmzasha.csv> ) {
        ( $family ) = $cvsfiles =~ /(RF\d{5})/;

        open IN, "$cvsfiles" || die("Can not open the file!\n");
        while( <IN> ) {
            if( $_ =~ /^Params:/ ) {
                $status = "HIT";
                @par = split ",", $_;

                # test the plausibility of the ravenna hits

                # check if the modeled subsequence has gaps inside (bug of the assembly software of Dr. Mike Gilchrist)
                # ravenna hits including gap regions causing modelling mistakes hence they should be ignored
                $subseq = $self->get_assest_subsequence($par[6],$par[9],$par[10]);
                $status = "GAPS" if $subseq=~/-/;

                # check if the modeled subsequence are longer as $LENGTH nucleotides
                $status = "SMALL" if( $status eq "HIT" && abs($par[10]-$par[9])<=$length );

                # check if the number of basepairs lies in a range of 20% around the basepair number of the model
                if( $status eq "HIT" ) {

                    # catch the dot-bracket notation of the predicted secondary structure from the .cmzasha-file
                    $cvsfiles =~ /^(.*).csv$/;
                    open IN2, "$1" || die("Can not open the file!\n");
                    $sshit=0;
                    foreach $line ( <IN2> ) {
                        ( $id ) = $line =~ /^----sequence:\s\#\d+,(\d+)/;
                        $sshit=1 if defined $id && $id==$par[6];
                        if( $sshit && $line=~/^----ssRNAplot:/ ) {
                            ( $subseq ) = $line =~ /^----ssRNAplot:\s+([\(\.\)]+)/;
                            last;
                        }
                    }
                    close IN2;

                    #$bprate = get_basepair_rate($subseq);
                    $bp = $subseq =~ tr/(//;
                    $modbpr = $$refbp[$index];
                    $status = "BP" if( $bp<=$modbpr-0.2*$modbpr || $bp>=$modbpr+0.2*$modbpr );
                }

                # removes a known ncrna from the 'candncrna'-table if it wasn't already removed and its status is a 'HIT'
                if( $status eq "HIT" ) {
                    $sth1->execute($par[6]);
                    @count = $sth1->fetchrow_array;
		    while( $sth1->fetchrow_array ) {}
                    unless( $count[0] ) {
                        $sth3->execute($par[6]);
                        $nr++;
                    }
                }

                # add every hits to the covariance models in the 'knownncrna'-table
                $sth2->execute($par[6], $par[9], $par[10], $par[11], $family, $bp, $status);
            }
        }
        close IN;
        $index++;
    }
    $self->dbh->commit;

    # delete temp files
    unlink "$subdir/db.fa.gz", "$subdir/ravenna.config.tab", "$subdir/default.heurcreationspec", "$subdir/pbs_ravenna.sh";

    return $nr;
}


# returns the subsequence of an EST
#
# input is the EST id, startindex and endindex
# returns the subsequence
sub get_assest_subsequence {
    my ($self, $id, $start, $end) = @_;
    my ($query, $sth, $seq, $tmp, @seq, $base);
    my $n = 0;
    my $subseq = "";

    $query = q{select ass_sequence from assest where ass_id=?};
    $sth = $self->dbh->prepare($query);
    $sth->execute($id);
    $seq = ( $sth->fetchrow_array ) [0];
    #while( $sth->fetchrow_array ) {}

    if( $start>$end ) {
        $tmp = $start;
        $start = $end;
        $end = $tmp;
    }
    @seq = split //,$seq;
    foreach $base (@seq) {
        $n++;
        if($n>=$start && $n<=$end) {
            $subseq = $subseq.$base;
        }
        elsif($n>$end) {
            last;
        }
    }

    return $subseq;
}


# add subject sequences to estcoverage table
#
# input are subject name,
#           a reference to a hash including as key the subject chromosome name and as value the genome FASTA-file,
#           the alignment type
sub update_estcov_subject_sequence {
    my ($self, $subject, $chr_hashref, $aligntype) = @_;
    my ($getestcov, $updateestcov, $sth1, $sth2);
    my ($chrom, $chrnr, $queries_ary_ref, $sseq_ref, $qid, $qstart);

    print "Get aligned subsequences of $subject.\n";
#    $getestcov = q{select estcov_source_id, estcov_source_start, estcov_subject_chr, estcov_subject_start, estcov_subject_end from estcoverage where estcov_subject_name=? and estcov_subject_chr=? and estcov_align_type=? order by estcov_subject_start};
     $getestcov = q{select estcov_source_id, estcov_source_start, blastn_subject_id, blastn_subject_start, blastn_subject_stop from estcoverage,blastn where estcov_blastnid=blastn_blastnid and estcov_subject_name=? and estcov_subject_chr=? and estcov_align_type=? order by blastn_subject_start};
     $updateestcov = q{update estcoverage set estcov_subject_sequence=? where estcov_source_id=? and estcov_source_start=? and estcov_subject_name=? and estcov_subject_chr=? and estcov_align_type=?};
    $sth1 = $self->dbh->prepare($getestcov);
    $sth2 = $self->dbh->prepare($updateestcov);

    foreach $chrom ( keys %$chr_hashref ) {
        #$sth1->execute("gnl|$subject|$chrom", $aligntype);
	$chrom =~ /chro?m?(\w+)/;
	$chrnr = $1;
	$sth1->execute($subject, $chrnr, $aligntype); print "$subject, $chrnr, $aligntype\n";
        $queries_ary_ref = $sth1->fetchall_arrayref;
        $sseq_ref = get_subject_subsequences_of_one_chromosome($$chr_hashref{$chrom},$queries_ary_ref);  print "$chrom\t".$$chr_hashref{$chrom}."\n";
        foreach( keys %$sseq_ref ) {
            # extract key data from hash
            ( $qid, $qstart ) = $_ =~ /^(\d+)\|(\d+)/;
            # add subsequence to estcoverage table
            #$sth2->execute($$sseq_ref{$_}, $qid, $qstart, "gnl|$subject|$chrom", $aligntype);
	    $sth2->execute($$sseq_ref{$_}, $qid, $qstart, $subject, $chrnr, $aligntype);
	    $self->dbh->commit;
        }
	
    }

    $self->dbh->commit;
}


sub test_assembly {
    my ($self, $subject, $subjanno, $ADVANCEDBLASTN) = @_;
    my ($aligntype, $assembly, $sth1, $bstart, $bend);

    $aligntype = $ADVANCEDBLASTN ? "advanced_blastn" : "standard_blastn";

    # check if blast to the subject assembly already exists
    $assembly = q{select blastnanno_blastnid_from, blastnanno_blastnid_to from blastnannotation where blastnanno_organism=? and blastnanno_assembly=? and blastnanno_parameter=?};
    $sth1 = $self->dbh->prepare($assembly);
    $sth1->execute($subject, $subjanno, $aligntype);
    $sth1->bind_columns(\$bstart,\$bend);

    while( $sth1->fetch ) {
        die("A blast to the subject $subject assembly $subjanno already exists. You have to change the assembly annotation.\n");
        # delete existing data of the release in 'blastn'-table
        #$delorg = qq{delete from blastn where blastn_blastnid>$bstart-1 and blastn_blastnid<$bend+1};
        #$self->dbh->do($delorg);
        #$delorg = qq{delete from blastnannotation where blastnanno_organism="$subject" and blastnanno_assembly="$subjanno" and blastnanno_parameter="$aligntype"};
        #$self->dbh->do($delorg);
    }
}


sub insert_blastn {
    my ($self, $subject, $subjanno, $blasttable, $ADVANCEDBLASTN) = @_;
    my ($assembly, $newblastn, $aligntype, $bstart, $bend);
    my ($sth1, $sth2);
    my (@par, $blastnid, $blastnidfrom);

    $aligntype = $ADVANCEDBLASTN ? "advanced_blastn" : "standard_blastn";

    # prepare the input-statement of the 'blastn'-table
    $newblastn = q{insert into blastn values (?,?,?,?,?,?,?,?,?,?,?,?,?,?)};
    $sth1 = $self->dbh->prepare($newblastn);

    # get highest blastnid from blastn table
    $newblastn = q{select max(blastn_blastnid) from blastn};
    $sth2 = $self->dbh->prepare($newblastn);
    $sth2->execute();
    $sth2->bind_columns(\$blastnid);
    $sth2->fetch;
    $blastnid = 0 if ! defined $blastnid;
    $blastnidfrom = $blastnid+1;

    # use kvlblast2table results to fill in blastn table
    open IN, "zcat $blasttable |" || die("Can not open the file!\n");
    <IN>;
    while( <IN> ) {
        @par = split " ", $_;
	$blastnid++;

        # add every blast hit with e-value<blast->evalue to the 'blastn'-table
        $sth1->execute($blastnid, $par[0], $par[1], $par[4], $par[5], $par[6], $par[7], $par[8], $par[9], $par[10], $par[11], $par[12], $par[13], $par[14]);
    }
    close IN;

    # update the blastnannotation table
    $assembly = q{insert into blastnannotation values (?,?,?,?,?)};
    $sth1 = $self->dbh->prepare($assembly);
    $sth1->execute($subject, $subjanno, $aligntype, $blastnidfrom, $blastnid);
 
    $self->dbh->commit;

    return 1;
}


sub insert_estcoverage_from_blastn {
    my ($self, $subject, $subjanno, $advancedblastn, $chr_hashref, $length, $identity) = @_;
    my ($assembly, $getblasthits, $getorigspecest, $updatecandidate, $newestcoverage, $deletecandidate, $neworigspecest);
    my ($sth1, $sth2, $sth3, $sth4, $sth5, $sth6);
    my ($assid, $qid, %range, $status, $qstart, $qstop, $sid, $sstart, $sstop, $e, $blastnid, $nr, $aligntype, $bstart, $bend, %blasthash);

    $aligntype = $advancedblastn ? "advanced_blastn" : "standard_blastn";
    # get the desired blastn table entries
    $assembly = q{select blastnanno_blastnid_from, blastnanno_blastnid_to from blastnannotation where blastnanno_organism=? and blastnanno_assembly=? and blastnanno_parameter=?};
    $sth1 = $self->dbh->prepare($assembly);
    $sth1->execute($subject, $subjanno, $aligntype);
    $sth1->bind_columns(\$bstart,\$bend);
    $sth1->fetch;
    $bstart = 0 if !defined $bstart;
    $bend = 0 if !defined $bend;

    # select the best non overlapping blast hits (lowest e-value) with a length>100nt and an identity>0.75 for each EST of the 'candidate'-table
    # if this alignment exists then update the 'candidate'-table and add the alignment in the 'estcoverage'-table
    # otherwise remove the EST from the 'candidate'-table and add it to the 'origspecest'-table
    $getblasthits = q{select blastn_query_id, blastn_query_start, blastn_query_stop, blastn_subject_id, blastn_subject_start, blastn_subject_stop, blastn_blastnid from blastn, (select cand_id from candidate group by cand_id) as tmp where blastn_blastnid>? and blastn_blastnid<? and blastn_query_id=tmp.cand_id and blastn_align_length>? and blastn_identity/blastn_align_length*100>? order by blastn_evalue};
    $getorigspecest = q{select cand_id from candidate left join (select blastn_query_id as qid from blastn where blastn_blastnid>? and blastn_blastnid<? and blastn_align_length>? and blastn_identity/blastn_align_length*100>? group by blastn_query_id) as tmp2 on candidate.cand_id=tmp2.qid where tmp2.qid is NULL group by candidate.cand_id};
 
    $updatecandidate = q{insert into candidate (cand_id, cand_start, cand_end, cand_blastnid) values (?,?,?,?)};
    $newestcoverage = q{insert into estcoverage values (?,?,?,?,?,?,NULL,?,?)};
    $deletecandidate = q{delete from candidate};
    $neworigspecest = q{insert into origspecest values (?)};

    $sth1 = $self->dbh->prepare($getorigspecest);
    $sth2 = $self->dbh->prepare($getblasthits);
    $sth3 = $self->dbh->prepare($updatecandidate);
    $sth4 = $self->dbh->prepare($newestcoverage);
    $sth5 = $self->dbh->prepare($deletecandidate);
    $sth6 = $self->dbh->prepare($neworigspecest);

    # get ESTs which are not conserved in source organism and add them to 'origspecest'-table
    $sth1->execute($bstart-1,$bend+1,$length,$identity); 
    $sth1->bind_columns(\$assid);
    $nr = 0;
    while( $sth1->fetch ) {
    	$sth6->execute($assid);
        $nr++;
    }

    # read blastn hits of all candidates in a hash (key=query ID) of arrays (references to blastn hits of the query ordered by their E-value) of arrays (blastn hit)
    $sth2->execute($bstart-1,$bend+1,$length,$identity);    
    $sth2->bind_columns(\$qid, \$qstart, \$qstop, \$sid, \$sstart, \$sstop, \$blastnid);
    while( $sth2->fetch ) {
	push (@{$blasthash{$qid}}, [$qstart, $qstop, $sid, $sstart, $sstop, $blastnid]); 
    }

    # clean the 'candidate'-table
    $sth5->execute();

    # fetch in loop, if hit don't overlap with previous ones then update candncrna, conservedest and estcoverage, if no entry then origspecest
    foreach my $assid ( keys %blasthash ) {
	%range = ();
	foreach my $bhEntry ( @{$blasthash{$assid}} ) {
		$status = 0;
		$qstart = $bhEntry->[0];
		$qstop = $bhEntry->[1];
		foreach my $start ( keys %range ) {
			if( $qstart>=$start && $qstart<=$range{$start} ) {
                   		$status = 1;
                    		last;
                	}
                	if( $qstop>=$start && $qstop<=$range{$start} ) {
                    		$status = 1;
                    		last;
                	}
		}
		next if $status;
		$range{$qstart} = $qstop;

		# add conserved EST subsequence to the 'candidate'-table
            	$sth3->execute($assid, $qstart, $qstop, $bhEntry->[5]);
		# get chromosome and offset if exist
            	if( $bhEntry->[2] =~ /(chr)?(\w+)\W(\d+)\W(\d+)/ ) {
			$sid = $2;
			$bhEntry->[3] += $3-1;
			$bhEntry->[4] += $3-1;
		}
		else {
			$bhEntry->[2] =~ /(chr)?(\S+)/;
			$sid = $2;
		}
		# add the alignment in the 'estcoverage'-table
           	$sth4->execute($assid, $qstart, $subject, $sid, $bhEntry->[3], $bhEntry->[4], $aligntype, $bhEntry->[5]);
	}
    }

    $self->dbh->commit;


    # add subject sequences to estcoverage table
    $self->update_estcov_subject_sequence($subject, $chr_hashref, $aligntype);

    # delete temp files
    #unlink "$SUBDIR/query.fa.gz", "$SUBDIR/query.blast.gz", "$SUBDIR/pbs_blastn.sh";
    
    return $nr;
}


# creates a BED-file including the subsequence coordinates of a query organism in the estcoverage table
#
# input are the name of the query organism and the subdirectory where the output file should be saved
# returns the output file location
sub get_estcoverage_bed {
    my ($self, $queryorg, $SUBDIR) = @_;
    my ($getestcov, $sth1);
    my ($qid, $qstart, $schr, $sstart, $sstop, $strand, $tmp, $start, $end, $blastnid);

    my $OLDFILE = "$SUBDIR/loIn.bed";

    # select all estcoverage data of the query organism from the mySQL-db
    $getestcov = qq{select estcov_source_id,estcov_source_start,estcov_subject_chr,estcov_subject_start,estcov_subject_end,estcov_blastnid from estcoverage where estcov_subject_name like "%$queryorg%"};
    $sth1 = $self->dbh->prepare($getestcov);
    $sth1->execute();
    $sth1->bind_columns(\$qid, \$qstart, \$schr, \$sstart, \$sstop, \$blastnid);

    # create liftOver input data
    open OUT, ">$OLDFILE" || die("Can not open the file!\n");

    while( $sth1->fetch ) {
        # test type of strand
        if( $sstart<=$sstop ) {
            $strand = "+";
        }
        else {
            $strand = "-";
            $tmp = $sstart;
            $sstart = $sstop;
            $sstop = $tmp;
        }

        # index conversion from blastn-convention to over.chain-convention (UCSC) on strand "+"
        $start = $sstart - 1;
        $end = $sstop;

        # fill $OLDFILE with query subsequence coordinates (chromosome \t start \t end \t sourcename|sourcestart|blastnid \t 1 \t strand)
        print OUT "chr$schr\t$start\t$end\t$qid|$qstart|$blastnid\t1\t$strand\n";
    }

    close OUT;

    return $OLDFILE;
}


# creates a file including the estcoverage entries with $queryorg as subject ordered by chromosome and startposition
#
# input are the name of the query organism and the subdirectory where the output file should be saved
# returns the output file location
sub get_estcoverage_for_maf {
    my ($self, $queryorg, $SUBDIR) = @_;
    my ($getestcov, $sth1, $getestcovchr, $sth2);
    my ($qchr, $queries_ary_ref, $query_ary_ref);

    my $OLDFILE = "estcand.txt";

    # select all estcoverage data of the query organism from the mySQL-db
    $getestcovchr = qq{select distinct estcov_subject_name from estcoverage where estcov_subject_name like "%$queryorg%"};
    $sth2 = $self->dbh->prepare($getestcovchr);
    $sth2->execute();
    $sth2->bind_columns(\$qchr);
    $getestcov = qq{select estcov_source_id,estcov_source_start,estcov_subject_name,estcov_subject_start,estcov_subject_end from estcoverage where estcov_subject_name=? order by estcov_subject_start};
    $sth1 = $self->dbh->prepare($getestcov);

    # print the estcoverage data sorted by chromosomes and start index in a file
    open OUT, ">$SUBDIR/$OLDFILE" || die("Can not open the file!\n");
    while( $sth2->fetch ) {

        $sth1->execute($qchr);
        $queries_ary_ref = $sth1->fetchall_arrayref;

        # chromosome name
        $qchr =~ /.*\|(\w+)$/;

        print OUT "#$1\n";
        foreach $query_ary_ref ( @$queries_ary_ref ) {
            print OUT "$$query_ary_ref[0] $$query_ary_ref[1] $$query_ary_ref[2] $$query_ary_ref[3] $$query_ary_ref[4]\n";
        }
    }
    close OUT;

    return $OLDFILE;
}


# insert alignments in the estcoverage table
#
# input are a reference to a hash including pairs "est_id:est_start:subject_name:align_type" = "subject_start:subject_end:(subject_seq)"
#       additional arguments are the subject name and
#       a reference to a hash including pairs "chromosome" = "fasta-file location" if the term $sseq is empty
sub insert_estcoverage {
    my ($self, $refcand, $subject, $chr_hashref) = @_;
    my ($qid, $qstart, $blastnid, $sid, $sstart, $sstop, $seq, $aligntype);
    my ($newestcov, $sth1);

    # prepare mySQL statement
    $newestcov = q{insert into estcoverage values (?,?,?,?,?,?,?,?,?)};
    $sth1 = $self->dbh->prepare($newestcov);

    foreach( keys %$refcand ) {
	# extract data from hash
	( $qid, $qstart, $blastnid, $sid, $aligntype ) = split ":";
	( $sstart, $sstop, $seq ) =  split ":", $$refcand{$_};
	#$sid = "gnl|$subject|$sid" if defined $subject;
	$seq = "NULL" if $seq eq "";

	# add the alignment in the 'estcoverage'-table
	$sid =~ /(chr)?(\S+)/;
        $sid = $2;
	$sth1->execute($qid, $qstart, $subject, $sid, $sstart, $sstop, $seq, $aligntype, $blastnid);
    }

    if( defined $chr_hashref) {
	# add subject sequences to estcoverage table
	$self->update_estcov_subject_sequence($subject, $chr_hashref, $aligntype);
    }

    $self->dbh->commit;
}


# creates fasta-files including one EST and its homologous sequences
#
# input is the output folder
# returns a reference to an array including the name of all fasta-files (without file-ending)
sub get_homologous_fasta {
    my ($self, $subdirfasta) = @_;
    my ($getcandncrna, $getclosedorgan, $getestcoverage, $sth1, $sth2, $sth3);
    my ($cblastnid, $cid, $schrom, $cstart, $cend, $aligntype, $csubseq, @fasta, $sname, $sseq, @name);
    
    # create for each candidate EST from the 'candncrna'-table a fasta file with all related sequences collecting in the 'estcoverage'-table
    $getcandncrna = qq{select cand_blastnid, estcov_source_id, estcov_source_start, cand_end, estcov_align_type from estcoverage, candidate where estcov_source_id=cand_id and estcov_source_start=cand_start group by estcov_source_id, estcov_source_start, estcov_align_type};
    $getclosedorgan = q{select estcov_subject_name, estcov_subject_chr, estcov_subject_sequence from estcoverage where estcov_source_id=? and estcov_source_start=? and estcov_align_type like "%blastn%"};
    $getestcoverage = q{select estcov_subject_name, estcov_subject_chr, estcov_subject_sequence from estcoverage where estcov_source_id=? and estcov_source_start=? and estcov_align_type=?};
    $sth1 = $self->dbh->prepare($getcandncrna);
    $sth1->execute();
    $sth1->bind_columns(\$cblastnid, \$cid, \$cstart, \$cend, \$aligntype);
    $sth2 = $self->dbh->prepare($getclosedorgan);
    $sth3 = $self->dbh->prepare($getestcoverage);

    print "Create fasta file for each alignment of each candidate EST with the related homologous sequences.\n";

    `rm -rf $subdirfasta`;
    `mkdir -p $subdirfasta`;
    
    # loop for each alignment of each candidate EST
    while( $sth1->fetch ) {
	next if index($aligntype, "blastn")>=0;

        # get candidate EST subsequence
	$csubseq = $self->get_assest_subsequence($cid, $cstart, $cend);

        # create one fasta file for each alignment of each candidate EST with covered mammalian subsequences
#        push @fasta, "$cid.$cstart.$aligntype";
	push @fasta, "$cblastnid";
	
	open OUT, ">$subdirfasta/$cblastnid.fa" || die("Can not open the file!\n");
	print OUT ">$cblastnid\n";
	print OUT "$csubseq\n";
	
	# add closed organism
	$sth2->execute($cid, $cstart);
	( $sname, $schrom, $sseq ) = $sth2->fetchrow_array;
	while( $sth2->fetchrow_array ) {}
	#@name = split '\|', $sname;
	#$sname = $name[1].".".$name[2];
#        $sname = "gnl|$sname|$schrom";
	$sname .= "." . $schrom;
	print OUT ">$sname\n";
	print OUT "$sseq\n";
	
	# add all other covered mammalian subsequence
	$sth3->execute($cid, $cstart, $aligntype);
	$sth3->bind_columns(\$sname, \$schrom, \$sseq);
	while( $sth3->fetch ) {
	    #@name = split '\|', $sname;
	    #$sname = $name[1].".".$name[2];
#            $sname = "gnl|$sname|$schrom";
	    $sname .= "." . $schrom;
	    print OUT ">$sname\n";
	    print OUT "$sseq\n";
	}
	
	close OUT;
    }	
    
    return \@fasta;
}


# first sequence index = 1
sub get_gapfree_indizes {
    my ($self, $id, $start, $end) = @_;
    my ($subseq, @subseq, $tmp, $strand, $nr, $size, $base);

    $subseq = $self->get_assest_subsequence($id, $start, $end);
    @subseq = split //,$subseq;
    if( $start>$end ) {
	$tmp = $start;
	$start = $end;
	$end = $tmp;
	$strand = "-";
    }
    else {
	$strand = "+";
    }
    $nr = $start-1;
    $size = $end-$start+1;
    foreach $base (@subseq) {
	$nr++;
	if( $start==$nr ) {
	    $start++ if $base eq "-";
	    $size-- if $base eq "-";
	}
	elsif( $start<$nr && $end>=$nr ) {
	    $size-- if $base eq "-";
	}
    }

    return [$start, $size, $strand];
}


# input  - start position (first index is 1),
#          end position
# output - new end position counting all gaps inside the subsequence
sub get_gapincluded_indizes {
    my ($self, $id, $start, $end) = @_;
    my ($subseq, @subseq, $nr, $base);
  
    $subseq = $self->get_assest_subsequence($id, $start, $end);
    @subseq = split //,$subseq;

    $nr = $start-1;
    foreach $base (@subseq) {
	$nr++;
	if( $end>=$nr ) {
	    $end++ if $base eq "-";
	}
    }
    
    return $end;
}


# builds for each EST in 'candncrna' a hash including its homologous sequences in 'estcoverage'
#
# returns a reference to a hash including 
#             as key "$cid.$cstart.$aligntype" of a ncRNA candidate and 
#             as value a reference to a hash about the aligned sequences including 
#                 as key the identifier and
#                 as value an anonymous array with startposition (first sequence index = 1), length without gaps, strand and size of entire source sequence ('0')
sub get_homologous {
    my ($self) = @_;
    my ($getcandncrna, $getclosedorgan, $getestcoverage, $sth1, $sth2, $sth3);
    my ($cblastnid, $cid, $cstart, $cend, $aligntype, $sname, $schrom, $sstart, $send, %homologous);
    my ($gapfree);
    
    # create maf-files each with $MAFLENGTH aln-files (use blastn indizes: starting with 1 and indizes every time from the positiv strand)
    $getcandncrna = q{select cand_blastnid, estcov_source_id, estcov_source_start, cand_end, estcov_align_type from estcoverage, candidate where estcov_source_id=cand_id and estcov_source_start=cand_start group by estcov_source_id, estcov_source_start, estcov_align_type};
    $getclosedorgan = q{select estcov_subject_name, estcov_subject_chr, estcov_subject_start, estcov_subject_end from estcoverage where estcov_source_id=? and estcov_source_start=? and estcov_align_type like "%blastn%"};
    $getestcoverage = q{select estcov_subject_name, estcov_subject_chr, estcov_subject_start, estcov_subject_end from estcoverage where estcov_source_id=? and estcov_source_start=? and estcov_align_type=?};
    $sth1 = $self->dbh->prepare($getcandncrna);
    $sth1->execute();
    $sth1->bind_columns(\$cblastnid, \$cid, \$cstart, \$cend, \$aligntype);
    $sth2 = $self->dbh->prepare($getclosedorgan);
    $sth3 = $self->dbh->prepare($getestcoverage);

    # loop for each alignment of each candidate EST
    while( $sth1->fetch ) {
	next if index($aligntype, "blastn")>=0;
        my %aln;
	
        # candidate ncRNA
	$gapfree = $self->get_gapfree_indizes($cid, $cstart, $cend);	
	$aln{$cblastnid} = [$$gapfree[0], $$gapfree[1], $$gapfree[2], 0];
	
	# closed organism
	$sth2->execute($cid, $cstart);
	( $sname, $schrom, $sstart, $send ) = $sth2->fetchrow_array;
	while( $sth2->fetchrow_array ) {}
	$aln{"gnl|$sname|$schrom"} = [$sstart, $send-$sstart+1, "+", 0] if $sstart<$send;
	$aln{"gnl|$sname|$schrom"} = [$send, $sstart-$send+1, "-", 0] if $sstart>$send;
	
        # homologous organisms
        $sth3->execute($cid, $cstart, $aligntype);
        $sth3->bind_columns(\$sname, \$schrom, \$sstart, \$send);
        while( $sth3->fetch ) {
            $aln{"gnl|$sname|$schrom"} = [$sstart, $send-$sstart+1, "+", 0] if $sstart<$send;
            $aln{"gnl|$sname|$schrom"} = [$send, $sstart-$send+1, "-", 0] if $sstart>$send;
        }
        
        # add the key=value pair "$cid.$cstart.$aligntype"=%aln to the returned hash
        my $aln_hashref = \%aln;
        $homologous{$cblastnid} = $aln_hashref;
    }
my $n = keys %homologous;print "nr: $n\n";	
    return \%homologous;
}


# analyse the results and update the mySQL-db; 
# stores for each pair of EST-id and EST-startindex the conserved structure with the highest p-value  
#
# input are the relative path to the RNAz output file and 
#           the relative path to the rnazCluster.pl output file
#           the version of used organisms for RNAz
sub insert_rnazwindows {
    my ($self, $rnazoutfile, $clusteredoutfile, $version) = @_;
    my ($selectblastnid, $deletecandncrna, $getcandncrna, $deletelocus, $deletewindow, $newcandncrna, $newrnazwindows, $getalignversion, $getmaxalignid, $newalignversion, $sth1, $sth2, $sth3, $sth4, $sth5, $sth6, $sth7, $sth8, $sth9, $sth10);
    my ($status, $id, $start, $seq, $str, %sstruct, $org, @line, $pval, $strand, $queryid);
    my (%versali, @aliorg, %alivers, $aliorg, $tmp, $max, $vers);
    
    # prepare mySQL statement
    $deletecandncrna = q{delete from candncrna};
    $getcandncrna = q{select candnc_pvalue from candncrna where candnc_id=? and candnc_start=?};
    $deletelocus = q{delete from candncrna where candnc_id=? and candnc_start=?};
    $deletewindow = q{delete from rnazwindows where rnaz_locus=?};
    $selectblastnid = q{select cand_id from candidate where cand_blastnid=?};
    $newcandncrna = q{insert into candncrna values (?,?,?,?,?,?,?,NULL,?,?)};
    $newrnazwindows = q{insert into rnazwindows values (?,?,?,?,?,?,?,?,?,?,?,?,?,?)};
    $getalignversion = q{select * from alignversion order by align_organism};
    $getmaxalignid = q{select max(align_id) from alignversion};
    $newalignversion = q{insert into alignversion values (?,?)};
    $sth1 = $self->dbh->prepare($deletecandncrna);
    $sth2 = $self->dbh->prepare($newcandncrna);
    $sth3 = $self->dbh->prepare($newrnazwindows);
    $sth4 = $self->dbh->prepare($getcandncrna);
    $sth5 = $self->dbh->prepare($deletelocus);
    $sth6 = $self->dbh->prepare($deletewindow);
    $sth7 = $self->dbh->prepare($getalignversion);
    $sth8 = $self->dbh->prepare($getmaxalignid);
    $sth9 = $self->dbh->prepare($newalignversion);
    $sth10 = $self->dbh->prepare($selectblastnid);

    # clean the 'candncrna'-table
    $sth1->execute();
    
    # fetch versions already exists in table alignversion and fill a hash
    $sth7->execute();
    $sth7->bind_columns(\$id, \$org);
    while( $sth7->fetch ) {
	$versali{$id} = ( defined $versali{$id} ) ? $versali{$id}.":".$org : $org;
    }
    foreach( keys %versali ) {
	$alivers{$versali{$_}} = $_;
    }
    # insert $version in table alignversion if not already exists
    # ("best alignments": this entry is the version of a process using all alignments in estcoverage and pre-processing them with rnazWindow)
    $sth8->execute();
    $max = ( $sth8->fetchrow_array )[0];
    $max = -1 unless defined $max; 
    $sth9->execute($max+1, $version) unless defined $alivers{$version};
    $alivers{$version} = $max+1;

    # fill a hash with conserved structure and sequence information and aligned organisms
    $status = 0;
    open IN, $rnazoutfile || die("Can not open the file!\n");
    while( <IN> ) {
	if( !$status && /^>/ ) {
	    $status = 1;
	    @line = split " ";
	    $line[0]=~s/>//;
	    $id = $line[0];
	    $start = $line[1];
	    $strand = $line[3];
	    #( $id, $start ) = /^>(\d+)\s(\d+)/;
	    $seq = <IN>;
	    chomp( $seq );
	}
	elsif( /^>consensus/ ) {
	    # get the version number of an alignment and create a new one if necessary
	    undef $aliorg;
            map { $aliorg = ( defined $aliorg ) ? $aliorg.":".$_ : $_ } sort @aliorg;
	    unless( defined $alivers{$aliorg} ) {

		# insert new version in alignversion with align_id = max(align_id)+1
		$sth8->execute();
		$max = ( $sth8->fetchrow_array )[0];
		$max = -1 unless defined $max; 
		map { $sth9->execute($max+1, $_ ) } @aliorg;
		$alivers{$aliorg} = $max+1;
		
	    }
	    $vers = $alivers{$aliorg};
            undef @aliorg;

	    # fill a hash with conserved structure and sequence information and aligned organisms
	    <IN>;
	    $str = <IN>;
	    $str = ( split " ", $str )[0];
	    $sstruct{"$id:$start:$strand"} = [$seq,$str,$vers,$org];
	    $status = 0;
	    undef $org;
	}
	elsif( $status && /^>/ ) {
	    @line = split " ", $_;
	    $line[0] =~ s/>gnl\|//;
	    $tmp = $line[0]."|".($line[1]+1)."|".$line[2]."|".$line[3];
            $org = ( defined $org ) ? $org.":".$tmp : $tmp;
            # add organism to aligned organism list
            $tmp = ( split '\|',$line[0] )[0];
            push @aliorg, lc($tmp);
	}
    }
    close IN;
    

    open IN, $clusteredoutfile || die("Can not open the file!\n");
    while( <IN> ) {
	@line = split " ", $_;
	if( @line==8 ) {
	    # extract number of locus
	    $line[0] =~ s/locus//;
	    # fetch blastn_blastnid
	    $sth10->execute($line[1]);
	    $queryid = ( $sth10->fetchrow_array )[0];
	    # update the EST length considering gaps and count them
	    $line[3] = $self->get_gapincluded_indizes($queryid, $line[2]+1, $line[3]);
	    # add predicted RNA genes to the 'candncrna'-table if primary key doesn't already exist with higher p-value
	    $sth4->execute($queryid, $line[2]+1);
	    $pval = ( $sth4->fetchrow_array )[0];
	    if( $pval && $pval<$line[6]) {
		# delete old entry
		$sth5->execute($queryid, $line[2]+1);
		$sth6->execute($line[0]);
		$sth2->execute($queryid, $line[2]+1, $line[3], $line[6], $line[7], $line[5], $line[0], $alivers{$version}, $line[1]);
	    }
	    elsif( !$pval ) {
		$sth2->execute($queryid, $line[2]+1, $line[3], $line[6], $line[7], $line[5], $line[0], $alivers{$version}, $line[1]);
	    }
	}
	else {
	    # extract number of locus and window
	    $line[1] =~ s/locus//;
	    $line[0] =~ s/window//;
	    # fetch blastn_blastnid
	    $sth10->execute($line[2]);
	    $queryid = ( $sth10->fetchrow_array )[0];
	    # update the EST length considering gaps and count them
	    $line[4] = $self->get_gapincluded_indizes($queryid, $line[3]+1, $line[4]);
	    # add the local conserved structures (windows) of the predicted RNA genes to the 'rnazwindows'-table
	    $sth3->execute($line[1], $line[0], $line[3]+1, $line[4], $line[5], $line[17], $line[14], $line[15], $line[13], $line[8], $sstruct{"$line[2]:$line[3]:$line[5]"}[0], $sstruct{"$line[2]:$line[3]:$line[5]"}[1], $sstruct{"$line[2]:$line[3]:$line[5]"}[2], $sstruct{"$line[2]:$line[3]:$line[5]"}[3]);
	}
    }
    close IN;
    
    $self->dbh->commit;
}


# updates the annotation of candncrna entries, if already an annotation exists then concatinate the new one seperated by ':'
#
# Input  - a reference to a hash including as key the ID (and the start position seperated by ":") and as value the annotation text
#  	   (query_start, query_end, subject_id, evalue, identity, subject_coverage and description seperated by ":")
# Output - the number of changes
sub insert_cand_annotation {
    my ($self, $anno_hashref) = @_;
    my ($sth1, $sth2, $sth3, $sth4, $id, $start, $oldanno, $tmp, $anno, @item);
    my $nr = 0;

    $sth1 = $self->dbh->prepare("select candnc_annotation from candncrna where candnc_id=?");
    $sth2 = $self->dbh->prepare("update candncrna set candnc_annotation=? where candnc_id=?");
    $sth3 = $self->dbh->prepare("select candnc_annotation from candncrna where candnc_id=? and candnc_start=?");
    $sth4 = $self->dbh->prepare("update candncrna set candnc_annotation=? where candnc_id=? and candnc_start=?");

    foreach( keys %$anno_hashref ) {
	@item = split ":", $$anno_hashref{$_};
        chomp $item[6];
	$anno = $item[6]." ($item[2])";

	if( index($_,":")>-1 ) {
		( $id, $start ) = split ":", $_;
		$sth3->execute($id, $start);
		$oldanno = ( $sth3->fetchrow_array )[0];
		$anno = $oldanno." : ".$anno if defined $oldanno;

		$sth4->execute($anno, $id, $start);
		$nr++;			
	}
	else {		
		$sth1->execute($_);
		$oldanno = ( $sth1->fetchrow_array )[0];
		$anno = $oldanno." : ".$anno if defined $oldanno;

		$sth2->execute($anno, $_);
		$nr++;
	}
    }

    $self->dbh->commit;

    return $nr;
}


sub insert_cand_utr {
    my ($self, $utr_hashref) = @_;
    my ($sth1, $sth2, $anno, $oldanno, $id, $start, $utr, @item);
    my $nr = 0;

    $sth1 = $self->dbh->prepare("update candncrna set cand_utr=? where cand_id=? and cand_start=?");
    $sth2 = $self->dbh->prepare("select cand_utr from candncrna where cand_id=? and cand_start=?");
    foreach( keys %$utr_hashref ) {
	$anno = $$utr_hashref{$_};
        ( $id, $start ) = split ":", $_;
	$sth2->execute($id, $start);
        $oldanno = ( $sth2->fetchrow_array )[0];
        $anno = $oldanno." : ".$anno if defined $oldanno;
        $sth1->execute($anno, $id, $start);
        $nr++;
    }

    $self->dbh->commit;

    return $nr;
}


# fill the annotatedncrna table
#
# Input  - a reference to a hash including as key the ID and as value the annotation
#          (query_start, query_end, subject_id, evalue, identity, subject_coverage and description seperated by ":")
# Output - the number of similar items in the ncRNA dbs for the ESTs

sub insert_annotatedncrna {
    my ($self, $anno_refhash) = @_;
    my ($sth, @item);
    my $nr = 0;

    $sth = $self->dbh->prepare("insert into annotatedncrna values (?,?,?,?,?,?,?,?)");
    foreach( keys %$anno_refhash ) {
	@item = split ":", $$anno_refhash{$_};
	$sth->execute($_, $item[0], $item[1], $item[2], $item[3], $item[4], $item[5], $item[6]);
	$nr++;
    }

    $self->dbh->commit;

    return $nr;
}


# updates the annotation of knownncrna entries
#
# Input  - a reference to a hash including as key the ID (id:start) and as value the annotation text
#	   (query_start, query_end, subject_id, evalue, identity, subject_coverage and description seperated by ":")
# Output - the number of changes
sub update_known_ncRNA {
    my ($self, $anno_refhash) = @_;
    my ($sth, $sth2, $fam, $anno, @item, $id, $start);
    my $nr = 0;

    $sth = $self->dbh->prepare("update knownncrna set known_annotation=? where known_id=? and known_start=?");
    $sth2 = $self->dbh->prepare("select known_annotation from knownncrna where known_id=? and known_start=?");
    foreach( keys %$anno_refhash ) {
	($id, $start) = split ":";
	@item = split ":", $$anno_refhash{$_};
	chomp $item[6];
	$anno = $item[6]." ($item[2])";
	$sth2->execute($id, $start);
	$fam = ( $sth2->fetchrow_array )[0];
	$anno = $fam." : ".$anno if defined $fam;
        $sth->execute($anno, $id, $start);
        $nr++;
    }

    $self->dbh->commit;

    return $nr;
}


# get filenames of clustalw alignments of candncrnas
sub cand_aln_files {
    my ($self) = @_;
    my ($query, $sth, $id, $start, $type, @files);

    $query = q{select cons_id, cons_start, estcov_align_type from conservedest, estcoverage where cons_id=estcov_source_id and cons_start=estcov_source_start and cons_id in (select cand_id from candncrna) and estcov_align_type like "%bosTau2%" group by cons_id, cons_start, estcov_align_type};
    
    $sth = $self->dbh->prepare($query);
    $sth->execute();
    $sth->bind_columns(\$id, \$start, \$type);
    while( $sth->fetch ) {
	# add file name to an array
	push @files, "$id.$start.$type.aln";
    }

    return \@files;
}


# fills the microRNA table with RNAmicro-results
# Input  - a reference to a hash including as key the ID and as value all hits seperated by ":" 
#          whereby each hit is presented by the start pos., end pos., strand, p-value seperated by space
# Output - the number ESTs predicted as microRNA
sub insert_microRNA {
    my ($self, $micro_hashref) = @_;
    my ($sth, $id, @cluster, @micro, $nr);
    
    $sth = $self->dbh->prepare("insert into microrna values (?,?,?,?,?)");

    foreach $id ( keys %$micro_hashref ) {
	@cluster = split ":", $$micro_hashref{$id};
	foreach( @cluster ) {
		@micro = split " ";
		$sth->execute($id, $micro[0], $micro[1], $micro[2], $micro[3]);
	}
	$nr++;
    }

    $self->dbh->commit;

    return $nr;
}


# fills the snoRNA table with snowRNA-results
# Input  - a reference to a hash including as key the ID and as value an anonymous hash with strand and p-value
# Output - the number ESTs predicted as snoRNA
sub insert_snoRNA {
    my ($self, $sno_hashref) = @_;
    my ($sth, $id, @cluster, @sno, $nr);
    
    $sth = $self->dbh->prepare("insert into snorna values (?,?,?)");

    foreach $id ( keys %$sno_hashref ) {
	$sth->execute($id, $$sno_hashref{$id}[1], $$sno_hashref{$id}[0]);
	$nr++;
    }
    
    $self->dbh->commit;

    return $nr;
}


# returns the number of ncRNA candidates
sub get_nr_candncrna {
    my ($self) = @_;
    my ($sth, $nr);
    
    $sth = $self->dbh->prepare("select count(*) from candncrna");
    $sth->execute();
    $nr = ( $sth->fetchrow_array )[0];
    while( $sth->fetchrow_array ) {}
    
    return $nr;
}


# get homologous sequences of an homologous organism to all candidates of ncRNAs
# Input  - the searched organism name (f.e. hg17)
# Output - a reference to a hash whereby the key describes the EST (id, start, end seperated by ":") and the value the chromosome, start pos., end pos. and strand of the homologous sequence seperated by ":"
sub cand_homologous_seq {
    my ($self, $org) = @_;
    my ($query, $sth1, $cid, $cstart, $cend, $align, $tmp, @hg, @org, %cchr, %cstart, %cend, %cstrand, %out);
    
    # get all candidates of ncRNAs and the homologous organism $org
    $query = q{select cand_id, cand_start, cand_end, rnaz_align_organisms_list from rnazwindows, candncrna where rnaz_locus=cand_locus and rnaz_pvalue>0.9 and rnaz_align_organisms_list like ?};
    $sth1 = $self->dbh->prepare($query);
    $sth1->execute("\%$org\%");
    $sth1->bind_columns(\$cid,\$cstart,\$cend,\$align);
    while($sth1->fetch) {
	@org = split ":", $align;
	map { $tmp=lc($_); @hg = split '\|' if $tmp=~/^$org/ } @org;
        if( $#hg > -1 ) {
        	$cchr{"$cid:$cstart:$cend"} = $hg[1] unless defined $cchr{"$cid:$cstart:$cend"};
                $cstart{"$cid:$cstart:$cend"} = $hg[2] unless defined $cstart{"$cid:$cstart:$cend"};
                $tmp = $hg[2]+$hg[3]-1;
                $cend{"$cid:$cstart:$cend"} = $tmp;
                $cstrand{"$cid:$cstart:$cend"} = $hg[4];
     	}
        @hg=();
    }

    # generate returned hash
    map { $out{$_} = $cchr{$_}.":".$cstart{$_}.":".$cend{$_}.":".$cstrand{$_} } keys %cchr;

    return \%out; 
}


# calculate the false positive rate of RNAz predictions for the given p-value (number of shuffled scanning windows/number of original scanning windows)
# Input  - (1) the clustered RNAz output file of the shuffled alignments
#          (2) the p-value
# Output - the false positive rate
sub RNAz_statistics {
    my ($self, $rnazrandomcluster, $p) = @_;
    my ($randnr, @line, $query, $sth, $orignr, $fprate);

    $randnr = 0;
    open IN, "$rnazrandomcluster" || die("Can not open the file!\n");
    while( <IN> ) {
	@line = split " ";
	$randnr++ if( index($line[0],"window")>=0 && $line[17]>$p ); 	
    }
    close IN;

    $query = q{select count(*) from rnazwindows where rnaz_pvalue>?};
    $sth = $self->dbh->prepare($query);
    $sth->execute($p);
    $orignr = ( $sth->fetchrow_array )[0];

    # calculate false positive rate
    $fprate = $randnr/$orignr;

    return $fprate;
}


# calculate the false positive rate of RNAmicro predictions for the given p-value (number of shuffled scanning windows/number of original scanning windows)
# Input  - (1) a reference to a hash including as key the ID and as value all hits seperated by ":" 
#          whereby each hit is presented by the start pos., end pos., strand, p-value seperated by space
#          (3) the p-value
# Output - the false positive rate
sub RNAmicro_statistics {
    my($self, $micro_refhash, $p) = @_;
    my ( $id, @cluster, @micro, $query, $sth, $orignr, $fprate);
    my $randnr = 0;

    # count the hits of the shuffled data with p-value higher as $p
    foreach $id ( keys %$micro_refhash ) {
    	@cluster = split ":", $$micro_refhash{$id};
        foreach( @cluster ) {
       		@micro = split " ";
                $randnr++ if $micro[3]>$p;
        }
    }

    # get hits of the original data with p-value higher as $p
    $query = q{select count(*) from microrna where micro_pvalue>?};
    $sth = $self->dbh->prepare($query);
    $sth->execute($p);
    $orignr = ( $sth->fetchrow_array )[0];

    # calculate false positive rate
    $fprate = $randnr/$orignr;

    return $fprate;
}


# calculate the false positive rate of snoRNA predictions for the given p-value (number of shuffled scanning windows/number of original scanning windows)
# Input  - (1) a reference to a hash including as key the ID and as value an anonymous hash with strand and p-value
#          (3) the p-value
# Output - the false positive rate
sub get_snoRNA_statistics {
    my($self, $sno_refhash, $p) = @_;
    my($id, $query, $sth, $orignr, $fprate);
    my $randnr = 0;

    # get hits of the shuffled data
    foreach $id ( keys %$sno_refhash ) {
	$randnr++ if $$sno_refhash{$id}[0]>$p;	
    }

    # get hits of the original data
    $query = q{select count(*) from snorna where sno_pvalue>?};
    $sth = $self->dbh->prepare($query);
    $sth->execute($p);
    $orignr = ( $sth->fetchrow_array )[0];

    # calculate false positive rate
    $fprate = $randnr/$orignr;

    return $fprate;
}


# fills the ncbiannotation table with blastn hits of candidate ncRNAs to the NCBI databases
# Input  - a reference to a hash including as key the EST id, EST start, subject id seperated by ":" and as value query start, query end, e-value, identity (identity/align_length), EST coverage (align_length/query_length) and description also seperated by ":"
# Output - the number of new database entries
sub insert_ncbi_annotation {
    my ($self, $anno_hashref) = @_;
    my ($sth, $id, @key, @val, $nr);

    $sth = $self->dbh->prepare("insert into ncbiannotation values (?,?,?,?,?,?,?,?,?)");

    foreach $id ( keys %$anno_hashref ) {
	@key = split ":", $id;
	@val = split ":", $$anno_hashref{$id};

        $sth->execute($key[0], $key[1], $val[0], $val[1], $key[2], $val[2], $val[3], $val[4], $val[5]);
     
        $nr++;
    }

    $self->dbh->commit;

    return $nr;
}
1;

__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

EST2ncRNA::MysqlInterface - Perl extension for blah blah blah

=head1 SYNOPSIS

  use EST2ncRNA::MysqlInterface;
  blah blah blah

=head1 DESCRIPTION

Stub documentation for EST2ncRNA::MysqlInterface, created by h2xs. It looks like the
author of the extension was negligent enough to leave the stub
unedited.

Blah blah blah.

=head2 EXPORT

None by default.



=head1 SEE ALSO

Mention other useful documentation such as the documentation of
related modules or operating system documentation (such as man pages
in UNIX), or any relevant external documentation such as RFCs or
standards.

If you have a mailing list set up for your module, mention it here.

If you have a WEB Site set up for your module, mention it here.

=head1 AUTHOR

Stefan Seemann, E<lt>seemann@bioinf.uni-leipzig.deE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2006 by Stefan Seemann

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.6 or,
at your option, any later version of Perl 5 you may have available.


=cut
