package EST2ncRNA::ServerInterface;

use strict;
use warnings;

require Exporter;

our @ISA = qw(Exporter);  # inherits from Exporter

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

our @EXPORT_OK = qw(
		    );

our @EXPORT = qw();

our $VERSION = '0.01';


# constructor
sub new {
    my $class = shift @_;
    my ($servername, $workdir, $uid) = @_;

    my $self = {
	_name         => $servername,
	_home         => undef,
	_userid       => undef,
	_queue        => undef,
	_nodes        => 1,
	_wallt        => "0:20:0",
	_pbsubmit     => undef,
	_locally      => 0
    };
    bless $self, $class;

    # get the user id on the server
    $self->{_userid} = ( $self->sshcall("whoami") )[0];

    # get user home directory
    $self->{_home} = ( $self->sshcall("pwd") )[0]."/".$workdir."/est2ncrna".$uid;

    # create workdir on remote server if not already exists
    $self->sshcall("mkdir -p ".$self->{_home});

    return $self;
}


# accessor method for ServerInterface name
sub name {
    my($self) = @_;
    return $self->{_name};
}


# accessor method for ServerInterface home
sub home {
    my($self) = @_;
    return $self->{_home};
}


# accessor method for ServerInterface userid
sub userid {
    my($self) = @_;
    return $self->{_userid};
}


# accessor method for ServerInterface queue
sub queue {
    my($self, $queue) = @_;
    $self->{_queue} = $queue if defined($queue);
    return $self->{_queue};
}


# accessor method for ServerInterface nodes
sub nodes {
    my($self, $nodes) = @_;
    $self->{_nodes} = $nodes if defined($nodes);
    return $self->{_nodes};
}


# accessor method for ServerInterface
sub wallt {
    my($self, $wallt) = @_;
    $self->{_wallt} = $wallt if defined($wallt);
    return $self->{_wallt};
}


# accessor method for path of pbsubmit
sub pbsubmit {
    my($self, $pbsubmit) = @_;
    $self->{_pbsubmit} = $pbsubmit if defined($pbsubmit);
    return $self->{_pbsubmit};
}

# accessor method for path of scratch
sub scratch {
    my($self, $scratch) = @_;
    $self->{_scratch} = $scratch if defined($scratch);
    return $self->{_scratch};
}

# accessor method for locally
sub locally {
    my($self, $locally) = @_;
    $self->{_locally} = 1 if defined($locally);
    return $self->{_locally};
}


# create temporary working directories on the server
sub prepare_dir_tree {
    my ($self, $uid) = @_;
    my ($server_home, $ravenna_home, $blast_home, $estcov_home, $rnaz_home, $alignments, $annotation, $pwd);

    $server_home = $self->home;

    $ravenna_home = "est2ncrna.ravenna";
    $self->sshcall("mkdir -p $server_home/$ravenna_home");

    $blast_home = "est2ncrna.blast";
    $self->sshcall("mkdir -p $server_home/$blast_home");

    $estcov_home = "est2ncrna.estcoverage";
    $self->sshcall("mkdir -p $server_home/$estcov_home");

    $rnaz_home = "est2ncrna.rnaz";
    $self->sshcall("mkdir -p $server_home/$rnaz_home");

    $alignments = "est2ncrna.aln";
    $self->sshcall("mkdir -p $server_home/$alignments");

    $annotation = "est2ncrna.annotation";
    $self->sshcall("mkdir -p $server_home/$annotation");
}


# execute a command on a remote machine and analyse the exit-status
# if the call was unsuccessful then it will be tried again after 60 seconds
#
# input is the executable command
# returns an array with the command output
sub sshcall {
    my $self = shift;
    my $cmd = shift;
    my $status = 1;
    my $counter = 11;
    my @res;

    my $server = $self->name;

    while( $status!=0 ) {
        die("'$cmd' can not execute on '$server'!\n") if --$counter==0;
        if( $server ne "localhost" ) {
            @res = `ssh $server \"$cmd\"`;
        }
        else {
            @res = `$cmd`;
        }
        $status = $? / 256;
        print "Try '$cmd' on '$server' again in 60 seconds!\n" if $status!=0;
        sleep 120 if $status!=0;
    }

    chomp @res;
    return @res;
}


# scp a file and analyse the exit-status
# if the copy was unsuccessful then it will be tried again after 60 seconds
#
# input is the source path and the destination path
sub scpcall {
    my $self = shift;
    my $source = shift;
    my $dest = shift;
    my $status = 1;
    my $counter = 11;

    my $server = $self->name;

    if( $server ne "localhost" ) {
	while( $status!=0 ) {
        	die("Can not copy to '$server'!\n") if --$counter==0;
		`scp $source $dest`;
        	$status = $? / 256;
       	 	print "Try copy from/to '$server' again in 10 seconds!\n" if $status!=0;
        	sleep 10 if $status!=0;
    	}
    }
}


# set environment variables in the server .cshrc or .bashrc file
#
# input is a hash consisting as keys the environment variable names and as values the paths
sub set_env {
    my $self = shift;
    my $envar = shift;
    my $input = "";
    my ($key, $shell);

    my $shome = $self->home;
    my $name = $self->name;

    $shell =  ( $self->sshcall("echo \\\$SHELL") )[0] if $self->name ne "localhost";
    $shell =  ( $self->sshcall("echo \$SHELL") )[0] if $self->name eq "localhost";
    my $tmp = index($shell, "bash");
    print "$shell $tmp\n";
    open OUT, ">setenv.sh" || die("Can not open the file!\n");

    if( index($shell, "bash") > -1 ) {
	foreach $key ( keys %$envar ) {
	    if($key eq 'path') {
		my $paths = join(":",split " ", $$envar{$key}); 
		$input = $input ."export PATH=$paths:\$PATH\n";
	    }
	    else {
		$input = $input ."export $key=$$envar{$key}\n";
	    }
	}	
	print OUT "#!/bin/bash\n if [ -e ~/.bashrc ]; then\n cp -p ~/.bashrc ~/.bashrc.1.bak\n fi\n $input\n source ~/.bashrc\n";
	print "#!/bin/bash\n if[ -e ~/.bashrc ]; then\n cp -p ~/.bashrc ~/.bashrc.1.bak\n fi\n $input\n source ~/.bashrc\n";
    }
    else {
	foreach $key ( keys %$envar ) {
	    if($key eq 'path') {
		$input = $input . "echo \"set $key = ( $$envar{$key} \$path )\" >> ~/.cshrc\n";
	    }
	    else {
		$input = $input . "echo \"setenv $key $$envar{$key}\" >> ~/.cshrc\n";
	    }
	}
	print OUT "#! /bin/csh\n if (-e ~/.cshrc) then\n cp -p ~/.cshrc ~/.cshrc.1.bak\n endif\n $input\n source ~/.cshrc\n";
    }

    close OUT;

    $self->scpcall('setenv.sh', $self->name.":".$self->home) unless $self->name eq "localhost";
    $self->sshcall("cd ".$self->home."; chmod u+x setenv.sh; ./setenv.sh");
}


# reset old .cshrc file on server
sub reset_env {
    my $self = shift;
    my ($shell);

    $shell =  ( $self->sshcall("echo \\\$SHELL") )[0] if $self->name ne "localhost";
    $shell =  ( $self->sshcall("echo \$SHELL") )[0] if $self->name eq "localhost";

    open OUT, ">resetenv.sh" || die("Can not open the file!\n");

    if( index($shell, "bash") > -1 ) {
	print OUT "#!/bin/bash\n if[ -e ~/.bashrc.1.bak ]; then\n mv ~/.bashrc.1.bak ~/.bashrc\n else\n rm -f ~/.bashrc\n fi\n";
    }
    else {
	print OUT "#! /bin/csh\n if (-e ~/.cshrc.1.bak) then\n mv ~/.cshrc.1.bak ~/.cshrc\n else\n rm -f ~/.cshrc\n endif\n";
    }

    close OUT;

    $self->scpcall('resetenv.sh', $self->name.":".$self->home) unless $self->name eq "localhost";
    $self->sshcall("cd ".$self->home."; chmod u+x resetenv.sh; ./resetenv.sh");
}


# wait until all processes of the user are finished
#
# input is the process name
sub wait {
    my ($self, $processname) = @_;
    my (@qstat, $index);
    
    while(1) {
	@qstat = $self->sshcall("qstat -u".$self->userid);
    	last if @qstat==0;
    	$index = 0;
    	map { $index++ if /$processname/ } @qstat;
    	last if $index==0;
	sleep 120;
    }
}


1;

__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

EST2ncRNA::ServerInterface - Perl extension for blah blah blah

=head1 SYNOPSIS

  use EST2ncRNA::ServerInterface;
  blah blah blah

=head1 DESCRIPTION

Stub documentation for EST2ncRNA::ServerInterface, created by h2xs. It looks like the
author of the extension was negligent enough to leave the stub
unedited.

Create a ssh-tunnel to establish the connection between the local machine and the server (no password prompt anymore):
(1) ssh-keygen -t dsa
(2) cat ~/.ssh/*.pub | ssh user@remote-system 'cat>>.ssh/authorized_keys'

Blah blah blah.

=head2 EXPORT

None by default.



=head1 SEE ALSO

Mention other useful documentation such as the documentation of
related modules or operating system documentation (such as man pages
in UNIX), or any relevant external documentation such as RFCs or
standards.

If you have a mailing list set up for your module, mention it here.

If you have a web site set up for your module, mention it here.

=head1 AUTHOR

Stefan Seemann, E<lt>seemann@bioinf.uni-leipzig.deE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2006 by Stefan Seemann

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.6 or,
at your option, any later version of Perl 5 you may have available.


=cut
