package EST2ncRNA::RNAzInterface;

use strict;
use warnings;

require Exporter;

our @ISA = qw(Exporter); # inherits from Exporter

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

our @EXPORT_OK = qw(
					);

our @EXPORT = qw();

our $VERSION = '0.01';


# constructor
sub new {
    my $class = shift @_;
    my ($workdir,$server,$uid) = @_;

    my $self = {
	_version      => "best alignments",
	_subdir       => $workdir."/est2ncrna.rnaz",
	_uid          => $uid,
	_server       => $server,
	_rnazhome     => undef,
	_rnazscripthome => undef
    };
    bless $self, $class;
    
    # create subdir if not already exists
    mkdir $self->{_subdir} unless -d $self->{_subdir};

    return $self;
}


# accessor method for RNAzInterface version
sub version {
    my($self, $version) = @_;
    $self->{_version} = $version if defined($version);
    return $self->{_version};
}


# accessor method for RNAzInterface rnazhome
sub rnazhome {
    my($self, $rnazhome) = @_;
    $self->{_rnazhome} = $rnazhome if defined($rnazhome);
    return $self->{_rnazhome};
}


# accessor method for RNAzInterface rnazscripthome
sub rnazscripthome {
    my($self, $rnazscripthome) = @_;
    $self->{_rnazscripthome} = $rnazscripthome if defined($rnazscripthome);
    return $self->{_rnazscripthome};
}


# accessor method for RNAzInterface subdir
sub subdir {
    my($self) = @_;
    return $self->{_subdir};
}


=head2 B<uid>

Accessor method for BlastInterface uid

=cut

sub uid {
    my($self) = @_;
    return $self->{_uid};
}


# accessor method for RNAzInterface server-interface
sub server {
    my($self) = @_;
    return $self->{_server};
}


# start rnazWindow.pl of the RNAz package to slice, pre-process and filter the alignments
#
# input is a reference to an array including the maf-file names
# returns a reference to an array including the prepared maf-file names
sub run_rnazWindow {
    my ($self, $refmaffiles) = @_;
    my ($file, @prepmaffiles);
    
    my $SERVER = $self->server->name;
    my $SERVER_HOME = $self->server->home;
    my $QUEUE = $self->server->queue;
    my $WALLT = $self->server->wallt;
    my $USERID = $self->server->userid;
    my $PBSUBMIT = $self->server->pbsubmit;
    my $SUBDIR = $self->subdir;
    my $SCRATCH = $self->server->scratch;
    my $LOCALLY = $self->server->locally;
    my $RNAZSCRIPTHOME = ($self->rnazscripthome) ? $self->rnazscripthome . "/" : "";
    	
    print "Start slicing, pre-processing and filtering of the alignments on $SERVER.\n";


    foreach $file ( @$refmaffiles ) {
    	push @prepmaffiles, "$file.prepared.maf";
        open OUT, ">$SUBDIR/pbs_rnazwindow.sh" || die("Can not open the file!\n");
	if( $LOCALLY ) {
		print OUT "#!/bin/tcsh\n set path = ( $PBSUBMIT \$path )\n cd $SERVER_HOME\n nohup ${RNAZSCRIPTHOME}rnazWindow.pl $file > $file.prepared.maf\n";
	}
	else {
        	print OUT "#!/bin/tcsh\n set path = ( $PBSUBMIT \$path )\n";
		print OUT " cd $SERVER_HOME\n pbsubmit.pl -o \"-N rnazwindow -l walltime=$WALLT\" $QUEUE -D -Q -B $SERVER_HOME -c \"${RNAZSCRIPTHOME}rnazWindow.pl $SERVER_HOME/$file > $SERVER_HOME/$file.prepared.maf\"\n";
	}
        close OUT;
	
        $self->server->scpcall("$SUBDIR/pbs_rnazwindow.sh", "$SERVER:$SERVER_HOME");
        $self->server->scpcall("$SUBDIR/$file.gz", "$SERVER:$SERVER_HOME");
        $self->server->sshcall("cd $SERVER_HOME; gunzip -f $file.gz; chmod u+x $SERVER_HOME/pbs_rnazwindow.sh; $SERVER_HOME/pbs_rnazwindow.sh");
    }
    # wait until all rnazwindow processes of user 'whoami' are finished
    $self->server->wait("rnazwindow");
    
    return \@prepmaffiles;
}


# run RNAz with a P value cutoff of 0.5
#
# input is a reference to an array including the maf-file names on the server
# returns the RNAz output file name
sub run_rnaz {
    my ($self, $refmaffiles, $output) = @_;
    my ($file, @rnazout);
    
    my $SERVER = $self->server->name;
    my $SERVER_HOME = $self->server->home;
    my $QUEUE = $self->server->queue;
    my $WALLT = $self->server->wallt;
    my $USERID = $self->server->userid;
    my $PBSUBMIT = $self->server->pbsubmit;
    my $SUBDIR = $self->subdir;
    my $SCRATCH = $self->server->scratch;
    my $LOCALLY = $self->server->locally;   
    my $RNAZHOME = ($self->rnazhome) ? $self->rnazhome . "/" : "";
     
    $output = "est.rnaz.out" unless defined $output;
	
    print "Scoring candidate ESTs with RNAz for conserved secondary structures on $SERVER.\n";
    
    foreach $file ( @$refmaffiles ) {
        open OUT, ">$SUBDIR/pbs_rnaz.sh" || die("Can not open the file!\n");
	if( $LOCALLY ) {
		print OUT "#!/bin/tcsh\n set path = ( $PBSUBMIT \$path )\n cd $SERVER_HOME/\n nohup ${RNAZHOME}RNAz --both-strands --show-gaps --cutoff=0.5 $file > $file.rnaz.out\n";
	}
	else {	
        	print OUT "#!/bin/tcsh\n set path = ( $PBSUBMIT \$path )\n cd $SERVER_HOME\n pbsubmit.pl -o \"-N rnaz -l walltime=$WALLT\" $QUEUE -D -Q -B $SERVER_HOME -c \"${RNAZHOME}RNAz --both-strands --show-gaps --cutoff=0.5 $SERVER_HOME/$file > $SERVER_HOME/$file.rnaz.out\"\n";
	}
        close OUT;

        $self->server->scpcall("$SUBDIR/pbs_rnaz.sh", "$SERVER:$SERVER_HOME");
        $self->server->sshcall("chmod u+x $SERVER_HOME/pbs_rnaz.sh; $SERVER_HOME/pbs_rnaz.sh");
        push @rnazout, "$file.rnaz.out";
    }
    # wait until all rnaz processes of user 'whoami' are finished
    $self->server->wait("rnaz");

    # concatenate RNAz results
    $self->server->sshcall("cd $SERVER_HOME; cat @rnazout > $output");
    
    # fetch data from server
    $self->server->scpcall("$SERVER:$SERVER_HOME/$output", "$SUBDIR/");
    
    return $output;
}


sub run_rnazRandomizeAln {
    my ($self, $alnfolder, $refmaffiles) = @_;
    my ($file, @randommaf);

    my $SERVER = $self->server->name;
    my $SERVER_HOME = $self->server->home;
    my $QUEUE = $self->server->queue;
    my $WALLT = $self->server->wallt;
    my $USERID = $self->server->userid;
    my $SUBDIR = $self->subdir;
    my $PBSUBMIT = $self->server->pbsubmit;
    my $SCRATCH = $self->server->scratch;
    my $LOCALLY = $self->server->locally;
    my $RNAZSCRIPTHOME = ($self->rnazscripthome) ? $self->rnazscripthome . "/" : "";
    
    print "Randomize alignments by shuffling the columns.\n";

    open OUT, ">$SUBDIR/pbs_rnaz_random.sh" || die("Can not open the file!\n");
    print OUT "#!/bin/tcsh\n cd $alnfolder\n";
    foreach $file ( @$refmaffiles ) {
	if( $LOCALLY ) {
		print OUT "nohup ${RNAZSCRIPTHOME}rnazRandomizeAln.pl $alnfolder/$file > $alnfolder/random_$file\n";
	}
	else {
        	print OUT "${RNAZSCRIPTHOME}rnazRandomizeAln.pl $alnfolder/$file > $alnfolder/random_$file\n";
	}

        push @randommaf, "random_$file";
    }
    close OUT;

    $self->server->scpcall("$SUBDIR/pbs_rnaz_random.sh", "$SERVER:$SERVER_HOME");
    $self->server->sshcall("chmod u+x $SERVER_HOME/pbs_rnaz_random.sh");
    `ssh $SERVER '$PBSUBMIT -o "-N shuffle -l walltime=$WALLT" $QUEUE -D -Q -B $SERVER_HOME -c "$SERVER_HOME/pbs_rnaz_random.sh"'`;

    # wait until all shuffle processes of user 'whoami' are finished
    $self->server->wait("shuffle");

    return \@randommaf;
}


# clustering the results generated by RNAz
#
# input is the RNAz output file name on the server
# returns the name of the rnazCluster.pl output file
sub run_rnazCluster {
    my ($self, $rnazoutfile, $output) = @_;
    my ($cmd);
    
    my $SERVER = $self->server->name;
    my $SERVER_HOME = $self->server->home;
    my $QUEUE = $self->server->queue;
    my $WALLT = $self->server->wallt;
    my $USERID = $self->server->userid;
    my $PBSUBMIT = $self->server->pbsubmit;
    my $SUBDIR = $self->subdir;
    my $SCRATCH = $self->server->scratch;
    my $LOCALLY = $self->server->locally;
    my $RNAZSCRIPTHOME = ($self->rnazscripthome) ? $self->rnazscripthome . "/" : "";
    	
    $output = "est.rnaz.results.dat" unless defined $output;
    
    print "Clustering of overlapping windows in the RNAz results on $SERVER.\n";
	
    $cmd = "${RNAZSCRIPTHOME}rnazCluster.pl $SERVER_HOME/$rnazoutfile > $SERVER_HOME/$output";
    open OUT, ">$SUBDIR/pbs_rnazcluster.sh" || die("Can not open the file!\n");
    if( $LOCALLY ) {
	print OUT "#!/bin/tcsh\n set path = ( $PBSUBMIT \$path )\n cd $SERVER_HOME\n nohup ${RNAZSCRIPTHOME}rnazCluster.pl $rnazoutfile > $output\n";
    }
    else {
    	print OUT "#!/bin/tcsh\n set path = ( $PBSUBMIT \$path )\n";
   	 print OUT "pbsubmit.pl -o \"-N rnazclust -l walltime=$WALLT\" $QUEUE -D -Q -B $SERVER_HOME -c \"$cmd \"\n";
    }
    close OUT;

    $self->server->scpcall("$SUBDIR/pbs_rnazcluster.sh", "$SERVER:$SERVER_HOME");
    $self->server->sshcall("chmod u+x $SERVER_HOME/pbs_rnazcluster.sh; $SERVER_HOME/pbs_rnazcluster.sh");

    # wait until all rnazcluster processes of user 'whoami' are finished
    $self->server->wait("rnazclust");
    
    # fetch data from server
    $self->server->scpcall("$SERVER:$SERVER_HOME/$output", "$SUBDIR/");
    
    return $output;
}


1;
__END__
# Below is stub documentation for your module. You'd better edit it!

=head1 NAME

EST2ncRNA::RNAzInterface - Perl extension for blah blah blah

=head1 SYNOPSIS

  use EST2ncRNA::RNAzInterface;
  blah blah blah

=head1 DESCRIPTION

Stub documentation for EST2ncRNA::RNAzInterface, created by h2xs. It looks like the
author of the extension was negligent enough to leave the stub
unedited.

Blah blah blah.

=head2 EXPORT

None by default.


=head1 SEE ALSO

Mention other useful documentation such as the documentation of
related modules or operating system documentation (such as man pages
in UNIX), or any relevant external documentation such as RFCs or
standards.

If you have a mailing list set up for your module, mention it here.

If you have a web site set up for your module, mention it here.

=head1 AUTHOR

Stefan Seemann, E<lt>seemann@bioinf.uni-leipzig.deE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2006 by Stefan Seemann

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.6 or,
at your option, any later version of Perl 5 you may have available.


=cut
