#!/bin/bash

# script to compute base pairing probability (P) matrix for a given single sequence or multiple sequence alignment in fasta format

# dependencies
# ===========
# this script requires prior local installation of PETfold (https://rth.dk/resources/petfold/download/PETfold2.1.tar.gz) and RNAfold from Veinna RNA package. 
# set the path information of PETfold and RNAfold in the below variable
# path for the local PETfold and RNAfold 
export PETFOLDBIN=/home/projects/rth/rnabound/multiz_mapping/scripts/PETfold/bin/ 
PETFOLDPATH=/home/projects/rth/rnabound/multiz_mapping/scripts/PETfold/bin/
RNAFOLDPATH=/home/users/sabari/miniconda3/envs/viennarna241/bin/

show_help() {
cat << EOF
Usage: ${0##*/} [-s <single_Seqfile> | -m <multiple sequence file>]
    -h  display this help and exit
    -s  single sequence file 
    -m  multiple sequence alignment (fasta file)
Example: ${0##*/} -m examples/tRNA_align.fa
EOF
}

if [[ $# == 0 ]];then show_help;exit 1;fi

# input parameter for rnabound
flanking=10
pnull=0.0005

while getopts "s:m:f:p:h" opt; do
    case "$opt" in
        h) show_help;exit 0;;
        s) sfile=$OPTARG;;
        m) mfile=$OPTARG;;
        f) flanking=$OPTARG;;
        p) pnull=$OPTARG;;
       '?')show_help >&2 exit 1 ;;
    esac
done

# check input variables
if [[ -z $sfile && -z $mfile ]];then 
   echo -e "\n\n\t***input missing check the usage below**\n\n"; show_help; exit 1;
fi

if [ ! -z $mfile ]
then
  # run petfold
  ofilename=$(mktemp)
  $PETFOLDPATH/PETfold -f $mfile -r ${ofilename}_pp.txt >${ofilename}.out

  # run rnabound
  perl scripts/rnabound.pl --relibmat ${ofilename}_pp.txt --flanking $flanking --pnull $pnull

  rm -rf ${ofilename}_pp.txt ${ofilename}.out
fi

if [ ! -z $sfile ]
then
   # run rnafold
   tmpdir=$(mktemp -d)
   grep -v ">" $sfile >$tmpdir/seq.txt
   cpath=$PWD
   cd $tmpdir
   $RNAFOLDPATH/RNAfold -p < seq.txt >out.txt
   cd $cpath

   # run rnabound
   perl scripts/rnabound.pl --dotfile $tmpdir/dot.ps --flanking $flanking --pnull $pnull

   rm -rf $tmpdir
fi
