#!/usr/bin/env bash
#
# perfoms complete RNAcop analysis
#
#    Copyright (C) 2015  Nikolai Hecker                                   #
#                                                                         #
#   This program is free software: you can redistribute it and/or modify  #
#   it under the terms of the GNU General Public License as published by  #
#   the Free Software Foundation, either version 3 of the License, or     #
#   (at your option) any later version.                                   #
#                                                                         #
#   This program is distributed in the hope that it will be useful,       #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#   GNU General Public License for more details.                          #
#                                                                         #
#   ADDITIONAL TERMS                                                      #
#    The program RNAcop itself is part of a different license.            #
#    In particular, it was implemented in the ViennaRNA package,          # 
#    thus it is not licensed under terms of the GNU General Public        #
#    License, but follows a similar license as the ViennaRNA package      #
#    itself.                                                              #
#                                                                         #
#   You should have received a copy of the GNU General Public License     #
#   along with this program.  If not, see <http://www.gnu.org/licenses/>. #
#                                                                         #
###########################################################################
usage="Usage:\n\t$0 [INPUT-FILE] [OUTPUT-TITLE]"

if [ -z $2 ]
then
    echo -e $usage
    exit
fi

inf=$(readlink -m $1)
title=$2
outf=rnacop_${title}.out
maxl=100
maxr=100

odir=$(pwd)
mkdir $title

#cp files into directory
cp utility/*.R $title/
cp utility/*.Rscript $title/
cp utility/*.pl $title/
cp findgoodregion-0.2/findgoodregion $title/
cp RNAcop-1.03/Progs/RNAcop $title/

#HTML output
echo "<html>" > $odir/$title.html


#change to working dir
cd $title/
titlepath=$(pwd)

#RNAcop computation
./RNAcop --maxleft $maxl --maxright $maxr < $inf > $outf

#rank flanking regions
./rank_flanks.pl -i $outf -o $title

##create output for each flanking region
for f in $title*rddG.tsv
do
    pref=${f%.*}
    pref=${pref%_*}

    #convert to log10 prob
    convf=${f%_rddG.tsv}
    convf=flanks_${convf}_logprob.tsv
    ./convert_rddG2logProb.Rscript $f $convf

    #plot landscapes
    ./plot_mat_contour.Rscript $f pl_${pref}.ps 100 100 
    ./plot_mat_contour_png.Rscript $f pl_${pref}.png 100 100

    #identify good regions and sort output
    ./findgoodregion -i $f -o goodregion_${pref}.tsv
    ./sort_goodregion.Rscript goodregion_${pref}.tsv goodregion_${pref}.tsv
    
    l5prime=$(awk 'NR==2 {print $1}' goodregion_${pref}.tsv)
    l3prime=$(awk 'NR==2 {print $2}' goodregion_${pref}.tsv)
    tolerance=$(awk 'NR==2 {print $4}' goodregion_${pref}.tsv)
    logp=$(awk 'NR==2 {print $5}' goodregion_${pref}.tsv)
    
    #get optimal solution
    optl5prime=$(awk 'NR==2 {print $1}' flanks_${pref}_logprob.tsv)
    optl3prime=$(awk 'NR==2 {print $2}' flanks_${pref}_logprob.tsv)
    optlogp=$(awk 'NR==2 {print $3}' flanks_${pref}_logprob.tsv)

    ./flank_viennaconstraint.pl $inf $l5prime $l3prime > flanked_${title}.vienna

    goodseq=$(awk 'NR==2 {print $_}' flanked_${title}.vienna)
    goodcons=$(awk 'NR==3 {print $_}' flanked_${title}.vienna)

    #get sequences for all good regions
    RNACOP_CONSTRAINTINF=$inf;export RNACOP_CONSTRAINTINF;
 
    #add sequence to good regions
    echo "sequence" > column_seq.txt
    ruby -nae 'f=ENV["RNACOP_CONSTRAINTINF"];if($_ !~ /^l5prime/); then system("./flank_viennaconstraint.pl #{f} #{$F[0].to_i} #{$F[1].to_i} | sed -n 2p"); end' goodregion_${pref}.tsv >> column_seq.txt
    
    cp goodregion_${pref}.tsv tmp_goodregion_${pref}.tsv
    paste tmp_goodregion_${pref}.tsv column_seq.txt > goodregion_${pref}.tsv
    rm tmp_goodregion_${pref}.tsv

    #HTML output
    echo "<hr><th2>Optimal flanking regions</th2>" >> $odir/$title.html
    echo "<table border='1px solid black'>" >> $odir/$title.html
    echo "<tr><th>entry</th><th>length 5' (nt)</th><th>length 3' (nt)</th><th>log<sub>10</sub> P</th></tr>" >> $odir/$title.html
    echo "<tr><td>$pref</td><td>$optl5prime</td><td>$optl3prime</td><td>$optlogp</td></tr>" >> $odir/$title.html
    echo "</table><br><br>" >> $odir/$title.html
    echo "<th2>Suggested flanking regions</th2>" >> $odir/$title.html
    echo "<table border='1px solid black'>" >> $odir/$title.html
    echo "<tr><th>entry</th><th>length 5' (nt)</th><th>length 3' (nt)</th><th>tolerance (nt)</th><th>log<sub>10</sub> P</th></tr>" >> $odir/$title.html
    echo "<tr><td>$pref</td><td>$l5prime</td><td>$l3prime</td><td>$tolerance</td><td>$logp</td></tr>" >> $odir/$title.html
    echo "</table><br><br>" >> $odir/$title.html
    echo "<p>Suggested sequence:</p><textarea style='width: 300px; height: 100px;' readonly='readonly'>$goodseq</textarea>" >> $odir/$title.html
    echo "<p>Structure constraint:</p><textarea style='width: 300px; height: 100px;' readonly='readonly'>$goodcons</textarea><br><br>" >> $odir/$title.html
    echo "<p>Alternative suggestions: <a href='${titlepath}/goodregion_${pref}.tsv'>[TSV]</a></p>" >> $odir/$title.html
    echo "<p>All pairwise extensions: <a href='${titlepath}/$convf'>[TSV]</a></p><br>" >> $odir/$title.html
    echo "<p>Proabability landscape <a href='${titlepath}/pl_${pref}.ps'>[PS]</a> <a href='${titlepath}/pl_${pref}.png'>[PNG]</a>:<br></p><img src='${titlepath}/pl_${pref}.png' alt='Probability landscape' width=500px ><br>" >> $odir/$title.html    
done
echo "</html>" >> $title.html

cd $odir
