/* disjoint.c
 *
 * contains disjoint set operations using parent, rank and size array
 *
 * based on Cormen Introduction to Algorithms ...
 *
 *    Copyright (C) 2014  Nikolai Hecker
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdlib.h>

/* --------------------- make_set -------------------
 *
 * creates a new set
 *
 * arrays have to allocated
 */
void
make_set(const unsigned long id, unsigned long *parent, unsigned long *rank, unsigned long *size)
{
  parent[id] = id;
  rank[id] = 0;
  size[id] = 1;
}


/* --------------------- find_set -------------------
 *
 * finds set
 *
 * arrays have to allocated
 */
unsigned long
find_set(const unsigned long id, unsigned long *parent)
{
  unsigned long cur_id, rid, oid;

  cur_id = id;
  rid = id;
  oid = id;

  /* get root */
  while( cur_id != parent[cur_id])
    {      
      cur_id = parent[cur_id];
    }
  rid = cur_id;
  
  /* compress path */
  while( cur_id != parent[cur_id])
    {
      oid = cur_id;
      cur_id = parent[cur_id];
      parent[oid] = rid;
    }

  return(rid);
}


/* --------------------- link_sets -------------------
 *
 * links two sets
 *
 * arrays have to allocated
 */
void
link_sets(const unsigned long id_a, const unsigned long id_b, unsigned long *parent, unsigned long *rank, unsigned long *size)
{
  if(rank[id_a] > rank[id_b])
    {
      parent[id_b] = id_a;
      size[id_a] += size[id_b];
    }
  else
    {
       parent[id_a] = id_b;
       size[id_b] += size[id_a];
       
       if(rank[id_a] == rank[id_b])
	 {
	   rank[id_b] = rank[id_b] + 1;
	 }
    }
  
}


/* --------------------- union_sets -------------------
 *
 * combines two sets
 * arrays have to allocated
 *
 * returns 1 if sets were joined and 0 otherwise (already same set)
 */
int
union_sets(const unsigned long id_a, const unsigned long id_b, unsigned long *parent, unsigned long *rank, unsigned long *size)
{
  unsigned long rid_a, rid_b;
  
  rid_a = find_set(id_a, parent);
  rid_b = find_set(id_b, parent);
  
  if(rid_a != rid_b)
    {
      link_sets(rid_a, rid_b, parent, rank, size);
      return 1;
    }
  return 0;
}
