/* testfile.c
 *
  *    Copyright (C) 2014  Nikolai Hecker
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <float.h>
#include "datatypes.h"
#include "file.h"
#include <math.h>
#include "matrix.h"
#include <gsl/gsl_statistics.h>
#include <gsl/gsl_cdf.h>
#include <limits.h>
#include "plateau.h"
#include "utility.h"

/* ---------------------- main ------------------------------------------------- */
int
main(int argc, char *argv[])
{
  NMMatrix *m;
  unsigned long max_bfsize, i, j, length;
  unsigned long idxi, idxj, k;
  double *ring;
  double min, max, median, q, p;

  ring = NULL;
  m = NULL;
  max_bfsize = 10000;
  i = j = 0;
  length = 0;

  median = min = max = q = 0;
  p = 0.25;

  /* process input */
  if(argc < 5)
    {
      fprintf(stderr, "Usage:\n\t%s [INPUT-FILE] [i] [j] [k]\nToo few arguments.\n", argv[0]);
      exit(EXIT_FAILURE);
    }

  /* read input */
  printf("...reading input matrix.\n");
  m = read_rddG_mat(argv[1], max_bfsize);
  idxi = atoi(argv[2]);
  idxj = atoi(argv[3]);
  k = atoi(argv[4]);

  if(m == NULL)
    {
      fprintf(stderr, "Error while reading '%s'! Check file format.\n", argv[1]);
      exit(EXIT_FAILURE);
    }

  /* allocate ring array */
  if((ring = malloc( m->ncols * m->nrows * sizeof(ring)) ) == NULL)
    {
      fprintf(stderr, "Error! Mem allocation for distance ring failed.\n");
      exit(EXIT_FAILURE);
    }

  /* print matrix */
  printf("*** INPUT MATRIX [%lu x %lu] ***\n", m->nrows, m->ncols);
  printf("* (%lu, %lu) - distance %lu: %f *\n", idxi, idxj, k, m->e[idxi][idxj]);
  length = get_k_distance_ring(m, k, idxi, idxj, ring);

  for(i=0; i < length; i++)
    printf(" %f", ring[i]);
  printf("\n");
    
  qsort(ring, length, sizeof(ring[0]), *compare_double);
  printf("*** sorted ***");

  for(i=0; i < length; i++)
    printf(" %f", ring[i]);
  printf("\n");

  max = gsl_stats_max(ring, 1, length);
  min = gsl_stats_min(ring, 1, length);
  median = gsl_stats_median_from_sorted_data( ring, 1, length);
  q = gsl_stats_quantile_from_sorted_data( ring, 1, length, p);

  printf("\nmin: %f --- med: %f ---  max: %f\n", min, median, max);
  printf("\n%f-quantile: %f\n", p, q);

  printf("********************************\n");

  free(ring);
  nmm_free(m);

  exit(EXIT_SUCCESS);
}
