/*********************************************************************

  nogap.c

  usage: nogap [FILE]

  This program removes gaps from sequences. See also man page.

  000919 Bjarne Knudsen (bk@daimi.au.dk)

  Copyright (C) 2000 Bjarne Knudsen

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

*********************************************************************/

#include "clib/col.h"

typedef struct tagColno {
  int align_bp, alignpos;
  int seq_bp, seqpos;
  int label;
} Colno;

void rnanogap(Entry *entry, int lineno, void *colno);
void nogap(Entry *entry, int lineno, void *colno);

int main(int argc, char **argv)
{
  FILE *fp;
  Header *header;
  Entry *entry;
  Colno *colno;     /* Column numbers */
  int read_error;   /* for keeping track of errors in reading entries */
  char *file;

  colno = (Colno *) malloc(sizeof(Colno));

  file = NULL;

  if (argc == 2)
    file = argv[1];
  else if (argc != 1) {
    fprintf(stderr, "Usage: nogap [FILE]\n");
    return 1; }

  if (file == NULL)
    fp = stdin;
  else if ((fp = fopen(file, "r")) == NULL) {
    fprintf(stderr, "Error in opening file '%s'\n", file);
    return 1; }

  header = MakeHeader();
  entry = MakeEntry();

  if (ReadHeader(fp, header) != 0)
    return 1;

  AddHeaderInfo(header, argc, argv);

  PrintHeader(stdout, header);

  while ((read_error = ReadEntry(fp, entry)) == 0) {
    colno->label = ReadColno(entry, "label");
    colno->align_bp = ReadColno(entry, "align_bp");
    colno->alignpos = ReadColno(entry, "alignpos");
    colno->seq_bp = ReadColno(entry, "seq_bp");
    colno->seqpos = ReadColno(entry, "seqpos");
    if (colno->label == 0) {
      fprintf(stderr, "nogap: Warning: Ignoring sequence,"
	              " insufficient column info\n");
      PrintEntry(stdout, entry);
      continue; }
    if (ReadType(entry, "RNA") && colno->seq_bp != 0) {
      RmvCol(entry, "align_bp");
      EntryMap(entry, &nogap, (void *)colno);
    }
    else if (ReadType(entry, "RNA") &&
	     (colno->align_bp != 0 &&
	      colno->seqpos != 0 &&
	      colno->alignpos != 0)) {
      EntryMap(entry, &rnanogap, (void *)colno);
      ChgCollabel(entry, colno->align_bp, "seq_bp");
    }
    else
      EntryMap(entry, &nogap, (void *)colno);

    if (colno->alignpos != 0)
      RmvCol(entry, "alignpos");

    PrintEntry(stdout, entry);
  }

  if (file != NULL && fclose(fp) != 0) {
    fprintf(stderr, "nogap: error in closing file '%s'\n", file);
    return 1; }

  if (read_error == 1)
    return 1;

  return 0;
}

void nogap(Entry *entry, int lineno, void *colno)
{
  char field[MAXCOLW];

  GetField(field, entry, lineno, ((Colno *)colno)->label);

  if (field[0] == 'G')    /* A gap is here */
    RmvLine(entry, lineno);
}

void rnanogap(Entry *entry, int lineno, void *colno)
{
  int pair;
  char field[MAXCOLW];

  pair = FindPair(entry, lineno, ((Colno *)colno)->align_bp,
		  ((Colno *)colno)->alignpos,
		  ((Colno *)colno)->seq_bp,
		  ((Colno *)colno)->seqpos);

  if (pair != 0) {  /* A pair is present */
    GetField(field, entry, pair, ((Colno *)colno)->seqpos);
    ChgField(entry, lineno, ((Colno *)colno)->align_bp, "%s", field);
  }

  GetField(field, entry, lineno, ((Colno *)colno)->label);

  if (field[0] == 'G')    /* A gap is here */
    RmvLine(entry, lineno);
}
