/*********************************************************************

  unknown.c

  usage: unknown
              [-e | --extended]  [-C<color_r>,<color_g>,<color_b> |
              --color=<color_r>,<color_g>,<color_b>] [<file>]

  See man page for more info

  000630 Bjarne Knudsen (bk@daimi.au.dk)

  Copyright (C) 2000 Bjarne Knudsen

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

*********************************************************************/

#include "clib/col.h"
#include "clib/rna.h"
#include "clib/dna.h"
#include "clib/protein.h"

typedef struct tagColno {
  int symbol;
  int red, green, blue;
  int red2, green2, blue2;
} Colno;

typedef struct tagColor {
  double r, g, b;
} Color;

void usage(void);
void colorpos(Entry *entry, Colno *colno, Color *color, Color *fgcolor,
	      int (*check)(char));

int main(int argc, char **argv)
{
  FILE *fp;
  Header *header;
  Entry *entry;
  Colno *colno;
  int read_error;   /* for keeping track of errors in reading entries */
  unsigned int len; /* Used with -C option */
  CmdArg *cmdarg;   /* Command line arguments */
  char *s;          /* String for arguments */
  Color *color;     /* The colour of unknown symbols pairs */
  Color *fgcolor;   /* The foreground colour of unknown symbols pairs */
  int option_extended;  /* Use the extended alphabet? */
  int (*check)(char);   /* The function used for checking symbols */
  int option_c;
  int option_fg;

  colno = (Colno *) malloc(sizeof(Colno));
  color = (Color *) malloc(sizeof(Color));
  fgcolor = (Color *) malloc(sizeof(Color));

  /* default options */
  option_c = 0;
  color->r = 0.0;
  color->g = 1.0;
  color->b = 0.0;

  option_fg = 0;
  fgcolor->r = 0.0;
  fgcolor->g = 1.0;
  fgcolor->b = 0.0;

  option_extended = 0;

  cmdarg = InitArgument(argc, argv);

  while ((s = GetArgument(cmdarg)) != NULL)
    if (strcmp(s, "c") == 0)
      option_c = 1;
    else if (strcmp(s, "-color") == 0)
      option_c = 1;
    else if (strcmp(s, "f") == 0)
      option_fg = 1;
    else if (strcmp(s, "-fgcolor") == 0)
      option_fg = 1;
    else if (strcmp(s, "e") == 0 ||
	strcmp(s, "-extended") == 0)
      option_extended = 1;
    else if (strncmp(s, "C", 1) == 0) {
      option_c = 1;
      if (sscanf(&s[1], "%lf,%lf,%lf%n",
		 &color->r, &color->g, &color->b, &len) != 3 ||
	  len+1 != strlen(s)) {
	usage();
	return 1; }
    }
    else if (strncmp(s, "-color=", 7) == 0) {
      option_c = 1;
      if (sscanf(&s[7], "%lf,%lf,%lf%n",
		 &color->r, &color->g, &color->b, &len) != 3 ||
	  len+7 != strlen(s)) {
	usage();
	return 1; }
    }
    else if (strncmp(s, "F", 1) == 0) {
      option_fg = 1;
      if (sscanf(&s[1], "%lf,%lf,%lf%n",
		 &fgcolor->r, &fgcolor->g, &fgcolor->b, &len) != 3 ||
	  len+1 != strlen(s)) {
	usage();
	return 1; }
    }
    else if (strncmp(s, "-fgcolor=", 9) == 0) {
      option_fg = 1;
      if (sscanf(&s[9], "%lf,%lf,%lf%n",
		 &fgcolor->r, &fgcolor->g, &fgcolor->b, &len) != 3 ||
	  len+9 != strlen(s)) {
	usage();
	return 1; }
    }
    else {
      usage();
      return 1; }

  if (option_c == 0 && option_fg == 0)
    option_c = 1;

  if ((s = GetFilename(cmdarg)) == NULL)
    fp = stdin;
  else if (GetFilename(cmdarg) != NULL) {
    usage();
    return 1; }
  else if ((fp = fopen(s, "r")) == NULL) {
    fprintf(stderr, "unknown: Error in opening file '%s'\n", s);
    return 1; }

  header = MakeHeader();
  entry = MakeEntry();

  if (ReadHeader(fp, header) != 0)
    return 1;

  AddHeaderInfo(header, argc, argv);

  PrintHeader(stdout, header);

  while ((read_error = ReadEntry(fp, entry)) == 0) {
    if (ReadType(entry, "RNA")) {
      if (option_extended == 0)
	check = &IsRNAnuc;
      else
	check = &IsRNAnucExt;
    }
    else if (ReadType(entry, "DNA")) {
      if (option_extended == 0)
	check = &IsDNAnuc;
      else
	check = &IsDNAnucExt;
    }
    else if (ReadType(entry, "protein")) {
      if (option_extended == 0)
	check = &IsAmino;
      else
	check = &IsAminoExt;
    }
    else {
      PrintEntry(stdout, entry);
      continue;
    }

    if ((colno->symbol = ReadColno(entry, "residue")) == 0 &&
	(colno->symbol = ReadColno(entry, "nucleotide")) == 0 &&
	(colno->symbol = ReadColno(entry, "aminoacid")) == 0 &&
	(colno->symbol = ReadColno(entry, "code")) == 0) {
      fprintf(stderr, "unknown: Warning: Ignoring sequence, "
	                 "insufficient column info\n");
      PrintEntry(stdout, entry);
      continue;
    }

    colno->red = colno->red2 = 0;
    if (option_c == 1) {
      colno->red = EnsureCol(entry, "color_r", "1.000");
      colno->green = EnsureCol(entry, "color_g", "1.000");
      colno->blue = EnsureCol(entry, "color_b", "1.000");
    }
    if (option_fg == 1) {
      colno->red2 = EnsureCol(entry, "color2_r", "0.000");
      colno->green2 = EnsureCol(entry, "color2_g", "0.000");
      colno->blue2 = EnsureCol(entry, "color2_b", "0.000");
    }
    colorpos(entry, colno, color, fgcolor, check);
    PrintEntry(stdout, entry);
  }

  if (fp != stdin && fclose(fp) != 0) {
    fprintf(stderr, "unknown: Error in closing file\n");
    return 1; }

  if (read_error == 1)
    return 1;

  return 0;
}

void usage(void)
{
  fprintf(stderr,
	  "usage: unknown\n"
	  "            [-e | --extended]  [-C<color_r>,<color_g>,<color_b> |\n"
	  "            --color=<color_r>,<color_g>,<color_b>] [<file>]\n");
}

/* check is the function used for checking the symbols */
void colorpos(Entry *entry, Colno *colno, Color *color, Color *fgcolor,
	      int (*check)(char))
{
  char field[MAXCOLW];
  int i;
  int len;

  len = EntryLength(entry);

  for (i = 1; i <= len; i++) {
    GetField(field, entry, i, colno->symbol);
    if (!(*check)(field[0])) {
      if (colno->red != 0) {
	ChgField(entry, i, colno->red, "%5.3f", color->r);
	ChgField(entry, i, colno->green, "%5.3f", color->g);
	ChgField(entry, i, colno->blue, "%5.3f", color->b);
      }
      if (colno->red2 != 0) {
	ChgField(entry, i, colno->red2, "%5.3f", fgcolor->r);
	ChgField(entry, i, colno->green2, "%5.3f", fgcolor->g);
	ChgField(entry, i, colno->blue2, "%5.3f", fgcolor->b);
      }
    }
  }
}
