/*********************************************************************

  col2line.c

  usage: col2line [FILE]

  This program transforms a col file to another format, which is more
  compact, and easier for gzip to pack.

  000125 Bjarne Knudsen (bk@daimi.au.dk)

  Copyright (C) 2000 Bjarne Knudsen

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

*********************************************************************/

#include "clib/col.h"
#include "clib/file.h"

int main(int argc, char **argv)
{
  FILE *fp;         /* For input */
  int sym_col, seqpos_col, alignpos_col;
  Header *header;
  Entry *entry;
  int read_error;   /* For keeping track of errors in reading entries */
  char *s;
  CmdArg *cmdarg;   /* Command line arguments */
  int numf;         /* Number of fields */
  int width;        /* Field widths */
  char fmt[20];
  char field[MAXCOLW];
  int i, j;
  int len;
  int ptr;
  int seqpos;

  cmdarg = InitArgument(argc, argv);

  if ((s = GetArgument(cmdarg)) != NULL) {
    fprintf(stderr, "Usage: col2line [FILE]\n");
    return 1; }

  if ((s = GetFilename(cmdarg)) == NULL)
    fp = stdin;
  else if (GetFilename(cmdarg) != NULL) {
    fprintf(stderr, "Usage: col2line [FILE]\n");
    return 1; }
  else if ((fp = fopen(s, "r")) == NULL) {
    fprintf(stderr, "col2line: Error in opening file '%s'\n", s);
    return 1; }

  header = MakeHeader();
  entry = MakeEntry();

  if (ReadHeader(fp, header) != 0)
    return 1;
  PrintHeader(stdout, header);

  while ((read_error = ReadEntry(fp, entry)) == 0) {
    if ((sym_col = ReadColno(entry, "residue")) != 0 ||
	(sym_col = ReadColno(entry, "nucleotide")) != 0 ||
	(sym_col = ReadColno(entry, "aminoacid")) != 0 ||
	(sym_col = ReadColno(entry, "code")) != 0)
      seqpos_col = ReadColno(entry, "seqpos");
    else
      seqpos_col = 0;
    alignpos_col = ReadColno(entry, "alignpos");

    PrintEntryText(stdout, entry);
    numf = CountField(entry, 1);
    len = EntryLength(entry);
    ptr = 0;
    for (i = 1; i <= numf; i++) {
      s = LineAddress(entry, 1);
      sscanf(&s[ptr], " %s%n", field, &width);
      ptr += width;
      printf("%d", width);
      if (i == seqpos_col) {
	/* Check seqpos column */
	seqpos = 0;
	for (j = 1; j <= len; j++) {
	  GetField(field, entry, j, sym_col);
	  if (strcmp(field, "-") != 0) {
	    seqpos++;
	    GetField(field, entry, j, seqpos_col);
	    if (atoi(field) != seqpos)
	      break;
	  }
	  else {
	    GetField(field, entry, j, seqpos_col);
	    if (strcmp(field, ".") != 0)
	      break;
	  }
	}
	if (j == len+1) {
	  printf("\n");
	  continue;  /* seqpos ok */
	}
      }
      else if (i == alignpos_col) {
	for (j = 1; j <= len; j++) {
	  GetField(field, entry, j, alignpos_col);
	  if (atoi(field) != j)
	    break;
	}
	if (j == len+1) {
	  printf("\n");
	  continue;  /* alignpos ok */
	}
      }
      width = 0;
      for (j = 1; j <= len; j++) {
	GetField(field, entry, j, i);
	if (strlen(field) > width)
	  width = strlen(field);
      }
      sprintf(fmt, " %%%ds", width);
      for (j = 1; j <= len; j++) {
	GetField(field, entry, j, i);
	printf(" %s", field);
      }
      printf("\n");
    }
    PrintEntryEnd(stdout, entry);
  }


  if (fp != stdin && fclose(fp) != 0) {
    fprintf(stderr, "col2line: Error in closing file\n");
    return 1; }

  if (read_error == 1)
    return 1;

  return 0;
}
