/*
 *  SARSE, Semi-Automated RNA Sequence Editor.
 *  Copyright (C) 2004 Allan Lind-Thomsen
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dk.kvl.alignmenttools;

import dk.kvl.controller.*;
import dk.kvl.gui.properties.*;
import java.util.*;
import dk.kvl.tools.*;
import dk.kvl.tools.io.*;
import javax.swing.*;
import java.io.*;
import dk.kvl.alignmenttools.io.*;
import java.awt.event.*;
import dk.kvl.gui.alignment.*;
import dk.kvl.gui.event.ProgramEvent;

import java.awt.*;

/**
 *  Description of the Class
 *
 * @author     allan
 * @created    February 12, 2004
 */
public class Analyzer extends Thread {
	
  private final Vector programs;
  private final Alignment alignment;
  private final AlignmentProperties properties;
  private ProjectProperties pProperties;
  private final String id;
  private BufferedWriter logWriter;
  private Log log;
  private File resultFile;
  private String projectName;
  private File finalResult;
  private ActionListener listener;
  private boolean error, pfold;
  private Program findphyl;


  /**
   *  Constructor for the Analyzer object
   *
   * @param  selectedPrograms  Description of the Parameter
   * @param  alignment         Description of the Parameter
   * @param  props             Description of the Parameter
   * @param  list              Description of the Parameter
   * @param  project           Description of the Parameter
   * @param  newId             Description of the Parameter
   * @param  pProperties       Description of the Parameter
   */
  public Analyzer(Vector selectedPrograms, Alignment alignment, AlignmentProperties props, ProjectProperties pProperties, ActionListener list, String project, String newId) {
    super(AlignmentFrame.sarseThreads, newId);
    
    programs = selectedPrograms;
    projectName = project;
    properties = props;
    this.pProperties = pProperties;
    this.alignment = alignment;
    schedulePrograms();
    id = newId;
    createLog();
    if (validProgramChain()) {
      listener = list;
      start();
    } else {
      JOptionPane.showMessageDialog(null, "Broken program chain, incompatible formats!");
    }
  }


  /**
   *  Gets the id attribute of the Analyzer object
   *
   * @return    The id value
   */
  public String getAnalyzerId() {
    return id;
  }


  /**
   *  Main processing method for the Analyzer object
   *
   * @return    The alignment value
   */
  public Alignment getAlignment() {
    return alignment;
  }


  /**
   *  Main processing method for the Analyzer object
   */
  
  public void run() {
    int counter = 1;
    try {
      finalResult = new File(pProperties.getPath() + "/result-" + id + ".col");
      //initialize data- and result files
      String ppFile = properties.getTmpDir() + "dotplot-" + id + ".pp";
      File dataFile = File.createTempFile("alignment-" + id, ".col", new File(pProperties.getPath()));
      resultFile = null;
      File ps = null;
      //
      new ColWriter(dataFile, alignment, true, false);
      File originalFile = dataFile;
      //resultfile
      resultFile = new File(properties.getTmpDir() + "/result-" + counter++ + "-" + id);

      //there must be a new file for the results
      if (resultFile.exists()) {
        resultFile.delete();
      }
      for (int i = 0; i < programs.size(); i++) {
        Program p = (Program) programs.get(i);
        if (p.getName().equals("cluster.tcsh")) {
          File clusterFile = new File(pProperties.getPath() + "/clusterfiles" + id);
          if (!clusterFile.exists()) {
            clusterFile.mkdir();
          }
          clusterFile = new File(clusterFile + "/" + pProperties.getFile());
          clusterFile.createNewFile();

          new ColWriter(clusterFile, alignment, true, false);
          p.execute(clusterFile, log);
          clusterFile.renameTo(resultFile);
          resultFile = clusterFile;
        } else {
          resultFile.createNewFile();
          if (p.getName().equals("findphyl")) {
            pfold = true;
            findphyl = p; 

            //dataFile = runDiscPmask(dataFile, log);
          }
          p.execute(dataFile, resultFile, log, ppFile);
        }
        if (p.hasErrorMessage()) {
          String progName = p.getName();
          String[] options = {"Yes", "No", "View log"};
          int selected = JOptionPane.showOptionDialog(null,
              "An error occured executing " + progName + " which might cause your analysis to be corrupted.\n Refer to the log to view error messages. \n Do you want to continue?",
              "Warning",
              JOptionPane.OK_OPTION,
              JOptionPane.QUESTION_MESSAGE,
              null,
              options,
              options[0]);
          if (options[selected].equals("No")) {
            throw new Exception("Execution aborted");
          } else if (options[selected].equals("View log")) {
            JTextArea jta = new JTextArea(log.readLog(), 10, 30);

            JScrollPane jsp = new JScrollPane(jta);
            jsp.setHorizontalScrollBarPolicy(jsp.HORIZONTAL_SCROLLBAR_ALWAYS);
            jsp.setVerticalScrollBarPolicy(jsp.VERTICAL_SCROLLBAR_ALWAYS);
            String[] opts = {"Continue", "Stop execution"};
            int ch = JOptionPane.showOptionDialog(null,
                jsp,
                "Log message",
                JOptionPane.OK_OPTION,
                JOptionPane.QUESTION_MESSAGE,
                null,
                opts,
                opts[0]);
            if (opts[ch].equals("Stop execution")) {
              throw new Exception("Execution aborted");
            }
          }
        }
        Vector output = ((Program) programs.get(i)).getOutputTypes();
        String pName = ((Program) programs.get(i)).getName();
        if (i < programs.size() - 1) {
          if (resultFile.length() == 0) {
            if (!pName.equals("cluster.tcsh")) {
              JOptionPane.showMessageDialog(null, "Error running " + p.getName() + ". Analysis aborted.");
              removeFiles();
              return;
            }
          } else {
            if (((String) output.get(0)).equals("ps")) {
              ps = new File(pProperties.getPath() + "/" + p.getName() + "-" + id + ".ps");
              resultFile.renameTo(ps);
            } else {
              dataFile = resultFile;
              resultFile = new File(properties.getTmpDir() + "/result-" + counter++ + "-" + id);
            }
          }
        } else {
          if (resultFile.length() == 0) {
            if (p.getName().equals("cluster.tcsh")) {
              JOptionPane.showMessageDialog(null, pName + " has finished \n Output files are placed here:\n " + pProperties.getPath() + "/clusterfiles" + id + "/");
              return;
            } else {
              JOptionPane.showMessageDialog(null, "Error running " + p.getName() + ". Analysis aborted.");
              removeFiles();
              return;
            }
          } else {
            if (((String) output.get(0)).equals("ps")) {
              ps = new File(pProperties.getPath() + "/" + p.getName() + "-" + id + ".ps");
              resultFile.renameTo(ps);
              if (pfold) {
                runParNum(dataFile, log);
              } else {
                dataFile.renameTo(finalResult);
              }
              //dataFile.renameTo(finalResult);
            } else {
              if (pfold) {
                runParNum(resultFile, log);
              } else {
                resultFile.renameTo(finalResult);
              }
            }
          }
        }
      }
      updateProjectFile();
      if (listener != null) {
    	  ProgramEvent pe = new ProgramEvent(this, 0, finalResult.getName() + ", ID: " + id);
    	  pe.setResultfile(finalResult.getName());
    	  pe.addFiles(getFiles());
        listener.actionPerformed(pe);
      }
    } catch (Exception e) {
      error = true;
      e.printStackTrace();
      if (listener != null) {
    	  ProgramEvent pe = new ProgramEvent(this, 1, finalResult.getName() + ", ID: " + id);
    	  pe.setResultfile(finalResult.getName());
    	  pe.addFiles(getFiles());
        listener.actionPerformed(pe);
      }
    }
    removeFiles();
    ((AlignmentContainer)listener).setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
    listener = null;
  }


  /**
   *  Description of the Method
   *
   * @param  dataFile  Description of the Parameter
   * @param  log       Description of the Parameter
   * @return           Description of the Return Value
   */
  public File runDiscPmask(File dataFile, Log log) {
    File discmask = new File(pProperties.getPath() + "/discmask.col");

    try {
      if (discmask.exists()) {
        discmask.delete();
      }
      discmask.createNewFile();
      String[] cmds = {"discpmask", dataFile.getAbsolutePath()};
      Runtime runtime = Runtime.getRuntime();
      Process p = runtime.exec(cmds);

      log.writeLine("---------discpmask started---------------");

      //initialize readers and writers
      BufferedReader resultReader = new BufferedReader(new InputStreamReader(p.getInputStream()));
      BufferedWriter resultWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(discmask)));
      BufferedReader errorReader = new BufferedReader(new InputStreamReader(p.getErrorStream()));

      //execute error reading and data reading in two threads to avoid blocking
      StreamReaderThread srt = new StreamReaderThread(resultReader, resultWriter);
      ErrorReaderThread ert = new ErrorReaderThread(errorReader, log, findphyl);

      srt.start();
      ert.start();

      p.waitFor();

      srt.join();
      ert.join();

      log.writeLine("--------- discpmask finished---------------");
      log.flush();
      resultWriter.flush();
      resultReader.close();
      resultWriter.close();
      errorReader.close();

      dataFile = discmask;
    } catch (Exception e) {
      JOptionPane.showMessageDialog(null, "Error removing pairingmask from file");
      e.printStackTrace();
    }
    return discmask;
  }


  /**
   *  Description of the Method
   *
   * @param  dataFile  Description of the Parameter
   * @param  log       Description of the Parameter
   */
  public void runParNum(File dataFile, Log log) {
    try {
      ArrayList allCmds = new ArrayList();
      //String name = finalResult.getAbsolutePath();
      //finalResult = new File(name.substring(0, name.length() - 4) + ".txt");
      finalResult.createNewFile();
      String[] cmds = {"addpmasknum3", dataFile.getAbsolutePath()};
      allCmds.add(cmds);

      for (int i = 0; i < allCmds.size(); i++) {
        String[] cmd = (String[]) allCmds.get(i);
        String proName = cmd[0];

        log.writeLine("---------" + proName + " started---------------");
        Runtime runtime = Runtime.getRuntime();
        Process p = runtime.exec(cmd);

        //initialize readers and writers
        BufferedReader resultReader = new BufferedReader(new InputStreamReader(p.getInputStream()));
        BufferedWriter resultWriter = null;
        //if (proName.equals("addpmasknum")) {
          resultWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(finalResult)));
        /*} else {
          if (proName.equals("col2txt")) {
            resultWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(col2txt)));
          } else {
            if (proName.equals("../cluster/par2num")) {
              resultWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(finalResult)));
            }*/
          
    
        BufferedReader errorReader = new BufferedReader(new InputStreamReader(p.getErrorStream()));

        //execute error reading and data reading in two threads to avoid blocking
        StreamReaderThread srt = new StreamReaderThread(resultReader, resultWriter);
        ErrorReaderThread ert = new ErrorReaderThread(errorReader, log, findphyl);

        srt.start();
        ert.start();

        p.waitFor();

        srt.join();
        ert.join();

        log.writeLine("---------" + proName + " finished---------------");
        log.flush();
        resultWriter.flush();
        resultReader.close();
        resultWriter.close();
        errorReader.close();
      }
      
    } catch (Exception e) {
      JOptionPane.showMessageDialog(null, "An error occurred running addp,masknum");
      e.printStackTrace();
    }
  }


  /**
   *  Description of the Method
   */
  public void removeFiles() {
    File[] tmpFiles = new File(properties.getTmpDir()).listFiles();
    for (int i = 0; i < tmpFiles.length; i++) {
      tmpFiles[i].delete();
    }
  }


  /**
   *  This method sort the programs to run in a specific order it is primary
   *  to be sure that all programs recieve a format they understand, and to be
   *  sure no information is lost in the pipeline
   */
  public void schedulePrograms() {
    Collections.sort(programs);
  }


  /**
   *  Gets the resultFile attribute of the Analyzer object
   *
   * @return    The resultFile value
   */
  public File getResultFile() {
    return resultFile;
  }


  /**
   *  Description of the Method
   *
   * @return    Description of the Return Value
   */
  protected boolean validProgramChain() {
    /*
     *  for (int i = 0; i < programs.size(); i++)
     *  {
     *  if (i == 0 && ((Program)programs.get(i)).getPriority() > 2)
     *  {
     *  return false;
     *  }
     *  else if (i > 0 && ((Program)programs.get(i)).getPriority() != ((Program)programs.get(i - 1)).getPriority() && ((Program)programs.get(i)).getPriority() != ((Program)programs.get(i - 1)).getPriority() + 1)
     *  {
     *  return false;
     *  }
     *  }
     */
    return true;
  }


  /**
   *  Description of the Method
   */
  public void createLog() {
    try {
      File logPath = new File(properties.getLogDir());
      if (!logPath.exists()) {
        logPath.mkdir();
      }

      File logFile = new File(logPath.getAbsolutePath() + "/log-" + id + ".log");
      if (!logFile.exists()) {
        logFile.createNewFile();
      }
      log = new Log(logFile, logPath);
      Program p;
      StringBuffer cmdLine;
      for (int i = 0; i < programs.size(); i++) {
        cmdLine = new StringBuffer();
        p = (Program) programs.get(i);
        cmdLine.append(p.getName());
        for (int j = 0; j < p.getSelectedParameters().size(); j++) {
          cmdLine.append(" " + p.getSelectedParameters().get(j));
        }
        log.writeLine(cmdLine.toString());

      }
    } catch (Exception e) {
      e.printStackTrace();
    }
  }


  /**
   *  Description of the Method
   */
  public void updateProjectFile() {
    try {
      String path = pProperties.getPath();
      ProjectProperties prop = new ProjectProperties(projectName + ".prj");
      prop.setFile(finalResult.getName());
      prop.setPath(path);
      prop.store();
    } catch (Exception e) {
      e.printStackTrace();
    }
  }
  
  public Vector getFiles()
  {
	  Vector files = new Vector();
	  for(int i = 0;i<programs.size();i++)
	  {
		  if(((Program)programs.get(i)).getFiles().size() > 0 )
		  {
			    files.add(((Program)programs.get(i)).getFiles());
		  }
	  }
	  return files;
  }
}

