/*
 *  SARSE, Semi-Automated RNA Sequence Editor. Copyright (C) 2004 Allan
 *  Lind-Thomsen This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version. This program is distributed in the hope that it
 *  will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 *  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *  Public License for more details. You should have received a copy of the GNU
 *  General Public License along with this program; if not, write to the Free
 *  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 *  02111-1307, USA.
 */
package dk.kvl.alignmenttools.io;

import java.util.*;
import java.io.*;
import dk.kvl.alignmenttools.*;
import dk.kvl.alignmenttools.io.*;
import dk.kvl.sequencetools.*;
import javax.swing.*;
import java.awt.*;

/**
 *  Description of the Class
 *
 * @author     allan
 * @created    March 12, 2004
 */
public class ColWriter extends BufferedWriter implements AlignmentWriter
{
    private final Alignment alignment;
    private boolean colors = true;
    private boolean pairingmasks = false;


    /**
     *  Constructor for the ColWriter object
     *
     * @param  outFile          Description of the Parameter
     * @param  alignment        Description of the Parameter
     * @param  preserveColors   Description of the Parameter
     * @param  allPairingMasks  Description of the Parameter
     * @exception  IOException  Description of the Exception
     */
    public ColWriter(File outFile, Alignment alignment, boolean preserveColors, boolean allPairingMasks) throws IOException
    {
        super(new FileWriter(outFile));
        this.alignment = alignment;
        pairingmasks = allPairingMasks;
        colors = preserveColors;
        writeAlignmentHeader();
        writeSequences();
        flush();
    }


    /**
     *  Constructor for the ColWriter object
     *
     * @param  outFile          Description of the Parameter
     * @param  alignment        Description of the Parameter
     * @exception  IOException  Description of the Exception
     */
    public ColWriter(File outFile, Alignment alignment) throws IOException
    {
        this(outFile, alignment, true, true);
    }


    /**
     *  Description of the Method
     *
     * @exception  IOException  Description of the Exception
     */
    protected void writeAlignmentHeader() throws IOException
    {
        try
        {
            char[] info;
            String inf = alignment.getInformation().trim();
            if (inf != null && inf.length() > 0)
            {
                info = inf.toCharArray();
                if (info[0] != ';')
                {
                    write(';');
                    write(' ');
                }
                for (int i = 0; i < info.length; i++)
                {
                    write(info[i]);
                    if (info[i] == '\r' || info[i] == '\n' || info[i] == '\f')
                    {
                        write(';');
                        write(' ');
                    }
                }
            }
            String sep = "\n; ==========\n";
            write(sep, 0, sep.length());
        }
        catch (Throwable e)
        {
            e.printStackTrace();
        }
    }


    /**
     *  Description of the Method
     *
     * @exception  IOException  Description of the Exception
     */
    protected void writeSequences() throws IOException
    {
        if (pairingmasks)
        {
            Enumeration e = alignment.getPairingMaskKeys();
            while (e.hasMoreElements())
            {
                //writePairingmask(alignment.getPairingMask((String)e.nextElement()));
            	Sequence seq =alignment.getPairingMask((String)e.nextElement()); 
            	writeSequenceHeader(seq);
            	writeSequence(seq);
            }
        }
        else
        {
            if (alignment.getPairingMask() != null)
            {
                //writePairingmask(alignment.getPairingMask());
            	Sequence seq = alignment.getPairingMask(); 
            	writeSequenceHeader(seq);
                writeSequence(seq);
            }
        }
        int oldSort = alignment.getSort();
        alignment.setSort(Alignment.ORIGINALSORT);
        String[] keys = alignment.getSortedNames();
        for (int i = 0; i < keys.length; i++)
        {
            writeSequence(keys[i]);
        }
        alignment.setSort(oldSort);
    }


    /**
     *  Description of the Method
     *
     * @param  name  Description of the Parameter
     */
    protected void writeSequence(String name)
    {
        Sequence seq = alignment.getSequence(name);
        try
        {
            writeSequenceHeader(seq);
            writeSequence(seq);
        }
        catch (IOException e)
        {
            e.printStackTrace();
            JOptionPane.showMessageDialog(null, "Exception thrown: ColWriter:writeSequence()");
        }
    }


    /**
     *  Description of the Method
     *
     * @param  seq              Description of the Parameter
     * @exception  IOException  Description of the Exception
     */
    public void writeSequenceHeader(Sequence seq) throws IOException
    {
        String type = "; TYPE  " + seq.getAlphabet(seq.getType()).getLabel() + "\n";
        write(type, 0, type.length());
        //always write these columns
        String column1 = "; COL 1 label\n";
        write(column1, 0, column1.length());
        String column2 = "; COL 2 residue\n";
        write(column2, 0, column2.length());
        String column3 = "; COL 3 seqpos\n";
        write(column3, 0, column3.length());
        String column4 = "; COL 4 alignpos\n";
        write(column4, 0, column4.length());
        int columnNr = 5;
        //align_bp if pairingmask exist
        if (seq.getPairing(0) != -2)
        {
            String column5 = "; COL 5  align_bp\n";
            write(column5, 0, column5.length());
            columnNr++;
        }
        //other optional columns
        String column;
        String[] colLabels = seq.getColumnLabels();
        for(int i = 0; i<colLabels.length;i++)
        {
            column = "; COL "+columnNr+"    "+colLabels[i]+"\n";
            write(column, 0, column.length());
            columnNr++;
        }
            if (seq.isBackgroundColred())
            {
                column = "; COL " + (columnNr++) + "   color_r\n";
                write(column, 0, column.length());
                column = "; COL " + (columnNr++) + "   color_g\n";
                write(column, 0, column.length());
                column = "; COL " + (columnNr++) + "   color_b\n";
                write(column, 0, column.length());
            }
            if (seq.isForegroundColred())
            {
                column = "; COL " + (columnNr++) + "   color2_r\n";
                write(column, 0, column.length());
                column = "; COL " + (columnNr++) + "   color2_g\n";
                write(column, 0, column.length());
                column = "; COL " + (columnNr++) + "   color2_b\n";
                write(column, 0, column.length());
            }
        
        String entry = "; ENTRY  " + seq.getLabel() + "\n";
        write(entry, 0, entry.length());
        //write(seq.getInformation(), 0, seq.getInformation().length());
        try
        {
            char[] info;
            String inf = seq.getInformation().trim();
            if (inf != null && inf.length() > 0)
            {
                info = inf.toCharArray();

                if (info[0] != ';')
                {
                    write(';');
                    write(' ');
                }
                for (int i = 0; i < info.length; i++)
                {
                    write(info[i]);
                    if (info[i] == '\r' || info[i] == '\n' || info[i] == '\f')
                    {
                        write(';');
                        write(' ');
                    }
                }
                if (info[info.length - 1] != '\r' || info[info.length - 1] != '\n' || info[info.length - 1] != '\f')
                {
                    write('\n');
                }
            }
        }
        catch (Throwable e)
        {
            e.printStackTrace();
        }

        write("; ----------\n", 0, 13);
    }


    /**
     *  Description of the Method
     *
     * @param  seq              Description of the Parameter
     * @exception  IOException  Description of the Exception
     */
    public void writeSequence(Sequence seq) throws IOException
    {
        int seqpos = 1;
//      for(int i = 0;i<seq.getPairings().length;i++)
//      {
//      System.out.print(seq.getPairing(i)+",");
//      }
//      System.out.print("\n");

        for (int i = 0; i < seq.getLength(); i++)
        {
            String col2 = seq.getSymbolAt(i) + "";
            String col1;
            String col3;
            if (col2.equals("-"))
            {
                col1 = "G";
                col3 = ".";
            }
            else
            {
                col1 = "N";
                col3 = "" + seqpos;
                seqpos++;
            }
            String col4 = "" + (i + 1);
            StringBuffer line = new StringBuffer(col1 + "    " + col2 + "    " + col3 + "    " + col4);
            String col5 = null;
//            if (seq.getPairing(i) != -2)
//            {
//            int pair = seq.getPairing(i);
//            //System.out.println((i+1)+" : "+(pair+1));
//            	//int pair = alignment.getPairingMask().getPairing(i);
//            int columnNr = 5;
//            
//            if (pair != -2 && pair != -1)
//            {
//                
//                	col5 = "" + (pair+1);
//                
//            }
//            else if(pair == -1 || pair == -2)
//            {
//                col5 = ".";
//                columnNr = 6;
//            }
//            }
            if (seq.getPairing(i) > 0)
            {
            	col5  = "" + (seq.getPairing(i)+1);
            }
            else if(seq.getPairing(0)!=-2)
            {
            	col5 = ".";
            }
            if(col5 != null)
            {
                line.append("    " + col5);
            }
            
            //optional columns
            String[] cols = seq.getColumnInfo(i);
            for(int j = 0; j<seq.getColumnLabels().length;j++)
            {
               line.append("    "+cols[j]);
                
            }
                if (seq.isBackgroundColred())
                {
                    float[] b = seq.getBackgroundColor(i).getRGBColorComponents(null);
                    if (b.length == 3)
                    {
                        line.append("   " + b[0] + "   " + b[1] + "   " + b[2]);
                    }
                }
                if (seq.isForegroundColred())
                {
                    float[] b = seq.getForegroundColor(i).getRGBColorComponents(null);
                    if (b.length == 3)
                    {
                        line.append("   " + b[0] + "   " + b[1] + "   " + b[2]);
                    }
                }
            
            line.append("\n");
            write(line.toString(), 0, line.length());
        }
        write("; **********\n", 0, 13);
    }


    /**
     *  Description of the Method
     *
     * @param  mask             Description of the Parameter
     * @exception  IOException  Description of the Exception
     */
    public void writePairingmask(PairingMask mask) throws IOException
    {
        String type = "; TYPE pairingmask\n";
        write(type, 0, type.length());
        String column1 = "; COL 1    label\n";
        write(column1, 0, column1.length());
        String column2 = "; COL 2    residue\n";
        write(column2, 0, column2.length());
        String column3 = "; COL 3    alignpos\n";
        write(column3, 0, column3.length());
        int columnNr = 4;
        String column;
        String[] colLabels = mask.getColumnLabels();
        //System.out.println(colLabels.length);
        for(int i = 0; i<colLabels.length;i++)
        {
            column = "; COL "+columnNr+"    "+colLabels[i]+"\n";
            write(column, 0, column.length());
            columnNr++;
        }
        if(mask.isBackgroundColred())
        {
          column = "; COL " + (columnNr++) + "   color_r\n";
          write(column, 0, column.length());
          column = "; COL " + (columnNr++) + "   color_g\n";
          write(column, 0, column.length());
          column = "; COL " + (columnNr++) + "   color_b\n";
          write(column, 0, column.length());
        }
        if(mask.isForegroundColred())
        {
          column = "; COL " + (columnNr++) + "   color2_r\n";
          write(column, 0, column.length());
          column = "; COL " + (columnNr++) + "   color2_g\n";
          write(column, 0, column.length());
          column = "; COL " + (columnNr++) + "   color2_b\n";
          write(column, 0, column.length());
        }
        String entry = "; ENTRY   " + mask.getLabel() + "\n";
        write(entry, 0, entry.length());
        write(mask.getInformation(), 0, mask.getInformation().length());
        write("; ----------\n", 0, 13);
        StringBuffer l;
        for (int i = 0; i < mask.getLength(); i++)
        {
             l = new StringBuffer();
            l.append("M    " + mask.getSymbolAt(i) + "    " + (i + 1) );
            if(mask.isBackgroundColred())
            {
              float[] b = mask.getBackgroundColor(i).getRGBColorComponents(null);
              if (b.length == 3)
              {
                  l.append("   " + b[0] + "   " + b[1] + "   " + b[2]);
              }
            }
            if(mask.isForegroundColred())
            {
              float[] b = mask.getForegroundColor(i).getRGBColorComponents(null);
              if (b.length == 3)
              {
                  l.append("   " + b[0] + "   " + b[1] + "   " + b[2]);
              }
            }
            String[] cols = mask.getColumnInfo(i);
            for(int j = 0; j<mask.getColumnLabels().length;j++)
            {
                l.append("    "+cols[j]);
                
            }
            l.append( "\n");
            write(l.toString(), 0, l.length());
        }
        //System.out.println(l.toString());
        write("; **********\n", 0, 13);
    }
}


