/*
 *  SARSE, Semi-Automated RNA Sequence Editor.
 *  Copyright (C) 2004 Allan Lind-Thomsen
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dk.kvl.gui.alignment;

import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.io.*;
import dk.kvl.alignmenttools.*;
import dk.kvl.sequencetools.*;
import dk.kvl.controller.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import dk.kvl.gui.tools.*;
import dk.kvl.gui.components.*;
import dk.kvl.gui.alignment.table.*;
import dk.kvl.gui.properties.*;
import dk.kvl.tools.*;

/**
 *  Description of the Class
 *
 * @author     allan
 * @created    January 12, 2004
 */
public class AlignmentPanel extends JPanel implements ListSelectionListener, ChangeListener {
  private AlignmentTable sequenceTable = null;
  private AlignmentTable sequenceTable2 = null;
  private AlignmentTable sequenceTable3 = null;

  //should not exist here only in sequenceTable's
  private TitleTable nameTable = null;
  private TitleTable nameTable2 = null;
  private TitleTable nameTable3 = null;
  private TwoTablePanel tableView = null;
  private TwoTablePanel tableView2 = null;
  private TwoTablePanel tableView3 = null;
  //
  private JPanel sequencePanel = null;
  private AlignmentProperties properties = null;
  private AlignmentController alignmentController = null;
  private String alignmentName = null;
  private JPanel splitView = null;
  private JSplitPane splitPane = null;
  private Vector sequenceSelectionListeners = new Vector();
  //
  private boolean horizontalSplit;
  private boolean split = false;
  private boolean selectionMode, trace = true;
  private int operationMode;
  private final int unsplitIndex = 1;
  private final int splitIndex = 2;
  private boolean mouse;
  //indicates that a mouseevent has taken place
  //
  private Vector listeners = new Vector();
  private String wDir = "";

  /**
   *  Constructor for the AlignmentPanel object
   *
   * @param  alignCtrl      Description of the Parameter
   * @param  properties     Description of the Parameter
   * @param  name           Description of the Parameter
   * @exception  Exception  Description of the Exception
   */

  public AlignmentPanel(AlignmentController alignCtrl, AlignmentProperties properties, String name) throws Exception {
    super(new CardLayout());
    this.properties = properties;
    alignmentName = name;
    alignmentController = alignCtrl;
    createView(name);
    updateGrid();
    nameTable.getSelectionModel().addListSelectionListener(this);
  }


  /**
   *  Description of the Method
   *
   * @param  ce  Description of the Parameter
   */
  public void stateChanged(ChangeEvent ce) {
    if (trace && tableView2 != null && !mouse) {
      if (ce.getSource() instanceof JViewport) {
        JViewport port = (JViewport) ce.getSource();
        Point view = port.getViewPosition();
        JViewport t2 = tableView2.getViewport();
        JViewport t3 = tableView3.getViewport();
        Point point2 = new Point();
        point2.setLocation(t2.getViewPosition().getX(), view.getY());
        Point point3 = new Point();
        point3.setLocation(t3.getViewPosition().getX(), view.getY());
        t2.setViewPosition(point2);
        t3.setViewPosition(point3);
      }
    }
    mouse = false;
  }


  /**
   *  Gets the alignmentName attribute of the AlignmentPanel object
   *
   * @return    The alignmentName value
   */
  public String getAlignmentName() {
    return alignmentName;
  }

  public String getWdir()
  {
	return properties.getLastPath();  
  }

  /**
   *  Description of the Method
   *
   * @return    Description of the Return Value
   */
  public boolean horizontalSplit() {
    return horizontalSplit;
  }


  /**
   *  Sets the splitAlign attribute of the AlignmentPanel object
   *
   * @param  horizontalSplit  The new splitAlign value
   */
  public void setSplitAlign(boolean horizontalSplit) {
    this.horizontalSplit = horizontalSplit;
    if (horizontalSplit) {
      splitPane.setOrientation(JSplitPane.VERTICAL_SPLIT);
    } else {
      splitPane.setOrientation(JSplitPane.HORIZONTAL_SPLIT);
    }
    setVisible(true);
    splitPane.setDividerLocation(0.5);
  }


  /**
   *  Description of the Method
   *
   * @param  name           Description of the Parameter
   * @exception  Exception  Description of the Exception
   */
  protected void createView(String name) throws Exception {
    createTables(name);
    tableView = new TwoTablePanel(nameTable, sequenceTable, properties);
    add(tableView, "unsplit");
  }


  /**
   *  Description of the Method
   *
   * @param  name  Description of the Parameter
   */
  public void createTables(String name) {
    //int index = alignmentController.getStartIndex(name);
    nameTable = new TitleTable(alignmentController, name, properties);
    sequenceTable = new AlignmentTable(new AlignmentDataModel(alignmentController, name, nameTable), properties, alignmentController.getMainAlignment(), this);
    sequenceTable.addMouseListener(
          new MouseAdapter() {
            public void mouseReleased(MouseEvent e) {
              if (!selectionMode) {
                if (alignmentController.getMainAlignment().getNumberOfPairingMasks() > 0) {
                  
                    int[] rows = sequenceTable.getSelectedRows();
                    int[] cols = sequenceTable.getSelectedColumns();
                    for (int i = 0; i < rows.length; i++) {
                      for (int j = 0; j < cols.length; j++) {
                        int row = rows[i];
                        int col = cols[j];
                        sequenceTable.addRowSelectionInterval(row, row);
                        int pair = getPairing(row, col);
                        if (pair > -1) {
                          sequenceTable.addColumnSelectionInterval(pair, pair);
                        }
                      }
                    }
                    mouse = true;
                    //centerTable(getPairing(sequenceTable2.getSelectedRow(), sequenceTable2.getSelectedColumn()), sequenceTable2.getSelectedRow());
                    //centerTable(getPairing(sequenceTable2.getSelectedRow(), sequenceTable2.getSelectedColumn()), new Double(tableView2.getViewport().getViewPosition().getY()).intValue());
                  
                  
                }
              }
            }


            public void mousePressed(MouseEvent e) {
              if (isSplit() && sequenceTable3 != null && !e.isControlDown()) {
                sequenceTable3.clearSelection();
              }
            }
          });
  }


  /**
   *  Adds a feature to the ColumnModelListener attribute of the
   *  AlignmentPanel object
   *
   * @param  listener  The feature to be added to the ColumnModelListener
   *      attribute
   */
  public void addColumnModelListener(TableColumnModelListener listener) {
    sequenceTable.getColumnModel().addColumnModelListener(listener);
  }


  /**
   *  Gets the selectedColumns attribute of the AlignmentPanel object
   *
   * @return    The selectedColumns value
   */
  public int[] getSelectedColumns() {
    if (isSplit()) {
      return StringTools.combineSortedArrays(getLeftSelectedColumns(), getRightSelectedColumns());
    } else {
      return sequenceTable.getSelectedColumns();
    }
  }


  /**
   *  Gets the leftSelectedColumns attribute of the AlignmentPanel object
   *
   * @return    The leftSelectedColumns value
   */
  protected int[] getLeftSelectedColumns() {
    if (isSplit()) {
      return sequenceTable2.getSelectedColumns();
    } else {
      return new int[0];
    }
  }


  /**
   *  Gets the rightSelectedColumns attribute of the AlignmentPanel object
   *
   * @return    The rightSelectedColumns value
   */
  protected int[] getRightSelectedColumns() {
    if (isSplit()) {
      return sequenceTable3.getSelectedColumns();
    } else {
      return new int[0];
    }
  }


  /**
   *  Gets the selectedRows attribute of the AlignmentPanel object
   *
   * @return    The selectedRows value
   */
  public int[] getSelectedRows() {
    if (isSplit()) {
      return StringTools.combineSortedArrays(getLeftSelectedRows(), getRightSelectedRows());
    } else {
      return sequenceTable.getSelectedRows();
    }
  }


  /**
   *  Gets the selectedNameRows attribute of the AlignmentPanel object
   *
   * @return    The selectedNameRows value
   */
  public int[] getSelectedNameRows() {
    if (isSplit()) {
      return nameTable2.getSelectedRows();
    } else {
      return nameTable.getSelectedRows();
    }
  }


  /**
   *  Gets the leftSelectedRows attribute of the AlignmentPanel object
   *
   * @return    The leftSelectedRows value
   */
  protected int[] getLeftSelectedRows() {
    if (isSplit()) {
      return sequenceTable2.getSelectedRows();
    } else {
      return new int[0];
    }
  }


  /**
   *  Gets the rightSelectedRows attribute of the AlignmentPanel object
   *
   * @return    The rightSelectedRows value
   */
  protected int[] getRightSelectedRows() {
    if (isSplit()) {
      return sequenceTable3.getSelectedRows();
    } else {
      return new int[0];
    }
  }


  /**
   *  Gets the sequenceName attribute of the AlignmentPanel object
   *
   * @param  row  Description of the Parameter
   * @return      The sequenceName value
   */
  public String getSequenceName(int row) {
    if (row >= 0 && row < sequenceTable.getRowCount()) {
      return nameTable.getSequenceName(row);
    }
    return null;
  }


  /**
   *  Gets the sequenceNames attribute of the AlignmentPanel object
   *
   * @param  rows  Description of the Parameter
   * @return       The sequenceNames value
   */
  public String[] getSequenceNames(int[] rows) {
    String[] names = new String[rows.length];
    for (int i = 0; i < rows.length; i++) {
      names[i] = getSequenceName(rows[i]);
    }
    return names;
  }


  /**
   *  Gets the columnNumber attribute of the AlignmentPanel object
   *
   * @param  col  Description of the Parameter
   * @return      The columnNumber value
   */
  public int getColumnNumber(int col) {
    return Integer.parseInt(sequenceTable.getColumnName(col));
  }


  /**
   *  Description of the Method
   */
  public void clearSelections() {
    nameTable.clearSelection();
    sequenceTable.clearSelection();
    if (nameTable2 != null) {
      nameTable2.clearSelection();
      sequenceTable2.clearSelection();
      nameTable3.clearSelection();
      sequenceTable3.clearSelection();
    }
  }


  /**
   *  Gets the splitActionListener attribute of the AlignmentPanel object
   *
   * @return    The splitActionListener value
   */
  public TwoTablePanel getSplitActionListener() {
    return tableView;
  }


  /**
   *  Description of the Method
   */
  public void split() {
    if (!isSplit()) {
      Point p = tableView.getViewport().getViewPosition();
      if (splitView == null) {
        createSplitViewComponent();
      }
      operationMode(operationMode);
      selectionMode(selectionMode);

      clearSelections();
      split = true;
      tableView2.getViewport().setViewPosition(p);
      ((CardLayout) getLayout()).show(this, "split");
      splitPane.setDividerLocation(0.5);
    }
  }


  /**
   *  Description of the Method
   */
  public void unsplit() {
    if (isSplit()) {
      ((CardLayout) getLayout()).show(this, "unsplit");
      clearSelections();
      split = false;
    }
  }


  /**
   *  Gets the split attribute of the AlignmentPanel object
   *
   * @return    The split value
   */
  public boolean isSplit() {
    return split;
  }


  /**
   *  Gets the primarySelection attribute of the AlignmentPanel object
   *
   * @return    The primarySelection value
   */
  public boolean isPrimarySelection() {
    return selectionMode;
  }


  /**
   *  Gets the selectedSequenceNames attribute of the AlignmentPanel object
   *
   * @return    The selectedSequenceNames value
   */
  public String[] getSelectedSequenceNames() {
    int[] rows = null;
    if (isSplit()) {
      rows = StringTools.combineSortedArrays(getLeftSelectedRows(), getRightSelectedRows());
    } else {
      rows = sequenceTable.getSelectedRows();
    }
    String[] names = getSequenceNames(rows);
    return names;
  }


  /**
   *  Gets the selectedNames attribute of the AlignmentPanel object
   *
   * @return    The selectedNames value
   */
  public Collection getSelectedNames() {
    if (isSplit()) {
      return new Vector();
    } else {
      return nameTable.getSelectedNames();
    }
  }


  /**
   *  Gets the selectedNames attribute of the AlignmentPanel object
   *
   * @return    The selectedNames value
   */
  protected Collection getLeftSelectedNames() {
    if (isSplit()) {
      return nameTable2.getSelectedNames();
    } else {
      return null;
    }
  }


  /**
   *  Gets the rightSelectedNames attribute of the AlignmentPanel object
   *
   * @return    The rightSelectedNames value
   */
  protected Collection getRightSelectedNames() {
    if (isSplit()) {
      return nameTable3.getSelectedNames();
    } else {
      return null;
    }
  }


  /**
   *  Description of the Method
   */
  protected void createSplitViewComponent() {
    try {
    	
      if (splitView == null) {
        nameTable2 = new TitleTable(alignmentController, alignmentName, properties);
        nameTable3 = new TitleTable(alignmentController, alignmentName, properties);
        sequenceTable2 = new AlignmentTable(new AlignmentDataModel(alignmentController, alignmentName, nameTable), properties, alignmentController.getMainAlignment(), this);
        sequenceTable3 = new AlignmentTable(new AlignmentDataModel(alignmentController, alignmentName, nameTable), properties, alignmentController.getMainAlignment(), this);
        splitView = new JPanel(new BorderLayout());
        tableView2 = new TwoTablePanel(nameTable2, sequenceTable2, properties);
        tableView3 = new TwoTablePanel(nameTable3, sequenceTable3, properties);
        if (horizontalSplit()) {
          splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, tableView2, tableView3);
        } else {
          splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, tableView2, tableView3);
        }
        splitView.add(BorderLayout.CENTER, splitPane);
        add(splitView, "split");

        sequenceTable2.addMouseListener(
          new MouseAdapter() {
            public void mouseReleased(MouseEvent e) {
            	if (!selectionMode) {
                //if (alignmentController.getMainAlignment().getNumberOfPairingMasks() > 0) {
                  if (isSplit()) {
                    int[] rows = sequenceTable2.getSelectedRows();
                    int[] cols = sequenceTable2.getSelectedColumns();
                    for (int i = 0; i < rows.length; i++) {
                      for (int j = 0; j < cols.length; j++) {
                        int row = rows[i];
                        int col = cols[j];
                        sequenceTable3.addRowSelectionInterval(row, row);
                        int pair = getPairing(row, col);
                        if (pair > -1) {
                          sequenceTable3.addColumnSelectionInterval(pair, pair);
                        }
                      }
                    }
                    mouse = true;
                    //centerTable(getPairing(sequenceTable2.getSelectedRow(), sequenceTable2.getSelectedColumn()), sequenceTable2.getSelectedRow());
                    centerTable(getPairing(sequenceTable2.getSelectedRow(), sequenceTable2.getSelectedColumn()), new Double(tableView2.getViewport().getViewPosition().getY()).intValue());
                  }
                  
                //}
              }
            }


            public void mousePressed(MouseEvent e) {
              if (isSplit() && sequenceTable3 != null && !e.isControlDown()) {
                sequenceTable3.clearSelection();
              }
            }
          });

        sequenceTable3.addMouseListener(
          new MouseAdapter() {
            public void mouseReleased(MouseEvent e) {
              if (isSplit()) {
                //if (alignmentController.getMainAlignment().getNumberOfPairingMasks() > 0) {
                  if (!selectionMode) {
                    int[] rows = sequenceTable3.getSelectedRows();
                    int[] cols = sequenceTable3.getSelectedColumns();
                    for (int i = 0; i < rows.length; i++) {
                      for (int j = 0; j < cols.length; j++) {
                        int row = rows[i];
                        int col = cols[j];
                        sequenceTable2.addRowSelectionInterval(row, row);
                        int pair = getPairing(row, col);
                        if (pair != -1) {
                          sequenceTable2.addColumnSelectionInterval(pair, pair);
                        }
                      }
                    }
                    mouse = true;
                    centerLeftTable(getPairing(sequenceTable3.getSelectedRow(), sequenceTable3.getSelectedColumn()), new Double(tableView3.getViewport().getViewPosition().getY()).intValue());
                  }
                //}
              }
            }


            public void mousePressed(MouseEvent e) {
              if (isSplit() && sequenceTable2 != null && !e.isControlDown()) {
                sequenceTable2.clearSelection();
              }
            }
          });

        tableView2.getViewport().addChangeListener(this);
        tableView3.getViewport().addChangeListener(this);
        tableView2.addListSelectionListener(this);
        tableView3.addListSelectionListener(this);
        /*
         *  for (int i = 0; i<listeners.size();i++)
         *  {
         *  tableView2.addListSelectionListener((ListSelectionListener)listeners.get(i));
         *  }
         */
      }
    } catch (Exception e) {
      e.printStackTrace();
    }
  }


  /**
   *  Description of the Method
   *
   * @param  column  Description of the Parameter
   * @param  row     Description of the Parameter
   */
  public void centerTable(int column, int row) {
    if (isSplit()) {
      tableView3.centerTable(column, row);
    } else {
      tableView.centerTable(column, row);
    }
  }


  /**
   *  Description of the Method
   *
   * @param  column  Description of the Parameter
   * @param  row     Description of the Parameter
   */
  public void centerLeftTable(int column, int row) {
    if (isSplit()) {
      tableView2.centerTable(column, row);
    }
  }


  /**
   *  Adds a feature to the SequenceSelectionListener attribute of the
   *  AlignmentPanel object
   *
   * @param  listener  The feature to be added to the
   *      SequenceSelectionListener attribute
   */
  public void addSequenceSelectionListener(ActionListener listener) {
    sequenceSelectionListeners.add(listener);
  }


  /**
   *  Description of the Method
   *
   * @param  listener  Description of the Parameter
   */
  public void removeSequenceSelectionListener(ActionListener listener) {
    sequenceSelectionListeners.remove(listener);
  }


  /**
   *  Description of the Method
   *
   * @param  lse  Description of the Parameter
   */
  public void valueChanged(ListSelectionEvent lse) {
    Collection selected = null;
    if (selected != null && selected.size() > 0) {
      fireActionEvent((String) selected.toArray()[0]);
    }
    /*
     *  if(isSplit() && (getLeftSelectedColumns().length > 1 || getLeftSelectedRows().length > 1))
     *  {
     *  if (alignmentController.getMainAlignment().getNumberOfPairingMasks() > 0){
     *  centerTable(alignmentController.getPairing(getSequenceName(sequenceTable2.getSelectedRow()), sequenceTable2.getSelectedColumn()),sequenceTable2.getSelectedRow());
     *  }
     *  sequenceTable3.repaint();
     *  }
     */
  }


  /**
   *  Description of the Method
   *
   * @param  seqName  Description of the Parameter
   */
  public void fireActionEvent(String seqName) {
    ActionEvent ae = new ActionEvent(this, 0, seqName);
    for (int i = 0; i < sequenceSelectionListeners.size(); i++) {
      ((ActionListener) sequenceSelectionListeners.get(i)).actionPerformed(ae);
    }
  }


  /**
   *  Sets the traceEnabled attribute of the AlignmentPanel object
   *
   * @param  value  The new traceEnabled value
   */
  public void setTraceEnabled(boolean value) {
    trace = value;
    if (tableView2 != null) {
      tableView2.setTraceEnabled(value);
      tableView3.setTraceEnabled(value);
    }
  }


  /**
   *  Adds a feature to the ListSelectionListener attribute of the
   *  AlignmentPanel object
   *
   * @param  listener  The feature to be added to the ListSelectionListener
   *      attribute
   */
  public void addListSelectionListener(ListSelectionListener listener) {
    tableView.addListSelectionListener(listener);
    listeners.add(listener);
    if (tableView2 != null) {
      tableView2.addListSelectionListener(listener);
      tableView3.addListSelectionListener(listener);
    }
  }


  /**
   *  Description of the Method
   *
   * @param  listener  Description of the Parameter
   */
  public void removeListSelectionListener(ListSelectionListener listener) {
    tableView.removeListSelectionListener(listener);
    listeners.remove(listener);
    if (tableView2 != null) {
      tableView2.removeListSelectionListener(listener);
      tableView3.removeListSelectionListener(listener);
    }
  }


  /**
   *  Description of the Method
   */
  public void toggleSplit() {
    if (isSplit()) {
      if (splitPane.getOrientation() == JSplitPane.VERTICAL_SPLIT) {
        splitPane.setOrientation(JSplitPane.HORIZONTAL_SPLIT);
      } else {
        splitPane.setOrientation(JSplitPane.VERTICAL_SPLIT);
      }
    }
  }


  /**
   *  Description of the Method
   *
   * @param  column   Description of the Parameter
   * @param  enabled  Description of the Parameter
   */
  public void columnInfo(String column, boolean enabled) {

  }


  /**
   *  Description of the Method
   *
   * @param  mode  Description of the Parameter
   */
  public void splitMode(boolean mode) {
    horizontalSplit = mode;
    if (isSplit()) {
      if (horizontalSplit) {
        splitPane.setOrientation(JSplitPane.VERTICAL_SPLIT);
      } else {
        splitPane.setOrientation(JSplitPane.HORIZONTAL_SPLIT);
      }
      splitPane.setDividerLocation(0.5);
    }
    setVisible(true);
    
  }


  /**
   *  Gets the splitMode attribute of the AlignmentPanel object
   *
   * @return    The splitMode value
   */
  public boolean getSplitMode() {
    return horizontalSplit;
  }


  /**
   *  Description of the Method
   *
   * @param  mode  Description of the Parameter
   */
  public void selectionMode(boolean mode) {
    if (sequenceTable != null) {
      selectionMode = mode;
      sequenceTable.clearSelection();
      if (sequenceTable3 != null) {
        if (mode) {
          sequenceTable3.clearSelection();
          sequenceTable2.repaint();
        } else {
          if (alignmentController.getMainAlignment().getNumberOfPairingMasks() > 0) {
            sequenceTable2.repaint();
            int[] rows = sequenceTable2.getSelectedRows();
            int[] cols = sequenceTable2.getSelectedColumns();
            for (int i = 0; i < rows.length; i++) {
              for (int j = 0; j < cols.length; j++) {
                int row = rows[i];
                int col = cols[j];
                sequenceTable3.addRowSelectionInterval(row, row);
                int pair = getPairing(row, col);
                sequenceTable3.addColumnSelectionInterval(pair, pair);
              }
            }
            mouse = true;
            centerTable(getPairing(sequenceTable2.getSelectedRow(), sequenceTable2.getSelectedColumn()), new Double(tableView2.getViewport().getViewPosition().getY()).intValue());
          }
        }
      }
    }
  }


  /**
   *  Gets the pairing attribute of the AlignmentPanel object
   *
   * @param  row  Description of the Parameter
   * @param  col  Description of the Parameter
   * @return      The pairing value
   */
  public int getPairing(int row, int col) {
    //if (alignmentController.getNumberOfPairingMasks() > 0) {
      int pair = alignmentController.getPairing(getSequenceName(row), col);
      //System.out.println(getSequenceName(row)+": "+pair);
return pair;
      //the base is part of a base pair
//      if (pair != -1) {
//        return pair;
//      } 
//      else {
//        //the base is within the pairingmask
//        String pm = alignmentController.getPairingmaskName();
//        pair = alignmentController.getPairing(pm, col);
//        if (pair != -1) {
//          return pair;
//        }
//        //not basepairing and not within pairingmask
//        else {
//          return -1;
//        }
//      }
    //}
    //return -1;
  }


  /**
   *  Gets the controller attribute of the AlignmentPanel object
   *
   * @return    The controller value
   */
  public AlignmentController getController() {
    return alignmentController;
  }


  /**
   *  Description of the Method
   *
   * @param  mode  Description of the Parameter
   */
  public void operationMode(int mode) {
    operationMode = mode;
    if (isSplit() && sequenceTable2 != null && sequenceTable3 != null) {
      if (alignmentController.getMainAlignment().getNumberOfPairingMasks() > 0) {
        sequenceTable2.repaint();
        mouse = true;
        centerTable(sequenceTable3.getSelectedColumn(), sequenceTable3.getSelectedRow());
      }
    }
  }


  /**
   *  Gets the operationMode attribute of the AlignmentPanel object
   *
   * @return    The operationMode value
   */
  public int getOperationMode() {
    return operationMode;
  }


  /**
   *  Gets the selectionMode attribute of the AlignmentPanel object
   *
   * @return    The selectionMode value
   */
  public boolean getSelectionMode() {
    return selectionMode;
  }


  /**
   *  Gets the table attribute of the AlignmentPanel object
   *
   * @return    The table value
   */
  public JTable getTable() {
    return sequenceTable;
  }


  /**
   *  Gets the table2 attribute of the AlignmentPanel object
   *
   * @return    The table2 value
   */
  public JTable getTable2() {
    return sequenceTable2;
  }


  /**
   *  Gets the table3 attribute of the AlignmentPanel object
   *
   * @return    The table3 value
   */
  public JTable getTable3() {
    return sequenceTable3;
  }


  /**
   *  Description of the Method
   */
  public void updateCellSize() {
    nameTable.setFont(properties.getTableFont());
    if (nameTable2 != null) {
      nameTable2.setFont(properties.getTableFont());
      nameTable3.setFont(properties.getTableFont());
    }
    tableView.updateCellSize();
    if (tableView2 != null) {
      tableView2.updateCellSize();
      tableView3.updateCellSize();
    }
  }


  /**
   *  Description of the Method
   */
  public void updateGrid() {
    tableView.updateGrid();
    if (tableView2 != null) {
      tableView2.updateGrid();
      tableView3.updateGrid();
    }
  }


  /**
   *  Gets the tableView attribute of the AlignmentPanel object
   *
   * @return    The tableView value
   */
  public TwoTablePanel getTableView() {
    return tableView;
  }


  /**
   *  Gets the tableView2 attribute of the AlignmentPanel object
   *
   * @return    The tableView2 value
   */
  public TwoTablePanel getTableView2() {
    return tableView2;
  }


  /**
   *  Gets the tableView3 attribute of the AlignmentPanel object
   *
   * @return    The tableView3 value
   */
  public TwoTablePanel getTableView3() {
    return tableView3;
  }


  /**
   *  Gets the splitPane attribute of the AlignmentPanel object
   *
   * @return    The splitPane value
   */
  public JSplitPane getSplitPane() {
    return splitPane;
  }


  /**
   *  Gets the alignmentController attribute of the AlignmentPanel object
   *
   * @return    The alignmentController value
   */
  public AlignmentController getAlignmentController() {
    return alignmentController;
  }


  /**
   *  Description of the Method
   *
   * @param  sequenceNumber  Description of the Parameter
   */
  public void verticalCenter(int sequenceNumber) {
    if (sequenceNumber > 0 && sequenceNumber <= alignmentController.getNumberOfSequences()) {
      if (isSplit()) {
        tableView2.verticalCenter((int) tableView2.getViewport().getViewPosition().getX(), sequenceNumber);
      } else {
        tableView.verticalCenter((int) tableView.getViewport().getViewPosition().getX(), sequenceNumber);
      }
    }
  }

  public void centerCell(int col, int row)
  {
    if(isSplit())
    {
      tableView2.centerCell(col,row);
    }
    else
    {
      tableView.centerCell(col,row);
    }
  }
  
  
}

