/*
    SARSE, Semi-Automated RNA Sequence Editor. Copyright (C) 2004 Allan
    Lind-Thomsen This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or (at your
    option) any later version. This program is distributed in the hope that it
    will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
    of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
    Public License for more details. You should have received a copy of the GNU
    General Public License along with this program; if not, write to the Free
    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
  */
package dk.kvl.sequencetools;

import java.util.*;
import dk.kvl.tools.*;

/**
 *  Implements a pairing mask sequence and keep track of pairings in both
 *  positions and a letter string
 *
 * @author     Allan Lind-Thomsen
 * @created    January 9, 2004
 */
public class PairingMask extends Sequence
{
    public static int OUT_OF_SEQUENCE = -3;
    public static int NO_PAIRINGMASK = -2;
    public static int NO_PAIRING = -1;


    /**
     *  Constructor for the PairingMask object
     */
    public PairingMask() { 
    	setAlphabet(SequenceAlphabet.PAIRINGMASK);    	
    }

    public PairingMask(int length)
    {
    	this();
    	char[] seq = new char[length];
    	for(int i = 0;i<length;i++)
    	{
    		seq[i] ='-';
    	}
    	setPairingmask(seq);
    }
    
    /**
     *  Constructor for the PairingMask object
     *
     * @param  label          Description of the Parameter
     * @param  sequence       Description of the Parameter
     * @param  alph           Description of the Parameter
     * @param  checkSequence  Description of the Parameter
     * @exception  Exception  Description of the Exception
     */
    public PairingMask(String label, String sequence, int alph, boolean checkSequence) throws Exception
    {
        super(label, sequence, alph, checkSequence);
        pairings = updatePairings(sequence.toCharArray());
    }


    /**
     *  Constructor for the PairingMask object
     *
     * @param  label          Description of the Parameter
     * @param  struc          Description of the Parameter
     * @param  alph           Description of the Parameter
     * @param  checkSequence  Description of the Parameter
     * @exception  Exception  Description of the Exception
     */
    public PairingMask(String label, int[] struc, int alph, boolean checkSequence) throws Exception
    {
        this(label, makePairingmask(struc), alph, checkSequence);
    }


    /**
     *  Gets the subsequence attribute of the PairingMask object
     *
     * @param  begin  Description of the Parameter
     * @param  end    Description of the Parameter
     * @return        The subsequence value
     */
    public Sequence getSubsequence(int begin, int end)
    {
        PairingMask seq = null;
        if (end < getLength() && begin < end && begin >= 0)
        {
            try
            {
                //end+1 because substring is excluding endindex
                seq = new PairingMask(getLabel(), getSequence().substring(begin, end + 1), getType(), false);
            }
            catch (Exception e)
            {
                //this exception should never be thrown here, because the program sends a valid
                //subsequence and therefore don't test if the sequence is valid, the check parameter is set to false
                e.printStackTrace();
            }
        }
        return seq;
    }


    /**
     *  Description of the Method
     *
     * @param  seq  Description of the Parameter
     * @return      Description of the Return Value
     */
    public static int[] resolvePairings(char[] seq)
    {
        Hashtable masks = new Hashtable();
        String symbol;
        IntArray ia;
        for (int i = 0; i < seq.length; i++)
        {
            if (seq[i] != '-' && seq[i] != '.')
            {
                symbol = "" + seq[i];
                ia = (IntArray)masks.get(symbol);
                if (ia == null)
                {//symbol dosn't exists
                    ia = new IntArray();
                    ia.add(i);
                    masks.put(symbol, ia);
                }
                else
                {
                    ia.add(i);
                }
            }
        }
        Enumeration e = masks.elements();
        int[] result = new int[seq.length];
        initPairings(result);
        while (e.hasMoreElements())
        {
            ia = (IntArray)e.nextElement();
            for (int i = 0; i < ia.getLength(); i++)
            {
                result[ia.get(i)] = ia.get(ia.getLength() - i - 1);
            }
        }
        return result;
    }


    /**
     *  Description of the Method
     *
     * @param  arr  Description of the Parameter
     * @return      Description of the Return Value
     */
    protected static int[] initPairings(int[] arr)
    {
        for (int i = 0; i < arr.length; i++)
        {
            arr[i] = NO_PAIRING;
        }
        return arr;
    }


    /**
     *  Description of the Method
     *
     * @param  start  Description of the Parameter
     * @param  end    Description of the Parameter
     * @return        Description of the Return Value
     */
    public boolean rotateLeft(int start, int end)
    {
        boolean b = super.rotateLeft(start, end);
        if (b)
        {
            rotatePairingsLeft(start, end);
        }
        return b;
    }


    /**
     *  Description of the Method
     *
     * @param  start  Description of the Parameter
     * @param  end    Description of the Parameter
     * @return        Description of the Return Value
     */
    public boolean rotateRight(int start, int end)
    {
        boolean b = super.rotateRight(start, end);
        if (b)
        {
            rotatePairingsRight(start, end);
        }
        return b;
    }


    /**
     *  Description of the Method
     *
     * @param  start  Description of the Parameter
     * @param  end    Description of the Parameter
     */
    public void rotatePairingsLeft(int start, int end)
    {
        int st = pairings[end];
        while (st == -1)
        {
            for (int i = end; i > start; i--)
            {
                pairings[i] = pairings[i - 1];
                if (pairings[i] != -1)
                {
                    pairings[pairings[i]] = i;
                }
            }
            pairings[start] = -1;
            st = pairings[end];
        }
    }


    /**
     *  Description of the Method
     *
     * @param  start  Description of the Parameter
     * @param  end    Description of the Parameter
     */
    public void rotatePairingsRight(int start, int end)
    {
        int st = pairings[start];
        while (st == -1)
        {
            for (int i = start; i < end; i++)
            {
                pairings[i] = pairings[i + 1];
                if (pairings[i] != -1)
                {
                    pairings[pairings[i]] = i;
                }
            }
            pairings[end] = -1;
            st = pairings[start];
        }
    }


    /**
     *  Description of the Method
     *
     * @param  base1  Description of the Parameter
     * @param  base2  Description of the Parameter
     * @return        Description of the Return Value
     */
    public char removePairing(int base1, int base2)
    {
        char oldChar = getSymbolAt(base1);
        if(oldChar == '(')
        {
            oldChar = ')';
        }
        else if(oldChar == ')')
        {
            oldChar = '(';
        }
        else if(oldChar == '}')
        {
            oldChar = '{';
        }
        else if(oldChar == '{')
        {
            oldChar = '}';
        }
        else if(oldChar == '[')
        {
            oldChar = ']';
        }
        else if(oldChar == ']')
        {
            oldChar = '[';
        }
        if (oldChar == getSymbolAt(base2))
        {
            pairings[base1] = -1;
            pairings[base2] = -1;
            setSymbolAt(base1, '-');
            setSymbolAt(base2, '-');
            return oldChar;
        }
        return '!';
    }


    /**
     *  Adds a feature to the Pairing attribute of the PairingMask object
     *
     * @param  base1   The feature to be added to the Pairing attribute
     * @param  base2   The feature to be added to the Pairing attribute
     * @param  symbol  The feature to be added to the Pairing attribute
     * @return         Description of the Return Value
     */
    public boolean addPairing(int base1, int base2, char symbol)
    {
        if (pairings[base1] != -1 || pairings[base2] != -1)
        {
            return false;
        }
        else
        {
            pairings[base1] = base2;
            pairings[base2] = base1;
            setSymbolAt(base1, symbol);
            setSymbolAt(base2, symbol);

            return true;
        }
    }


    /**
     *  Description of the Method
     *
     * @param  seq  Description of the Parameter
     * @return      Description of the Return Value
     */
    public static int[] updatePairings(char[] seq)
    {
        boolean parantese = false;
        int[] pMask;
        for (int i = 0; i < seq.length; i++)
        {
            if (seq[i] == '(' || seq[i] == '[' || seq[i] == '{'||seq[i] == '<')
            {
                parantese = true;
                break;
            }
        }
        if (parantese)
        {
            pMask = resolveParanteseStructure(seq);
        }
        else
        {
            pMask = resolvePairings(seq);
        }
        return pMask;
    }


    /**
     *  Description of the Method
     *
     * @param  seq  Description of the Parameter
     * @return      Description of the Return Value
     */
    public static int[] resolveParanteseStructure(char[] seq)
    {
        int[] pairs = new int[seq.length];

        initPairings(pairs);
        Stack stack1 = new Stack();// for '('
        Stack stack2 = new Stack();// for '['
        Stack stack3 = new Stack();// for '{'
        Stack stack4 = new Stack();// for '<'
        for (int i = 0; i < seq.length; i++)
        {
            if (seq[i] == '(')
            {
                //System.out.println("seq[" + i + "] == " + seq[i] + " == '('");
                stack1.push(new Integer(i));
            }
            else if (seq[i] == '[')
            {
                stack2.push(new Integer(i));
            }
            else if (seq[i] == '{')
            {
                stack3.push(new Integer(i));
            }
            else if (seq[i] == '<')
            {
                stack4.push(new Integer(i));
            }
            
            else if (seq[i] == ')')
            {
                if (!stack1.empty())
                {
                    int val = ((Integer)stack1.pop()).intValue();
                    pairs[i] = val;
                    pairs[val] = i;
                }
            }
            else if (seq[i] == ']')
            {
                if (!stack2.empty())
                {
                    int val = ((Integer)stack2.pop()).intValue();
                    pairs[i] = val;
                    pairs[val] = i;
                }
            }
            else if (seq[i] == '}')
            {
                if (!stack3.empty())
                {
                    int val = ((Integer)stack3.pop()).intValue();
                    pairs[i] = val;
                    pairs[val] = i;
                }
            }
            else if (seq[i] == '>')
            {
                if (!stack4.empty())
                {
                    int val = ((Integer)stack4.pop()).intValue();
                    pairs[i] = val;
                    pairs[val] = i;
                }
            }
        }
        return pairs;
    }


    /**
     *  Description of the Method
     *
     * @param  pairs  Description of the Parameter
     * @return        Description of the Return Value
     */
    public static String makePairingmask(int[] pairs)
    {
        //no support for  pseudoknots
        StringBuffer mask = new StringBuffer();
        for (int i = 0; i < pairs.length; i++)
        {
            if (pairs[i] == -1)
            {
                mask.append("-");
            }
            else if (pairs[i] > i)
            {
                mask.append("(");
            }
            else if (pairs[i] < i)
            {
                mask.append(")");
            }
        }
        return mask.toString();
    }


    /**
     *  Description of the Method
     *
     * @return    Description of the Return Value
     */
    public int[] finalProcessing()
    {
        super.finalProcessing();
        pairings = updatePairings(getSequenceArray());
        return pairings;
    }


    /**
     *  Gets the paranteseStructure attribute of the PairingMask object
     *
     * @return    The paranteseStructure value
     */
    public boolean isParanteseStructure()
    {
        char symbol = '-';
        for (int i = 0; i < getLength(); i++)
        {
            if (getSequenceArray()[i] != '-' && getSequenceArray()[i] != '.')
            {
                symbol = getSequenceArray()[i];
                break;
            }
        }
        if (symbol == '{' || symbol == '[' || symbol == '('|| symbol == '<')
        {
            return true;
        }
        else
        {
            return false;
        }
    }


    /**
     *  Description of the Method
     *
     * @param  col1  Description of the Parameter
     * @param  col2  Description of the Parameter
     * @param  mask  Description of the Parameter
     */
    public boolean pairColumns(int col1, int col2, char mask)
    {
        if (pairings[col1] != -1 || pairings[col2] != -1)
        {
            return false;
        }
        else
        {
            if (mask == '(' || mask == ')')
            {
                if (col1 < col2)
                {
                    changeSymbol('(', col1);
                    changeSymbol(')', col2);
                }
                else
                {
                    changeSymbol('(', col2);
                    changeSymbol(')', col1);
                }
            }
            else if (mask == '{' || mask == '}')
            {
                if (col1 < col2)
                {
                    changeSymbol('{', col1);
                    changeSymbol('}', col2);
                }
                else
                {
                    changeSymbol('{', col2);
                    changeSymbol('}', col1);
                }
            }
            else if (mask == '[' || mask == ']')
            {
                if (col1 < col2)
                {
                    changeSymbol('[', col1);
                    changeSymbol(']', col2);
                }
                else
                {
                    changeSymbol('[', col2);
                    changeSymbol(']', col1);
                }
            }
            else if (mask == '<' || mask == '>')
            {
                if (col1 < col2)
                {
                    changeSymbol('<', col1);
                    changeSymbol('>', col2);
                }
                else
                {
                    changeSymbol('<', col2);
                    changeSymbol('>', col1);
                }
            }
            else
            {
                changeSymbol(mask, col1);
                changeSymbol(mask, col2);
            }
            pairings[col1] = col2;
            pairings[col2] = col1;
            return true;
        }
    }
    
    public void setPairings(int[] pairs)
    {
      pairings = pairs;
    }
    
    public char[] empty()
    {
    	char[] oldPairs = getSequence().toCharArray();
    	int[] emptyPairs = new int[getSequence().length()];
    	for(int i = 0;i<emptyPairs.length;i++)
    	{
    		emptyPairs[i] = -1;
    		changeSymbol('-',i);
    	}
    	setPairings(emptyPairs);
    	return oldPairs;
    }
}

