/**
 *  SARSE, Semi-Automated RNA Sequence Editor. Copyright (C) 2004 Allan
 *  Lind-Thomsen This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version. This program is distributed in the hope that it
 *  will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 *  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *  Public License for more details. You should have received a copy of the GNU
 *  General Public License along with this program; if not, write to the Free
 *  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 *  02111-1307, USA.
 */
package dk.kvl.tools;

import javax.xml.parsers.*;
import org.w3c.dom.*;
import org.xml.sax.*;
import java.util.*;
import java.io.*;
import dk.kvl.tools.io.*;
import dk.kvl.alignmenttools.*;
import javax.swing.*;
/**
 *  Description of the Class
 *
 * @author     allan
 * @created    February 10, 2004
 */
public class Program implements Comparable {
  private String name;
  private String type, pack;
  private String sequenceType;
  private String description;
  private boolean error;
  private Vector inputTypes = new Vector();
  private Vector outputTypes = new Vector();
  private Hashtable parameters = new Hashtable();
  private Vector selectedParams = new Vector();
  private boolean selected;
  private int priority;
  private String path;
  private boolean errorMessage = false;
  private String depends = null;
  private Vector outputFiles = new Vector();


/**
   *  Constructor for the Program object
   *
   * @param  programNode  Description of the Parameter
   */
  public Program(Node programNode) {
    createProgram(programNode);
  }


  /**
   *  Description of the Method
   *
   * @param  programNode  Description of the Parameter
   */
  public void createProgram(Node programNode) {
    NamedNodeMap nnm = programNode.getAttributes();
    setName(nnm.getNamedItem("name").getNodeValue());
    setType(nnm.getNamedItem("type").getNodeValue());
    setPackage(nnm.getNamedItem("package").getNodeValue());
    if (nnm.getNamedItem("depends").getNodeValue() != null) {
      depends = nnm.getNamedItem("depends").getNodeValue();
    }
    setPriority(Integer.parseInt(nnm.getNamedItem("priority").getNodeValue()));
    setSequenceType(nnm.getNamedItem("sequencetype").getNodeValue());
    setSelected(Boolean.getBoolean(nnm.getNamedItem("selected").getNodeValue()));
    NodeList nl = programNode.getChildNodes();
    Node node = null;
    for (int i = 0; i < nl.getLength(); i++) {
      node = nl.item(i);
      if (node.getNodeName().equals("programdescription")) {
        setDescription(node.getFirstChild().getNodeValue());
      } else if (node.getNodeName().equals("parameters")) {
        extractParamters(node);
      } else if (node.getNodeName().equals("inputformats")) {
        extractInputFormats(node);
      } else if (node.getNodeName().equals("outputformats")) {
        extractOutputFormats(node);
      } else if (node.getNodeName().equals("files")) {
        extractFiles(node);
      }
    }

  }


  private void extractFiles(Node node) {
	  NodeList nl = node.getChildNodes();
	    for (int i = 0; i < nl.getLength(); i++)
	    {
	    	Node n = nl.item(i);
	    	if(n.getNodeName().equals("file"))
	    	{
	    		addFile(n.getTextContent());
	    	}
	    }
}


/**
   *  Description of the Method
   *
   * @param  parameterNode  Description of the Parameter
   */
  public void extractParamters(Node parameterNode) {
    Parameter param = null;
    NodeList nl = parameterNode.getChildNodes();
    for (int i = 0; i < nl.getLength(); i++) {
      if (nl.item(i).getNodeName().equals("param")) {
        param = new Parameter(nl.item(i));
        addParameter(param);
        if (param.isSelected()) {
          selectParameter(param.getName());
        }
      }
    }
  }


  /**
   *  Description of the Method
   *
   * @param  inputNode  Description of the Parameter
   */
  public void extractInputFormats(Node inputNode) {
    NodeList nl = inputNode.getChildNodes();
    Node node = null;
    for (int i = 0; i < nl.getLength(); i++) {
      if (nl.item(i).getNodeName().equals("fileextension")) {
        NodeList list = nl.item(i).getChildNodes();
        for (int j = 0; j < list.getLength(); j++) {
          addInputFormat(list.item(j).getNodeValue());
        }
      }
    }
  }


  /**
   *  Description of the Method
   *
   * @param  outputNode  Description of the Parameter
   */
  public void extractOutputFormats(Node outputNode) {
    NodeList nl = outputNode.getChildNodes();
    for (int i = 0; i < nl.getLength(); i++) {
      if (nl.item(i).getNodeName().equals("fileext")) {
        NodeList list = nl.item(i).getChildNodes();
        for (int j = 0; j < list.getLength(); j++) {
          addOutputFormat(list.item(j).getNodeValue());
        }
        //prog.addOutputFormat(nl.item(i).getNodeValue());
      }
    }
  }


  /**
   *  Sets the name attribute of the Program object
   *
   * @param  name  The new name value
   */
  public void setName(String name) {
    this.name = name;
  }


  /**
   *  Gets the name attribute of the Program object
   *
   * @return    The name value
   */
  public String getName() {
    return name;
  }


  /**
   *  Sets the type attribute of the Program object
   *
   * @param  type  The new type value
   */
  public void setType(String type) {
    this.type = type;
  }


  /**
   *  Sets the package attribute of the Program object
   *
   * @param  pack  The new package value
   */
  public void setPackage(String pack) {
    this.pack = pack;
  }


  /**
   *  Gets the type attribute of the Program object
   *
   * @return    The type value
   */
  public String getType() {
    return type;
  }


  /**
   *  Gets the package attribute of the Program object
   *
   * @return    The package value
   */
  public String getPackage() {
    return pack;
  }


  /**
   *  Sets the sequenceType attribute of the Program object
   *
   * @param  sequenceType  The new sequenceType value
   */
  public void setSequenceType(String sequenceType) {
    this.sequenceType = sequenceType;
  }


  /**
   *  Gets the parameters attribute of the Program object
   *
   * @return    The parameters value
   */
  public Enumeration getParameters() {
    return parameters.elements();
  }


  /**
   *  Gets the selectedParameters attribute of the Program object
   *
   * @return    The selectedParameters value
   */
  public Vector getSelectedParameters() {
    Collections.sort(selectedParams);
    return selectedParams;
  }



  /**
   *  Description of the Method
   *
   * @param  param  Description of the Parameter
   */
  public void selectParameter(String param) {
    if(getParameter(param) != null)
    {
    getParameter(param).setSelected(true);
    selectedParams.add(getParameter(param));
    }
  }


  /**
   *  Gets the parameter attribute of the Program object
   *
   * @param  key  Description of the Parameter
   * @return      The parameter value
   */
  public Parameter getParameter(String key) {
    return (Parameter) parameters.get(key);
  }


  /**
   *  Description of the Method
   *
   * @param  param  Description of the Parameter
   */
  public void unselectParameter(String param) {
    if(getParameter(param) != null)
    {
    getParameter(param).setSelected(false);
    selectedParams.remove(getParameter(param));
    }
  }


  /**
   *  Gets the parameterSelected attribute of the Program object
   *
   * @param  paramName  Description of the Parameter
   * @return            The parameterSelected value
   */
  public boolean isParameterSelected(String paramName) {
    return getParameter(paramName).isSelected();
  }


  /**
   *  Gets the selected attribute of the Program object
   *
   * @return    The selected value
   */
  public boolean isSelected() {
    return selected;
  }


  /**
   *  Sets the selected attribute of the Program object
   *
   * @param  sel  The new selected value
   */
  public void setSelected(boolean sel) {
    selected = sel;
  }


  /**
   *  Gets the sequenceType attribute of the Program object
   *
   * @return    The sequenceType value
   */
  public String getSequenceType() {
    return sequenceType;
  }


  /**
   *  Gets the inputTypes attribute of the Program object
   *
   * @return    The inputTypes value
   */
  public Vector getInputTypes() {
    return inputTypes;
  }


  /**
   *  Gets the outputTypes attribute of the Program object
   *
   * @return    The outputTypes value
   */
  public Vector getOutputTypes() {
    return outputTypes;
  }


  /**
   *  Sets the description attribute of the Program object
   *
   * @param  description  The new description value
   */
  public void setDescription(String description) {
    this.description = description;
  }


  /**
   *  Gets the description attribute of the Program object
   *
   * @return    The description value
   */
  public String getDescription() {
    return description;
  }


  /**
   *  Adds a feature to the InputFormat attribute of the Program object
   *
   * @param  ext  The feature to be added to the InputFormat attribute
   */
  public void addInputFormat(String ext) {
    inputTypes.add(ext);
  }


  /**
   *  Adds a feature to the OutputFormat attribute of the Program object
   *
   * @param  ext  The feature to be added to the OutputFormat attribute
   */
  public void addOutputFormat(String ext) {
    outputTypes.add(ext);
  }


  /**
   *  Adds a feature to the Parameter attribute of the Program object
   *
   * @param  param  The feature to be added to the Parameter attribute
   */
  public void addParameter(Parameter param) {
    parameters.put(param.getName(), param);
  }


  /**
   *  Description of the Method
   *
   * @return    Description of the Return Value
   */
  public String toString() {
    /*
     *  StringBuffer sb = new StringBuffer("Name: " + name + ", Type: " + type + ", Sequencetype: " + sequenceType + "\n");
     *  sb.append("Description: "+description+"\n");
     *  sb.append("Inputtypes: ");
     *  for (int i = 0; i < inputTypes.size(); i++)
     *  {
     *  sb.append(inputTypes.get(i) + ",");
     *  }
     *  sb.append("\nOutputtypes: ");
     *  for (int i = 0; i < outputTypes.size(); i++)
     *  {
     *  sb.append(outputTypes.get(i) + ",");
     *  }
     *  sb.append("\nParameters: \n");
     *  for (int i = 0; i < parameters.size(); i++)
     *  {
     *  sb.append("\t" + parameters.get(i) + "\n");
     *  }
     *  return sb.toString();
     */
    return name;
  }



  /**
   *  Main processing method for the Program object
   *
   * @param  dataFile       Description of the Parameter
   * @param  resultFile     Description of the Parameter
   * @param  log            Description of the Parameter
   * @param  ppFile         Description of the Parameter
   * @exception  Exception  Description of the Exception
   */
  public void execute(File dataFile, File resultFile, Log log, String ppFile) throws Exception {
    Runtime runtime = Runtime.getRuntime();
    //write to log
    log.writeLine("---------------" + name + " started----------------");
    //setup cmd-lind
    java.util.List cmds = new ArrayList();
    cmds.add(name);
    log.write(name, 0, name.length());

    for (int i = 0; i < getSelectedParameters().size(); i++) {
      String s = ((Parameter) getSelectedParameters().get(i)).toString();
      cmds.add(s);
      if (s.equals("--ppfile")) {
        new File(ppFile);
        cmds.add(ppFile);
        log.write(" " + s);
        log.write(" " + ppFile);
      } else {
        log.write(" " + s);
      }
    }

    if (name.equals("drawdot")) {
      cmds.add(ppFile);
    } else {
      cmds.add(dataFile.getAbsolutePath());
    }

    //write cmd-line to log
    log.writeLine(" " + dataFile.getAbsolutePath());
    //
    Process p = runtime.exec((String[]) cmds.toArray(new String[cmds.size()]),null,new File(dataFile.getParent()));

    //initialize readers and writers
    BufferedReader resultReader = new BufferedReader(new InputStreamReader(p.getInputStream()));
    BufferedWriter resultWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(resultFile)));
    BufferedReader errorReader = new BufferedReader(new InputStreamReader(p.getErrorStream()));

    //execute error reading and data reading in two threads to avoid blocking
    StreamReaderThread srt = new StreamReaderThread(resultReader, resultWriter);
    ErrorReaderThread ert = new ErrorReaderThread(errorReader, log, this);

    srt.start();
    ert.start();

    p.waitFor();

    srt.join();
    ert.join();

    log.writeLine("---------" + name + " finished---------------");
    log.flush();
    resultWriter.flush();
    resultReader.close();
    resultWriter.close();
    errorReader.close();
    if (error) {
      //throw new Exception("ERROR: see logfile");
      errorMessage = true;
    }
  }


  /**
   *  Description of the Method
   *
   * @param  dataFile  Description of the Parameter
   * @param  log       Description of the Parameter
   */
  public void execute(File dataFile, Log log) {
    File newFile = null;
    try {
      //convert col to fasta
      String oldName = dataFile.getAbsolutePath();
      String newName = oldName.substring(0, oldName.length() - 4);
      newFile = new File(newName + ".fasta");
      newFile.createNewFile();

      String[] cmds = new String[2];
      cmds[0] = "col2fasta";
      cmds[1] = oldName;
      Runtime runtime = Runtime.getRuntime();
      Process p = runtime.exec(cmds,new String[]{},new File(dataFile.getParent()));

      //initialize readers and writers
      BufferedReader resultReader = new BufferedReader(new InputStreamReader(p.getInputStream()));
      BufferedWriter resultWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(newFile)));
      BufferedReader errorReader = new BufferedReader(new InputStreamReader(p.getErrorStream()));

      //execute error reading and data reading in two threads to avoid blocking
      StreamReaderThread srt = new StreamReaderThread(resultReader, resultWriter);
      ErrorReaderThread ert = new ErrorReaderThread(errorReader, log, this);

      srt.start();
      ert.start();

      p.waitFor();

      srt.join();
      ert.join();

      resultWriter.flush();
      resultReader.close();
      resultWriter.close();
      errorReader.close();

      if (error) {
        errorMessage = true;
      }

      //run clusterscripts
      java.util.List allCommands = new ArrayList();
      java.util.List cmd = new ArrayList();
      String range = null;
      String group = null;

      cmd.add("cluster.tcsh");
      cmd.add(newFile.getAbsolutePath());

      for (int i = 0; i < getSelectedParameters().size(); i++) {
        String s = ((Parameter) getSelectedParameters().get(i)).toString();
        if (s.indexOf("Group file") != -1) {
          String[] split = s.split(" ");
          group = split[split.length - 1];
        } else {
          if (s.indexOf("Range") != -1) {
            String[] split = s.split(" ");
            range = split[split.length - 1];
            if (range != null) {
              cmd.add(range);
            }
          }
        }
      }
      allCommands.add(cmd);

      List cmd2 = new ArrayList();
      cmd2.add("makeplots.tcsh");
      cmd2.add(newFile.getAbsolutePath());
      cmd2.add(newName + "_groups_" + group + ".txt");

      if (range != null) {
        cmd2.add(range);
      }
      allCommands.add(cmd2);

      List cmd3 = new ArrayList();
      cmd3.add("makestruc.tcsh");
      cmd3.add(newFile.getAbsolutePath());
      cmd3.add(newName + "_groups_" + group + ".txt");
      allCommands.add(cmd3);
      /*
      List cmd4 = new ArrayList();
      cmd4.add("process_cluster");
      cmd4.add(newName + "_groups_" + group + ".txt");
      cmd4.add(newName + "_groups_" + group + "_makestruc.fasta");
      cmd4.add(newName);
      allCommands.add(cmd4);
      */
      runtime = Runtime.getRuntime();
      for (int i = 0; i < allCommands.size(); i++) {
        if (i > 0) {
          File groupFile = new File(newName + "_groups_" + group + ".txt");
          while (!groupFile.exists()) {
          }
        }

        ArrayList list = (ArrayList) allCommands.get(i);
        String proName = (String) list.get(0);
        log.writeLine("---------" + proName + " started---------------");
        p = runtime.exec((String[]) list.toArray(new String[cmd.size()]),new String[]{},new File(dataFile.getParent()));

        File tmpOut = File.createTempFile("tmpOut", "txt");
        //initialize readers and writers
        resultReader = new BufferedReader(new InputStreamReader(p.getInputStream()));
        resultWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(tmpOut.getAbsolutePath())));
        errorReader = new BufferedReader(new InputStreamReader(p.getErrorStream()));

        //execute error reading and data reading in two threads to avoid blocking
        srt = new StreamReaderThread(resultReader, resultWriter);
        ert = new ErrorReaderThread(errorReader, log, this);

        srt.start();
        ert.start();

        p.waitFor();

        srt.join();
        ert.join();

        log.writeLine("---------" + proName + " finished---------------");
        log.flush();
        resultWriter.flush();
        resultReader.close();
        resultWriter.close();
        errorReader.close();

        tmpOut.delete();

        if (error) {
          //throw new Exception("ERROR: see logfile");
          errorMessage = true;
        }
      }
    } catch (Exception e) {
      System.out.println("Error Program.execute: " + e);
    }

  }


  /**
   *  Sets the error attribute of the Program object
   *
   * @param  error  The new error value
   */
  public void setError(boolean error) {
    this.error = error;
  }


  /**
   *  Description of the Method
   *
   * @return    Description of the Return Value
   */
  public boolean hasErrorMessage() {
    return errorMessage;
  }


  /**
   *  Gets the priority attribute of the Program object
   *
   * @return    The priority value
   */
  public int getPriority() {
    return priority;
  }


  /**
   *  Sets the priority attribute of the Program object
   *
   * @param  prior  The new priority value
   */
  public void setPriority(int prior) {
    priority = prior;
  }


  /**
   *  Description of the Method
   *
   * @param  programObject  Description of the Parameter
   * @return                Description of the Return Value
   */
  public int compareTo(Object programObject) {
    final int before = -1;
    final int after = 1;
    final int equal = 0;
    Program p = (Program) programObject;

    if (getPriority() < p.getPriority()) {
      return before;
    }
    if (getPriority() > p.getPriority()) {
      return after;
    }
    return equal;
  }


  public void addFile(String file)
  {
	  //System.out.println("Adding: "+file);
	  outputFiles.add(file);  
  }
  
  public Vector getFiles()
  {
	  return outputFiles;
  }
  
  public String getFile(int index)
  {
	  if(index >= 0 && index < outputFiles.size())
	  {
		  return (String)outputFiles.get(index);
	  }
	  return null;
  }
  
  /**
   *  Gets the depends attribute of the Program object
   *
   * @return    The depends value
   */
  public String getDepends() {
    return depends;
  }
  
  public void reset(){
	  selectedParams = new Vector();
	  Enumeration e = parameters.keys();
	  while(e.hasMoreElements()){
		  Parameter p = (Parameter)parameters.get((String)e.nextElement());
		  p.reset();
		  if(p.isSelected()){
			  selectedParams.add(p);
		  }
	  }
  }
}

