/*
 *  SARSE, Semi-Automated RNA Sequence Editor.
 *  Copyright (C) 2004 Allan Lind-Thomsen
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package dk.kvl.controller;

import java.util.*;
import java.io.*;
import dk.kvl.alignmenttools.*;
import dk.kvl.alignmenttools.io.*;
import dk.kvl.sequencetools.*;
import dk.kvl.gui.event.*;
import javax.swing.*;
import java.awt.*;
import dk.kvl.tools.*;
import dk.kvl.tools.history.*;
import dk.kvl.tools.io.*;
import dk.kvl.gui.properties.*;
/**
 *  This class is a controller for working with alignmentViews All access should
 *  go through here to implement a MVC pattern if used correctly it should be
 *  easy to implement other userinterfaces because the userinterface dosn't need
 *  to know about the datamodels
 *
 * @author     allan
 * @created    January 12, 2004
 */
public class AlignmentController {
  private Alignment mainAlignment = null;
  private Vector updateListeners = new Vector();
  private Vector reloadListeners = new Vector();
  private boolean changed, error;
  private String label, fileName, projectDir;
  private History history;
  private FileHistory fileHistory;


  /**
   *  Constructor for the AlignmentController object
   *
   * @param  file             Description of the Parameter
   * @param  name             Description of the Parameter
   * @param  projectDir       Description of the Parameter
   * @exception  IOException  Description of the Exception
   */
  public AlignmentController(File file, String name, String projectDir) throws IOException {
    label = name;
    this.projectDir = projectDir;
    setMainAlignmentFromFile(file);
  }


  /**
   *  Constructor for the AlignmentController object
   *
   * @param  fileName         Description of the Parameter
   * @param  name             Description of the Parameter
   * @param  projectDir       Description of the Parameter
   * @exception  IOException  Description of the Exception
   */
  public AlignmentController(String fileName, String name, String projectDir) throws IOException {
    this(new File(fileName), name, projectDir);
  }


  /**
   *  Sets the alignment from a specified file
   *
   * @param  fileName                   the file with the alignment
   * @exception  FileNotFoundException  thrown if fileName dosn't exsists
   * @exception  IOException            thrown when other io errors
   */
  public void setMainAlignmentFromFile(String fileName) throws FileNotFoundException, IOException {
    setMainAlignmentFromFile(new File(fileName));
  }


  /**
   *  Sets the alignmentFromFile attribute of the AlignmentController object
   *
   * @param  file                       The new alignmentFromFile value
   * @exception  FileNotFoundException  Description of the Exception
   * @exception  IOException            Description of the Exception
   */
  protected void setMainAlignmentFromFile(File file) throws FileNotFoundException, IOException {
    String fName = file.getName();
    //this method assumes at least one pairingmask
    if (fName.endsWith(".col")) {
           mainAlignment = AlignmentIOController.loadAlignment(file, AlignmentWriter.COL);

      setActivePairingmask();
      mainAlignment.emptyInfo();
      fireReloadEvent();

      /*
       *  discmask.delete();
       *  pmask.delete();
       *  col2txt.delete();
       *  par2num.delete();
       */
    } else if (fName.endsWith(".fasta")||fName.endsWith(".fa")){ 
    	mainAlignment = AlignmentIOController.loadAlignment(file, AlignmentWriter.FASTA);
        setActivePairingmask();
        mainAlignment.emptyInfo();
        fireReloadEvent();
    
    }
    else {
      if (fName.endsWith(".txt") || fName.endsWith(".widetxt")) {
        mainAlignment = AlignmentIOController.loadAlignment(file, AlignmentWriter.TXT);
        setActivePairingmask();
        mainAlignment.emptyInfo();
      }
      /*
       *  else{
       *  if (fName.endsWith(".fasta")){
       *  try{
       *  Runtime runtime = Runtime.getRuntime();
       *  String newName = fName.substring(0, fName.length() - 6);
       *  File newFile = new File(file.getParent() + "/" + newName + ".col");
       *  newFile.createNewFile();
       *  String[] cmds = new String[2];
       *  cmds[0] = "../rnadbtools/fasta2col";
       *  cmds[1] = file.getAbsolutePath();
       *  Process p = runtime.exec(cmds);
       *  /initialize readers and writers
       *  BufferedReader resultReader = new BufferedReader(new InputStreamReader(p.getInputStream()));
       *  BufferedWriter resultWriter = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(newFile)));
       *  BufferedReader errorReader = new BufferedReader(new InputStreamReader(p.getErrorStream()));
       *  /execute error reading and data reading in two threads to avoid blocking
       *  StreamReaderThread srt = new StreamReaderThread(resultReader, resultWriter);
       *  ControllerErrorReaderThread ert = new ControllerErrorReaderThread(errorReader, new Log(new File(projectDir + "/" + "log-fasta2col.log"), newFile), this);
       *  srt.start();
       *  ert.start();
       *  p.waitFor();
       *  srt.join();
       *  ert.join();
       *  resultWriter.flush();
       *  resultReader.close();
       *  resultWriter.close();
       *  errorReader.close();
       *  mainAlignment = AlignmentIOController.loadAlignment(newFile, AlignmentWriter.COL);
       *  if (error)
       *  {
       *  JOptionPane.showMessageDialog(null, "Error converting .fasta to .col");
       *  }
       *  }
       *  catch(Exception e){
       *  System.out.println("Error AlignmentController.setMainAlignmentFromFile: " + e);
       *  }
       *  }
       */
          else {
        JOptionPane.showMessageDialog(null, "Unknown file format");
      }
    }
    
    mainAlignment.setLabel(label);
  }


  /**
   *  Sets the error attribute of the AlignmentController object
   *
   * @param  error  The new error value
   */
  public void setError(boolean error) {
    this.error = error;
  }


  /**
   *  Sets the mainAlignment attribute of the AlignmentController object
   *
   * @param  a  The new mainAlignment value
   */
  public void setMainAlignment(Alignment a) {
    mainAlignment = a;
  }


  /**
   *  Sets the activePairingmask attribute of the AlignmentController object
   */
  protected void setActivePairingmask() {
    String value = null;
    if (mainAlignment.getNumberOfPairingMasks() > 1) {
      Enumeration e = mainAlignment.getPairingMaskKeys();
      String[] names = new String[mainAlignment.getNumberOfPairingMasks()];
      int count = 0;
      while (e.hasMoreElements()) {
        names[count] = (String) e.nextElement();
        count++;
      }
      value = (String) JOptionPane.showInputDialog(null, "There is more than one pairingmask.\nWhich one should be used ?", "Select pairingmask", JOptionPane.QUESTION_MESSAGE, null, names, names[0]);
    } else {
      Enumeration e = mainAlignment.getPairingMaskKeys();
      if (e.hasMoreElements()) {
        value = (String) e.nextElement();
      }
    }
    //If no pairingmask is available the value is null
    mainAlignment.setActivePairingmask(value);
  }


  /**
   *  Gets the pairingmaskName attribute of the AlignmentController object
   *
   * @return    The pairingmaskName value
   */
  public String getPairingmaskName() {
    return mainAlignment.getPairingmaskName();
  }


  /**
   *  Gets the alignmentLength attribute of the AlignmentController object
   *
   * @return    The alignmentLength value
   */
  public int getAlignmentLength() {
    return mainAlignment.getLength();
  }


  /**
   *  Gets the sequenceKeys attribute of the AlignmentController object
   *
   * @return    The sequenceKeys value
   */
  public Enumeration getSequenceKeys() {
    return mainAlignment.getSequenceKeys();
  }



  /**
   *  Gets the label attribute of the AlignmentController object
   *
   * @return    The label value
   */
  public String getLabel() {
    return "RNA";
    //hack
  }


  /**
   *  Sets the fileName attribute of the AlignmentController object
   *
   * @param  fileName  The new fileName value
   */
  public void setFileName(String fileName) {
    this.fileName = fileName;
  }


  /**
   *  Gets the fileName attribute of the AlignmentController object
   *
   * @return    The fileName value
   */
  public String getFileName() {
    return fileName;
  }


  /**
   *  Gets the numberOfSequences attribute of the AlignmentController object
   *
   * @return    The numberOfSequences value
   */
  public int getNumberOfSequences() {
    return mainAlignment.getNumberOfSequences();
  }


  /**
   *  Gets the pairingMask attribute of the AlignmentController object
   *
   * @param  seqKey  Description of the Parameter
   * @return         The pairingMask value
   */
  public boolean isPairingMask(String seqKey) {
    if (mainAlignment.getPairingMask(seqKey) != null) {
      return true;
    }
    return false;
  }


  /**
   *  Gets the numberOfPairingMasks attribute of the AlignmentController
   *  object
   *
   * @return    The numberOfPairingMasks value
   */
  public int getNumberOfPairingMasks() {
    return mainAlignment.getNumberOfPairingMasks();
  }


  /**
   *  Gets the totalNumberOfSequences attribute of the AlignmentController
   *  object
   *
   * @param  aKey  Description of the Parameter
   * @return       The totalNumberOfSequences for alignment aKey, or for
   *      public static void main(String[] args) Alignment. }
   */
  public int getTotalNumberOfSequences(String aKey) {
  	int numPair = 0;
  	if(mainAlignment.getNumberOfPairingMasks() > 0)
  	{
  		numPair=1;
  	}
    return mainAlignment.getNumberOfSequences() + numPair;//mainAlignment.getNumberOfPairingMasks();
  }



  /**
   *  Gets the sequence attribute of the AlignmentController object
   *
   * @param  name  Description of the Parameter
   * @return       The sequence value
   */
  public Sequence getSequence(String name) {
    Sequence seq = null;
    if (name != null) {
      seq = mainAlignment.getSequence(name);
      if (seq == null) {
        seq = mainAlignment.getPairingMask(name);
      }
    }
    return seq;
  }



  /**
   *  Gets the mainAlignment attribute of the AlignmentController object
   *
   * @return    The mainAlignment value
   */
  public Alignment getMainAlignment() {
    return mainAlignment;
  }



  /**
   *  Gets the charAt attribute of the AlignmentController object
   *
   * @param  sequenceKey  Description of the Parameter
   * @param  col          Description of the Parameter
   * @return              The charAt value
   */
  public char getCharAt(String sequenceKey, int col) {
    Sequence seq = getSequence(sequenceKey);
    if (seq != null) {
      return seq.getSymbolAt(col);
    }
    return '#';
  }


  /**
   *  Description of the Method
   *
   * @param  names  Description of the Parameter
   * @param  index  Description of the Parameter
   * @return        Description of the Return Value
   */
  public HistoryItem toLowerCase(String[] names, int[] index) {
    if (mainAlignment != null) {
      int[][] result = mainAlignment.toLowerCase(names, index);
      updateViews();
      ToLowerCase historyItem = new ToLowerCase();
      historyItem.setSequences(names);
      historyItem.setIndex(index);
      historyItem.setResult(result);
      //history.add(historyItem);
      return historyItem;
    }
    return null;
  }


  /**
   *  Description of the Method
   *
   * @param  elements  Description of the Parameter
   */
  public void toLowerCase(ArrayList elements) {
    ArrayList historyItems = new ArrayList();
    for (int i = 0; i < elements.size(); i++) {
      HistoryElement element = (HistoryElement) elements.get(i);
      HistoryItem item = toLowerCase(element.getNames(), element.getIndex());
      if (item != null) {
        historyItems.add(item);
      }
    }
    historyItems.add("To Lower Case");
    if (!historyItems.isEmpty()) {
      history.add(historyItems);
    }
  }



  /**
   *  Description of the Method
   *
   * @param  names  Description of the Parameter
   * @param  index  Description of the Parameter
   * @return        Description of the Return Value
   */
  public HistoryItem toUpperCase(String[] names, int[] index) {
    //index = indexCorrection(index);
    if (mainAlignment != null) {
      int[][] result = mainAlignment.toUpperCase(names, index);
      updateViews();
      ToUpperCase historyItem = new ToUpperCase();
      historyItem.setSequences(names);
      historyItem.setIndex(index);
      historyItem.setResult(result);
      //history.add(historyItem);
      return historyItem;
    }
    return null;
  }


  /**
   *  Description of the Method
   *
   * @param  elements  Description of the Parameter
   */
  public void toUpperCase(ArrayList elements) {
    ArrayList historyItems = new ArrayList();
    for (int i = 0; i < elements.size(); i++) {
      HistoryElement element = (HistoryElement) elements.get(i);
      HistoryItem item = toUpperCase(element.getNames(), element.getIndex());
      if (item != null) {
        historyItems.add(item);
      }
    }
    historyItems.add("To Upper Case");
    if (!historyItems.isEmpty()) {
      history.add(historyItems);
    }
  }


  /**
   *  Adds a feature to the MainPairingMask attribute of the
   *  AlignmentController object
   *
   * @param  mask  The feature to be added to the MainPairingMask attribute
   * @param  name  The feature to be added to the MainPairingMask attribute
   */
  public void addMainPairingMask(PairingMask mask, String name) {
    mainAlignment.addPairingMask(mask);
  }



  /**
   *  Gets the mainPairingmask attribute of the AlignmentController object
   *
   * @param  key  Description of the Parameter
   * @return      The mainPairingmask value
   */
  public PairingMask getMainPairingmask(String key) {
    return mainAlignment.getPairingMask(key);
  }


  /**
   *  Adds a feature to the PairingMask attribute of the AlignmentController
   *  object
   *
   * @param  mask  The feature to be added to the PairingMask attribute
   * @param  name  The feature to be added to the PairingMask attribute
   */
  public void addPairingMask(PairingMask mask) {
    if (mainAlignment != null) {
      mainAlignment.addPairingMask(mask);
      mainAlignment.setActivePairingmask(mask.getLabel());
      fireReloadEvent();
      updateViews();
    }
  }



  /**
   *  Gets the pairingmask attribute of the AlignmentController object
   *
   * @return    The pairingmask value
   */
  public PairingMask getPairingmask() {
    if (mainAlignment != null) {
      return mainAlignment.getPairingMask();
    }
    return null;
  }


  /**
   *  Gets the pairingmaskKeys attribute of the AlignmentController object
   *
   * @return    The pairingmaskKeys value
   */
  public Enumeration getPairingMaskKeys() {
    if (mainAlignment != null) {
      return mainAlignment.getPairingMaskKeys();
    }
    return null;
  }



  /**
   *  Description of the Method
   *
   * @param  sequenceNames  Description of the Parameter
   * @param  index          Description of the Parameter
   * @param  newChar        Description of the Parameter
   * @return                Description of the Return Value
   */
  public HistoryItem changeSymbols(String[] sequenceNames, int[] index, char newChar) {
    if (mainAlignment != null) {
      char[][] result = mainAlignment.changeSymbol(sequenceNames, index, newChar);
      updateViews();
      ChangeSymbols historyItem = new ChangeSymbols();
      historyItem.setSequences(sequenceNames);
      historyItem.setIndex(index);
      historyItem.setNewChar(newChar);
      historyItem.setResult(result);
      //history.add(historyItem);
      return historyItem;
    }
    return null;
  }


  /**
   *  Description of the Method
   *
   * @param  elements  Description of the Parameter
   */
  public void changeSymbols(ArrayList elements) {
    ArrayList historyItems = new ArrayList();
    for (int i = 0; i < elements.size(); i++) {
      HistoryElement element = (HistoryElement) elements.get(i);
      HistoryItem item = changeSymbols(element.getNames(), element.getIndex(), element.getNewChar());
      if (item != null) {
        historyItems.add(item);
      }
    }
    historyItems.add("Change Symbol");
    if (!historyItems.isEmpty()) {
      history.add(historyItems);
    }
  }


  /**
   *  Adds a feature to the UpdateListener attribute of the
   *  AlignmentController object
   *
   * @param  ul  The feature to be added to the UpdateListener attribute
   */
  public void addUpdateListener(UpdateListener ul) {
    updateListeners.add(ul);
  }


  /**
   *  Description of the Method
   */
  public void updateViews() {
    UpdateEvent ue = new UpdateEvent(this, 1, "update");
    for (int i = 0; i < updateListeners.size(); i++) {
      ((UpdateListener) updateListeners.get(i)).updatePerformed(ue);
    }
  }


  /**
   *  Description of the Method
   *
   * @param  file  Description of the Parameter
   * @return       Description of the Return Value
   */
  public String saveAlignment(File file) {
    try {
      if (file.getName().indexOf('.') == -1) {
        file = new File(file.getAbsolutePath() + ".col");
      }
      if (!file.getName().endsWith(".col")) {
        String oldName = file.getAbsolutePath();
        String newName = oldName.substring(0, oldName.lastIndexOf('.')) + ".col";
        file = new File(newName);
      }
      AlignmentIOController.saveAlignment(mainAlignment, file, AlignmentWriter.COL);
    } catch (IOException ie) {
      ie.printStackTrace();
      return null;
    }
    mainAlignment.setChanged(false);
    updateViews();
    return file.getAbsolutePath();
  }


  /**
   *  Gets the alphabet attribute of the AlignmentController object
   *
   * @return    The alphabet value
   */
  public char[] getAlphabet() {
    return mainAlignment.getAlphabet();
  }


  /**
   *  Gets the changed attribute of the AlignmentController object
   *
   * @return    The changed value
   */
  public boolean isChanged() {
    return changed;
  }


  /**
   *  Gets the changed attribute of the AlignmentController object
   *
   * @param  key  Description of the Parameter
   * @return      The changed value
   */
  public boolean isChanged(String key) {
    if (mainAlignment != null) {
      return mainAlignment.isChanged();
    }
    return false;
  }


  /**
   *  Gets the pairing attribute of the AlignmentController object
   *
   * @param  seq    Description of the Parameter
   * @param  index  Description of the Parameter
   * @return        The pairing value
   */
  public int getPairing(String seq, int index) {
    int result = -1;
    Sequence s = getSequence(seq);//getPairingmask()
    
    if (s != null) {
      result = s.getPairing(index);
    }
    return result;
  }


  /**
   *  This method is used to change the mainalignment, it deletes all views
   *
   * @param  fileName         Description of the Parameter
   * @exception  IOException  Description of the Exception
   */
  public void changeMainAlignment(String fileName) throws IOException {
    setMainAlignmentFromFile(fileName);
    //        alignmentViews = new Hashtable();

  }


  /**
   *  Gets the foreground attribute of the AlignmentController object
   *
   * @param  sequenceKey  Description of the Parameter
   * @param  column       Description of the Parameter
   * @return              The foreground value
   */
  public Color getForeground(String sequenceKey, int column) {
    return getSequence(sequenceKey).getForegroundColor(column);
  }


  /**
   *  Gets the background attribute of the AlignmentController object
   *
   * @param  sequenceKey  Description of the Parameter
   * @param  column       Description of the Parameter
   * @return              The background value
   */
  public Color getBackground(String sequenceKey, int column) {
    if (getSequence(sequenceKey) != null) {
      return getSequence(sequenceKey).getBackgroundColor(column);
    }
    return Color.WHITE;
  }


  /**
   *  Gets the sequenceInfo attribute of the AlignmentController object
   *
   * @param  sequence  Description of the Parameter
   * @return           The sequenceInfo value
   */
  public String getSequenceInfo(String sequence) {
    return mainAlignment.getSequenceInfo(sequence);
  }


  /**
   *  Gets the alignmentInfo attribute of the AlignmentController object
   *
   * @return    The alignmentInfo value
   */
  public String getAlignmentInfo() {
    return mainAlignment.getInformation();
  }


  /**
   *  Description of the Method
   *
   * @param  sequences  Description of the Parameter
   * @param  start      Description of the Parameter
   * @param  end        Description of the Parameter
   * @return            Description of the Return Value
   */
  public HistoryItem rotateLeft(String[] sequences, int start, int end) {
  	if (mainAlignment != null) {
      String[] result = mainAlignment.rotateLeft(sequences, start, end);
      if(result != null)
      {
      RotateLeft historyItem = new RotateLeft();
      historyItem.setSequences(sequences);
      historyItem.setStart(start);
      historyItem.setEnd(end);
      historyItem.setResult(result);
      //history.add(historyItem);
      updateViews();
      return historyItem;
      }
    }

    return null;
  }


  /**
   *  Description of the Method
   *
   * @param  elements  Description of the Parameter
   * @return 
   */
  public boolean rotateLeft(ArrayList elements) {
    boolean succes = true;
    ArrayList historyItems = new ArrayList();
    for (int i = 0; i < elements.size(); i++) {
      HistoryElement element = (HistoryElement) elements.get(i);
      HistoryItem item = rotateLeft(element.getNames(), element.getStart(), element.getEnd());
      if (item != null) {
        historyItems.add(item);
      }
      else
      {
        succes = false;
      }
    }
    historyItems.add("Move right");
    if (!historyItems.isEmpty()) {
      history.add(historyItems);
    }
    return succes;
  }


  /**
   *  Description of the Method
   *
   * @param  elements  Description of the Parameter
   * @return 
   */
  public boolean rotateRight(ArrayList elements) {
  	boolean succes = true;
    ArrayList historyItems = new ArrayList();
    for (int i = 0; i < elements.size(); i++) {
      HistoryElement element = (HistoryElement) elements.get(i);
      HistoryItem item = rotateRight(element.getNames(), element.getStart(), element.getEnd());
      if (item != null) {
        historyItems.add(item);
      }
      else
      {
        succes = false;
      }
    }
    historyItems.add("Move left");
    if (!historyItems.isEmpty()) {
      history.add(historyItems);
    }
    return succes;
  }


  /**
   *  Description of the Method
   *
   * @param  sequences  Description of the Parameter
   * @param  start      Description of the Parameter
   * @param  end        Description of the Parameter
   * @return            Description of the Return Value
   */
  public HistoryItem rotateRight(String[] sequences, int start, int end) {
    if (mainAlignment != null) {
      String[] result = mainAlignment.rotateRight(sequences, start, end);
      if(result != null)
      {
      RotateRight historyItem = new RotateRight();
      historyItem.setSequences(sequences);
      historyItem.setStart(start);
      historyItem.setEnd(end);
      historyItem.setResult(result);
      //history.add(historyItem);
      updateViews();
      return historyItem;
      }
    }
    return null;
  }


  /**
   *  Gets the columnInfo attribute of the AlignmentController object
   *
   * @param  name   Description of the Parameter
   * @param  index  Description of the Parameter
   * @return        The columnInfo value
   */
  public String[] getColumnInfo(String name, int index) {
    Sequence seq = getSequence(name);
    if (seq != null && !(seq instanceof dk.kvl.sequencetools.PairingMask)) {
      return seq.getColumnInfo(index);
    } else {
      return null;
    }
  }


  /**
   *  Description of the Method
   *
   * @param  base1  Description of the Parameter
   * @param  base2  Description of the Parameter
   * @return        Description of the Return Value
   */
  public boolean removePairing(int base1, int base2) {
    if (mainAlignment != null && mainAlignment.getPairingMask() != null) {
      char result = mainAlignment.removePairing(base1, base2);
      if (result != '!') {
        changed = true;
        updateViews();
        RemovePairing historyItem = new RemovePairing();
        historyItem.setIndexA(base1);
        historyItem.setIndexB(base2);
        historyItem.setResult(result);
        history.add(historyItem);
        return true;
      }
    }
    return false;
  }


  /**
   *  Adds a feature to the Pairing attribute of the AlignmentController
   *  object
   *
   * @param  base1   The feature to be added to the Pairing attribute
   * @param  base2   The feature to be added to the Pairing attribute
   * @param  symbol  The feature to be added to the Pairing attribute
   * @return         Description of the Return Value
   */
  public boolean addPairing(int base1, int base2, char symbol) {
    boolean result = mainAlignment.addPairing(base1, base2, symbol);
    if (result) {
      changed = true;
      updateViews();
      AddPairing historyItem = new AddPairing();
      historyItem.setIndexA(base1);
      historyItem.setIndexB(base2);
      historyItem.setSymbol(symbol);
      history.add(historyItem);
    }
    return result;
  }


  /**
   *  Sets the alignmentInfo attribute of the AlignmentController object
   *
   * @param  info  The new alignmentInfo value
   */
  public void setAlignmentInfo(String info) {
    mainAlignment.setInformation(info);
  }


  /**
   *  Sets the sequenceInfo attribute of the AlignmentController object
   *
   * @param  sequenceName  The new sequenceInfo value
   * @param  info          The new sequenceInfo value
   */
  public void setSequenceInfo(String sequenceName, String info) {

    getSequence(sequenceName).setInformation(info);
  }


  /**
   *  Gets the sortedNames attribute of the AlignmentController object
   *
   * @return    The sortedNames value
   */
  public String[] getSortedNames() {
    return mainAlignment.getSortedNames();
  }


  /**
   *  Gets the history attribute of the AlignmentController object
   *
   * @return    The history value
   */
  public History getHistory() {
    return history;
  }


  /**
   *  Gets the fileHistory attribute of the AlignmentController object
   *
   * @return    The fileHistory value
   */
  public FileHistory getFileHistory() {
    return fileHistory;
  }


  /**
   *  Description of the Method
   */
  public void undo() {
    ArrayList items = history.undo();
    if (items != null) {
      for (int i = 0; i < items.size(); i++) {
        if (items.get(i) instanceof dk.kvl.tools.history.HistoryItem) {

          HistoryItem hi = (HistoryItem) items.get(i);
          if (hi != null &&
              (hi.getLabel().equals("Add column") ||
              (hi.getLabel().equals("Remove column")) ||
              (hi.getLabel().equals("Remove gaps")) ||
              (hi.getLabel().equals("Delete sequences")) ||
              (hi.getLabel().equals("Sequences Added")))) {
            fireReloadEvent();
            updateViews();
          } else {
            updateViews();
          }
        }
      }
    }
  }


  /**
   *  Description of the Method
   *
   * @param  index  Description of the Parameter
   */
  public void undo(int index) {
    ArrayList lists = history.undo(index);
    boolean reload = false;
    if (lists != null) {
      for (int i = 0; i < lists.size(); i++) {
        ArrayList items = (ArrayList) lists.get(i);
        for (int j = 0; j < items.size() - 1; j++) {
          if (items.get(j) instanceof dk.kvl.tools.history.HistoryItem) {

            HistoryItem hi = (HistoryItem) items.get(j);
            if (hi != null &&
                (hi.getLabel().equals("Add column") ||
                (hi.getLabel().equals("Remove column")) ||
                (hi.getLabel().equals("Remove gaps")) ||
                (hi.getLabel().equals("Delete sequences")) ||
                (hi.getLabel().equals("Sequences Added"))
                )) {
              reload = true;
            }
          }
        }
      }
    }
    if (reload) {
      fireReloadEvent();
      updateViews();
    } else {
      updateViews();
    }
  }


  /**
   *  Description of the Method
   */
  public void redo() {
    ArrayList items = history.redo();
    if (items != null) {
      for (int i = 0; i < items.size(); i++) {
        if (items.get(i) instanceof dk.kvl.tools.history.HistoryItem) {
          if (items.get(i) instanceof dk.kvl.tools.history.HistoryItem) {

            HistoryItem hi = (HistoryItem) items.get(i);
            if (hi != null &&
                (hi.getLabel().equals("Add column") ||
                (hi.getLabel().equals("Remove column")) ||
                (hi.getLabel().equals("Remove gaps")) ||
                (hi.getLabel().equals("Delete sequences")) ||
                (hi.getLabel().equals("Sequences Added"))
                )) {
              fireReloadEvent();
              updateViews();
            } else {
              updateViews();
            }
          }
        }
      }
    }
  }


  /**
   *  Description of the Method
   *
   * @param  index  Description of the Parameter
   */
  public void redo(int index) {
    ArrayList lists = history.redo(index);
    boolean reload = false;
    if (lists != null) {
      for (int i = 0; i < lists.size(); i++) {
        ArrayList items = (ArrayList) lists.get(i);
        for (int j = 0; j < items.size() - 1; j++) {
          HistoryItem hi = (HistoryItem) items.get(j);
          if (hi != null &&
              (hi.getLabel().equals("Add column") ||
              (hi.getLabel().equals("Remove column")) ||
              (hi.getLabel().equals("Remove gaps")) ||
              (hi.getLabel().equals("Delete sequences")) ||
              (hi.getLabel().equals("Sequences Added")))) {
            reload = true;
            updateViews();
          }
        }
      }
    }
    if (reload) {
      fireReloadEvent();
    } else {
      updateViews();
    }
  }


  /**
   *  Description of the Method
   */
  protected void updatePairings() {
    mainAlignment.updatePairings();
  }


  /**
   *  Description of the Method
   *
   * @param  index  Description of the Parameter
   * @return        Description of the Return Value
   */
  public HistoryItem removeColumn(int index) {
    char[] removed;
    if (mainAlignment != null) {
      removed = mainAlignment.removeColumn(index);
      RemoveColumn item = new RemoveColumn(index, removed);
      //history.add(item);
      fireReloadEvent();
      return item;
    }
    return null;
  }


  /**
   *  Description of the Method
   *
   * @param  elements  Description of the Parameter
   */
  public void removeColumn(ArrayList elements) {
    ArrayList historyItems = new ArrayList();
    for (int i = 0; i < elements.size(); i++) {
      HistoryElement element = (HistoryElement) elements.get(i);
      HistoryItem item = removeColumn(element.getColIndex());
      if (item != null) {
        historyItems.add(item);
      }
    }
    historyItems.add("Remove column");
    if (!historyItems.isEmpty()) {
      history.add(historyItems);
    }
  }


  /**
   *  Description of the Method
   *
   * @return    Description of the Return Value
   */
  public int removeGaps() {
    IntArray ia = new IntArray();
    int removed = 0;
    for (int i = mainAlignment.getLength() - 1; i >= 0; i--) {
      boolean gapCheck = true;
      Enumeration e = getSequenceKeys();
      while (e.hasMoreElements()) {
        if (getSequence((String) e.nextElement()).getSymbolAt(i) != '-') {
          gapCheck = false;
          break;
        }
        //removeColumn(i);
        //mainAlignment.removeColumn(i);

      }
      if (gapCheck) {
        ia.add(i);
        removed++;
      }
    }
    if (ia.getLength() > 0) {
      char[] mask = new char[ia.getLength()];
      for (int i = 0; i < ia.getLength(); i++) {
        if (getPairingmask() != null) {
          mask[i] = getPairingmask().getSymbolAt(ia.get(i));
        }
        mainAlignment.removeColumn(ia.get(i));
      }
      RemoveGaps item = new RemoveGaps(ia, mask);
      ArrayList list = new ArrayList();
      list.add(item);
      list.add(new String("Remove gaps"));
      history.add(list);
      //history.add(item);
      fireReloadEvent();
      updatePairings();
      updateViews();
    }
    return removed;
  }


  /**
   *  Adds a feature to the Column attribute of the AlignmentController object
   *
   * @param  index  The feature to be added to the Column attribute
   * @return        Description of the Return Value
   */
  public HistoryItem addColumn(int index) {
    mainAlignment.addColumn(index, '-');
    AddColumn item = new AddColumn(index);
    //history.add(item);
    fireReloadEvent();
    return item;
  }


  /**
   *  Adds a feature to the Column attribute of the AlignmentController object
   *
   * @param  elements  The feature to be added to the Column attribute
   */
  public void addColumn(ArrayList elements) {
    ArrayList historyItems = new ArrayList();
    for (int i = 0; i < elements.size(); i++) {
      HistoryElement element = (HistoryElement) elements.get(i);
      HistoryItem item = addColumn(element.getColIndex());
      if (item != null) {
        historyItems.add(item);
      }
    }
    historyItems.add("Add column");
    if (!historyItems.isEmpty()) {
      history.add(historyItems);
    }
  }


  /**
   *  Description of the Method
   */
  public void fireReloadEvent() {
    for (int i = 0; i < reloadListeners.size(); i++) {
      try
	  {
    	((ReloadListener) reloadListeners.get(i)).reload();
	  }
      catch(Exception e)
	  {}
    }
  }


  /**
   *  Adds a feature to the ReloadListener attribute of the
   *  AlignmentController object
   *
   * @param  rl  The feature to be added to the ReloadListener attribute
   */
  public void addReloadListener(ReloadListener rl) {
    reloadListeners.add(rl);
  }


  /**
   *  Description of the Method
   *
   * @param  rl  Description of the Parameter
   */
  public void removeReloadListener(ReloadListener rl) {
    reloadListeners.remove(rl);
  }


  /**
   *  Gets the columnLabels attribute of the AlignmentController object
   *
   * @return    The columnLabels value
   */
  public String[] getColumnLabels() {
    //get column from 1. sequence, all sequences has the same columns except pairingmask
    Enumeration e = mainAlignment.getSequenceKeys();
    if (e.hasMoreElements()) {
      return mainAlignment.getSequence((String) e.nextElement()).getColumnLabels();
    } else {
      return null;
    }
  }


  /**
   *  Sets the history attribute of the AlignmentController object
   *
   * @param  hist  The new history value
   */
  public void setHistory(History hist) {
    history = hist;
    //history.setAlignment(mainAlignment);
  }


  /**
   *  Sets the fileHistory attribute of the AlignmentController object
   *
   * @param  fileHist  The new fileHistory value
   */
  public void setFileHistory(FileHistory fileHist) {
    fileHistory = fileHist;
    //history.setAlignment(mainAlignment);
  }
  
  public void addFileHistoryItem(FileHistoryItem newItem)
  {
	  fileHistory.add(newItem);
	  updateViews();
  }

  /**
   *  Adds a feature to the Sequences attribute of the AlignmentController
   *  object
   *
   * @param  newSequences  The feature to be added to the Sequences attribute
   * @return               Description of the Return Value
   */
  public int addSequences(Alignment newSequences) {
    int result = 0;
    String[] seqnames = newSequences.getSortedNames();
    Sequence[] sequences = new Sequence[seqnames.length];
    for (int i = 0; i < seqnames.length; i++) {
      Sequence seq = newSequences.getSequence(seqnames[i]);
      sequences[i] = seq;
      seq.setNumber(mainAlignment.getNumberOfSequences() + 1);
      if (mainAlignment.addSequence(seq)) {
        result++;
      }
    }
    if (result > 0) {
      fireReloadEvent();
    }
    AddSequences historyItem = new AddSequences(sequences);
    ArrayList list = new ArrayList();
    list.add(historyItem);
    list.add("Sequences added");
    history.add(list);
    return result;
  }


  /**
   *  Description of the Method
   *
   * @param  delSequences  Description of the Parameter
   * @return               Description of the Return Value
   */
  public boolean deleteSequences(String[] delSequences) {
    java.util.List seqs = Arrays.asList(delSequences);
    java.util.List histSeqs = new ArrayList();
    Enumeration e = mainAlignment.getSequences();
    boolean deleted = false;
    //search for pairngmask
    //delete sequences
    for(int i= 0;i<delSequences.length;i++)
    {
      if(delSequences[i].equals(mainAlignment.getPairingmaskName()))
      {
        mainAlignment.removePairingmask(getPairingmaskName());
        deleted = true;
      }
    }
    while (e.hasMoreElements()) {
      Sequence s = (Sequence) e.nextElement();
      if (seqs.contains(s.getLabel())) {
        Sequence deletedSeq = mainAlignment.removeSequence(s.getLabel());
        if (deletedSeq != null) {
          histSeqs.add(deletedSeq);
          deleted = true;
        }
      }
      
      
        
    }
    if (deleted) {
      ArrayList al = new ArrayList();
      DeleteSequences item = new DeleteSequences((Sequence[]) histSeqs.toArray(new Sequence[histSeqs.size()]));
      al.add(item);
      al.add("Sequences deleted");
      history.add(al);
      mainAlignment.sort();
      fireReloadEvent();
      updateViews();
      return true;
    }
    
    
    return false;
  }


  /**
   *  Description of the Method
   */
  public void removeColors() {
    Enumeration e = mainAlignment.getSequences();
    while (e.hasMoreElements()) {
      ((Sequence) e.nextElement()).removeColors();
    }
    if(getPairingmask() != null)
    {
    	getPairingmask().removeColors();
    }
    if(mainAlignment.getPairingMask()!=null){
    mainAlignment.getPairingMask().removeColors();
    }
    
  }


  /**
   *  Description of the Method
   *
   * @param  col1  Description of the Parameter
   * @param  col2  Description of the Parameter
   * @param  mask  Description of the Parameter
   * @return       Description of the Return Value
   */
  public HistoryItem pairColumns(int col1, int col2, char mask) {
    if (mainAlignment.pairColumns(col1, col2, mask)) {
      HistoryItem item = new PairColumns(col1, col2, mask);
      //history.add(item);
      updateViews();
      return item;
    }
    return null;
  }


  /**
   *  Description of the Method
   *
   * @param  elements  Description of the Parameter
   */
  public void pairColumns(ArrayList elements) {
    ArrayList historyItems = new ArrayList();
    for (int i = 0; i < elements.size(); i++) {
      HistoryElement element = (HistoryElement) elements.get(i);
      HistoryItem item = pairColumns(element.getCol1(), element.getCol2(), element.getMask());
      if (item != null) {
        historyItems.add(item);
      }
    }
    historyItems.add("Pair columns");
    if (!historyItems.isEmpty()) {
      history.add(historyItems);
    }
  }


  /**
   *  Description of the Method
   *
   * @param  names  Description of the Parameter
   * @param  cols   Description of the Parameter
   * @return        Description of the Return Value
   */
  public HistoryItem unpair(String[] names, int[] cols) {
    if (cols.length % 2 == 0) {
      Hashtable histData = mainAlignment.unpair(names, cols);
      HistoryItem item = new UnpairBases(names, histData);
      //history.add(item);
      updateViews();
      return item;
    }
    return null;
  }


  /**
   *  Description of the Method
   *
   * @param  elements  Description of the Parameter
   */
  public void unpair(ArrayList elements) {
    ArrayList historyItems = new ArrayList();
    for (int i = 0; i < elements.size(); i++) {
      HistoryElement element = (HistoryElement) elements.get(i);
      HistoryItem item = unpair(element.getNames(), element.getIndex());
      if (item != null) {
        historyItems.add(item);
      }
    }
    historyItems.add("Unpair bases");
    if (!historyItems.isEmpty()) {
      history.add(historyItems);
    }
  }


  /**
   *  Description of the Method
   *
   * @param  names  Description of the Parameter
   * @param  cols   Description of the Parameter
   * @return        Description of the Return Value
   */
  public HistoryItem pair(String[] names, int[] cols) {
    if (cols.length % 2 == 0) {
      mainAlignment.pair(names, cols);
      HistoryItem item = new PairBases(names, cols);
      //history.add(item);
      updateViews();
      return item;
    }
    return null;
  }


  /**
   *  Description of the Method
   *
   * @param  elements  Description of the Parameter
   */
  public void pair(ArrayList elements) {
    ArrayList historyItems = new ArrayList();
    for (int i = 0; i < elements.size(); i++) {
      HistoryElement element = (HistoryElement) elements.get(i);
      HistoryItem item = pair(element.getNames(), element.getIndex());
      if (item != null) {
        historyItems.add(item);
      }
    }
    historyItems.add("Pair bases");
    if (!historyItems.isEmpty()) {
      history.add(historyItems);
    }
  }

  /**
   *  Description of the Method
   */
  public void fitCases() {
    getMainAlignment().fitCases();
  }
  
  public void emptyPairingmask()
  {
  	if(getPairingmask()!=null)
  	{
  		char[] pm = mainAlignment.getPairingMask().empty();
  		updateViews();
  	}
  }
}

