/*
 *  SARSE, Semi-Automated RNA Sequence Editor.
 *  Copyright (C) 2004 Allan Lind-Thomsen
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dk.kvl.gui.alignment;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.text.*;
import java.io.*;
import java.awt.event.*;
import java.awt.*;

import dk.kvl.gui.properties.*;
import dk.kvl.gui.event.*;
import dk.kvl.gui.components.*;

import java.util.*;
import dk.kvl.controller.*;
import dk.kvl.tools.*;
import java.awt.datatransfer.*;

/**
 *  Description of the Class
 *
 * @author     allan
 * @created    January 12, 2004
 */
public class AlignmentFrame extends JFrame implements ActionListener, ClipboardOwner, ComponentListener {
	
	/**
	 *  Description of the Field
	 */
	
	public final static ThreadGroup sarseThreads = new ThreadGroup("Sarse Threads");
	private String releaseDate = "12. oct. 2007";
	private String version = "1.37";
	private JFileChooser fileChooser = null;
	private String visibleComponent;
	private Hashtable projects = new Hashtable();
	private static String SARSE_HOME;
	private JMenu fileMenu;
	private ProjectController projectController;
	private JFrame historyWindow;
	private JFrame overviewWindow;
	private JPanel overviewCards;
	private JPanel historyCards;
	//
	private File newProjectFile = null;
	private DependingTextFields newProjectInputFields = null;
	private JFrame newProjectInputFrame = null;
	private Hashtable propertyFiles;
	private boolean analyzing = false;
	private final JPanel basePanel = new JPanel();
	//
	private JCheckBoxMenuItem history;
	private JCheckBoxMenuItem overview;
	private JCheckBoxMenuItem grid;
	
	//command line args
	private static String openFile, configurationName, openProject;
	private boolean selectionMode, splitMode, splitAlign;
	private int operationMode;
	private Point p1, p2, p3;
	private int divider;
	private JRadioButtonMenuItem primaryOp, bothSel;
	private AlignmentProperties properties;
	private JMenuItem wDir;
	private static AlignmentFrame thisFrame;
	private JTabbedPane tabs = new JTabbedPane();
	private String cwd;
	
	/**
	 *  Constructor for the AlignmentFrame object
	 *
	 * @param  title  Description of the Parameter
	 */
	public AlignmentFrame(String title,String[] args) throws Exception {
		super(title);
		windowListener();
		loadProperties();
		if (properties.getLastPath() == null) {
			properties.setLastPath(AlignmentProperties.getProjectDir());
		}
		projectController = new ProjectController();
		getContentPane().setLayout(new BorderLayout());
		//System.out.println("user.dir = "+System.getProperty("user.dir"));
		cwd = System.getProperty("user.dir");
		getContentPane().add(tabs);
		tabs.addChangeListener(new ChangeListener(){

		public void stateChanged(ChangeEvent e) {
			
			int index = tabs.getSelectedIndex();
			if(index > -1)
			{
				showHistory(currentProject());
				showOverview(currentProject());
				setWdir(currentProject());
			}
		}
		});
		createMenuBar();
		fileChooser = new JFileChooser(System.getProperty("user.dir"));
		setLocation(50,50);
		setSize(800, 600);
		setVisible(true);
		createOverviewWindow();
		createHistoryWindow();
		if (openFile != null) {
			newProject(openFile);
		}
		thisFrame = this;
		if(args.length > 0)
		{
			doArgs(args);
		}
	}
	
	private void doArgs(String[] args) {
		String file = null;
		String project = null;
		for(int i= 0;i<args.length;i++)
		{
			if(args[i].startsWith("--file="))
			{
				file= args[i].substring(7);
			}
			else if (args[i].startsWith("--project="))
			{
				project= args[i].substring(10);
			}
		}
		if(file == null )
		{
			if(project!=null)
			{
				openProject(project);
			}
			
		}
		else if (project == null) {
			newProject(file);
		}
		else
		{
			newProjectFile = new File(file);
			if(projectExists(project)){
				int confirm = JOptionPane.showConfirmDialog(null, "The project already exists. Continue anyway?");
				if(confirm== JOptionPane.OK_OPTION){
					createProject(project, System.getProperty("user.dir")+"/"+project);
				}else{
				projectname();
				}
			}else{
				createProject(project, System.getProperty("user.dir")+"/"+project);
			}
		}
	}

	private void showHistory(String name) {
		((CardLayout)historyCards.getLayout()).show(historyCards, name);
		historyCards.validate();
		historyCards.repaint();
	}
	private void showOverview(String name) {
		((CardLayout)overviewCards.getLayout()).show(overviewCards, name);
		overviewCards.validate();
		overviewCards.repaint();
	}
	/**
	 *  Description of the Method
	 */
	public void loadProperties() {
		try {
			File f = new File(AlignmentProperties.getProjectDir() + "properties.obj");
			if (f.exists()) {
				try
				{
				ObjectInputStream ois = new ObjectInputStream(new FileInputStream(f));
				properties = (AlignmentProperties) ois.readObject();
				ois.close();
				properties.setFontName(properties.getTableFont().getName());
				properties.setFontSize(properties.getFontSize());
				validate();
				repaint();
				}
				catch (Exception e)
				{
					properties = new AlignmentProperties();
				}
			} else {
				properties = new AlignmentProperties();
			}
			properties.loadProgramController();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	
	/**
	 *  Description of the Method
	 */
	protected void createMenuBar() {
		JMenuBar menuBar = new JMenuBar();
		menuBar.add(getFileMenu());
		menuBar.add(getEditMenu());
		menuBar.add(getViewMenu());
		menuBar.add(getToolsMenu());
		menuBar.add(getInfoMenu());
		menuBar.add(Box.createHorizontalGlue());
		menuBar.add(getHelpMenu());
		menuBar.add(Box.createHorizontalStrut(5));
		setJMenuBar(menuBar);
	}
	
	
	/**
	 *  Gets the editMenu attribute of the AlignmentFrame object
	 *
	 * @return    The editMenu value
	 */
	public JMenu getEditMenu() {
		JMenu editMenu = new JMenu("Edit");
		
		JMenuItem undoItem = new JMenuItem("Undo");
		undoItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z, ActionEvent.CTRL_MASK));
		undoItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							//System.out.print("undo");
							getActiveProjectContainer().getAlignmentController().undo();
						}
					}
				});
		editMenu.add(undoItem);
		
		JMenuItem redoItem = new JMenuItem("Redo");
		redoItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z, 3));
		redoItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							getActiveProjectContainer().getAlignmentController().redo();
						}
					}
				});
		editMenu.add(redoItem);
		
		JMenuItem moveLeftItem = new JMenuItem("Move left");
		moveLeftItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_M, ActionEvent.CTRL_MASK));
		moveLeftItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							getActiveProjectContainer().rotateRight();
						}
					}
				});
		editMenu.add(moveLeftItem);
		
		JMenuItem moveRightItem = new JMenuItem("Move right");
		moveRightItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_M, 3));
		moveRightItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							getActiveProjectContainer().rotateLeft();
						}
					}
				});
		editMenu.add(moveRightItem);
		
//		JMenuItem toUpperItem = new JMenuItem("To upper case");
//		toUpperItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T, ActionEvent.CTRL_MASK));
//		toUpperItem.addActionListener(
//				new ActionListener() {
//					public void actionPerformed(ActionEvent ae) {
//						if(getActiveProjectContainer() != null)
//						{
//							getActiveProjectContainer().toUpperCase();
//						}
//					}
//				});
//		editMenu.add(toUpperItem);
//		
//		JMenuItem toLowerItem = new JMenuItem("To lower case");
//		toLowerItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T, 3));
//		toLowerItem.addActionListener(
//				new ActionListener() {
//					public void actionPerformed(ActionEvent ae) {
//						if(getActiveProjectContainer() != null)
//						{
//							getActiveProjectContainer().toLowerCase();
//						}
//					}
//				});
//		editMenu.add(toLowerItem);
		
		JMenuItem changeItem = new JMenuItem("Change symbol");
		changeItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_T, ActionEvent.ALT_MASK));
		changeItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							getActiveProjectContainer().change();
						}
					}
				});
		editMenu.add(changeItem);
		
		JMenuItem pairItem = new JMenuItem("Pair bases");
		pairItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_B, ActionEvent.CTRL_MASK));
		pairItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							getActiveProjectContainer().pairBases();
						}
					}
				});
		editMenu.add(pairItem);
		
		JMenuItem columnItem = new JMenuItem("Pair columns");
		columnItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_K, ActionEvent.CTRL_MASK));
		columnItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							getActiveProjectContainer().pairColumns();
						}
					}
				});
		editMenu.add(columnItem);
		
		JMenuItem gotoItem = new JMenuItem("Go to ..");
		gotoItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_J, ActionEvent.CTRL_MASK));
		gotoItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							getActiveProjectContainer().jump();
						}
					}
				});
		editMenu.add(gotoItem);
		
		 history = new JCheckBoxMenuItem("History", false);
		history.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						//setHistoryEnabled();
							historyWindow.setVisible(history.isSelected());
						
					}
				});
		editMenu.add(history);
		
		return editMenu;
	}
	
	
	/**
	 *  Creates a dialog prompting for which project to delete
	 */
	public void deleteProject() {
		String[] projectFiles = projectController.getProjectNames();
		boolean exists = false;
		if (projectFiles.length > 0) {
			String project = (String) JOptionPane.showInputDialog(null, "Select a project", "Select", JOptionPane.QUESTION_MESSAGE, null, projectFiles, projectFiles[0]);
			if (project != null) {
				Enumeration e = projects.keys();
				while (e.hasMoreElements()) {
					if (project.equals((String) e.nextElement())) {
						exists = true;
					}
				}
				if (exists) {
					JOptionPane.showMessageDialog(this, "You cannot delete an open project");
				} else {
					deleteProject(project);
				}
			}
		} else {
			JOptionPane.showMessageDialog(null, "No projects");
		}
	}
	
	
	/**
	 *  Deletes file name.prj and directory name
	 *
	 * @param  name  Description of the Parameter
	 */
	public void deleteProject(String name) {
		boolean fileDeleted = false;
		boolean dirDeleted = false;
		if (JOptionPane.showConfirmDialog(this, "Delete project " + name + "?") == 0) {
			ProjectProperties prop = projectController.getProjectProperties(name);
			File file = new File(AlignmentProperties.getProjectDir() + "/" + name + ".prj");
			fileDeleted = file.delete();
			//delete history
			file = new File(prop.getPath() + "/"+history);
			if(file.exists())
			{
				if(file.isDirectory())
				{
					File files[] = file.listFiles();
					for(int i = 0;i<files.length;i++)
					{
						files[i].delete();
					}
				}
				dirDeleted = file.delete(); 
			}
		}
		if (fileDeleted && dirDeleted) {
			JOptionPane.showMessageDialog(this, "Project " + name + " deleted");
		}
	}
	
	
	/**
	 *  Description of the Method
	 */
	public void saveProject() {
		if (getActiveProjectContainer() != null) {
			setCursor(new Cursor(Cursor.WAIT_CURSOR));
			getActiveProjectContainer().save();
			setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
		}
	}
	
	
	/**
	 *  Gets the viewMenu attribute of the AlignmentFrame object
	 *
	 * @return    The viewMenu value
	 */
	public JMenu getViewMenu() {
		
		JMenu view = new JMenu("View");
		//
		view.add(getColorMenu());
		
		JMenu selectionMode = new JMenu("Selection mode");
		ButtonGroup selModes = new ButtonGroup();
		JRadioButtonMenuItem primarySel = new JRadioButtonMenuItem("Primary selection");
		primarySel.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						setSelectionMode(true);
						selectionMode(true);
						setOperationMode(1);
						operationMode(1);
						primaryOp.setSelected(true);
					}
				});
		selectionMode.add(primarySel);
		selModes.add(primarySel);
		
		bothSel = new JRadioButtonMenuItem("Both");
		bothSel.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						setSelectionMode(false);
						selectionMode(false);
					}
				});
		bothSel.setSelected(true);
		selectionMode.add(bothSel);
		selModes.add(bothSel);
		//might be added later but must then be syncronized with the icons in AlignmentControlPanel
		//view.add(selectionMode);
		
		//JMenu operationMode = new JMenu("Operation mode");
		ButtonGroup opModes = new ButtonGroup();
		primaryOp = new JRadioButtonMenuItem("Primary selection");
		primaryOp.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						setOperationMode(1);
						operationMode(1);
					}
				});
		//operationMode.add(primaryOp);
		opModes.add(primaryOp);
		
		JRadioButtonMenuItem bothOp = new JRadioButtonMenuItem("Both");
		bothOp.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						setOperationMode(0);
						operationMode(0);
						setSelectionMode(false);
						selectionMode(false);
						bothSel.setSelected(true);
					}
				});
		bothOp.setSelected(true);
		//operationMode.add(bothOp);
		opModes.add(bothOp);
		
		JRadioButtonMenuItem secondaryOp = new JRadioButtonMenuItem("Secondary selection");
		secondaryOp.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent e) {
						setOperationMode(2);
						operationMode(2);
						setSelectionMode(false);
						selectionMode(false);
						bothSel.setSelected(true);
					}
				});
		//operationMode.add(secondaryOp);
		opModes.add(secondaryOp);
		//view.add(operationMode);
		//
		JMenu splitModes = new JMenu("Split mode");
		ButtonGroup modes = new ButtonGroup();
		JRadioButtonMenuItem horizontalMode = new JRadioButtonMenuItem("Horizontal split");
		horizontalMode.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						splitMode(true);
					}
				});
		splitModes.add(horizontalMode);
		modes.add(horizontalMode);
		JRadioButtonMenuItem verticalMode = new JRadioButtonMenuItem("Vertical split");
		verticalMode.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						splitMode(false);
					}
				});
		splitModes.add(verticalMode);
		modes.add(verticalMode);
		verticalMode.setSelected(true);
		splitAlign = false;
		view.add(splitModes);
		
		JMenuItem split = new JMenuItem("Split screen");
		split.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_2, ActionEvent.CTRL_MASK));
		split.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							
							getActiveProjectContainer().getControlPanel().enableSplitButtons(true);
							getActiveProjectContainer().split();
						}
					}
				});
		view.add(split);
		
		JMenuItem unsplit = new JMenuItem("Unsplit screen");
		unsplit.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_1, ActionEvent.CTRL_MASK));
		unsplit.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							getActiveProjectContainer().getControlPanel().enableSplitButtons(false);
							getActiveProjectContainer().unsplit();
						}
					}
				});
		view.add(unsplit);
		
		//
		view.add(getFontMenu());
		view.add(getFontSizeMenu());
		//
		JMenuItem sub = new JMenuItem("New view");
		sub.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							getActiveProjectContainer().subalignment();
						}
					}
				});
		//view.add(sub);
		//
		JMenuItem clear = new JMenuItem("Clear selection(s)");
		clear.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, 3));
		clear.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							getActiveProjectContainer().clear();
						}
					}
				});
		view.add(clear);
		//
//		grid = new JCheckBoxMenuItem("Tablegrid", false);
//		grid.addActionListener(
//				new ActionListener() {
//					public void actionPerformed(ActionEvent ae) {
//						if(getActiveProjectContainer() != null)
//						{
//							
//							getActiveProjectContainer().getProperties().setGrid(grid.isSelected());
//							getActiveProjectContainer().updateGrid();
//						}
//					}
//				});
//		view.add(grid);
		
		//
		JCheckBoxMenuItem cb = new JCheckBoxMenuItem("Fixed scrolling", true);
		cb.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						setTraceEnabled(((JCheckBoxMenuItem) ae.getSource()).isSelected());
					}
				});
		view.add(cb);
		//base coloring
		JCheckBoxMenuItem bcItem = new JCheckBoxMenuItem("Base coloring", false);
		bcItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						setBasecoloringEnabled(((JCheckBoxMenuItem) ae.getSource()).isSelected());
						repaint();
					}
				});
		view.add(bcItem);
		//overview
		overview = new JCheckBoxMenuItem("Overview", false);
		overview.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						//setOverviewEnabled();
						setOverviewVisible(overview.isSelected());
					}
				});
		view.add(overview);
		//commit view
		return view;
	}
	
	
	/**
	 *  Sets the operationMode attribute of the AlignmentFrame object
	 *
	 * @param  mode  The new operationMode value
	 */
	public void setOperationMode(int mode) {
		operationMode = mode;
	}
	
	
	/**
	 *  Sets the selectionMode attribute of the AlignmentFrame object
	 *
	 * @param  mode  The new selectionMode value
	 */
	public void setSelectionMode(boolean mode) {
		selectionMode = mode;
	}
	
	
	/**
	 *  Gets the toolsMenu attribute of the AlignmentFrame object
	 *
	 * @return    The toolsMenu value
	 */
	public JMenu getToolsMenu() {
		JMenu tools = new JMenu("Tools");
		JMenuItem runItem = new JMenuItem("Programs");
		
		runItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F9, 0));
		runItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							Runnable r = new Runnable()
							{
								public void run()
								{
									analyze();
									//setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
								}
							};
							SwingUtilities.invokeLater(r);
						}
					}
				});
		tools.add(runItem);
		
		JMenuItem addSeqs = new JMenuItem("Add sequences");
		addSeqs.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D, ActionEvent.CTRL_MASK));
		addSeqs.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().addSequences();
						}
					}
				});
		tools.add(addSeqs);
//		JMenuItem addPairingmask = new JMenuItem("Add Pairingmask");
//		//addSeqs.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D, ActionEvent.CTRL_MASK));
//		addPairingmask.addActionListener(
//				new ActionListener() {
//					public void actionPerformed(ActionEvent ae) {
//						if (getActiveProjectContainer() != null) {
//							getActiveProjectContainer().addPairingmask();
//						}
//					}
//				});
//		tools.add(addPairingmask);
		JMenuItem delSeqs = new JMenuItem("Delete sequences");
		delSeqs.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_D, 3));
		delSeqs.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().removeSequences();
						}
					}
				});
		tools.add(delSeqs);
		
		JMenuItem addCol = new JMenuItem("Add column");
		addCol.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_E, ActionEvent.CTRL_MASK));
		addCol.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().addColumn();
						}
					}
				});
		tools.add(addCol);
		
		JMenuItem removeCol = new JMenuItem("Remove column");
		removeCol.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_E, 3));
		removeCol.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().removeColumn();
						}
					}
				});
		tools.add(removeCol);
		
		JMenuItem removeColors = new JMenuItem("Remove coloring");
		removeColors.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R, ActionEvent.CTRL_MASK));
		removeColors.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().removeColors();
						}
					}
				});
		tools.add(removeColors);
		
		JMenuItem removeItem = new JMenuItem("Remove gaps");
		removeItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_G, ActionEvent.CTRL_MASK));
		removeItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().removeGaps();
						}
					}
				});
		tools.add(removeItem);
		JMenuItem emptyPair = new JMenuItem("Empty pairingmask");
		//emptyPair.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F, ActionEvent.CTRL_MASK));
		emptyPair.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						
						if(JOptionPane.showConfirmDialog(getActiveProjectContainer(), "Do you want to empty the pairingmask?","Please confirm",JOptionPane.YES_NO_OPTION)==JOptionPane.OK_OPTION){
						getActiveProjectContainer().emptyPairingmask();
						}
					}
				});
		tools.add(emptyPair);
		JMenuItem ulItem = new JMenuItem("Fit alignment to pairingmask");
		//ulItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F, ActionEvent.CTRL_MASK));
		ulItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						getActiveProjectContainer().fitCases();
					}
				});
		tools.add(ulItem);
		return tools;
	}
	
	
	/**
	 *  Gets the helpMenu attribute of the AlignmentFrame object
	 *
	 * @return    The helpMenu value
	 */
	public JMenu getHelpMenu() {
		JMenu helpMenu = new JMenu("Help");
		
		JMenuItem docItem = new JMenuItem("Documentation");
		docItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F1, 0));
		docItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						showDocumentation();
					}
				});
		helpMenu.add(docItem);
		
		
		return helpMenu;
	}
	
	
	/**
	 *  Description of the Method
	 */
	public void showDocumentation() {
		JFrame helpWindow = new JFrame("Documentation");
		JTextComponent text = new JEditorPane();
		text.setEditable(false);
		try {
			text.read(new BufferedReader(new FileReader(AlignmentProperties.getHomeDir() + "/README")), null);
		} catch (Exception e) {
			JOptionPane.showMessageDialog(helpWindow, "An error occurred reading the documentation");
		}
		
		JScrollPane docScroll = new JScrollPane(text, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		helpWindow.getContentPane().add(docScroll);
		helpWindow.setSize(600, 500);
		helpWindow.setVisible(true);
	}
	
	
	/**
	 *  Description of the Method
	 */
	public void showTutorial() {
		JFrame helpWindow = new JFrame("Tutorial");
		JTextComponent text = new JEditorPane();
		text.setEditable(false);
		try {
			text.read(new BufferedReader(new FileReader(AlignmentProperties.getHomeDir() + "/documentaion/tutorial.txt")), null);
		} catch (Exception e) {
			JOptionPane.showMessageDialog(helpWindow, "An error occurred reading the tutorial");
		}
		JScrollPane docScroll = new JScrollPane(text, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		helpWindow.getContentPane().add(docScroll);
		helpWindow.setSize(600, 500);
		helpWindow.setVisible(true);
	}
	
	
	/**
	 *  Gets the fileMenu attribute of the AlignmentFrame object
	 *
	 * @return    The fileMenu value
	 */
	public JMenu getFileMenu() {
		fileMenu = new JMenu("File");
		
		JMenuItem newItem = new JMenuItem("New");
		newItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, ActionEvent.CTRL_MASK));
		newItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						Runnable r = new Runnable()
						{
							public void run()
							{
								newProject();
							}
						};
						SwingUtilities.invokeLater(r);
					}
				});
		fileMenu.add(newItem);
		//
		JMenuItem openItem = new JMenuItem("Open project");
		openItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, ActionEvent.CTRL_MASK));
		openItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						Runnable r = new Runnable()
						{
							public void run()
							{
								openProject();
							}
						};
						SwingUtilities.invokeLater(r);
					}
				});
		fileMenu.add(openItem);
		//
		JMenuItem saveItem = new JMenuItem("Save project");
		saveItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, ActionEvent.CTRL_MASK));
		saveItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							saveProject();
						}
					}
				});
		fileMenu.add(saveItem);
		JMenuItem export = new JMenuItem("Export file");
		export.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().export();
						}
					}
				});
		fileMenu.add(export);
		//close
		JMenuItem closeItem = new JMenuItem("Close");
		closeItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_W, ActionEvent.CTRL_MASK));
		closeItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if(getActiveProjectContainer() != null)
						{
							getActiveProjectContainer().save();
							close();
						}
					}
				});
		fileMenu.add(closeItem);
		//
		JMenuItem deleteItem = new JMenuItem("Delete project");
		deleteItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0));
		deleteItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						deleteProject();
					}
				});
		fileMenu.add(deleteItem);
		//save as..
		/*
		 *  JMenuItem saveAsItem = new JMenuItem("Save alignment as...");
		 *  saveAsItem.addActionListener(
		 *  new ActionListener()
		 *  {
		 *  public void actionPerformed(ActionEvent ae)
		 *  {
		 *  if (getActiveProjectContainer() != null)
		 *  {
		 *  getActiveProjectContainer().saveAs();
		 *  }
		 *  }
		 *  });
		 *  fileMenu.add(saveAsItem);
		 */
		//
		
		fileMenu.addSeparator();
		
		//exit
		JMenuItem exitItem = new JMenuItem("Exit");
		exitItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q, ActionEvent.CTRL_MASK));
		exitItem.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						exit();
					}
				});
		fileMenu.add(exitItem);
		//adding menubar
		return fileMenu;
	}
	
	
	/**
	 *  Gets the infoMenu attribute of the AlignmentFrame object
	 *
	 * @return    The infoMenu value
	 */
	public JMenu getInfoMenu() {
		JMenu infoMenu = new JMenu("Info");
		
		JMenu dirMenu = new JMenu("Working directory");
		wDir = new JMenuItem(System.getProperty("user.dir"));
		wDir.setToolTipText("Click to copy path to clipboard");
		dirMenu.add(wDir);
		wDir.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						Clipboard clipboard = getToolkit().getSystemClipboard();
						clipboard.setContents(new StringSelection(wDir.getText()), thisFrame);
					}
				});
		infoMenu.add(dirMenu);
		JMenuItem release = new JMenuItem("Version: "+version);
		
		infoMenu.add(release);
		JMenuItem date = new JMenuItem(releaseDate);
		infoMenu.add("Released:");
		infoMenu.add(date);
		return infoMenu;
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  clipboard  Description of the Parameter
	 * @param  contents   Description of the Parameter
	 */
	public void lostOwnership(Clipboard clipboard, Transferable contents) {
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @return    The alignmentName value
	 */
	/*
	 *  protected void open()
	 *  {
	 *  newProject();
	 *  }
	 */
	
	/**
	 *  Gets the alignmentName attribute of the AlignmentFrame object
	 *
	 * @return    The alignmentName value
	 */
	protected String getAlignmentName() {
		return JOptionPane.showInputDialog("Write a name for the alignment");
	}
	
	
	/**
	 *  Description of the Method
	 *
	 */
	/*
	 *  public void open(File file, String name)
	 *  {
	 *  try
	 *  {
	 *  addProject(new AlignmentContainer(file, name, selectionMode, operationMode, false, true, null, null, null), name);
	 *  }
	 *  catch (FileNotFoundException fnf)
	 *  {
	 *  JOptionPane.showMessageDialog(null, fnf.getMessage());
	 *  fnf.printStackTrace();
	 *  }
	 *  catch (IOException ioe)
	 *  {
	 *  JOptionPane.showMessageDialog(null, ioe.getMessage());
	 *  ioe.printStackTrace();
	 *  }
	 *  catch (Exception e)
	 *  {
	 *  JOptionPane.showMessageDialog(null, e.getMessage());
	 *  e.printStackTrace();
	 *  }
	 *  }
	 */
	
	/**
	 *  Description of the Method
	 */
	public void exit() {
		boolean analyzing = false;
		while (visibleComponent != null) {
			close();
		}
		if (sarseThreads.activeCount() > 0) {
			dispose();
		} else {
			System.exit(0);
		}
	}
	
	
	/**
	 *  Description of the Method
	 */
	public void removeProjectName() {
		MenuElement[] popupElement = fileMenu.getSubElements();
		for (int i = 0; i < popupElement.length; i++) {
			MenuElement[] menuItemElements = popupElement[i].getSubElements();
			for (int j = 0; j < menuItemElements.length; j++) {
				if (menuItemElements[j] instanceof javax.swing.JMenuItem) {
					if (((JMenuItem) menuItemElements[j]).getText().equals(visibleComponent)) {
						fileMenu.remove((JMenuItem) menuItemElements[j]);
					}
				}
			}
		}
	}
	
	
	
	/**
	 *  Description of the Method
	 */
	public void close() {
		if (getActiveProjectContainer().isAnalyzing()) {
			analyzing = true;
		}
		historyWindow.remove(tabs.getComponentAt(tabs.getSelectedIndex()));
		overviewWindow.remove(tabs.getComponentAt(tabs.getSelectedIndex()));
		saveSplitMode();
		getActiveProjectContainer().cleanup();
		tabs.remove(getActiveProjectContainer());
		getContentPane().remove(getProjectContainer(visibleComponent));
		removeProjectName();
		projects.remove(visibleComponent);
		//select a new project if one exist
		Enumeration e = projects.keys();
		//
		if (e.hasMoreElements()) {
			String newProject = (String) e.nextElement();
			//((CardLayout)getContentPane().getLayout()).show(getContentPane(), newProject);
			visibleComponent = newProject;
			wDir.setText(projectController.getProjectProperties(newProject).getPath());
		} else {
			visibleComponent = null;
			wDir.setText("No working directory");
			//((CardLayout) getContentPane().getLayout()).show(getContentPane(), "base");
		}
		System.gc();
		if(currentProject()!=null)
		{
		showHistory(currentProject());
		showOverview(currentProject());
		}
		else
		{
			showHistory("default");
			showOverview("default");
			
		}
		getContentPane().validate();
		getContentPane().repaint();
	}
	
	
	/**
	 *  Gets the projectContainer attribute of the AlignmentFrame object
	 *
	 * @param  name  Description of the Parameter
	 * @return       The projectContainer value
	 */
	public AlignmentContainer getProjectContainer(String name) {
		return (AlignmentContainer) projects.get(name);
	}
	
	
	/**
	 *  Gets the activeProjectContainer attribute of the AlignmentFrame object
	 *
	 * @return    The activeProjectContainer value
	 */
	public AlignmentContainer getActiveProjectContainer() {
		if (visibleComponent != null) {
			//AlignmentContainer ac = (AlignmentContainer) projects.get(visibleComponent);
			if(tabs.getSelectedIndex() >-1)
			{
			AlignmentContainer ac = (AlignmentContainer)tabs.getComponentAt(tabs.getSelectedIndex());
			
			if (ac == null) {
				JOptionPane.showMessageDialog(null, "You must start a project first!");
			}
			return ac;
			}
		}
		return null;
	}
	
	
	/**
	 *  The main program for the AlignmentFrame class
	 *
	 * @param  args  The command line arguments
	 */
	public static void main(String[] args) {
		SARSE_HOME = System.getProperty("SARSE_HOME");
		if (SARSE_HOME == null) {
			System.out.println("You must set the $SARSE_HOME variable before running the application!");
			System.exit(1);
		}
		AlignmentProperties.setHomeDir(SARSE_HOME);
		AlignmentProperties.setProjectDir(System.getProperty("user.home")+"/.sarse/");
//		for (int i = 0; i < args.length; i++) {
//			System.out.println(args[i]);
//		}
//		if (args.length > 0) {
//			System.out.println("args > 0");
//			for (int i = 0; i < args.length; i++) {
//				String s = null;
//				if (args[i].equals("--al")) {
//					System.out.println("args[i].equals(--alignment)" + i);
//					if (i + 1 < args.length) {
//						s = args[i + 1].trim();
//						System.out.println("args + 1: " + s);
//						String[] formats = s.split("\\.");
//						for (int h = 0; h < formats.length; h++) {
//							System.out.println(formats[h]);
//						}
//						if (formats.length > 0) {
//							String format = formats[formats.length - 1];
//							if (format.equals("col") || format.equals("txt") || format.equals("widetxt")) {
//								openFile = args[i + 1];
//								System.out.println(openFile.toString());
//							} else {
//								System.out.println("You need to specify a .col or .txt file to open.");
//								System.exit(1);
//							}
//						}
//					}
//				}
//			}
//		}
		//System.out.println("Working dir: "+System.getProperty("user.dir"));
		try
		{
		new AlignmentFrame("SARSE - Semi-Automated RNA Sequence Editor",args		);
		}
		catch( Exception e)
		{
			e.printStackTrace();
		}
	}
	
	
	/**
	 *  Adds a feature to the Project attribute of the AlignmentFrame object
	 *
	 * @param  container  The feature to be added to the Project attribute
	 * @param  name       The feature to be added to the Project attribute
	 */
	public void addProject(AlignmentContainer container, String name) {
		tabs.add(container, name);
		projects.put(name, container);
		//((CardLayout)getContentPane().getLayout()).show(getContentPane(), name);
		//tabs.setSelectedComponent(container);
		visibleComponent = name;
//		JMenuItem projectItem = new JMenuItem(name);
//		projectItem.addActionListener(
//				new ActionListener() {
//					public void actionPerformed(ActionEvent ae) {
//						String pName = ae.getActionCommand();
//						if (!pName.equals(visibleComponent)) {
//							
//							visibleComponent = pName;
//							wDir.setText(projectController.getProjectProperties(pName).getPath());
//						}
//					}
//				});
//		fileMenu.add(projectItem, 6);
		historyCards.add(container.getHistoryPanel(),name);
		overviewCards.add(container.getOverviewPanel(overviewWindow),name);
		showHistory(currentProject());
		showOverview(currentProject());
	}
	
	
	/**
	 *  Description of the Method
	 */
	public void properties() {
		
	}
	
	
	/**
	 *  Description of the Method
	 */
	public void analyze() {
		
		AlignmentContainer ac = getActiveProjectContainer();
		ac.save();
		ac.analyze();
		
		//close();
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  fileName  Description of the Parameter
	 */
	/*  public void changeAlignment(String fileName) {
	 //assuming the project is the same
	  String oldName = visibleComponent;
	  close();
	  projectController = new ProjectController();
	  openProject(oldName);
	  }
	  */
	
	/**
	 *  Sets the traceEnabled attribute of the AlignmentFrame object
	 *
	 * @param  trace  The new traceEnabled value
	 */
	public void setTraceEnabled(boolean trace) {
		getActiveProjectContainer().setTraceEnabled(trace);
	}
	
	
	/**
	 *  Description of the Method
	 */
	public void openProject() {
		String[] names = projectController.getProjectNames();
		if (names.length > 0) {
			String project = (String) JOptionPane.showInputDialog(null, "Select a project", "Select", JOptionPane.QUESTION_MESSAGE, null, names, names[0]);
			if (project != null) {
				openProject(project);
			}
		} else {
			JOptionPane.showMessageDialog(null, "No projects");
		}
	}
	
	
	/**
	 *  Description of the Method
	 */
	public void saveSplitMode() {
		AlignmentPanel ap = getActiveProjectContainer().getSelectedProject();
		splitMode = ap.isSplit();
		if (splitMode) {
			divider = ap.getSplitPane().getDividerLocation();
			p2 = ap.getTableView2().getViewport().getViewPosition();
			p3 = ap.getTableView3().getViewport().getViewPosition();
			splitAlign = ap.horizontalSplit();
		} else {
			p1 = ap.getTableView().getViewport().getViewPosition();
		}
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  name  Description of the Parameter
	 */
	public void openProject(String name) {
		Cursor orig = getCursor();
		if(projects.get(name)!=null)
		{
			JOptionPane.showMessageDialog(null, "\nProject already open!");
			return;
		}
		try{
			setCursor(new Cursor(Cursor.WAIT_CURSOR));
			ProjectProperties prop = projectController.getProjectProperties(name);
			String file = prop.getPath() + "/" + prop.getFile();
			AlignmentContainer newPrj = new AlignmentContainer(this, new File(file), name, selectionMode, operationMode, splitMode, splitAlign, p1, p2, p3, divider, prop.getPath(), properties);
			addProject(newPrj, name);
			//
			newPrj.setHistoryEnabled(history.getState());
			newPrj.setOverviewEnabled(overview.getState());
			//wDir.setText(prop.getPath());
			
		} catch (IOException e) {
			//e.printStackTrace();
			JOptionPane.showMessageDialog(null, e.getMessage() + "\nOpen Project failed!");
			if(projects.get(name)!=null)
			{
				projects.remove(name);
			}
			if(tabs.indexOfTab(name)!= -1)
			{
				tabs.remove(tabs.indexOfTab(name));
			}
			e.printStackTrace();
		} catch (Exception e) {
			//e.printStackTrace();
			JOptionPane.showMessageDialog(null, e.getMessage() + "\nOpen Project failed!");
			if(projects.get(name)!=null)
			{
				projects.remove(name);
			}
			if(tabs.indexOfTab(name)!= -1)
			{
				tabs.remove(tabs.indexOfTab(name));
			}
			e.printStackTrace();
		}
		setCursor(orig);
		
	}
	
	
	/**
	 *  Gets the historyEnabled attribute of the AlignmentFrame object
	 *
	 * @return    The historyEnabled value
	 */
	public boolean isHistoryEnabled() {
		return history.getState();
	}
	
	
	/**
	 *  Gets the overviewEnabled attribute of the AlignmentFrame object
	 *
	 * @return    The overviewEnabled value
	 */
	public boolean isOverviewEnabled() {
		return overview.getState();
	}
	
	
	/**
	 *  Description of the Method
	 */
	public void newProject() {
		int chooserStatus = fileChooser.showOpenDialog(null);
		if (chooserStatus == JFileChooser.APPROVE_OPTION) {
			newProjectFile = fileChooser.getSelectedFile();
			projectname();
		}
	}

	private void projectname() {
		newProjectInputFrame = new JFrame("Projectname");
		//newProjectInputFields = new DependingTextFields("Write a name for your project", "Project name: ", "Directory: ", "", properties.getLastPath(), this);
		java.util.Calendar rightNow = java.util.Calendar.getInstance();
		String date = rightNow.get(java.util.Calendar.YEAR)+"-"+(rightNow.get(java.util.Calendar.MONTH)+1)+"-"+rightNow.get(java.util.Calendar.DAY_OF_MONTH);
		//newProjectFile.getAbsolutePath().substring(0,newProjectFile.getAbsolutePath().lastIndexOf(System.getProperty("file.separator")))
		newProjectInputFields = new DependingTextFields("Write a name for your project", "Project name: ", "Working Directory: ", newProjectFile.getName().substring(0, newProjectFile.getName().lastIndexOf(".")),	cwd+"/"+newProjectFile.getName().substring(0, newProjectFile.getName().lastIndexOf(".")),  this);
		newProjectInputFields.getField1().addKeyListener(
				new KeyAdapter() {
					public void keyReleased(KeyEvent e) {
						if (e.getKeyCode() == KeyEvent.VK_ENTER) {
							checkInput();
						}
					}
				});
		newProjectInputFrame.getContentPane().add(newProjectInputFields);
		//newProjectInputFrame.setSize(400, 150);
		newProjectInputFrame.pack();
		newProjectInputFrame.setVisible(true);
	}
	public void checkInput(){
		checkInput(newProjectInputFields.getValue1());
	}
	
	/**
	 *  Description of the Method
	 */
	public void checkInput(String newName) {
		//String newName = newProjectInputFields.getValue1();
		if (newName != null && !newName.equals("")) {
			boolean projectExists = projectExists(newName);
			if (!projectExists) {
				createNewProject();
			} else {
				int confirm = JOptionPane.showConfirmDialog(null, "The project already exists. Continue anyway?");
				if (confirm == 0) {
					deleteProject(newName);
					createNewProject();
				} else {
					newProjectInputFrame.setTitle("Please specify new name");
					newProjectInputFields.getField1().requestFocus();
				} 
			}
		} else {
			newProjectInputFrame.setTitle("You need to specify a project name");
			newProjectInputFields.getField1().requestFocus();
		}
	}

	protected boolean projectExists(String newName) {
		String[] projectNames = projectController.getProjectNames();
		boolean projectExists = false;
		for (int i = 0; i < projectNames.length; i++) {
			if (projectNames[i].equals(newName)) {
				projectExists = true;
			}
		}
		return projectExists;
	}
	
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  filePath  Description of the Parameter
	 */
	public void newProject(String filePath) {
		newProjectFile = new File(filePath);
		newProjectInputFrame = new JFrame("Projectname");
		int start = Math.max(0, filePath.lastIndexOf('/')+1);
		newProjectInputFields = new DependingTextFields("Change the projectname and the working directory if you are not satisfied.", "Project name: ", "Working Directory: ", filePath.substring(start, filePath.lastIndexOf(".")), cwd+"/"+filePath.substring(start, filePath.lastIndexOf(".")), this);
		
		newProjectInputFields.getField1().addKeyListener(
				new KeyAdapter() {
					public void keyReleased(KeyEvent e) {
						if (e.getKeyCode() == KeyEvent.VK_ENTER) {
							createNewProject();
						}
					}
				});
		newProjectInputFrame.getContentPane().add(newProjectInputFields);
		//newProjectInputFrame.setSize(400, 150);
		newProjectInputFrame.pack();
		newProjectInputFrame.setVisible(true);
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  ae  Description of the Parameter
	 */
	public void actionPerformed(ActionEvent ae) {
		if (ae.getActionCommand().equals("OK")) {
			checkInput();
		} else {
			if (ae.getActionCommand().equals("Browse")) {
				changeDirectory();
			} else {
				newProjectInputFields.removeActionListener(this);
				newProjectInputFrame.dispose();
			}
		}
	}
	
	
	/**
	 *  Description of the Method
	 */
	public void changeDirectory() {
		JFileChooser jfc = new JFileChooser(AlignmentProperties.getProjectDir() + newProjectInputFields.getValue1());
		jfc.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		int chosen = jfc.showDialog(null, "Select");
		
		if (chosen == JFileChooser.APPROVE_OPTION) {
			//newProjectInputFields.getField2().setText(jfc.getSelectedFile().getAbsolutePath() + "/" + newProjectInputFields.getValue1());
			newProjectInputFields.setDir(jfc.getSelectedFile().getAbsolutePath()+ "/" + newProjectInputFields.getValue1());
		}
	}
	
	
	/**
	 *  Description of the Method
	 */
	public void createNewProject() {
		String projectName = newProjectInputFields.getValue1();
		String projectDirectory = newProjectInputFields.getValue2();
		createProject(projectName, projectDirectory);
		
	}

	private void createProject(String projectName, String projectDirectory) {
		String lastPath = projectDirectory;
		//String lastPath = projectDirectory.substring(0, projectDirectory.lastIndexOf(newProjectInputFields.getValue1()));
		//projectDirectory = newProjectInputFields.getValue2();//+"/"+newProjectInputFields.getValue1();
		properties.setLastPath(lastPath);
		File pDir = new File(projectDirectory);
		try {
			if (!pDir.exists()) {
				pDir.mkdirs();
			}
			if (newProjectFile.exists()) {
				File projectFile = new File(projectDirectory + "/" + newProjectFile.getName() + "/");
				StringTools.copyFile(newProjectFile.getAbsolutePath(), projectFile.getAbsolutePath(), false);
				if(newProjectInputFrame != null)
				{
				newProjectInputFields.removeActionListener(this);
				newProjectInputFrame.dispose();
				}
				try {
					AlignmentContainer ac = new AlignmentContainer(this, projectFile, projectName, selectionMode, operationMode, false, true, null, null, null, 0, pDir.getAbsolutePath(), properties);
					ac.setProjectDir(projectDirectory);
					addProject(ac, projectName);
					projectController.addProject(projectName, projectFile.getName(), projectDirectory);
				} catch (Exception e) {
					e.printStackTrace();
					JOptionPane.showMessageDialog(null, e.getMessage() + "\nAlignmentFrame.createNewProject()");
				}
				
			} else {
				if(newProjectInputFrame != null)
				{
				newProjectInputFields.removeActionListener(this);
				newProjectInputFrame.dispose();
				}
				JOptionPane.showMessageDialog(null, "The file containing your alignment doesn't exist.");
			}
		} catch (IOException ie) {
			ie.printStackTrace();
			JOptionPane.showMessageDialog(null, "Couldn't create project: Sourcefile not readable");
		}
		wDir.setText(projectDirectory);
	}
	
	
	/**
	 *  Description of the Method
	 */
	public void windowListener() {
		addWindowListener(
				new WindowAdapter() {
					public void windowClosing(WindowEvent we) {
						exit();
					}
				});
	}
	
	
	/**
	 *  Sets the basecoloringEnabled attribute of the AlignmentFrame object
	 *
	 * @param  enabled  The new basecoloringEnabled value
	 */
	public void setBasecoloringEnabled(boolean enabled) {
		getActiveProjectContainer().setBasecoloringEnabled(enabled);
	}
	
	
	/**
	 *  Sets the historyEnabled attribute of the AlignmentFrame object
	 */
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  mode  Description of the Parameter
	 */
	public void splitMode(boolean mode) {
		getActiveProjectContainer().splitMode(mode);
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  mode  Description of the Parameter
	 */
	public void selectionMode(boolean mode) {
		Collection c = projects.values();
		AlignmentContainer[] containers = (AlignmentContainer[]) c.toArray(new AlignmentContainer[c.size()]);
		for (int i = 0; i < containers.length; i++) {
			containers[i].selectionMode(mode);
		}
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  mode  Description of the Parameter
	 */
	public void operationMode(int mode) {
		Collection c = projects.values();
		AlignmentContainer[] containers = (AlignmentContainer[]) c.toArray(new AlignmentContainer[c.size()]);
		for (int i = 0; i < containers.length; i++) {
			containers[i].operationMode(mode);
		}
	}
	
	
	/**
	 *  Gets the fontMenu attribute of the AlignmentFrame object
	 *
	 * @return    The fontMenu value
	 */
	public JMenuItem getFontMenu() {
		JMenuItem fontMenu = new JMenuItem("Fonttype");
		fontMenu.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							Font[] fonts = new Font[3];
							String[] fontNames = new String[3];
							try {
								
								//Font[] fonts = GraphicsEnvironment.getLocalGraphicsEnvironment().getAllFonts();
								
								fonts[0] = Font.createFont(Font.TRUETYPE_FONT, new FileInputStream(new File(getActiveProjectContainer().getProperties().getHomeDir() + "/fonts/georgia.ttf")));
								fonts[1] = Font.createFont(Font.TRUETYPE_FONT, new FileInputStream(new File(getActiveProjectContainer().getProperties().getHomeDir() + "/fonts/arial.ttf")));
								fonts[2] = Font.createFont(Font.TRUETYPE_FONT, new FileInputStream(new File(getActiveProjectContainer().getProperties().getHomeDir() + "/fonts/verdana.ttf")));
								//fontNames = new String[fonts.length];
							} catch (Exception e) {
								System.out.println("error creating font: " + e);
								e.printStackTrace();
							}
							for (int i = 0; i < fonts.length; i++) {
								fontNames[i] = fonts[i].getName();
							}
							String newFont = (String) JOptionPane.showInputDialog(null, "Select font:", "Font", JOptionPane.PLAIN_MESSAGE, null, fontNames, getActiveProjectContainer().getProperties().getTableFont().getName());
							if (newFont != null) {
								getActiveProjectContainer().getProperties().setFontName(newFont);
								getActiveProjectContainer().updateCellSize();
								validate();
								repaint();
							}
						}
					}
				});
		return fontMenu;
	}
	
	
	/**
	 *  Gets the fontSizeMenu attribute of the AlignmentFrame object
	 *
	 * @return    The fontSizeMenu value
	 */
	public JMenuItem getFontSizeMenu() {
		JMenuItem fontSizeMenu = new JMenuItem("Fontsize");
		fontSizeMenu.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							String[] sizes = {"8", "9", "10", "11", "12", "13", "14", "15", "16"};
							String newSize = (String) JOptionPane.showInputDialog(null, "Select fontsize:", "Size", JOptionPane.PLAIN_MESSAGE, null, sizes, getActiveProjectContainer().getProperties().getTableFont().getSize() + "");
							if (newSize != null) {
								getActiveProjectContainer().getProperties().setFontSize(Integer.parseInt(newSize));
								getActiveProjectContainer().updateCellSize();
								validate();
								repaint();
							}
						}
					}
				});
		return fontSizeMenu;
	}
	
	
	/**
	 *  Gets the colorMenu attribute of the AlignmentFrame object
	 *
	 * @return    The colorMenu value
	 */
	public JMenu getColorMenu() {
		JMenu colors = new JMenu("Colors");
		//background
		JMenuItem bg = new JMenuItem("Background color");
		bg.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().setBackground();
							validate();
							repaint();
						}
					}
				});
		colors.add(bg);
		//sequencename color
		JMenuItem sColor = new JMenuItem("Sequence name color");
		sColor.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().setSequenceColor();
							validate();
							repaint();
						}
					}
				});
		colors.add(sColor);
		//text
		JMenuItem txt = new JMenuItem("Text color");
		txt.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().setTextColor();
							validate();
							repaint();
						}
					}
				});
		colors.add(txt);
		//primary selection
		JMenuItem primary = new JMenuItem("Primary selection color");
		primary.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().setPrimarySelectionColor();
							validate();
							repaint();
						}
					}
				});
		colors.add(primary);
		//secondary selection
		JMenuItem secondary = new JMenuItem("Secondary selection color");
		secondary.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().setSecondarySelectionColor();
							validate();
							repaint();
						}
					}
				});
		colors.add(secondary);
		//c-color
		JMenuItem c = new JMenuItem("C color");
		c.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().setCColor();
							validate();
							repaint();
						}
					}
				});
		colors.add(c);
		//g-color
		JMenuItem g = new JMenuItem("G color");
		g.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().setGColor();
							validate();
							repaint();
						}
					}
				});
		colors.add(g);
		//a-color
		JMenuItem a = new JMenuItem("A color");
		a.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().setAColor();
							validate();
							repaint();
						}
					}
				});
		colors.add(a);
		//u-color
		JMenuItem u = new JMenuItem("U color");
		u.addActionListener(
				new ActionListener() {
					public void actionPerformed(ActionEvent ae) {
						if (getActiveProjectContainer() != null) {
							getActiveProjectContainer().setUColor();
							validate();
							repaint();
						}
					}
				});
		colors.add(u);
		return colors;
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  ce  Description of the Parameter
	 */
	public void componentHidden(ComponentEvent ce) {
		history.setSelected(false);
	}
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  ce  Description of the Parameter
	 */
	public void componentMoved(ComponentEvent ce) { }
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  ce  Description of the Parameter
	 */
	public void componentResized(ComponentEvent ce) { }
	
	
	/**
	 *  Description of the Method
	 *
	 * @param  ce  Description of the Parameter
	 */
	public void componentShown(ComponentEvent ce) { }
	
	public void setWdir(String path)
	{
		if(currentProject() != null)
		{
			try
			{
			wDir.setText(projectController.getProjectProperties(currentProject()).getPath());
			System.setProperty("user.dir", projectController.getProjectProperties(currentProject()).getPath());
			repaint();
			}
			catch(Exception e)
			{
				//not setting wDir
			}
		}
	}
	
	public String currentProject()
	{
		if(tabs.getSelectedIndex() > -1)
		{
			return tabs.getTitleAt(tabs.getSelectedIndex());
		}
		return null;
	}
	
	protected void createHistoryWindow()
	{
		historyWindow = new JFrame("History");
		historyWindow.addWindowListener(new WindowAdapter(){
			public void windowClosing(WindowEvent e)
			{
				history.setSelected(false);
			}
		});
        historyWindow.setLocation(50,50);
        historyWindow.getContentPane().setLayout(new BorderLayout());
        historyCards = new JPanel(new CardLayout());
        historyWindow.add(historyCards);
        historyCards.add(new JPanel(),"default");
        historyWindow.setSize(300, 200);
        historyWindow.setDefaultCloseOperation(WindowConstants.HIDE_ON_CLOSE);
        
	}
	

	private void createOverviewWindow() {
		overviewWindow = new JFrame("Overview");
		overviewWindow.addWindowListener(new WindowAdapter(){
			public void windowClosing(WindowEvent e)
			{
				overview.setSelected(false);
			}
			
			
		});
		overviewWindow.setLocation(0,0);
        overviewWindow.getContentPane().setLayout(new BorderLayout());
        overviewCards = new JPanel(new CardLayout());
        overviewWindow.add(overviewCards);
        overviewCards.add(new JPanel(),"default");
		overviewWindow.setSize(300, 200);
		overviewWindow.setDefaultCloseOperation(WindowConstants.HIDE_ON_CLOSE);
		
	}


	public void setHistoryVisible(boolean visible)
	{
		historyWindow.setVisible(visible);
		if(visible)
		{
		showHistory(currentProject());
		}
	}
	
	public void setOverviewVisible(boolean visible)
	{
		overviewWindow.setVisible(visible);
		if(visible)
		{
		showOverview(currentProject());
		}
	}
}

