/*
 *  SARSE, Semi-Automated RNA Sequence Editor.
 *  Copyright (C) 2004 Allan Lind-Thomsen
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dk.kvl.gui.alignment.table;

import javax.swing.*;
import javax.swing.table.DefaultTableColumnModel;

import dk.kvl.alignmenttools.*;
import dk.kvl.gui.properties.*;
import java.awt.*;
import java.awt.event.*;
import javax.accessibility.*;
import dk.kvl.gui.alignment.AlignmentPanel;
import dk.kvl.tools.history.*;
import java.util.*;

/**
 *  Description of the Class
 *
 * @author     allan
 * @created    November 8, 2004
 */
public class AlignmentTable extends JTable implements MouseListener,MouseMotionListener {
  
  private int posX = 0;
  private AlignmentProperties properties;
  private final char[] alphabet = {'A', 'a', 'U', 'u', 'C', 'c', 'G', 'g', '-', 'B', 'D', 'K', 'M', 'N', 'R', 'S', 'T', 'V', 'W', 'X', 'Y', '0', '1', '2', '3', '4', '5', '6', '7', '8', '9'};
  private int preferredCellHeight, preferredCellWidth, minCellHeight, minCellWidth;
  private AlignmentPanel ap;
  private JWindow warningWindow = new JWindow();
  
  /**
   *Constructor for the AlignmentTable object
   *
   * @param  adm    Description of the Parameter
   * @param  prop   Description of the Parameter
   * @param  align  Description of the Parameter
   * @param  ap     Description of the Parameter
   */
  public AlignmentTable(AlignmentDataModel adm, AlignmentProperties prop, Alignment align, AlignmentPanel ap) {
    super(adm);
    setDoubleBuffered(true);
    setTransferHandler(new AlignmentTableTransferHandler());
    setDragEnabled(true);
    //setSelectionModel(new AlignmentSelectionModel());
    addMouseListener(this);
    addMouseMotionListener(this);
    properties = prop;
    this.ap = ap;
    getTableHeader().setReorderingAllowed(false);
    getTableHeader().setResizingAllowed(false);
    getTableHeader().setDefaultRenderer(new SmallHeader(align, prop));
    getSelectionModel().setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    for (int i = 0; i < getColumnCount(); i++) {
      if (prop == null) {
        getColumn(getColumnName(i)).setCellRenderer(new AlignmentRenderer());
      } else {
        getColumn(getColumnName(i)).setCellRenderer(new AlignmentRenderer(properties, ap));
      }
      //getColumn(getColumnName(i)).setCellEditor(new DefaultCellEditor(getCharCombobox()));
    }
  }


  /**
   *  Sets preferred and minimum sizes for table cells depending on size of displayed text
   *
   */
  public void setPreferredCellSize() {
    FontMetrics fm = getAccessibleContext().getAccessibleComponent().getFontMetrics(properties.getTableFont());
    minCellHeight = fm.getHeight();
    minCellWidth = 0;

    for (int i = 0; i < alphabet.length; i++) {
      int charWidth = fm.charWidth(alphabet[i]);
      if (charWidth > minCellWidth) {
        minCellWidth = charWidth;
      }
    }
    preferredCellWidth = minCellWidth + 2;
    preferredCellHeight = minCellHeight + 2;
  }


  /**
   *  Gets the preferredCellWidth attribute of the AlignmentTable object
   *
   * @return    The preferredCellWidth value
   */
  public int getPreferredCellWidth() {
    setPreferredCellSize();
    return preferredCellWidth;
  }


  /**
   *  Gets the preferredCellHeight attribute of the AlignmentTable object
   *
   * @return    The preferredCellHeight value
   */
  public int getPreferredCellHeight() {
    setPreferredCellSize();
    return preferredCellHeight;
  }


  /**
   *  Gets the minCellWidth attribute of the AlignmentTable object
   *
   * @return    The minCellWidth value
   */
  public int getMinCellWidth() {
    setPreferredCellSize();
    return minCellWidth;
  }


  /**
   *  Gets the minCellHeight attribute of the AlignmentTable object
   *
   * @return    The minCellHeight value
   */
  public int getMinCellHeight() {
    setPreferredCellSize();
    return minCellHeight;
  }


  /**
   *  Description of the Method
   *
   * @param  e  Description of the Parameter
   */
  public void mouseClicked(MouseEvent e) { }


  /**
   *  Description of the Method
   *
   * @param  e  Description of the Parameter
   */
  public void mouseEntered(MouseEvent e) { }


  /**
   *  Description of the Method
   *
   * @param  e  Description of the Parameter
   */
  public void mouseExited(MouseEvent e) { }


  /**
   *  Description of the Method
   *
   * @param  e  Description of the Parameter
   */
  public void mousePressed(MouseEvent e) { 
  if(e.getButton() == e.BUTTON3)
    {
      if(continousCols(e))
      {
        posX = e.getX();
        e.consume();
      }
    }
  }


  /**
   *  Description of the Method
   *
   * @param  e  Description of the Parameter
   */
  public void mouseReleased(MouseEvent e) { 
    if(e.getButton() == e.BUTTON3)
    {
    posX = 0;
    }
    warningWindow.setVisible(false);
  }
  
  public void mouseDragged(MouseEvent e) {
    ArrayList history = new ArrayList();
    if (posX != 0 ) {
        int dx = e.getX() - posX;
        if (Math.abs(dx) >= getPreferredCellWidth() ) {
            String[] sequences = ap.getSelectedSequenceNames();
            int[] cols = getSelectedColumns();
            int maxCol = Math.max(cols[0],cols[cols.length-1]);
            int minCol = Math.min(cols[0],cols[cols.length-1]);
            ListSelectionModel selectModel = getSelectionModel();
            if(dx > 0)
            {
              if(isGaps(sequences,maxCol+1))
              {
              //HistoryItem hi = ap.getAlignmentController().rotateLeft(sequences,minCol,maxCol+1);
              	RotateLeft hi = new RotateLeft();
                hi.setSequences(sequences);
                hi.setStart(minCol);
                hi.setEnd(maxCol+1);
                if(hi != null )
                {
                  HistoryElement element = new HistoryElement(sequences, minCol, maxCol+1);
                  history.add(element);
                  if(ap.getAlignmentController().rotateLeft(history))
                  {
                    removeColumnSelectionInterval(minCol,minCol);
                    addColumnSelectionInterval(maxCol+1,maxCol+1);
                  }
                }
              }
            }
            else
            {
              if(isGaps(sequences,minCol-1))
              {
              
              //HistoryItem hi = ap.getAlignmentController().rotateRight(sequences,minCol-1,maxCol);
              	RotateRight hi = new RotateRight();
                hi.setSequences(sequences);
                hi.setStart(minCol);
                hi.setEnd(maxCol+1);
              if(hi != null)
              {
                HistoryElement element = new HistoryElement(sequences, minCol-1, maxCol);
                history.add(element);
                if (ap.getAlignmentController().rotateRight(history))
                {
                addColumnSelectionInterval(minCol-1,minCol-1);
                removeColumnSelectionInterval(maxCol,maxCol);
                }
              }
              }
            }
            posX = e.getX();
        }
        e.consume();
    }
  }
  
  public boolean isGaps(String[] sequences,int column)
  {
    boolean isGaps = false;
    for(int i = 0;i<sequences.length;i++)
    {
      if(ap.getAlignmentController().getCharAt(sequences[i],column) == '-'||ap.getAlignmentController().getCharAt(sequences[i],column) == '.')
      {
        isGaps = true;
        break;
      }
    }
    return isGaps;
  }
  
  public void mouseMoved(MouseEvent e) {}
  
  public boolean continousCols(MouseEvent e)
  {
    int[] cols = getSelectedColumns();
    if(cols.length > 1)
    {
    for(int i = 1;i<cols.length;i++)
    {
      if(cols[i] != cols[i-1]+1 && cols[i] != cols[i-1]-1)
      {
        JLabel multiple = new JLabel("<html>Your have multiple selections and can't drag<br>Probably you have set selectionmode to 'both'<html>");
        warningWindow.getContentPane().add(multiple);
        warningWindow.setLocation((int)e.getComponent().getLocationOnScreen().getX()+e.getX(),(int)e.getComponent().getLocationOnScreen().getY()+e.getY());
        warningWindow.pack();
        warningWindow.setVisible(true);
        //warningWindow.setLocation();
        return false;
      }
    }
    }
    return true;
    
  }
  
}

