/**
 *  SARSE, Semi-Automated RNA Sequence Editor. Copyright (C) 2004 Allan
 *  Lind-Thomsen This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version. This program is distributed in the hope that it
 *  will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 *  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *  Public License for more details. You should have received a copy of the GNU
 *  General Public License along with this program; if not, write to the Free
 *  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 *  02111-1307, USA.
 */
package dk.kvl.gui.components;

import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import dk.kvl.controller.*;
import dk.kvl.gui.alignment.*;
import dk.kvl.sequencetools.*;

/**
 *  Description of the Class
 *
 * @author     allan
 * @created    March 12, 2004
 */
public class InformationPanel extends JPanel implements ActionListener, ListSelectionListener
{
    private AlignmentController controller = null;
    private JTextArea infoArea = new JTextArea();
    private JTextArea pairingInfo = new JTextArea();
    private JTextArea positionInfo = new JTextArea();
    private JTextArea alignInfo;
    //
    private AlignmentContainer parent;
    private JLabel seqLabel = new JLabel("");
    

    /**
     *  Constructor for the InformationPanel object
     *
     * @param  aController  Description of the Parameter
     * @param  parent       Description of the Parameter
     */
    public InformationPanel(AlignmentContainer parent, AlignmentController aController)
    {
        super(new BorderLayout());
        this.parent = parent;
        JTabbedPane tp = new JTabbedPane();
        controller = aController;
        //
        JPanel selSeq = new JPanel(new BorderLayout());
        infoArea.setLineWrap(true);
        infoArea.setWrapStyleWord(true);
        selSeq.add(BorderLayout.CENTER, infoArea);
        selSeq.add(BorderLayout.NORTH, seqLabel);
        tp.add(selSeq, "Sequence", 0);
        //
        alignInfo = new JTextArea(controller.getAlignmentInfo());
        alignInfo.setLineWrap(true);
        alignInfo.setWrapStyleWord(true);
        tp.add(new JScrollPane(alignInfo), "Alignment", 1);
        //
        pairingInfo.setEditable(false);
        pairingInfo.setLineWrap(true);
        pairingInfo.setWrapStyleWord(true);
        tp.add(new JScrollPane(pairingInfo), "Pairing", 2);
        //
        positionInfo.setEditable(false);
        positionInfo.setLineWrap(true);
        positionInfo.setWrapStyleWord(true);
        tp.add(new JScrollPane(positionInfo), "Position", 3);
        
        //
        tp.setSelectedIndex(1);
        add(BorderLayout.CENTER, tp);
        //
        createSavePanel();
    }


    /**
     *  Description of the Method
     *
     * @param  ae  Description of the Parameter
     */
    public void actionPerformed(ActionEvent ae)
    {
        infoArea.setText(controller.getSequenceInfo(ae.getActionCommand()));
        setSelectedSequence(ae.getActionCommand());
    }


    /**
     *  Description of the Method
     *
     * @param  lse  Description of the Parameter
     */
    public void valueChanged(ListSelectionEvent lse)
    {
        if (!lse.getValueIsAdjusting())
        {
            int min;
            int max;
            if (((DefaultListSelectionModel)lse.getSource()).getMaxSelectionIndex() > ((DefaultListSelectionModel)lse.getSource()).getMinSelectionIndex())
            {
                max = ((DefaultListSelectionModel)lse.getSource()).getMaxSelectionIndex();
                min = ((DefaultListSelectionModel)lse.getSource()).getMinSelectionIndex();
            }
            else
            {
                max = ((DefaultListSelectionModel)lse.getSource()).getMinSelectionIndex();
                min = ((DefaultListSelectionModel)lse.getSource()).getMaxSelectionIndex();
            }
            StringBuffer info = new StringBuffer();
            int pairing;
            if (parent.getSelectedSequence() != null)
            {
                setSelectedSequence(parent.getSelectedSequence());
                info.append(parent.getSelectedSequence() + "\n");
                infoArea.setText(controller.getSequenceInfo(getSelectedSequence()));
                for (int i = min; i <= max; i++)
                {
                    pairing = parent.getPairing(i);
                    if (pairing == PairingMask.NO_PAIRINGMASK)
                    {
                        info.append("No pairingmask");
                    }
                    else if(pairing == PairingMask.NO_PAIRING)
                    {
                    }
                    else
                    {
                        info.append((i + 1) + "->" + ++pairing + "\n");
                    }
                }
                pairingInfo.setText(info.toString());
            }
            //position info
            positionInfo.setText(parent.getSelectedSequence()+", pos: "+(min+1)+"\n");
            
            String[] labels = controller.getColumnLabels();
            String[] values = controller.getColumnInfo(parent.getSelectedSequence(), min);
            if (values != null)
            {
                for (int i = 0; i < labels.length; i++)
                {
                    positionInfo.append(labels[i] + ": " + values[i] + "\n");
                }
            }
        }
    }


    /**
     *  Description of the Method
     */
    public void createSavePanel()
    {
        JPanel savePanel = new JPanel();
        JButton saveButton = new JButton("Save");
        saveButton.addActionListener(
            new ActionListener()
            {
                public void actionPerformed(ActionEvent ae)
                {
                    save();
                }
            });
        savePanel.add(saveButton);
        add(BorderLayout.SOUTH, savePanel);
    }


    /**
     *  Description of the Method
     */
    public void save()
    {
        //alignmentinfo !!
        controller.setAlignmentInfo(alignInfo.getText());
        if (seqLabel.getText() != null && !seqLabel.getText().equals(""))
        {
            controller.setSequenceInfo(seqLabel.getText(), infoArea.getText());
        }
    }


    /**
     *  Sets the selectedSequence attribute of the InformationPanel object
     *
     * @param  name  The new selectedSequence value
     */
    public void setSelectedSequence(String name)
    {
        seqLabel.setText(name);
    }


    /**
     *  Gets the selectedSequence attribute of the InformationPanel object
     *
     * @return    The selectedSequence value
     */
    public String getSelectedSequence()
    {
        return seqLabel.getText();
    }
}

