/**
 *  SARSE, Semi-Automated RNA Sequence Editor. Copyright (C) 2004 Allan
 *  Lind-Thomsen This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version. This program is distributed in the hope that it
 *  will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 *  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *  Public License for more details. You should have received a copy of the GNU
 *  General Public License along with this program; if not, write to the Free
 *  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 *  02111-1307, USA.
 */
package dk.kvl.sequencetools;
import java.io.*;
/**
 *  RNA alphabet including gaps
 *
 * @author     allan
 * @created    January 7, 2004
 */
public class RNAAlphabet implements SequenceAlphabet, Serializable
{
    private static char[] alphabet = {'A', 'a', 'U', 'u', 'C', 'c', 'G', 'g', '-', 'B', 'D', 'K', 'M', 'N', 'R', 'S', 'T', 'V', 'W', 'X', 'Y'};
    private final String label = "RNA";
    private final int alphabetType = SequenceAlphabet.RNAALPHABET;


    /**
     *  Gets the RNA alphabet
     *
     * @return    The alphabet
     */
    public char[] getAlphabet()
    {
        return alphabet;
    }


    /**
     *  Examines the sequence to see if it is valid It searches until it finds
     *  the first invalid character
     *
     * @param  sequence  Description of the Parameter
     * @return           The validSequence value
     */
    public boolean isValidSequence(String sequence)
    {
        boolean result = true;
        boolean tmp = false;
        for (int i = 0; i < sequence.length(); i++)
        {
            tmp = false;
            for (int j = 0; j < alphabet.length; j++)
            {
                if (sequence.charAt(i) == alphabet[j])
                {
                    tmp = true;
                    break;
                }
            }
            if (!tmp)
            {
                result = false;
                break;
            }
        }
        return result;
    }


    /**
     *  Gets the label attribute of the RNAAlphabet object
     *
     * @return    The label value
     */
    public String getLabel()
    {
        return label;
    }


    /**
     *  Gets the alphabetType attribute of the RNAAlphabet object
     *
     * @return    The alphabetType value
     */
    public int getAlphabetType()
    {
        return alphabetType;
    }


    /**
     *  Gets the validSymbol attribute of the RNAAlphabet object
     *
     * @param  aChar  Description of the Parameter
     * @return        The validSymbol value
     */
    public boolean isValidSymbol(char aChar)
    {
        boolean result = false;
        for (int j = 0; j < alphabet.length; j++)
        {
            if (aChar == alphabet[j])
            {
                result = true;
                break;
            }
        }
        return result;
    }


    /**
     *  return true if symbol1 and symbol2 is one of 
     *  the following pairs: au,gu,gc
     *
     * @param  symbol1  Description of the Parameter
     * @param  symbol2  Description of the Parameter
     * @return          The pairing value
     */
    public boolean isPairing(char symbol1, char symbol2)
    {
        if ((symbol1 == 'c'||symbol1 == 'C') && (symbol2 == 'g'||symbol2 == 'G'))
        {
            return true;
        }
        else if ((symbol1 == 'g'||symbol1 == 'G') && (symbol2 == 'c' || symbol2 == 'u' || symbol2 == 'C' || symbol2 == 'U'))
        {
            return true;
        }
        else if ((symbol1 == 'u'||symbol1 == 'U') && (symbol2 == 'A' || symbol2 == 'a' || symbol2 == 'G' || symbol2 == 'g'))
        {
            return true;
        }
        if ((symbol1 == 'a'||symbol1 == 'A') && (symbol2 == 'u'||symbol2 == 'U'))
        {
            return true;
        }
        else
        {
            return false;
        }

    }
}

