/*
 SARSE, Semi-Automated RNA Sequence Editor. Copyright (C) 2004 Allan
 Lind-Thomsen This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or (at your
 option) any later version. This program is distributed in the hope that it
 will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 Public License for more details. You should have received a copy of the GNU
 General Public License along with this program; if not, write to the Free
 Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 02111-1307, USA.
 */
package dk.kvl.alignmenttools;

import dk.kvl.sequencetools.*;
import java.util.*;

/**
 *  Description of the Class
 *
 * @author     Allan
 * @created    January 8, 2004
 */
public class Alignment
{
  /**
   *  Description of the Field
   */
  public static int ALPHABETSORT = 1;
  /**
   *  Description of the Field
   */
  public static int ORIGINALSORT = 2;
  
  private Hashtable sequences = null;
  private Hashtable pairingMasks = null;
  private int type = -1;
  private int length = -1;
  private String label = null;
  private char[] alphabet = null;
  private boolean changed = false;
  private String pairingMaskName = null;
  private StringBuffer header;
  private int sort = ORIGINALSORT;
  private String[] sortedNames;
  
  
  /**
   *  Constructor for the Alignment object
   *
   * @param  seqs    Description of the Parameter
   * @param  pMasks  Description of the Parameter
   */
  public Alignment(Hashtable seqs, Hashtable pMasks)
  {
    if (pMasks != null)
    {
      this.pairingMasks = pMasks;
    }
    else
    {
      pairingMasks = new Hashtable();
    }
    if (seqs != null)
    {
      sequences = seqs;
    }
    else
    {
      sequences = new Hashtable();
    }
    checkAlignment();
    sort();
  }
  
  public boolean hasPairingmask()
  {
	  if(pairingMasks == null || pairingMasks.size() == 0)
	  {
		 return false; 
	  }
	  return true;
	  
  }
  /**
   *  Constructor for the Alignment object
   */
  public Alignment()
  {
    sequences = new Hashtable();
    pairingMasks = new Hashtable();
  }
  
  
  /**
   *  Adds a feature to the PairingMask attribute of the Alignment object
   *
   * @param  mask  The feature to be added to the PairingMask attribute
   */
  public void addPairingMask(PairingMask mask)
  {
    pairingMasks.put(mask.getLabel(), mask);
    pairingMaskName = mask.getLabel();
    changed = true;
  }
  
  
  /**
   *  Sets the activePairingmask attribute of the Alignment object
   *
   * @param  name  The new activePairingmask value
   */
  public void setActivePairingmask(String name)
  {
    pairingMaskName = name;
    //updatePairings();
  }
  
  
  /**
   *  Gets the validSymbol attribute of the Alignment object
   *
   * @param  symbol  Description of the Parameter
   * @return         The validSymbol value
   */
  public boolean isValidSymbol(char symbol)
  {
    boolean result = false;
    Enumeration e = getSequenceKeys();
    if (e.hasMoreElements())
    {
      String name = (String)e.nextElement();
      result = getSequence(name).getAlphabet().isValidSymbol(symbol);
    }
    return result;
  }
  
  
  /**
   *  Gets the pairingmaskName attribute of the Alignment object
   *
   * @return    The pairingmaskName value
   */
  public String getPairingmaskName()
  {
    return pairingMaskName;
  }
  
  
  /**
   *  Gets the pairingMask attribute of the Alignment object
   *
   * @param  key  Description of the Parameter
   * @return      The pairingMask value
   */
  public PairingMask getPairingMask(String key)
  {
    if (key != null)
    {
      return (PairingMask)pairingMasks.get(key);
    }
    else
    {
      return null;
    }
  }
  
  
  /**
   *  Gets the pairingMask attribute of the Alignment object
   *
   * @return    The pairingMask value
   */
  public PairingMask getPairingMask()
  {
    if (pairingMaskName != null)
    {
      return (PairingMask)pairingMasks.get(pairingMaskName);
    }
    else
    {
      return null;
    }
  }
  
  
  /**
   *  Adds a feature to the Sequence attribute of the Alignment object
   *
   * @param  newSequence  The feature to be added to the Sequence attribute
   * @return              Description of the Return Value
   */
  public boolean addSequence(Sequence newSequence)
  {
	  newSequence.setNumber(sequences.size());
    //System.out.println("Adding: "+newSequence.getLabel());
    //System.out.println("Number: "+newSequence.getNumber()+", length: "+newSequence.getLength()+", type: "+newSequence.getType());
    //-------------------------------------------
    boolean valid = false;
    if (newSequence != null && getSequence(newSequence.getLabel()) == null)
    {
      if (getPairingMask() != null)
      {
        //newSequence.setPairingmask(getPairingMask().getSequenceArray());
      }
      if (length == -1 && type == -1)
      {//first sequence
        length = newSequence.getLength();
        type = newSequence.getType();
        //sequences.put(newSequence.getLabel(), newSequence);
        //changed = true;
        //valid = true;
        
      }
      //else
      //{
      	
      	//additional sequence
//        if (type == SequenceAlphabet.UNDEFINED && newSequence.getType() != SequenceAlphabet.UNDEFINED)
//        {
//          type = SequenceAlphabet.UNDEFINED;
//        }
//        else //if (type == newSequence.getType())
//        {//length == newSequence.getLength() &&
        //	System.out.println("adding: "+newSequence.getLabel());
          sequences.put(newSequence.getLabel(), newSequence);
          if (length < newSequence.getLength())
          {
            length = newSequence.getLength();
            //System.out.println("length changed:"+length);
          }
          changed = true;
          valid = true;
        //}
      //}
    }
    //System.out.println("number of sequences: "+getNumberOfSequences());
    return valid;
  }
  
  public boolean addOldSequence(Sequence sequence){
    Object o = sequences.put(sequence.getLabel(), sequence);
    if (o != null){
      return true;
    }
    return false;
  }
  
  
  /**
   *  Checks the hashtable content to see if it is Sequences, all matches the
   *  type variable and all have the same length as the length variable.
   *  Sequences of type UNDEFINED is only checked for length.
   *
   * @return    true if all sequences is the same type and length, false
   *      otherwise
   */
  public boolean checkAlignment()
  {
    boolean check = true;
    Sequence seq = null;
    Enumeration enumeration = sequences.elements();
    while (enumeration.hasMoreElements())
    {
      seq = (Sequence)enumeration.nextElement();
      if (length == -1 && type == -1)
      {
        length = seq.getLength();
        type = seq.getType();
      }
      else
      {
        if (type != seq.getType())
        {
          check = false;
        }
        else
        {
          if (length < seq.getLength())
          {
            length = seq.getLength();
          }
        }
      }
    }
    if (pairingMaskName == null)
    {
      if (pairingMasks.size() > 0)
      {
        Enumeration e = pairingMasks.keys();
        {
          String key = (String)e.nextElement();
          pairingMaskName = key;
        }
        
      }
    }
    return check;
  }
  
  
  /**
   *  Sets the typeAndLength attribute of the Alignment object
   */
  protected void setTypeAndLength()
  {
    Enumeration enumeration = sequences.elements();
    Sequence seq = null;
    int counter = 0;
    while (enumeration.hasMoreElements())
    {
      seq = (Sequence)enumeration.nextElement();
      if (seq.getType() != SequenceAlphabet.ARBITRARY)
      {
        type = seq.getType();
        length = seq.getLength();
        break;
      }
    }
  }
  
  
  /**
   *  Gets the length attribute of the Alignment object
   *
   * @return    The length value
   */
  public int getLength()
  {
    return length;
  }
  
  
  /**
   *  Gets the type attribute of the Alignment object
   *
   * @return    The type value
   */
  public int getType()
  {
    return type;
  }
  
  
  /**
   *  Description of the Method
   *
   * @return    Description of the Return Value
   */
  public String toString()
  {
    return "Type: " + type + ", length: " + length + ", number of sequences: " + sequences.size() + ", number of pairingmasks = " + pairingMasks.size();
  }
  
  
  /**
   *  Gets the sequence attribute of the Alignment object e.nextElement()
   *
   * @param  key  Description of the Parameter
   * @return      The sequence value
   */
  public Sequence getSequence(String key)
  {
    if (key != null)
    {
      return (Sequence)sequences.get(key);
    }
    else
    {
      return null;
    }
  }
  
  
  /**
   *  Gets the keys attribute of the Alignment object
   *
   * @return    The keys value
   */
  public Enumeration getSequenceKeys()
  {
    return sequences.keys();
  }
  
  public Sequence removeSequence(String key){
    Object o = sequences.remove(key);
    if (o != null){
      changed = true;
      return (Sequence)o;
    }
    return null;
  }
  
  /**
   *  Gets the numberOfSequences attribute of the Alignment object
   *
   * @return    The numberOfSequences value
   */
  public int getNumberOfSequences()
  {
    return sequences.size();
  }
  
  
  /**
   *  Gets the numberOfPairingMasks attribute of the Alignment object
   *
   * @return    The numberOfPairingMasks value
   */
  public int getNumberOfPairingMasks()
  {
    return pairingMasks.size();
  }
  
  
  /**
   *  Gets the pairingMaskKeys attribute of the Alignment object
   *
   * @return    The pairingMaskKeys value
   */
  public Enumeration getPairingMaskKeys()
  {
    return pairingMasks.keys();
  }
  
  
  /**
   *  Gets the sequences attribute of the Alignment object
   *
   * @return    The sequences value
   */
  public Enumeration getSequences()
  {
    return sequences.elements();
  }
  
  
  
  
  
  
  /**
   *  Description of the Method
   *
   * @param  name   Description of the Parameter
   * @param  index  Description of the Parameter
   * @return        Description of the Return Value
   */
  public boolean toUpperCase(String name, int index)
  {
    boolean result = false;
    if (getSequence(name) != null && index >= 0 && index <= length)
    {
      result = getSequence(name).toUpperCase(index);
      if (result)
      {
        //updatePairings();
        changed = true;
      }
    }
    return result;
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  name   Description of the Parameter
   * @param  index  Description of the Parameter
   * @return        Description of the Return Value
   */
  public int[] toUpperCase(String name, int[] index)
  {
    int[] result = new int[index.length];
    for (int i = 0; i < index.length; i++)
    {
      result[i] = -1;
      if (toUpperCase(name, index[i]))
      {
        result[i] = index[i];
      }
    }
    return result;
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  names  Description of the Parameter
   * @param  index  Description of the Parameter
   * @return        Description of the Return Value
   */
  public int[][] toUpperCase(String[] names, int[] index)
  {
    int[][] result = new int[names.length][];
    for (int i = 0; i < names.length; i++)
    {
      result[i] = toUpperCase(names[i], index);
    }
    //updatePairings();
    return result;
  }
  
  
  /**
   *  Description of the Method
   */
  public void toLowerCase()
  {
    Enumeration e = getSequences();
    while (e.hasMoreElements())
    {
      Sequence seq = (Sequence)e.nextElement();
      seq.toLowerCase();
    }
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  name   Description of the Parameter
   * @param  index  Description of the Parameter
   * @return        Description of the Return Value
   */
  public boolean toLowerCase(String name, int index)
  {
    boolean result = false;
    if (getSequence(name) != null && index >= 0 && index <= length)
    {
      result = getSequence(name).toLowerCase(index);
      if (result)
      {
        //updatePairings();
        changed = true;
      }
    }
    return result;
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  name   Description of the Parameter
   * @param  index  Description of the Parameter
   * @return        Description of the Return Value
   */
  public int[] toLowerCase(String name, int[] index)
  {
    int[] result = new int[index.length];
    for (int i = 0; i < index.length; i++)
    {
      result[i] = -1;
      if (toLowerCase(name, index[i]))
      {
        result[i] = index[i];
      }
    }
    return result;
  }
  
  
  /**
   *  Description of method
   *
   * @param  names  Description of the Parameter
   * @param  index  Description of the Parameter
   * @return        Description of the Return Value
   */
  public int[][] toLowerCase(String[] names, int[] index)
  {
    int[][] result = new int[names.length][];
    for (int i = 0; i < names.length; i++)
    {
      result[i] = toLowerCase(names[i], index);
    }
    //updatePairings();
    return result;
  }
  
  
  /**
   *  Gets the changed attribute of the Alignment object
   *
   * @return    The changed value
   */
  public boolean isChanged()
  {
    return changed;
  }
  
  
  /**
   *  Sets the changed attribute of the Alignment object
   *
   * @param  chg  The new changed value
   */
  public void setChanged(boolean chg)
  {
    changed = chg;
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  sequenceName  Description of the Parameter
   * @param  index         Description of the Parameter
   * @param  newChar       Description of the Parameter
   * @return               Description of the Return Value
   */
  public char changeSymbol(String sequenceName, int index, char newChar)
  {
    changed = true;
    Sequence seq = getSequence(sequenceName);
    if (seq == null)
    {
      seq = getPairingMask(sequenceName);
      if (seq == null)
      {
        return '!';
      }
    }
    return seq.changeSymbol(newChar, index);
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  sequenceNames  Description of the Parameter
   * @param  index          Description of the Parameter
   * @param  newChar        Description of the Parameter
   * @return                Description of the Return Value
   */
  public char[][] changeSymbol(String[] sequenceNames, int[] index, char newChar)
  {
    char[][] result = new char[sequenceNames.length][];
    boolean pm = false;
    for (int i = 0; i < sequenceNames.length; i++)
    {
      Sequence seq = getSequence(sequenceNames[i]);
      if (seq == null)
      {
        seq = getPairingMask(sequenceNames[i]);
        pm = true;
      }
      if (seq != null)
      {
        result[i] = seq.changeSymbol(newChar, index);
      }
    }
    updatePairings();
    changed = true;
    return result;
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  changedAlignment  Description of the Parameter
   * @return                   Description of the Return Value
   */
  public boolean updateAlignment(Subalignment changedAlignment)
  {
    boolean result = false;
    //check TODO!!!!!!!
    String name = null;
    int[] cell = new int[0];
    char symbol;
    Enumeration e = changedAlignment.getSequenceKeys();
    while (e.hasMoreElements())
    {
      name = (String)e.nextElement();
      for (int i = 0; i < changedAlignment.getLength(); i++)
      {
        symbol = changedAlignment.getSequence(name).getSymbolAt(i);
        changeSymbol(name, i + changedAlignment.getStartIndex(), symbol);
      }
    }
    e = changedAlignment.getPairingMaskKeys();
    while (e.hasMoreElements())
    {
      name = (String)e.nextElement();
      for (int i = 0; i < changedAlignment.getLength(); i++)
      {
        symbol = changedAlignment.getPairingMask(name).getSymbolAt(i);
        changeSymbol(name, i + changedAlignment.getStartIndex(), symbol);
      }
    }
    return result;
  }
  
  
  /**
   *  Gets the alphabet attribute of the Alignment object
   *
   * @return    The alphabet value
   */
  public char[] getAlphabet()
  {
    return Sequence.getAlphabet(type).getAlphabet();
  }
  
  
  /**
   *  Sets the header attribute of the Alignment object
   *
   * @param  newHead  The new header value
   */
  public void setHeader(StringBuffer newHead)
  {
    header = newHead;
  }
  
  
  /**
   *  used to prepare the alignment after creation from a colfile
   */
  public void prepareAlignment()
  {
    checkAlignment();
    if (getPairingMask() != null)
    {
      //updatePairings();
    }
    else
    {/*
    //check for structure in sequences
     Enumeration e = getSequences();
     Sequence seq = (Sequence)e.nextElement();
     //assumes common pairingmask for all sequences
      int[] pairs = seq.getPairings();
      if (pairs != null)
      {
      try
      {
      PairingMask p = new PairingMask("Pairingmask", pairs, SequenceAlphabet.UNDEFINED, true);
      addPairingMask(p);
      setActivePairingmask(p.getLabel());
      
      }
      catch (Exception ex)
      {
      ex.printStackTrace();
      }
      }
      /*else
       {
       toLowerCase();
       }*/
    }
  }
  
  
  /**
   *  Gets the sequenceInfo attribute of the Alignment object
   *
   * @param  sequence  Description of the Parameter
   * @return           The sequenceInfo value
   */
  public String getSequenceInfo(String sequence)
  {
    Sequence seq = getSequence(sequence);
    if (seq != null)
    {
      return seq.getInformation();
    }
    return null;
  }
  
  
  /**
   *  Gets the information attribute of the Alignment object
   *
   * @return    The information value
   */
  public String getInformation()
  {
    if (header != null)
    {
      return header.toString();
    }
    return null;
  }
  
  
  /**
   *  Sets the information attribute of the Alignment object
   *
   * @param  text  The new information value
   */
  public void setInformation(String text)
  {
    header = new StringBuffer(text);
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  sequences  Description of the Parameter
   * @param  start      Description of the Parameter
   * @param  end        Description of the Parameter
   * @return            Description of the Return Value
   */
  public String[] rotateLeft(String[] sequences, int start, int end)
  {
    String[] rotated = new String[sequences.length];
    boolean allowed = true;
    for (int i = 0; i < sequences.length; i++)
    {
      Sequence seq = getSequence(sequences[i]);
      if (seq == null)
      {
        seq = getPairingMask(sequences[i]);
      }
      if (seq != null)
      {
        if(seq.getSymbolAt(end) != '-')
        {
          allowed = false;
          rotated = null;
        }
      }
    }
    while(allowed)
    {
      for (int i = 0; i < sequences.length; i++)
      {
        Sequence seq = getSequence(sequences[i]);
        if (seq == null)
        {
          seq = getPairingMask(sequences[i]);
        }
        if (seq != null)
        {
          if (seq.rotateLeft(start, end))
          {
            rotated[i] = sequences[i];
          }
        }
      }
      for (int i = 0; i < sequences.length; i++)
      {
        Sequence seq = getSequence(sequences[i]);
        if (seq == null)
        {
          seq = getPairingMask(sequences[i]);
        }
        if (seq != null)
        {
          if(seq.getSymbolAt(end) != '-')
          {
            allowed = false;
          }
        }
      }
    }
    return rotated;
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  sequences  Description of the Parameter
   * @param  start      Description of the Parameter
   * @param  end        Description of the Parameter
   * @return            Description of the Return Value
   */
  public String[] rotateRight(String[] sequences, int start, int end)
  {
	String[] rotated = new String[sequences.length];
    boolean allowed = true;
    for (int i = 0; i < sequences.length; i++)
    {
      Sequence seq = getSequence(sequences[i]);
      if (seq == null)
      {
        seq = getPairingMask(sequences[i]);
      }
      if (seq != null)
      {
        if(seq.getSymbolAt(start) != '-')
        {
          allowed = false;
          rotated = null;
        }
      }
    }
    while(allowed)
    {
    	Sequence seq = null;
      for (int i = 0; i < sequences.length; i++)
      {
        seq = getSequence(sequences[i]);
        if (seq == null)
        {
          seq = getPairingMask(sequences[i]);
        }
        if (seq != null)
        {
          if (seq.rotateRight(start, end))
          {
            rotated[i] = sequences[i];
          }
        }
      }
      for (int i = 0; i < sequences.length; i++)
      {
        seq = getSequence(sequences[i]);
        if (seq == null)
        {
          seq = getPairingMask(sequences[i]);
        }
        if (seq != null)
        {
          if(seq.getSymbolAt(start) != '-')
          {
            allowed = false;
          }
        }
      }
    }
    return rotated;
  }
  
  
  /**
   *  Gets the label attribute of the Alignment object
   *
   * @return    The label value
   */
  public String getLabel()
  {
    return label;
  }
  
  
  /**
   *  Sets the label attribute of the Alignment object
   *
   * @param  name  The new label value
   */
  public void setLabel(String name)
  {
    label = name;
  }
  
  
  /**
   *  Description of the Method
   */
  public void emptyInfo()
  {
    header = new StringBuffer();
    Enumeration e = sequences.keys();
    while (e.hasMoreElements())
    {
      ((Sequence)sequences.get((String)e.nextElement())).emptyInfo();
    }
    e = pairingMasks.keys();
    while (e.hasMoreElements())
    {
      ((PairingMask)pairingMasks.get(e.nextElement())).emptyInfo();
    }
  }
  
  
  /**
   *  Sets the sort attribute of the Alignment object
   *
   * @param  sortType  The new sort value
   */
  public void setSort(int sortType)
  {
    if (sortType == ALPHABETSORT)
    {
      sort = ALPHABETSORT;
    }
    else
    {
      sort = ORIGINALSORT;
    }
    sort();
  }
  
  
  /**
   *  Gets the sort attribute of the Alignment object
   *
   * @return    The sort value
   */
  public int getSort()
  {
    return sort;
  }
  
  
  /**
   *  Description of the Method
   */
  public void sort()
  {
    Enumeration e = getSequenceKeys();
    if (sort == ORIGINALSORT)
    {
      java.util.List sorted = new ArrayList();
      Sequence[] sortedSequences = new Sequence[getNumberOfSequences()];
      sortedNames = new String[getNumberOfSequences()];
      while (e.hasMoreElements())
      {
        sorted.add(getSequence((String)e.nextElement()));
      }
      sortedSequences = (Sequence[])sorted.toArray(new Sequence[sorted.size()]);
      Arrays.sort(sortedSequences);
      
      for (int i = 0; i < sortedSequences.length; i++){
        sortedNames[i] = sortedSequences[i].getLabel();
      }
    }
    else
    {
      Vector v = new Vector(); 
      while (e.hasMoreElements())
      {
        v.add(getSequence((String)e.nextElement()).getLabel());
        Collections.sort(v);
        sortedNames = (String[])v.toArray(new String[0]);
      }
    }
  }
  
  
  /**
   *  Gets the sortedNames attribute of the Alignment object
   *
   * @return    The sortedNames value
   */
  public String[] getSortedNames()
  {
    sort();
    return sortedNames;
  }
  
  
  /**
   *  Adds a feature to the Pairing attribute of the Alignment object
   *
   * @param  indexA  The feature to be added to the Pairing attribute
   * @param  indexB  The feature to be added to the Pairing attribute
   * @param  symbol  The feature to be added to the Pairing attribute
   * @return         Description of the Return Value
   */
  public boolean addPairing(int indexA, int indexB, char symbol)
  {
    if (getPairingMask() != null)
    {
      return false;
    }
    boolean result = getPairingMask().addPairing(indexA, indexB, symbol);
    if (result)
    {
      updatePairings();
    }
    return result;
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  indexA  Description of the Parameter
   * @param  indexB  Description of the Parameter
   * @return         Description of the Return Value
   */
  public char removePairing(int indexA, int indexB)
  {
    if (getPairingMask() != null)
    {
      return '!';
    }
    char result = getPairingMask().removePairing(indexA, indexB);
    if (result != '!')
    {
      updatePairings();
    }
    return result;
  }
  
  
  /**
   *  Description of the Method
   */
  public void updatePairings()
  {
    //System.out.println("Update pairingsw");
    if (getPairingMask() != null)
    {
      Enumeration e = getSequenceKeys();
      getPairingMask().setPairings(PairingMask.updatePairings(getPairingMask().getSequenceArray()));
      while (e.hasMoreElements())
      {
        Sequence seq = getSequence((String)e.nextElement());
        seq.setPairingmask(getPairingMask().getSequenceArray());
      }
    }
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  index  Description of the Parameter
   * @return        Description of the Return Value
   */
  public char[] removeColumn(int index)
  {
    char[] removed = new char[getNumberOfSequences() + 1];
    int i = 0;
    if (index >= 0 && index < getLength())
    {
      Enumeration e = getSequenceKeys();
      while (e.hasMoreElements())
      {
        removed[i] = getSequence((String)e.nextElement()).removeColumn(index);
        i++;
      }
      if (getPairingMask() != null)
      {
        removed[i] = getPairingMask().removeColumn(index);
      }
      length--;
    }
    return removed;
  }
  
  
  
  /**
   *  Adds a feature to the Column attribute of the Alignment object
   *
   * @param  index   The feature to be added to the Column attribute
   * @param  symbol  The feature to be added to the Column attribute
   */
  public void addColumn(int index, char symbol)
  {
    if (index >= 0 && isValidSymbol(symbol))
    {
      Enumeration e = getSequenceKeys();
      while (e.hasMoreElements())
      {
        getSequence((String)e.nextElement()).addColumn(index, symbol);
      }
      if (getPairingMask() != null)
      {
        getPairingMask().addColumn(index, symbol);
      }
      length++;
      
      //updatePairings();
    }
  }
  
  /**
   *  Description of the Method
   */
  public void updateLength()
  {
    length = length + 1;
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  col1  Description of the Parameter
   * @param  col2  Description of the Parameter
   * @param  mask  Description of the Parameter
   * @return       Description of the Return Value
   */
  public boolean pairColumns(int col1, int col2, char mask)
  {
//    if (getPairingMask() != null && getPairingMask().getPairing(col1) == -1 && getPairingMask().getPairing(col2) == -1)
//    {
      Enumeration e = getSequences();
      getPairingMask().pairColumns(col1, col2, mask);
      Sequence seq;
      while (e.hasMoreElements())
      {
        seq = (Sequence)e.nextElement();
        seq.pairBases(col1, col2);
      }
      //updatePairings();
      return true;
//    }
//    else
//    {
//      return false;
//    }
  }
  
  public boolean unpairColumns(int col1, int col2)
  {
    char c = getPairingMask().removePairing(col1,col2);
    if(c != '!')
    {
      Enumeration e = getSequenceKeys();
      while(e.hasMoreElements())
      {
        String seq = (String)e.nextElement();
        toLowerCase(seq, col1);
        toLowerCase(seq, col2);
      }
      return true;
    }
    else
    {
      return false;
    }
  }
  
  public void pair(String[] names, int[] cols)
  {
    if(cols.length%2 == 0 )//&& getPairingMask()!= null
    {
      //assume that the first pairs with the last etc.
      for(int i = 0;i<cols.length/2;i++)
      {
//        if(getPairingMask().getPairing(cols[i]) >= 0 && getPairingMask().getPairing(cols[cols.length-1-i]) >= 0)
//        {
//          Enumeration e = getSequences();
//          while(e.hasMoreElements())
    	  for(int j = 0;j<names.length;j++)
          {
            Sequence seq = getSequence(names[j]);
             
            seq.pairBases(cols[i],cols[cols.length-1-i]);
          }
//        }
      }
      updatePairings();
    }
  }
  
  
  /**
   *  changes the case of the sequences according to pairingmask all symbols
   *  in a given column is changed the same
   *
   * @return    Description of the Return Value
   */
  public boolean fitCases()
  {
    Enumeration e = getSequenceKeys();
    while (e.hasMoreElements())
    {
      String name = (String)e.nextElement();
      Sequence seq = getSequence(name);
      if(seq != null)
      {
        for (int i = 0; i < getLength(); i++)
        {
          if (getPairingMask()!= null && getPairingMask().getPairing(i) >= 0 && seq.getAlphabet().isPairing(seq.getSymbolAt(i),seq.getSymbolAt(getPairingMask().getPairing(i))))
          {
            seq.pairBases(i,getPairingMask().getPairing(i));
          }
          else
          {
            //seq.unpair(new int[]{i,getPairingMask().getPairing(i)});
            seq.toLowerCase(i);
            seq.deletePairing(i);
          }
        }
      }
    }
    return true;
  }
  
  
  /**
   *  Description of the Method
   *
   * @param  names  Description of the Parameter
   * @param  cols   Description of the Parameter
   */
  public Hashtable unpair(String[] names, int[] cols)
  {
    Hashtable ht = new Hashtable();
    for (int i = 0; i < names.length; i++)
    {
      ht.put(names[i], getSequence(names[i]).unpair(cols));
    }
    return ht;
  }
  
  public PairingMask removePairingmask(String name)
  {
    Object o = pairingMasks.remove(name);
    if(o!= null)
    {
      pairingMaskName = null;
      return (PairingMask)o;
    }
    return null;
  }
}

