/*
 *  SARSE, Semi-Automated RNA Sequence Editor.
 *  Copyright (C) 2004 Allan Lind-Thomsen
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dk.kvl.gui.alignment;

import javax.swing.*;
import javax.swing.event.*;
import dk.kvl.controller.*;
import dk.kvl.tools.*;
import dk.kvl.tools.history.*;
import java.io.*;
import java.awt.*;
import java.awt.event.*;
import dk.kvl.gui.properties.*;
import dk.kvl.gui.event.*;
import dk.kvl.alignmenttools.*;
import dk.kvl.sequencetools.*;
import dk.kvl.alignmenttools.io.*;
import java.util.*;
import dk.kvl.gui.components.*;
import dk.kvl.tools.io.*;

/**
 *  Description of  the Class
 *
 * @author     allan
 * @created    January 16, 2004
 */
public class AlignmentContainer extends Container implements ActionListener, UpdateListener, MouseListener, ReloadListener {
  private JTabbedPane views = null;
  private AlignmentController controller = null;
  private AlignmentControlPanel controlPanel = null;
  private AlignmentProperties properties = null;
  //private JLabel alignmentNameLabel;
  private String alignmentName;
  //private InformationPanel infoPanel;
  private String alignmentFileName;
  private String projectDir;
  private String[] recentProjects;
  private AnalyzeFrame analyzeFrame;
  private ProjectProperties projectProperties;
  //canvas
  private AlignmentCanvas overview = null;
  private JFrame canvasWindow;
  private int overviewX = 50;
  private int overviewY = 50;
  //history
  private JFrame historyWindow;
  private int historyX = 100;
  private int historyY = 100;
  //
  private JFrame infoFrame;
  private FileHistory fileHist;
  //
  private boolean horizontalSplit, splitMode, historyEnabled, prevHistoryEnabled;
  private Point p1, p2, p3;
  private boolean selectionMode, splitAlign, newHistory;
  private int operationMode, divider;
  private File file;
  private final AlignmentFrame parent;


  /**
   *  Constructor for the AlignmentContainer object
   *
   * @param  file                       Description of the Parameter
   * @param  aname                      Description of the Parameter
   * @param  selectionMode              Description of the Parameter
   * @param  operationMode              Description of the Parameter
   * @param  splitMode                  Description of the Parameter
   * @param  splitAlign                 Description of the Parameter
   * @param  p1                         Description of the Parameter
   * @param  p2                         Description of the Parameter
   * @param  p3                         Description of the Parameter
   * @param  divider                    Description of the Parameter
   * @param  projectDir                 Description of the Parameter
   * @param  properties                 Description of the Parameter
   * @exception  IOException            Description of the Exception
   * @exception  FileNotFoundException  Description of the Exception
   * @exception  Exception              Description of the Exception
   */
  public AlignmentContainer(AlignmentFrame frame, File file, String aname, boolean selectionMode, int operationMode, boolean splitMode, boolean splitAlign, Point p1, Point p2, Point p3, int divider, String projectDir, AlignmentProperties properties) throws Exception, IOException, FileNotFoundException {
    //alignmentNameLabel = new JLabel("Project: " + aname);
    parent = frame;
    alignmentName = aname;
    alignmentFileName = file.getAbsolutePath();
    this.file = file;
    this.projectDir = projectDir;
    setLayout(new BorderLayout());
    //add(BorderLayout.NORTH, alignmentNameLabel);
    controller = new AlignmentController(file, aname, projectDir);
    controller.addUpdateListener(this);
    controller.addReloadListener(this);
    //loadProperties();
    this.properties = properties;
    views = new JTabbedPane();
//    views.addChangeListener(new ChangeListener(){
//    	public void stateChanged(ChangeEvent e)
//    	{ 
//    		System.out.println("state changed: " + e);    		
//    	}});
    //createInfoPanel();
    //JScrollPane scrollPane = new JScrollPane(infoPanel);
    //infoFrame = new JFrame("Alignment Information");
    //infoFrame.getContentPane().add(scrollPane);
    //infoFrame.setSize(200, 200);
    //infoFrame.setVisible(true);
    //scrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
    add(BorderLayout.CENTER, views);
    this.selectionMode = selectionMode;
    this.operationMode = operationMode;
    this.splitMode = splitMode;
    this.splitAlign = splitAlign;
    this.p1 = p1;
    this.p2 = p2;
    this.p3 = p3;
    this.divider = divider;
    addControlPanel(controller.getColumnLabels());

    if (projectProperties == null) {
      projectProperties = new ProjectProperties(alignmentName, file.getName(), projectDir);
      projectProperties.store();
    }
    //history
    FileHistory fileHist = AlignmentIOController.loadHistory(projectDir);
    if (fileHist != null) {
      controller.setFileHistory(fileHist);
    } else {
      newHistory = true;
      fileHist = new FileHistory(projectDir);

      controller.setFileHistory(fileHist);
    }

    if (file.getName().endsWith(".widetxt") || file.getName().endsWith(".txt") || file.getName().endsWith(".fasta")) {
      String[] split = file.getName().split("\\.");
      if (newHistory) {
        FileHistoryItem fhi = new ColFileHistoryItem(controller.getMainAlignment(), projectProperties.getPath(), split[0] + ".col");
        fhi.setHistory();
        //fileHist.add(fhi);
        controller.addFileHistoryItem(fhi);
        newHistory = false;
      } else {
        //fileHist.add(new ColFileHistoryItem(controller.getMainAlignment(), projectProperties.getPath(), split[0] + ".col"));
    	  controller.addFileHistoryItem(new ColFileHistoryItem(controller.getMainAlignment(), projectProperties.getPath(), split[0] + ".col"));
      }
      projectProperties.setFile(split[0] + ".col");
      controller.setFileName(split[0] + ".col");
      controller.setHistory(fileHist.get(split[0] + ".col").getHistory());
      fileHist.get(split[0] + ".col").getHistory().setAlignment(controller.getMainAlignment());
      if (properties.createMainViewFromBeginning()) {
        addView(split[0] + ".col");
      }
    } else {
      if (newHistory) {
        FileHistoryItem fhi = new ColFileHistoryItem(controller.getMainAlignment(), projectProperties.getPath(), file.getName());
        fhi.setHistory();
        fileHist.add(fhi);
        newHistory = false;
      } else {
        fileHist.add(new ColFileHistoryItem(controller.getMainAlignment(), projectProperties.getPath(), file.getName()));
      }
      controller.setHistory(fileHist.get(file.getName()).getHistory());
      fileHist.get(file.getName()).getHistory().setAlignment(controller.getMainAlignment());
      controller.setFileName(file.getName());
      if (properties.createMainViewFromBeginning()) {
        addView(file.getName());
      }
    }
  }


  /**
   *  Gets the alignmentController attribute of the AlignmentContainer object
   *
   * @return    The alignmentController value
   */
  public AlignmentController getAlignmentController() {
    return controller;
  }


  /**
   *  Sets the overviewEnabled attribute of the AlignmentContainer object
   *
   * @param  enabled  The new overviewEnabled value
   */
  public void setOverviewEnabled(boolean enabled) {
//    if (enabled) {
//      if (canvasWindow == null) {
//        canvasWindow = new JFrame("Overview");
//        canvasWindow.setLocation(overviewX,overviewY);
//        canvasWindow.getContentPane().setLayout(new BorderLayout());
//        overview = new AlignmentCanvas(canvasWindow, controller);
//        final JScrollPane jsp = new JScrollPane(overview);
//        jsp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
//        jsp.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
//        canvasWindow.getContentPane().add(BorderLayout.CENTER, jsp);
//        JButton zoomIn = new JButton("+");
//        zoomIn.addActionListener(
//          new ActionListener() {
//            public void actionPerformed(ActionEvent ae) {
//              overview.setZoom(overview.getZoom() + 1);
//            }
//          });
//        JButton fitZoom = new JButton("Fit");
//        fitZoom.addActionListener(
//          new ActionListener() {
//            public void actionPerformed(ActionEvent ae) {
//              overview.adjustZoom();
//            }
//          });
//
//        JButton zoomOut = new JButton("-");
//        zoomOut.addActionListener(
//          new ActionListener() {
//            public void actionPerformed(ActionEvent ae) {
//              overview.setZoom(overview.getZoom() - 1);
//            }
//          });
//        JPanel buttonPanel = new JPanel();
//        buttonPanel.add(zoomOut);
//        buttonPanel.add(fitZoom);
//        buttonPanel.add(zoomIn);
//        canvasWindow.getContentPane().add(BorderLayout.SOUTH, buttonPanel);
//        canvasWindow.setSize(400, 200);
//        overview.addMouseListener(this);
//        overview.adjustZoom();
//        canvasWindow.setVisible(true);
//      } else {
//        canvasWindow.setVisible(true);
//      }
//    } else {
//      if (canvasWindow != null) {
//        overview.removeMouseListener(this);
//        overview = null;
//        canvasWindow.dispose();
//        canvasWindow = null;
//      }
//    }
  }


  /**
   *  Adds a feature to the ControlPanel attribute of the AlignmentContainer
   *  object
   *
   * @param  columnLabels  The feature to be added to the ControlPanel
   *      attribute
   */
  public void addControlPanel(String[] columnLabels) {
    controlPanel = new AlignmentControlPanel(properties, columnLabels);
    controlPanel.addActionListener(this);
    controlPanel.enableSplitButtons(splitMode);
    add(BorderLayout.WEST, controlPanel);
  }


  /**
   *  Gets the controlPanel attribute of the AlignmentContainer object
   *
   * @return    The controlPanel value
   */
  public AlignmentControlPanel getControlPanel() {
    return controlPanel;
  }


  /**
   *  Description of the Method
   *
   * @param  ae  Description of the Parameter
   */
  public void actionPerformed(ActionEvent ae) {
    String cmd = ae.getActionCommand();
    String name = "";
    if (ae.getSource() instanceof javax.swing.JButton) {
      name = ((JButton) ae.getSource()).getAccessibleContext().getAccessibleName();
    } else if (ae.getSource() instanceof javax.swing.JCheckBox) {
      name = ((JCheckBox) ae.getSource()).getAccessibleContext().getAccessibleName();
    }

    if (cmd.equals("Subalignment")) {
      subalignment();
    } else if (cmd.equals("Clear selection")) {
      ((AlignmentPanel) views.getSelectedComponent()).clearSelections();
    } else if (name.equals("split")) {
      split();
    } else if (name.equals("uppercase")) {
      toUpperCase();
    } else if (name.equals("unpair bases")) {
      unpairBases();
    } else if (name.equals("rotate left")) {
      rotateRight();
    } else if (name.equals("rotate right")) {
      rotateLeft();
    } else if (name.equals("change")) {
      change();
    } else if (name.equals("unsplit")) {
      unsplit();
    } else if (ae.getSource() instanceof dk.kvl.alignmenttools.Analyzer) {
      changeMainAlignment(ae);
    } else if (name.equals("close_history")) {
      setHistoryEnabled(false);
    } else if (name.startsWith("Show")) {
      //It is a JCheckBox
      JCheckBox cb = (JCheckBox) ae.getSource();
      String column = cb.getText().substring(cb.getText().lastIndexOf(" ") + 1, cb.getText().length());
    } else if (name.equals("Pair bases")) {
      pairBases();
    } else if (name.equals("Pair columns")) {
      pairColumns();
    } else if (name.equals("single")) {
      selectionMode(true);
    } else if (name.equals("double")) {
      selectionMode(false);
    }
  }


  /**
   *  Pairs any equal number of selected columns If unequal number of columns
   *  selected it does nothing if any bases in a column pairs with another
   *  that column isn't changed for the moment it pairs any gc, au and gu and
   *  change it to GC,AU og GU. It only works when a pairingmask exists. if
   *  the pairingmask is with parentheses it ads a () to the mask in the
   *  columns and pairs them, else it asks what character it should insert. It
   *  pairs the columns from the outside in, e.g. if you have selected column
   *  1 to 10 it pairs 1-10, 2-9,....,5-6.
   */
  public void pairColumns() {
    int[] cols = getSelectedProject().getSelectedColumns();
    String input = null;
    if (cols.length % 2 == 0) {
      char mask = 0;
      while (mask == 0) {
        input = JOptionPane.showInputDialog("Write the character you want to use in the pairingmask:");
        if (input == null || input.length() == 0) {
          break;
        } else if (input.length() > 0) {
          mask = input.charAt(0);
        }
      }
      if(mask != 0 && Character.isLetterOrDigit(mask))
      {
      for (int i = 0; i < cols.length / 2; i++) {
        if (controller.getPairing(controller.getPairingmaskName(), cols[i]) == -1 && controller.getPairing(controller.getPairingmaskName(), cols[cols.length - 1 - i]) == -1) {
          ArrayList elements = new ArrayList();
          elements.add(new HistoryElement(cols[i], cols[cols.length - 1 - i], mask));
          controller.pairColumns(elements);
          //controller.pairColumns(cols[i], cols[cols.length - 1 - i], mask);
        } else {
          JOptionPane.showMessageDialog(null, "One or both of the columns " + (cols[i] + 1) + ", " + (cols[cols.length - 1 - i] + 1) + " allready pairs and is not changed");
        }
      }
      }
      else if(input != null)
      {
        JOptionPane.showMessageDialog(null, "You must write a letter or a digit");
      }
      
    } else {
      JOptionPane.showMessageDialog(null, "You must select an equal number af columns!");
    }
  }


  /**
   *  Description of the Method
   *
   * @param  ae  Description of the Parameter
   */
  public void changeMainAlignment(ActionEvent ae) {
	  int res = -1; 
    String cmd = ae.getActionCommand();
    
    if (ae.getID() == 0) {
    	try{
    	String optStr = "resultfile: " + cmd + " has finished.\nDo you want to load it?\n";
    	    if(((ProgramEvent)ae).getFiles().size()>0)
    	    {
    	    	optStr += "Additional files where created:\n";
    	    	for(int i =0 ;i<((ProgramEvent)ae).getFiles().size();i++)
    	    {
    	    	optStr += ((ProgramEvent)ae).getFiles().get(i)+"\n";
    	    }
    	    	optStr += "The files are placed in: "+((ProgramEvent)ae).getResultfile();
    	    }
       res = JOptionPane.showConfirmDialog(null, optStr);
    	}
    	catch(Exception e)
    	{
    		e.printStackTrace();
    	}
      if (res == JOptionPane.OK_OPTION) {
        /*
         *  Component c = getParent();
 
         *  {
         *  c = c.getParent();
         *  }
         *  ((AlignmentFrame)c).changeAlignment(projectProperties.getPath() + "/" + cmd);
         */
         //System.out.println("Alignmentname: "+alignmentName);
        //addView(alignmentName);
    	  try
          {
    		  controller.addFileHistoryItem(new ColFileHistoryItem(controller.getMainAlignment(), projectProperties.getPath(), ((ProgramEvent)ae).getResultfile() ));          
    	  controller.setMainAlignmentFromFile(projectProperties.getPath()+"/"+((ProgramEvent)ae).getResultfile());
    	  
          validate();
          repaint();
        }
        catch(Exception e)
        {
          e.printStackTrace();
        }
      } else {
        JOptionPane.showMessageDialog(null, "The result file is named: result-" + cmd);
      }
    } else {
      if (ae.getID() == 3) {
        res = JOptionPane.showConfirmDialog(null, "Do you wish to launch gv to view: " + cmd + "?");
        if (res == JOptionPane.OK_OPTION) {
          Runtime runtime = Runtime.getRuntime();
          String[] cmds = {"gv", cmd};
          try {
            Process p = runtime.exec(cmds);
          } catch (Exception e) {
            JOptionPane.showMessageDialog(this, "An error occurred while running gv");
          }
        }
      } else {
        JOptionPane.showMessageDialog(null, "An error occurred in processing id:" + cmd);
      }
    }
    analyzeFrame = null;
  }



  /**
   *  Description of the Method
   */
  public void close() {
    if (canvasWindow != null) {
      setOverviewEnabled(false);
    }
    if (historyWindow != null) {
      prevHistoryEnabled = historyEnabled;
      setHistoryEnabled(false);
    }
    if (infoFrame != null) {
      infoFrame.dispose();
      infoFrame = null;
    }
    String name = getSelectedName();
    checkView(name);
    //getSelectedProject().removeListSelectionListener(infoPanel);
    //getSelectedProject().removeSequenceSelectionListener(infoPanel);
    views.remove(((AlignmentPanel) views.getSelectedComponent()));
    //controller.removeView(name);
    if (views.getTabCount() > 0) {
      views.setSelectedIndex(0);
    } else {
      views.setSelectedIndex(-1);
    }
  }


  /**
   *  Description of the Method
   */
  /*
   *  public void update()
   *  {
   *  controller.updateMainAlignment(getSelectedName());
   *  }
   */
  /**
   *  Description of the Method
   */
  public void split() {
    splitMode = true;
    ((AlignmentPanel) views.getSelectedComponent()).split();
  }


  /**
   *  Description of the Method
   */
  public void unsplit() {
    splitMode = false;
    ((AlignmentPanel) views.getSelectedComponent()).unsplit();
  }


  /**
   *  Description of the Method
   */
  public void change() {
    if (views.getSelectedComponent() != null) {
      int[] selectedColumns = null;
      int[] selectedRows = null;
      ArrayList elements = new ArrayList();
      if (operationMode == 1) {
        selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getSelectedColumns();
        selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getSelectedRows();
        String[] seqs = new String[selectedRows.length];
        String pMask = null;
        String name;
        for (int i = 0; i < selectedRows.length; i++) {
          name = getSelectedProject().getSequenceName(selectedRows[i]);
          if (controller.isPairingMask(name)) {
            //assumes only one pairingmask
            pMask = name;
          } else {
            seqs[i] = name;
          }
        }
        if (pMask != null) {
          String symbol = JOptionPane.showInputDialog(null, "What symbol do you want to use?");
          if (symbol != null && symbol.length() >= 1) {
	     
            //only 1 character
            String[] mask = new String[1];
            mask[0] = pMask;
            elements.add(new HistoryElement(mask, selectedColumns, symbol.charAt(0)));
            controller.changeSymbols(elements);
            //controller.changeSymbols(mask, selectedColumns, symbol.charAt(0));
          } else {
            JOptionPane.showMessageDialog(null, "Only 1 character in the symbol");
          }
        } else if (seqs.length > 0) {
          char[] symbols = controller.getAlphabet();
          String[] alphabet = new String[symbols.length];
          for (int i = 0; i < symbols.length; i++) {
            alphabet[i] = "" + symbols[i];
          }
          String ch = (String) JOptionPane.showInputDialog(null, "What character do you want to insert?", "Select character", JOptionPane.PLAIN_MESSAGE, null, alphabet, null);
          if (ch != null && ch.length() == 1) {
            elements.add(new HistoryElement(seqs, selectedColumns, ch.charAt(0)));
            controller.changeSymbols(elements);
            //controller.changeSymbols(seqs, selectedColumns, ch.charAt(0));
          }
        }
      } else {
        if (operationMode == 2) {
          selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getSelectedColumns();
          selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getSelectedRows();
          String[] seqs = new String[selectedRows.length];
          String pMask = null;
          String name;
          for (int i = 0; i < selectedRows.length; i++) {
            name = getSelectedProject().getSequenceName(selectedRows[i]);
            if (controller.isPairingMask(name)) {
              //assumes only one pairingmask
              pMask = name;
            } else {
              seqs[i] = name;
            }
          }
          if (pMask != null) {
            String symbol = JOptionPane.showInputDialog(null, "What symbol do you want to use?");
            if (symbol != null &&  symbol.length() >= 1) {
              //only 1 character
              String[] mask = new String[1];
              mask[0] = pMask;
              elements.add(new HistoryElement(mask, selectedColumns, symbol.charAt(0)));
              controller.changeSymbols(elements);
              //controller.changeSymbols(mask, selectedColumns, symbol.charAt(0));
            } else {
              JOptionPane.showMessageDialog(null, "Only 1 character in the symbol");
            }
          } else if (seqs.length > 0) {
            char[] symbols = controller.getAlphabet();
            String[] alphabet = new String[symbols.length];
            for (int i = 0; i < symbols.length; i++) {
              alphabet[i] = "" + symbols[i];
            }
            String ch = (String) JOptionPane.showInputDialog(null, "What character do you want to insert?", "Select character", JOptionPane.PLAIN_MESSAGE, null, alphabet, null);
            if (ch.length() == 1) {
              elements.add(new HistoryElement(seqs, selectedColumns, ch.charAt(0)));
              controller.changeSymbols(elements);
              //controller.changeSymbolch != null && s(seqs, selectedColumns, ch.charAt(0));
            }
          }
        } else {
          selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getSelectedColumns();
          selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getSelectedRows();
          String[] seqs = new String[selectedRows.length];
          String pMask = null;
          String name;
          for (int i = 0; i < selectedRows.length; i++) {
            name = getSelectedProject().getSequenceName(selectedRows[i]);
            if (controller.isPairingMask(name)) {
              //assumes only one pairingmask
              pMask = name;
            } else {
              seqs[i] = name;
            }
          }
          if (pMask != null) {
            String symbol = JOptionPane.showInputDialog(null, "What symbol do you want to use?");
            if (symbol.length() >= 1) {
              //only 1 character
              String[] mask = new String[1];
              mask[0] = pMask;
              elements.add(new HistoryElement(mask, selectedColumns, symbol.charAt(0)));
              controller.changeSymbols(elements);
              //controller.changeSymbols(mask, selectedColumns, symbol.charAt(0));
            } else {
              JOptionPane.showMessageDialog(null, "Only 1 character in the symbol");
            }
          } else if (seqs.length > 0) {
            char[] symbols = controller.getAlphabet();
            String[] alphabet = new String[symbols.length];
            for (int i = 0; i < symbols.length; i++) {
              alphabet[i] = "" + symbols[i];
            }
            String ch = (String) JOptionPane.showInputDialog(null, "What character do you want to insert?", "Select character", JOptionPane.PLAIN_MESSAGE, null, alphabet, null);
            if (ch.length() == 1) {
              elements.add(new HistoryElement(seqs, selectedColumns, ch.charAt(0)));
              controller.changeSymbols(elements);
              //controller.changeSymbols(seqs, selectedColumns, ch.charAt(0));
            }
          }
        }
      }
    }
  }


  /**
   *  Description of the Method
   */
  public void toLowerCase() {
    if (views.getSelectedComponent() != null) {
      if (!((AlignmentPanel) views.getSelectedComponent()).isSplit()) {
        String name = getSelectedName();
        int[] selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getSelectedColumns();
        int[] selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getSelectedRows();

        if (selectedColumns.length > 0) {
          Vector v = getSequenceNames(selectedRows);
          ArrayList elements = new ArrayList();
          elements.add(new HistoryElement((String[]) v.toArray(new String[0]), selectedColumns));
          controller.toLowerCase(elements);
        }
      } else {
        toLowerCaseSplit();
      }
    } else {
      JOptionPane.showMessageDialog(this, "You need to create or open a project before changing cases");
    }
  }


  /**
   *  Description of the Method
   */
  public void toLowerCaseSplit() {
    String name = getSelectedName();
    int[] selectedColumns = null;
    int[] selectedRows = null;
    ArrayList elements = new ArrayList();
    if (operationMode == 1) {
      selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getLeftSelectedColumns();
      selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getLeftSelectedRows();

      if (selectedColumns.length > 0) {
        Vector v = getSequenceNames(selectedRows);
        elements.add(new HistoryElement((String[]) v.toArray(new String[0]), selectedColumns));
        //controller.toLowerCase((String[])v.toArray(new String[0]), selectedColumns);
        controller.toLowerCase(elements);
      }
    } else {
      if (operationMode == 2) {
        selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getRightSelectedColumns();
        selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getRightSelectedRows();
        if (selectedColumns.length > 0) {
          Vector v = getSequenceNames(selectedRows);
          elements.add(new HistoryElement((String[]) v.toArray(new String[0]), selectedColumns));
          controller.toLowerCase(elements);
        }
      } else {
        selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getLeftSelectedColumns();
        selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getLeftSelectedRows();

        if (selectedColumns.length > 0) {
          Vector v = getSequenceNames(selectedRows);
          //controller.toLowerCase((String[])v.toArray(new String[0]), selectedColumns);
          elements.add(new HistoryElement((String[]) v.toArray(new String[0]), selectedColumns));
        }
        selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getRightSelectedColumns();
        selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getRightSelectedRows();
        if (selectedColumns.length > 0) {
          Vector v = getSequenceNames(selectedRows);
          //controller.toLowerCase((String[])v.toArray(new String[0]), selectedColumns);
          elements.add(new HistoryElement((String[]) v.toArray(new String[0]), selectedColumns));
        }
        controller.toLowerCase(elements);
      }
    }
  }


  /**
   *  Adds a feature to the View attribute of the AlignmentContainer object
   *
   * @param  name  The feature to be added to the View attribute
   */
  public void addView(String name) throws Exception{
    try {
      setHistoryEnabled(prevHistoryEnabled);
      AlignmentPanel aPanel = new AlignmentPanel(controller, properties, name);
      if (splitMode && p2 != null && p3 != null && divider > 0) {
        aPanel.split();
        aPanel.setSplitAlign(splitAlign);
        aPanel.getTableView2().getViewport().setViewPosition(p2);
        aPanel.getTableView3().getViewport().setViewPosition(p3);
        aPanel.getSplitPane().setDividerLocation(divider);
      } else {
        if (p1 != null) {
          aPanel.getTableView().getViewport().setViewPosition(p1);
        }
      }
      //aPanel.addSequenceSelectionListener(infoPanel);
      //aPanel.addListSelectionListener(infoPanel);
      aPanel.selectionMode(selectionMode);
      aPanel.operationMode(operationMode);
      views.addTab(name, aPanel);
      views.setSelectedComponent(aPanel);
      if(overview == null)
      {
      setOverviewEnabled(parent.isOverviewEnabled());
      }
    } catch (Exception e) {
      //e.printStackTrace();
      JOptionPane.showMessageDialog(null, "Couldn't create view");
      throw e;
    }
  }


  /**
   *  Description of the Method
   *
   * @param  ue  Description of the Parameter
   */
  public void updatePerformed(UpdateEvent ue) {
    if (views.getTabCount() > 0) {
      if (controller.isChanged(getSelectedName())) {
        views.setTitleAt(views.getSelectedIndex(), "*" + getSelectedName());
      } else {
        views.setTitleAt(views.getSelectedIndex(), getSelectedName());
      }
    }
    if (controller.isChanged()) {
      //alignmentNameLabel.setText("Alignment: " + alignmentName + "*");
    } else {
      //alignmentNameLabel.setText("Alignment: " + alignmentName);
    }
    repaint();
  }


  /**
   *  Description of the Method
   */
  public void toUpperCase() {
    if (views.getSelectedComponent() != null) {
      String name = getSelectedName();
      if (!((AlignmentPanel) views.getSelectedComponent()).isSplit()) {
        int[] selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getSelectedColumns();
        int[] selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getSelectedRows();
        Vector v = getSequenceNames(selectedRows);
        ArrayList elements = new ArrayList();
        elements.add(new HistoryElement((String[]) v.toArray(new String[0]), selectedColumns));
        controller.toUpperCase(elements);
        //controller.toUpperCase((String[])v.toArray(new String[0]), selectedColumns);
      } else {
        toUpperCaseSplit();
      }
    }
  }


  /**
   *  Description of the Method
   */
  public void toUpperCaseSplit() {
    String name = getSelectedName();
    int[] selectedColumns = null;
    int[] selectedRows = null;
    Vector v = new Vector();
    ArrayList elements = new ArrayList();
    if (operationMode == 1) {
      selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getLeftSelectedColumns();
      selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getLeftSelectedRows();
      v = getSequenceNames(selectedRows);
      elements.add(new HistoryElement((String[]) v.toArray(new String[0]), selectedColumns));
      controller.toUpperCase(elements);
      //controller.toUpperCase((String[])v.toArray(new String[0]), selectedColumns);
    } else {
      if (operationMode == 2) {
        selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getRightSelectedColumns();
        selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getRightSelectedRows();
        v = getSequenceNames(selectedRows);
        elements.add(new HistoryElement((String[]) v.toArray(new String[0]), selectedColumns));
        controller.toUpperCase(elements);
        //controller.toUpperCase((String[])v.toArray(new String[0]), selectedColumns);
      } else {
        selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getLeftSelectedColumns();
        selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getLeftSelectedRows();
        v = getSequenceNames(selectedRows);
        elements.add(new HistoryElement((String[]) v.toArray(new String[0]), selectedColumns));
        //controller.toUpperCase((String[])v.toArray(new String[0]), selectedColumns);
        selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getRightSelectedColumns();
        selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getRightSelectedRows();
        v = getSequenceNames(selectedRows);
        elements.add(new HistoryElement((String[]) v.toArray(new String[0]), selectedColumns));
        controller.toUpperCase(elements);
        //controller.toUpperCase((String[])v.toArray(new String[0]), selectedColumns);
      }
    }
  }


  /**
   *  Gets the selectedColumns attribute of the AlignmentContainer object
   *
   * @return    The selectedColumns value
   */
  public int[] getSelectedColumns() {
    int[] selectedColumns = null;
    if (views.getSelectedComponent() != null) {
      if (!((AlignmentPanel) views.getSelectedComponent()).isSplit()) {
        selectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getSelectedColumns();
      } else {
        int[] rightselectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getRightSelectedColumns();
        int[] leftSelectedColumns = ((AlignmentPanel) views.getSelectedComponent()).getRightSelectedColumns();
      }
    }
    return selectedColumns;
  }



  /**
   *  Description of the Method
   */
  public void subalignment(){
    //find den tab der er i focus
    AlignmentPanel selected = getSelectedProject();
    Enumeration e = controller.getSequenceKeys();
    //        Vector v = new Vector();
    //        while (e.hasMoreElements())
    //      {
    //          v.add(e.nextElement());
    //      }
    //      Collections.sort(v);
    String[] selNames = new String[0];
    int start = 1;
    int end = controller.getAlignmentLength();
    if (selected != null) {
      int[] selectedColumns = selected.getSelectedColumns();
      //kald getSelectedColumns p� den
      if (selectedColumns != null && selectedColumns.length >= 2) {
        start = selected.getColumnNumber(selectedColumns[0]);
        end = selected.getColumnNumber(selectedColumns[selectedColumns.length - 1]);
      }
      selNames = (String[]) selected.getSelectedNames().toArray(new String[0]);
    }
    //(String[])v.toArray(new String[0])
    SubalignmentSelector.showSubalignmentSelector(selNames, controller.getSortedNames(), start, end,
      new ActionListener() {
        public void actionPerformed(ActionEvent ae) {
          SubalignmentSelector selector = (SubalignmentSelector) ae.getSource();
          String name = selector.getName();
          int start = selector.getStart();
          int end = selector.getEnd();
          Vector names = selector.getSelected();
          if (names.size() > 0 && name != null) {
            //controller.createAlignmentView(start, end, names, name);
           try
		   {
          	addView(name);
		   }
           catch(Exception e)
		   {
           	//TODO: this is a hack
           	//I dont think this method is ever called
		   }
          }
          selector.dispose();
        }
      });

  }


  /**
   *  Gets the sequenceNames attribute of the AlignmentContainer object
   *
   * @param  names  Description of the Parameter
   * @return        The sequenceNames value
   */
  public Vector getSequenceNames(int[] names) {
    Vector v = new Vector();
    for (int i = 0; i < names.length; i++) {
      v.add(((AlignmentPanel) views.getSelectedComponent()).getSequenceName(names[i]));
    }
    return v;
  }


  /**
   *  Gets the selectedName attribute of the AlignmentContainer object
   *
   * @return    The selectedName value
   */
  public String getSelectedName() {
    return ((AlignmentPanel) views.getSelectedComponent()).getAlignmentName();
  }


  /**
   *  Description of the Method
   */
  public void saveAs() {
    File workDir = new File(projectProperties.getPath() + "/");
    if (!workDir.exists()) {
      workDir.mkdirs();
    }
    JFileChooser chooser = new JFileChooser(workDir);
    int choice = chooser.showSaveDialog(null);
    if (choice == chooser.APPROVE_OPTION) {
      File f = chooser.getSelectedFile();
      saveAs(f);
      alignmentFileName = f.getName();
    }
  }


  /**
   *  Description of the Method
   */
  public void export() {
    AlignmentIOController.export(controller.getMainAlignment(), projectProperties);
  }


  /**
   *  Description of the Method
   *
   * @param  file  Description of the Parameter
   */
  public void saveAs(File file) {
    alignmentFileName = controller.saveAlignment(file);
    try {
      AlignmentIOController.saveHistory(controller.getFileHistory(), projectProperties.getPath());
      //AlignmentIOController.saveSubHistory(controller.getFileHistory().get(controller.getFileName().
      projectProperties.setFile(alignmentFileName.substring(alignmentFileName.lastIndexOf('/') + 1, alignmentFileName.length()));
      //linux hack
      projectProperties.store();
      storeProperties();
    } catch (IOException e) {
      e.printStackTrace();
      JOptionPane.showMessageDialog(null, "couldn't save property file");
    }
  }


  /**
   *  Description of the Method
   *
   * @param  fileName  Description of the Parameter
   */
  public void saveAs(String fileName) {
    saveAs(new File(fileName));
  }


  /**
   *  Description of the Method
   */
  public void save() {
    if (alignmentFileName == null || alignmentFileName.equals("")) {
      saveAs();
    } else {
      saveAs(projectProperties.getPath() + "/" + projectProperties.getFile());
    }
  }


  /**
   *  Gets the spJCheckBoxMenuItemlitMode attribute of the AlignmentContainer
   *  object
   *
   * @return    The splitMode value
   */
  public boolean isSplitMode() {
    if(getSelectedProject() != null)
    {
	  return getSelectedProject().isSplit();
    }
    return false;
  }


  /**
   *  Gets the selectedProject attribute of the AlignmentContainer object � *
   *
   * @return    The selectedProject value
   */
  public AlignmentPanel getSelectedProject() {
    return (AlignmentPanel) views.getSelectedComponent();
  }


  /**
   *  Description of the Method
   */
  public void analyze() {
    AlignmentContainer parent = this;
    	setCursor(new Cursor(Cursor.WAIT_CURSOR));    
    	checkAllViews();
        save();
        try
        {
        	analyzeFrame = new AnalyzeFrame(properties, projectProperties, controller.getMainAlignment(), parent, projectProperties.getName());
        }
        catch (Exception e)
        {
        	JOptionPane.showMessageDialog(parent,
        		    "Error message: "+e.getMessage(),
        		    "Error",
        		    JOptionPane.WARNING_MESSAGE);
        }
    }


  /**
   *  Description of the Method
   */
  public void createInfoPanel() {
    //infoPanel = new InformationPanel(this, controller);
  }


  /**
   *  Gets the analyzing attribute of the AlignmentContainer object
   *
   * @return    The analyzing value
   */
  public boolean isAnalyzing() {
    if (analyzeFrame == null) {
      return false;
    } else {
      return true;
    }
  }


  /**
   *  Description of the Method
   */
  public void rotateLeft() {
    //System.out.println("cont: left");
  	ArrayList elements = new ArrayList();
    if (getSelectedProject().isSplit()) {
      int[] columns = null;
      if (operationMode == 1) {
        columns = getSelectedProject().getLeftSelectedColumns();
        if (columns.length > 1) {
          //left table
          int[] rows = getSelectedProject().getLeftSelectedRows();
          String[] seqs = new String[rows.length];
          for (int i = 0; i < rows.length; i++) {
            seqs[i] = getSelectedProject().getSequenceName(rows[i]);
          }
          elements.add(new HistoryElement(seqs, columns[0], columns[columns.length - 1]));
          controller.rotateLeft(elements);
        }
      } else {
        if (operationMode == 2) {
          if (!selectionMode) {
            //right table
            columns = getSelectedProject().getRightSelectedColumns();
            if (columns.length > 1) {
              int[] rows = getSelectedProject().getRightSelectedRows();
              String[] seqs = new String[rows.length];
              for (int i = 0; i < rows.length; i++) {
                seqs[i] = getSelectedProject().getSequenceName(rows[i]);
              }
              elements.add(new HistoryElement(seqs, columns[0], columns[columns.length - 1]));
              controller.rotateLeft(elements);
              //controller.rotateLeft(seqs, columns[0], columns[columns.length - 1]);
            }
          }
        } else {
          if (!selectionMode) {
            columns = getSelectedProject().getLeftSelectedColumns();
            if (columns.length > 1) {
              //left table
              int[] rows = getSelectedProject().getLeftSelectedRows();
              String[] seqs = new String[rows.length];
              for (int i = 0; i < rows.length; i++) {
                seqs[i] = getSelectedProject().getSequenceName(rows[i]);
              }
              elements.add(new HistoryElement(seqs, columns[0], columns[columns.length - 1]));
              //controller.rotateLeft(seqs, columns[0], columns[columns.length - 1]);
            }
            //right table
            columns = getSelectedProject().getRightSelectedColumns();
            if (columns.length > 1) {
              int[] rows = getSelectedProject().getRightSelectedRows();
              String[] seqs = new String[rows.length];
              for (int i = 0; i < rows.length; i++) {
                seqs[i] = getSelectedProject().getSequenceName(rows[i]);
              }
              elements.add(new HistoryElement(seqs, columns[0], columns[columns.length - 1]));
              //controller.rotateLeft(seqs, columns[0], columns[columns.length - 1]);
            }
            controller.rotateLeft(elements);
          }
        }
      }
    } else {
      if(selectionMode)
      {
      int[] columns = getSelectedProject().getSelectedColumns();
      if (columns.length > 1) {
        int[] rows = getSelectedProject().getSelectedRows();
        String[] seqs = new String[rows.length];
        for (int i = 0; i < rows.length; i++) {
          seqs[i] = getSelectedProject().getSequenceName(rows[i]);
        }
        elements.add(new HistoryElement(seqs, columns[0], columns[columns.length - 1]));
        controller.rotateLeft(elements);
        //controller.rotateLeft(seqs, columns[0], columns[columns.length - 1]);
      } else {
        JOptionPane.showMessageDialog(null, "No columns selected!");
      }
      }
      else
      {
        showMoveWarning();
      }
    }
  }

  protected void showMoveWarning()
  {
    JOptionPane.showMessageDialog(this, "Move left/right works only in Single Select mode");
  }

  /**
   *  Description of the Method
   */
  public void rotateRight() {
  	//System.out.println("cont: right");
    ArrayList elements = new ArrayList();
    if (getSelectedProject().isSplit()) {
      int[] columns = null;
      if (operationMode == 1) {
        columns = getSelectedProject().getLeftSelectedColumns();
        if (columns.length > 1) {
          //left table
          int[] rows = getSelectedProject().getLeftSelectedRows();
          String[] seqs = new String[rows.length];
          for (int i = 0; i < rows.length; i++) {
            seqs[i] = getSelectedProject().getSequenceName(rows[i]);
          }
          elements.add(new HistoryElement(seqs, columns[0], columns[columns.length - 1]));
          controller.rotateRight(elements);
          //controller.rotateRight(seqs, columns[0], columns[columns.length - 1]);
        }
      } else {
        if (operationMode == 2) {
          if (!selectionMode) {
            //right table
            columns = getSelectedProject().getRightSelectedColumns();
            if (columns.length > 1) {
              int[] rows = getSelectedProject().getRightSelectedRows();
              String[] seqs = new String[rows.length];
              for (int i = 0; i < rows.length; i++) {
                seqs[i] = getSelectedProject().getSequenceName(rows[i]);
              }
              elements.add(new HistoryElement(seqs, columns[0], columns[columns.length - 1]));
              controller.rotateRight(elements);
              //controller.rotateRight(seqs, columns[0], columns[columns.length - 1]);
            }
          }
        } else {
          if (!selectionMode) {
            columns = getSelectedProject().getLeftSelectedColumns();
            if (columns.length > 1) {
              //left table
              int[] rows = getSelectedProject().getLeftSelectedRows();
              String[] seqs = new String[rows.length];
              for (int i = 0; i < rows.length; i++) {
                seqs[i] = getSelectedProject().getSequenceName(rows[i]);
              }
              elements.add(new HistoryElement(seqs, columns[0], columns[columns.length - 1]));
              //controller.rotateRight(seqs, columns[0], columns[columns.length - 1]);
            }
            //right table
            columns = getSelectedProject().getRightSelectedColumns();
            if (columns.length > 1) {
              int[] rows = getSelectedProject().getRightSelectedRows();
              String[] seqs = new String[rows.length];
              for (int i = 0; i < rows.length; i++) {
                seqs[i] = getSelectedProject().getSequenceName(rows[i]);
              }
              elements.add(new HistoryElement(seqs, columns[0], columns[columns.length - 1]));
              //controller.rotateRight(seqs, columns[0], columns[columns.length - 1]);
            }
          }
          controller.rotateRight(elements);
        }
      }
    } else {
      if(selectionMode) {
      int[] columns = getSelectedProject().getSelectedColumns();
      if (columns.length > 1) {
        int[] rows = getSelectedProject().getSelectedRows();
        String[] seqs = new String[rows.length];
        for (int i = 0; i < rows.length; i++) {
          seqs[i] = getSelectedProject().getSequenceName(rows[i]);
        }
        elements.add(new HistoryElement(seqs, columns[0], columns[columns.length - 1]));
        controller.rotateRight(elements);
        //controller.rotateRight(seqs, columns[0], columns[columns.length - 1]);
      } else {
        JOptionPane.showMessageDialog(null, "No columns selected!");
      }
      }
      else
      {
        showMoveWarning();
      }
    }
  }


  /**
   *  Sets the traceEnabled attribute of the AlignmentContainer object
   *
   * @param  value  The new traceEnabled value
   */
  public void setTraceEnabled(boolean value) {
    Component[] children = views.getComponents();
    for (int i = 0; i < children.length; i++) {
      ((AlignmentPanel) views.getComponent(i)).setTraceEnabled(value);
    }
  }


  /**
   *  Sets the projectDir attribute of the AlignmentContainer object
   *
   * @param  dir  The new projectDir value
   */
  public void setProjectDir(String dir) {
    projectDir = dir;
    projectProperties.setPath(projectDir);
  }


  /**
   *  Description of the Method
   */
  public void clear() {
    ((AlignmentPanel) views.getSelectedComponent()).clearSelections();
  }


  /**
   *  Description of the Method
   */
  public void jump() {
    String number = JOptionPane.showInputDialog("Which column will you go to");
    try {
      int num = Integer.parseInt(number);
      getSelectedProject().centerTable(num - 1, -1);
    } catch (NumberFormatException nfe) {
      JOptionPane.showMessageDialog(null, "Not a number!");
    }
  }


  /**
   *  Description of the Method
   */
  public void closeAll() {
    while (views.getTabCount() > 0) {
      close();
    }
  }


  /**
   *  Description of the Method
   *
   * @param  name  Description of the Parameter
   */
  public void checkView(String name) {

    if (controller.getMainAlignment() != null && controller.isChanged()) {
      int i = JOptionPane.showConfirmDialog(null, "There is uncommited changes to: " + name + ".\n\nDo you want to save them?");
      if (i == JOptionPane.OK_OPTION) {
        save();
      }
    }
  }


  /**
   *  Description of the Method
   */
  public void checkAllViews() {
    for (int i = 0; i < views.getTabCount(); i++) {
      checkView(views.getTitleAt(i));
    }
  }


  /**
   *  Description of the Method
   */
  public void cleanup() {
    //saveSplitMode();
    //close all windows
    closeAll();
    //remove all connections to analyzeFrame
    if (analyzeFrame != null) {
      analyzeFrame.removeActionListener();
      analyzeFrame = null;
    }
  }


  /**
   *  Sets the basecoloringEnabled attribute of the AlignmentContainer object
   *
   * @param  enabled  The new basecoloringEnabled value
   */
  public void setBasecoloringEnabled(boolean enabled) {
    properties.setBaseColoring(enabled);
  }


  /**
   *  Description of the Method
   *
   * @param  me  Description of the Parameter
   */
  public void mouseClicked(MouseEvent me) {
    if (me.getX() <= controller.getAlignmentLength() && me.getY() < controller.getNumberOfSequences()) {
        getSelectedProject().centerCell(me.getX(), me.getY());
    }
    me.consume();
  }


  /**
   *  Description of the Method
   *
   * @param  me  Description of the Parameter
   */
  public void mouseEntered(MouseEvent me) {
  }


  /**
   *  Description of the Method
   *
   * @param  me  Description of the Parameter
   */
  public void mouseExited(MouseEvent me) {
  }


  /**
   *  Description of the Method
   *
   * @param  me  Description of the Parameter
   */
  public void mousePressed(MouseEvent me) {
  }


  /**
   *  Description of the Method
   *
   * @param  me  Description of the Parameter
   */
  public void mouseReleased(MouseEvent me) {
  }


  /**
   *  Sets the historyEnabled attribute of the AlignmentContainer object
   *
   * @param  enabled  The new historyEnabled value
   */
  public void setHistoryEnabled(boolean enabled) {
    historyEnabled = enabled;
//    if (enabled) {
//      if (historyWindow == null) {
//        historyWindow = new JFrame("History");
//        historyWindow.setLocation(historyX,historyY);
//        HistoryPanel hPanel = getHistoryPanel();
//        historyWindow.getContentPane().add(hPanel);
//        historyWindow.setSize(300, 200);
//        historyWindow.setDefaultCloseOperation(WindowConstants.HIDE_ON_CLOSE);
//        Component c = getParent();
//        /*System.out.println("qwe");
//        while (!c.getClass().getName().equals("dk.kvl.gui.alignment.AlignmentFrame")) {
//          c = c.getParent();
//        }*/
//        //historyWindow.addComponentListener((AlignmentFrame) c);
//        historyWindow.addComponentListener(parent);
//        historyWindow.setVisible(true);
//      } else {
//        historyWindow.setVisible(true);
//      }
//    } else {
//       {
//        if (historyWindow != null) {
//          historyWindow.dispose();
//          historyWindow = null;
//        }
//      }

   // }
  }


public HistoryPanel getHistoryPanel() {
	HistoryPanel hPanel = new HistoryPanel(controller, this, properties);
	hPanel.addReloadListener(this);
	return hPanel;
}


  /**
   *  Description of the Method
   *
   * @param  g  Description of the Parameter
   */
  public void paint(Graphics g) {
    if (historyWindow != null) {
      historyWindow.repaint();
    }
    super.paint(g);
  }


  /**
   *  Gets the pairing attribute of the AlignmentContainer object
   *
   * @param  index  Description of the Parameter
   * @return        The pairing value
   */
  public int getPairing(int index) {
//    if (controller.getPairingmask() == null) {
//      return -1;
//    }
    int[] selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getSelectedRows();
    if (selectedRows.length > 0) {
      String name = getSelectedProject().getSequenceName(selectedRows[0]);
      return controller.getPairing(name, index);
    }
    return -1;
  }


  /**
   *  Gets the selectedSequence attribute of the AlignmentContainer object
   *
   * @return    The selectedSequence value
   */
  public String getSelectedSequence() {
    int[] selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getSelectedRows();
    if (selectedRows.length > 0) {
      return getSelectedProject().getSequenceName(selectedRows[0]);
    }
    return null;
  }


  /**
   *  Gets the selectedSequences attribute of the AlignmentContainer object
   *
   * @return    The selectedSequence value
   */
  public String[] getSelectedSequences() {
    int[] selectedRows = ((AlignmentPanel) views.getSelectedComponent()).getSelectedNameRows();
    String[] deleteRows = null;
    if (selectedRows != null && selectedRows.length > 0) {
      deleteRows = new String[selectedRows.length];
      for (int i = 0; i < selectedRows.length; i++) {
        deleteRows[i] = getSelectedProject().getSequenceName(selectedRows[i]);
      }
    }
    return deleteRows;
  }


  /**
   *  Description of the Method
   */
  public void removeGaps() {
    int removed = controller.removeGaps();
  }


  /**
   *  Description of the Method
   */
  public void removeColumn() {
    String res = JOptionPane.showInputDialog(null, "What column do you want to remove");
    if (res != null) {
      try {
        int col = Integer.parseInt(res);
        if (col > 0 && col <= controller.getAlignmentLength()) {
          ArrayList elements = new ArrayList();
          elements.add(new HistoryElement(col - 1));
          controller.removeColumn(elements);
          //controller.removeColumn(col - 1);
        }
      } catch (NumberFormatException nf) {
        JOptionPane.showMessageDialog(null, "Not a number!");
      }

    }
  }


  /**
   *  Adds a feature to the Column attribute of the AlignmentContainer object
   */
  public void addColumn() {
    String res = JOptionPane.showInputDialog(null, "At what column do you want to insert?");
    if (res != null) {
      try {
        int col = Integer.parseInt(res);
        if (col > 0 && col <= controller.getAlignmentLength()) {
          ArrayList elements = new ArrayList();
          elements.add(new HistoryElement(col - 1));
          controller.addColumn(elements);
          //controller.addColumn(col - 1);
        }
      } catch (NumberFormatException nf) {
        JOptionPane.showMessageDialog(null, "Not a number!");
      }
    }
  }


  /**
   *  Description of the Method
   */
  public void toggleSplit() {
    getSelectedProject().toggleSplit();
  }


  /**
   *  Description of the Method
   */
  public void reload()throws Exception {
    saveSplitMode();
    closeAll();
    projectProperties.setFile(controller.getFileName());
    addView(controller.getFileName());
    
    //controller.setHistory(fileHist.get(controller.getFileName()).getHistory());
  }


  /**
   *  Description of the Method
   */
  public void saveSplitMode() {
    AlignmentPanel ap = getSelectedProject();
    splitMode = isSplitMode();
    if(ap != null)
    {
    if (isSplitMode()) {
      divider = getSelectedProject().getSplitPane().getDividerLocation();
      p2 = ap.getTableView2().getViewport().getViewPosition();
      p3 = ap.getTableView3().getViewport().getViewPosition();
      splitAlign = ap.horizontalSplit();
    } else {
      p1 = ap.getTableView().getViewport().getViewPosition();
    }
    }
    else
    {
    	p1 = new Point(0,0);
    }
  }


  /**
   *  Description of the Method
   *
   * @param  mode  Description of the Parameter
   */
  public void splitMode(boolean mode) {
    splitAlign = mode;
    ((AlignmentPanel) views.getSelectedComponent()).splitMode(mode);
  }


  /**
   *  Description of the Method
   *
   * @param  mode  Description of the Parameter
   */
  public void selectionMode(boolean mode) {
    selectionMode = mode;
    ((AlignmentPanel) views.getSelectedComponent()).selectionMode(mode);
  }


  /**
   *  Description of the Method
   *
   * @param  mode  Description of the Parameter
   */
  public void operationMode(int mode) {
    operationMode = mode;
    ((AlignmentPanel) views.getSelectedComponent()).operationMode(operationMode);
  }


  /**
   *  Sets the history attribute of the AlignmentContainer object
   *
   * @param  hist  The new history value
   */
  public void setHistory(FileHistory hist) {
    controller.setFileHistory(hist);
  }


  /**
   *  Gets the properties attribute of the AlignmentContainer object
   *
   * @return    The properties value
   */
  public AlignmentProperties getProperties() {
    return properties;
  }


  /**
   *  Description of the Method
   */
  public void updateCellSize() {
    getSelectedProject().updateCellSize();
  }


  /**
   *  Description of the Method
   */
  /*
   *  public void loadProperties()
   *  {
   *  try
   *  {
   *  File f = new File(AlignmentProperties.getPropertyDir() + "propereties.obj");
   *  if (f.exists())
   *  {
   *  ObjectInputStream ois = new ObjectInputStream(new FileInputStream(f));
   *  properties = (AlignmentProperties)ois.readObject();
   *  ois.close();
   *  }
   *  else
   *  {
   *  properties = new AlignmentProperties();
   *  }
   *  }
   *  catch (Exception e)
   *  {
   *  e.printStackTrace();
   *  }
   *  }
   */

  /**
   *  Description of the Method
   */
  public void storeProperties() {
    try {
      File f = new File(AlignmentProperties.getProjectDir() + "/properties.obj");
      if (f.exists()) {
        f.delete();
      }
      ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(f));
      oos.writeObject(properties);
      oos.close();
    } catch (Exception e) {
      e.printStackTrace();
    }
  }


  /**
   *  Description of the Method
   */
  public void updateGrid() {
    getSelectedProject().updateGrid();
  }


  /**
   *  Sets the background attribute of the AlignmentContainer object
   */
  public void setBackground() {
    Color c = JColorChooser.showDialog(this, "Choose Background Color", properties.getBackground());
    properties.setBackground(c);
  }


  /**
   *  Sets the textColor attribute of the AlignmentContainer object
   */
  public void setTextColor() {
    Color c = JColorChooser.showDialog(this, "Choose Text Color", properties.getBackground());
    properties.setTextColor(c);
  }


  /**
   *  Sets the primarySelectionColor attribute of the AlignmentContainer object
   */
  public void setPrimarySelectionColor() {
    Color c = JColorChooser.showDialog(this, "Choose Color for Primary Selection", properties.getBackground());
    properties.setPrimarySelectionColor(c);
  }


  /**
   *  Sets the secondarySelectionColor attribute of the AlignmentContainer object
   */
  public void setSecondarySelectionColor() {
    Color c = JColorChooser.showDialog(this, "Choose Color for Secondary Selection", properties.getBackground());
    properties.setSecondarySelectionColor(c);
  }


  /**
   *  Sets the sequenceColor attribute of the AlignmentContainer object
   */
  public void setSequenceColor() {
    Color c = JColorChooser.showDialog(this, "Choose Color for Sequence Names", properties.getBackground());
    properties.setSequenceColor(c);
  }


  /**
   *  Sets the gColor attribute of the AlignmentContainer object
   */
  public void setGColor() {
    Color c = JColorChooser.showDialog(this, "Choose Color for G", properties.getBackground());
    properties.setGColor(c);
  }


  /**
   *  Sets the cColor attribute of the AlignmentContainer object
   */
  public void setCColor() {
    Color c = JColorChooser.showDialog(this, "Choose Color for C", properties.getBackground());
    properties.setCColor(c);
  }


  /**
   *  Sets the aColor attribute of the AlignmentContainer object
   */
  public void setAColor() {
    Color c = JColorChooser.showDialog(this, "Choose Color for A", properties.getBackground());
    properties.setAColor(c);
  }


  /**
   *  Sets the uColor attribute of the AlignmentContainer object
   */
  public void setUColor() {
    Color c = JColorChooser.showDialog(this, "Choose Color for U", properties.getBackground());
    properties.setUColor(c);
  }


  /**
   *  Adds a feature to the Sequences attribute of the AlignmentContainer
   *  object
   */
  public void addSequences() {
    //user must select a file
    //JFileChooser fileChooser = new JFileChooser(AlignmentProperties.getProjectDir());
	  JFileChooser fileChooser = new JFileChooser(System.getProperty("user.dir"));
	    
	  fileChooser.addChoosableFileFilter(new ColFilter());
    fileChooser.addChoosableFileFilter(new TxtFilter());
    int chooserStatus = fileChooser.showOpenDialog(null);
    if (chooserStatus == JFileChooser.APPROVE_OPTION) {
      File f = fileChooser.getSelectedFile();
      //read-file
      Alignment a = null;
      try {
        if (fileChooser.getFileFilter() instanceof ColFilter || f.getName().endsWith("col")) {
          a = AlignmentIOController.loadAlignment(f, AlignmentWriter.COL);
        } else if (fileChooser.getFileFilter() instanceof TxtFilter || f.getName().endsWith("txt") || f.getName().endsWith("widetext")) {
          a = AlignmentIOController.loadAlignment(f, AlignmentWriter.TXT);
        }
        if (a != null) {
          //extract & add sequences
          int added = controller.addSequences(a);
          JOptionPane.showMessageDialog(null, added + " sequences imported");
        }
        validate();
        repaint();
      } catch (Exception e) {
        e.printStackTrace();
      }
    }
  }


  /**
   *  Description of the Method
   */
  public void unpairBases() {
    int[] cols = getSelectedProject().getSelectedColumns();
    String[] names = getSelectedProject().getSelectedSequenceNames();
    ArrayList elements = new ArrayList();
    elements.add(new HistoryElement(names, cols));
    controller.unpair(elements);
    //controller.unpair(names, cols);
  }


  /**
   *  Description of the Method
   */
  public void pairBases() {
    int[] cols = getSelectedProject().getSelectedColumns();
    if (cols.length % 2 == 0) {
      String[] names = getSelectedProject().getSelectedSequenceNames();
      ArrayList elements = new ArrayList();
      elements.add(new HistoryElement(names, cols));
      controller.pair(elements);
      //controller.pair(names, cols);
    } else {
      JOptionPane.showMessageDialog(null, "You must select an equal number of columns");
    }

  }


  /**
   *  Description of the Method
   */
  public void removeColors() {
    controller.removeColors();
    repaint();
  }


  /**
   *  Description of the Method
   */
  public void removeSequences() {
    if (views.getSelectedComponent() != null) {
      String[] delSequences = getSelectedSequences();
      if (delSequences != null) {
        StringBuffer sb = new StringBuffer("Delete the following sequences? \n");
        for (int i = 0; i < delSequences.length; i++) {
          sb.append(delSequences[i]).append("\n");
        }
        if (JOptionPane.showConfirmDialog(this, sb.toString()) == 0) {
          if (controller.deleteSequences(delSequences)) {
            //JOptionPane.showMessageDialog(this, "Sequences deleted");
          }
        } else {
          JOptionPane.showMessageDialog(this, "The sequences cannot be deleted");
        }
      } else {
        JOptionPane.showMessageDialog(this, "Select sequences to delete");
      }
    } else {
      JOptionPane.showMessageDialog(this, "You need to create or open a project before deleting");
    }
  }


  /**
   *  Description of the Method
   */
  public void fitCases() {
    //selects all sequences and run pair bases
    controller.fitCases();
    controller.getMainAlignment().updatePairings();
    validate();
    repaint();
  }
  
  
  public void emptyPairingmask()
  {
    controller.emptyPairingmask(); 
    
  }
  
  public AlignmentFrame parent()
  {
  	return parent;
  }
  
  public JPanel getOverviewPanel(JFrame owner)
  {
	  JPanel overviewPanel = new JPanel(new BorderLayout());
	  overview = new AlignmentCanvas(owner, controller);
      final JScrollPane jsp = new JScrollPane(overview);
      jsp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED);
      jsp.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
      JButton zoomIn = new JButton("+");
      zoomIn.addActionListener(
        new ActionListener() {
          public void actionPerformed(ActionEvent ae) {
            overview.setZoom(overview.getZoom() + 1);
          }
        });
      JButton fitZoom = new JButton("Fit");
      fitZoom.addActionListener(
        new ActionListener() {
          public void actionPerformed(ActionEvent ae) {
            overview.adjustZoom();
          }
        });

      JButton zoomOut = new JButton("-");
      zoomOut.addActionListener(
        new ActionListener() {
          public void actionPerformed(ActionEvent ae) {
            overview.setZoom(overview.getZoom() - 1);
          }
        });
      JPanel buttonPanel = new JPanel();
      buttonPanel.add(zoomOut);
      buttonPanel.add(fitZoom);
      buttonPanel.add(zoomIn);
      overview.addMouseListener(this);
      overview.adjustZoom();
      
      overviewPanel.add(jsp);
      overviewPanel.add(buttonPanel,BorderLayout.SOUTH);
    return overviewPanel;
  }


public void addPairingmask() {
	PairingMask mask = new PairingMask(controller.getAlignmentLength());
	controller.addPairingMask(mask);
}
  
}

