/**
 *		SARSE, Semi-Automated RNA Sequence Editor.
 * 		Copyright (C) 2004 Allan Lind-Thomsen
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version 2
 *	of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package dk.kvl.gui.components;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;
import dk.kvl.tools.*;
import dk.kvl.tools.history.*;
import dk.kvl.controller.*;
import dk.kvl.gui.components.*;
import dk.kvl.gui.event.*;
import java.util.*;
import dk.kvl.gui.alignment.*;
import dk.kvl.gui.properties.*;

/**
 *  Description of the Class
 *
 * @author     allan
 * @created    November 12, 2004
 */
public class HistoryPanel extends JPanel implements TreeSelectionListener, UpdateListener, MouseListener {

  private final AlignmentController controller;
  private JTree fileTree;
  private FileHistory history;
  private ReloadListener listener;
  private DefaultMutableTreeNode root;
  private AlignmentContainer container;
  private AlignmentProperties properties;


  /**
   *Constructor for the HistoryPanel object
   *
   * @param  controller  Description of the Parameter
   * @param  container   Description of the Parameter
   * @param  properties  Description of the Parameter
   */
  public HistoryPanel(AlignmentController controller, AlignmentContainer container, AlignmentProperties properties) {
    super(new BorderLayout());
    this.controller = controller;
    this.container = container;
    this.properties = properties;
    createFileTree();
    JScrollPane jsp = new JScrollPane(fileTree, JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
    add(jsp, BorderLayout.CENTER);
  }


  /**
   *  Description of the Method
   */
  public void createFileTree() {
    history = controller.getFileHistory();
    root = new DefaultMutableTreeNode("Files in project");

    for (int i = 0; i < history.size(); i++) {
      DefaultMutableTreeNode node = new DefaultMutableTreeNode(((FileHistoryItem) history.get(i)).getLabel());
      History subHistory = ((FileHistoryItem) history.get(i)).getHistory();
      if (subHistory != null) {
        subHistory.addUpdateListener(this);
        for (int j = 0; j < subHistory.size(); j++) {
          ArrayList list = (ArrayList) subHistory.get(j);
          node.add(new DefaultMutableTreeNode((String) list.get(list.size() - 1)));
        }
      }
      root.add(node);
    }

    fileTree = new JTree(new DefaultTreeModel(root));
    fileTree.setCellRenderer(new HistoryTreeRenderer(controller.getFileName(), controller.getHistory(), root, properties));
    fileTree.addTreeSelectionListener(this);
    fileTree.addMouseListener(this);
  }


  /**
   *  Description of the Method
   *
   * @param  e  Description of the Parameter
   */
  public void updatePerformed(UpdateEvent e) {
    Enumeration enumeration = root.children();
    int index = -1;
    DefaultMutableTreeNode node = null;
    if (e.getID() == 0) {
      while (enumeration.hasMoreElements()) {
        node = (DefaultMutableTreeNode) enumeration.nextElement();
        if (node.getUserObject().equals(controller.getFileName())) {
          ArrayList list = (ArrayList) e.getSource();
          node.add(new DefaultMutableTreeNode((String) list.get(list.size() - 1)));
          break;
        }
      }
      fileTree.expandPath(new TreePath(node.getPath()));
      int[] indices = {node.getIndex(node.getLastChild())};
      ((DefaultTreeModel) fileTree.getModel()).nodesWereInserted(node, indices);
    } else {
      if (e.getID() == 1) {
        int[] items = (int[]) e.getSource();
        DefaultMutableTreeNode[] nodes = new DefaultMutableTreeNode[1];
        while (enumeration.hasMoreElements()) {
          node = (DefaultMutableTreeNode) enumeration.nextElement();
          if (node.getUserObject().equals(controller.getFileName())) {
            nodes[0] = (DefaultMutableTreeNode) node.getChildAt(items[0]);
            node.remove(items[0]);
            break;
          }
        }
        fileTree.expandPath(new TreePath(node.getPath()));
        ((DefaultTreeModel) fileTree.getModel()).nodesWereRemoved(node, items, nodes);
      }
    }
  }


  /**
   *  Description of the Method
   *
   * @param  e  Description of the Parameter
   */
  public void valueChanged(TreeSelectionEvent e) {
    DefaultMutableTreeNode node = (DefaultMutableTreeNode) fileTree.getLastSelectedPathComponent();

    if (node != null) {
      if (node.getLevel() == 1) {
        for (int i = 0; i < history.size(); i++) {
          FileHistoryItem item = (FileHistoryItem) history.get(i);
          if (item.getLabel().equals(node.getUserObject())) {
            try {
              container.save();
              controller.changeMainAlignment(item.getFilePath() + "/" + item.getLabel());
              controller.setFileName(item.getLabel());
              controller.setHistory(item.getHistory());
              controller.getHistory().setAlignment(controller.getMainAlignment());
              ((HistoryTreeRenderer) fileTree.getCellRenderer()).setFileName(item.getLabel());
              ((HistoryTreeRenderer) fileTree.getCellRenderer()).setHistory(item.getHistory());
              listener.reload();
            } catch (Exception ex) {
              JOptionPane.showMessageDialog(this, "Could not change file");
              ex.printStackTrace();
            }
            return;
          }
        }
      }
    }
  }


  /**
   *  Adds a feature to the ReloadListener attribute of the HistoryPanel object
   *
   * @param  listener  The feature to be added to the ReloadListener attribute
   */
  public void addReloadListener(ReloadListener listener) {
    this.listener = listener;
  }


  /**
   *  Description of the Method
   *
   * @param  e  Description of the Parameter
   */
  public void mouseClicked(MouseEvent e) {
    DefaultMutableTreeNode node = (DefaultMutableTreeNode) fileTree.getLastSelectedPathComponent();
    if (node != null) {
      if (node.getLevel() == 2 && ((DefaultMutableTreeNode) node.getParent()).getUserObject().equals(controller.getFileName())) {
        History subHistory = ((FileHistoryItem) controller.getFileHistory().get(controller.getFileName())).getHistory();
        DefaultMutableTreeNode parent = (DefaultMutableTreeNode) node.getParent();
        int newIndex = parent.getIndex(node);
        int oldIndex = subHistory.getPreviousIndex();
        subHistory.setPreviousIndex(newIndex);
        if (newIndex < oldIndex || oldIndex == -1) {
          controller.undo(newIndex);
          subHistory.setUndone(true);
        } else {
          if (newIndex > oldIndex) {
            controller.redo(newIndex);
            subHistory.setUndone(false);
          } else {
            if (subHistory.getUndone()) {
              controller.redo(newIndex);
              subHistory.setUndone(false);
            } else {
              controller.undo(newIndex);
              subHistory.setUndone(true);
            }
          }
        }
      }
    }
  }


  /**
   *  Description of the Method
   *
   * @param  e  Description of the Parameter
   */
  public void mousePressed(MouseEvent e) { }


  /**
   *  Description of the Method
   *
   * @param  e  Description of the Parameter
   */
  public void mouseReleased(MouseEvent e) { }


  /**
   *  Description of the Method
   *
   * @param  e  Description of the Parameter
   */
  public void mouseExited(MouseEvent e) { }


  /**
   *  Description of the Method
   *
   * @param  e  Description of the Parameter
   */
  public void mouseEntered(MouseEvent e) { }
}

