/*
    SARSE, Semi-Automated RNA Sequence Editor. Copyright (C) 2004 Allan
    Lind-Thomsen This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or (at your
    option) any later version. This program is distributed in the hope that it
    will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
    of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
    Public License for more details. You should have received a copy of the GNU
    General Public License along with this program; if not, write to the Free
    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/
package dk.kvl.sequencetools;

import java.awt.*;
import java.util.*;
import java.util.regex.*;
import java.io.*;



import dk.kvl.tools.*;

/**
 *  This class contains a sequence
 *
 * @author     allan
 * @created    January 7, 2004
 */

public class Sequence implements Serializable, Comparable
{
    private boolean valid = true;
    private int number = -1;
    private SequenceAlphabet alphabet;
    private String label = null;
    private char[] sequence = null;
    protected int[] pairings = null;
    protected Vector columnLabels = new Vector();
    //sequenceColumns: consists of a String[] for each position with length = numberOfColumns
    protected Vector sequenceColumns = new Vector();
    private StringBuffer tmpSequence;
    private IntArray tmpPairings = null;
    private StringBuffer information = new StringBuffer();
    //coloring
    private boolean background = false;
    private boolean foreground = false;
    private Color[] foregroundColors;
    private Color[] backgroundColors;
    //columns
    int align_bpPosition = -1;
    int numberOfColumns = -1;
    int residue = -1;
    int color_b = -1;
    int color_r = -1;
    int color_g = -1;
    int color2_b = -1;
    int color2_r = -1;
    int color2_g = -1;


    /**
     *  Constructor for the Sequence object
     */
    public Sequence()
    {
        this(SequenceAlphabet.RNAALPHABET);
    }


    /**
     *  Constructor for the Sequence object
     *
     * @param  alpha  Description of the Parameter
     */
    public Sequence(int alpha)
    {
        alphabet = getAlphabet(alpha);
        tmpSequence = new StringBuffer();
    }


    /**
     *  Basic constructor
     *
     * @param  alph           An alphabet
     * @param  sequence       A sequence
     * @param  checkSequence  true if you want to check if the sequence is legal
     * @param  label          Description of the Parameter
     */
    public Sequence(String label, String sequence, int alph, boolean checkSequence)
    {
        alphabet = getAlphabet(alph);
        if (checkSequence)
        {
            if (!alphabet.isValidSequence(sequence))
            {
                valid = false;
            }
        }
        this.label = label;
        this.sequence = sequence.toCharArray();
    }


    /**
     *  Gets the alphabet attribute of the Sequence object
     *
     * @param  alph  Description of the Parameter
     * @return       The alphabet value
     */
    public static SequenceAlphabet getAlphabet(int alph)
    {
        SequenceAlphabet sa = null;
        if (alph == SequenceAlphabet.RNAALPHABET)
        {
            sa = new RNAAlphabet();
        }
        else if (alph == SequenceAlphabet.PAIRINGMASK)
        {
            sa = new PairingmaskAlphabet();
        }
        else 
        {
            sa = new ArbitraryAlphabet();
        }
        return sa;
    }


    /**
     *  Gets the alphabet attribute of the Sequence object
     *
     * @return    The alphabet value
     */
    public SequenceAlphabet getAlphabet()
    {
        return alphabet;
    }


    /**
     *  Gets the label attribute of the Sequence object
     *
     * @return    The label value
     */
    public String getLabel()
    {
        return label;
    }


    /**
     *  Gets the sequence attribute of the Sequence object
     *
     * @return    The sequence value
     */
    public String getSequence()
    {
        return new String(sequence);
    }


    /**
     *  Gets the sequenceArray attribute of the Sequence object
     *
     * @return    The sequenceArray value
     */
    public char[] getSequenceArray()
    {
        return sequence;
    }


    /**
     *  Gets the type attribute of the Sequence object
     *
     * @return    The type value
     */
    public int getType()
    {
        return alphabet.getAlphabetType();
    }

    public void setAlphabet(int newType)
    {
        alphabet = getAlphabet(newType);
    }
    /**
     *  Gets the length attribute of the Sequence object
     *
     * @return    The length value
     */
    public int getLength()
    {
        return sequence.length;
    }


    /**
     *  Gets the subsequence attribute of the Sequence object
     *
     * @param  begin  Description of the Parameter
     * @param  end    Description of the Parameter
     * @return        The subsequence value
     */
    public Sequence getSubsequence(int begin, int end)
    {
        Sequence seq = null;
        if (end < getLength() && begin < end && begin >= 0)
        {
            try
            {
                seq = new Sequence(getLabel(), getSequence().substring(begin, end + 1), getType(), false);
                seq.setBackground(background);
                seq.setForeground(foreground);
                Color[] bColors = new Color[end + 1 - begin];
                System.arraycopy(backgroundColors, begin, bColors, 0, end + 1 - begin);
                seq.setBackgroundColors(bColors);
                Color[] fColors = new Color[end + 1 - begin];
                System.arraycopy(foregroundColors, begin, fColors, 0, end + 1 - begin);
                seq.setForegroundColors(fColors);
                Vector cLabels = new Vector();
                Vector sColumns = new Vector();
                for (int i = begin; i < end + 1; i++)
                {
                    cLabels.add(columnLabels.get(i));
                    sColumns.add(sequenceColumns.get(i));
                }
                seq.setColumnLabels(cLabels);
                seq.setSequenceColumns(sColumns);
            }
            catch (Exception e)
            {
                //this exception should never be thrown here, because the program sends a valid
                //subsequence and therefore don't test if the sequence is valid, the check parameter is set to false
                e.printStackTrace();
            }
        }
        return seq;
    }


    /**
     *  Gets the symbol attribute of the Sequence object
     *
     * @param  index  Value= [0;length[
     * @return        The symbol value
     */
    public char getSymbolAt(int index)
    {
        if (index >= 0 && index < getLength())
        {
            return sequence[index];
        }
        else
        {
            return ' ';
        }
    }


    /**
     *  Sets the symbolAt attribute of the Sequence object
     *
     * @param  index   The new symbolAt value
     * @param  symbol  The new symbolAt value
     */
    public void setSymbolAt(int index, char symbol)
    {
        sequence[index] = symbol;
    }


    /**
     *  the symbol in place indexA change place with the symbol in place indexB
     *
     * @param  indexA  Description of the Parameter
     * @param  indexB  Description of the Parameter
     * @return         Description of the Return Value
     */
    public boolean swapSymbols(int indexA, int indexB)
    {
        boolean result = true;
        if (indexA < 0 || indexB < 0)
        {
            result = false;
        }
        else if (indexA >= sequence.length || indexB >= sequence.length)
        {
            result = false;
        }
        else
        {
            char tmp = sequence[indexA];
            sequence[indexA] = sequence[indexB];
            sequence[indexB] = tmp;
            if(background)
            {
            	Color tmpCol = backgroundColors[indexA];
            	backgroundColors[indexA]=backgroundColors[indexB];
            	backgroundColors[indexB]=tmpCol;
            }
            if(foreground)
            {
            	Color tmpCol = foregroundColors[indexA];
            	foregroundColors[indexA]=foregroundColors[indexB];
            	backgroundColors[indexB]=tmpCol;
            }
            if(pairings != null && pairings.length==sequence.length)
            {
            	if(pairings[indexA]>=0 && pairings[pairings[indexA]]==indexA)
            	{
            		pairings[pairings[indexA]]=indexB;
            	}
            	if(pairings[indexB]>=0 && pairings[pairings[indexB]]==indexB)
            	{
            		pairings[pairings[indexB]]=indexA;
            	}
            	//
            	int tmpP = pairings[indexA];
            	pairings[indexA] = pairings[indexB];
            	pairings[indexB]=tmpP;
            	//swap other column info
            	String[] max = (String[])sequenceColumns.remove(Math.max(indexA, indexB));
            	String[] min = (String[])sequenceColumns.remove(Math.min(indexA, indexB));
            	sequenceColumns.add(Math.min(indexA, indexB), max);
            	sequenceColumns.add(Math.max(indexA, indexB), min);
            }
            result = true;
        }
        return result;
    }


    /**
     *  Description of the Method
     *
     * @param  newChar  Description of the Parameter
     * @param  index    Description of the Parameter
     * @return          the old values, or '!' if the operation couldn't be done
     */
    public char[] changeSymbol(char newChar, int[] index)
    {
        char[] oldSymbols = new char[index.length];
        for (int i = 0; i < index.length; i++)
        {
            //initialize
            oldSymbols[i] = '!';
            if (index[i] >= 0 && index[i] < sequence.length && alphabet.isValidSymbol(newChar))
            {
                oldSymbols[i] = sequence[index[i]];
                sequence[index[i]] = newChar;

            }
        }
        return oldSymbols;
    }


    /**
     *  Description of the Method
     *
     * @param  newChar  Description of the Parameter
     * @param  index    Description of the Parameter
     * @return          Description of the Return Value
     */
    public char changeSymbol(char newChar, int index)
    {
        char oldSymbol = '!';
        if (index >= 0 && index < sequence.length && alphabet.isValidSymbol(newChar))
        {
            oldSymbol = sequence[index];
            sequence[index] = newChar;
        }
        return oldSymbol;
    }


    /**
     *  Description of the Method
     *
     * @param  index  Description of the Parameter
     * @return        Description of the Return Value
     */
    public boolean toUpperCase(int index)
    {
        boolean result = false;
        if (index >= 0 &&
                index < sequence.length &&
                Character.isLowerCase(sequence[index]))
        {
            char newChar = Character.toUpperCase(sequence[index]);
                sequence[index] = newChar;
                result = true;
        }
        return result;
    }


    /**
     *  Description of the Method
     *
     * @param  index  Description of the Parameter
     * @return        Description of the Return Value
     */
    public int[] toUpperCase(int[] index)
    {
        int[] changed = new int[index.length];
        for (int i = 0; i < index.length; i++)
        {
            changed[i] = -1;
            if (toUpperCase(index[i]))
            {
                changed[i] = index[i];
            }
        }
        return changed;
    }


    /**
     *  Description of the Method
     *
     * @param  index  Description of the Parameter
     * @return        Description of the Return Value
     */
    public boolean toLowerCase(int index)
    {
        boolean result = false;
        if (index >= 0 &&
                index < sequence.length &&
                Character.isUpperCase(sequence[index]))
        {
            char newChar = Character.toLowerCase(sequence[index]);
            sequence[index] = newChar;
            result = true;
            
        }
        return result;
    }


    /**
     *  Description of the Method
     */
    public void toLowerCase()
    {
        for (int i = 0; i < getLength(); i++)
        {
            sequence[i] = Character.toLowerCase(sequence[i]);
        }
    }


    /**
     *  Description of the Method
     *
     * @param  index  Description of the Parameter
     * @return        Description of the Return Value
     */
    public int[] toLowerCase(int[] index)
    {
        int[] changed = new int[index.length];
        for (int i = 0; i < index.length; i++)
        {
            changed[i] = -1;
            if (toLowerCase(index[i]))
            {
                changed[i] = index[i];
            }
        }
        return changed;
    }


    /**
     *  Description of the Method
     *
     * @return    Description of the Return Value
     */
    public String toString()
    {
        return getLabel() + ":\t" + getSequence();
    }


    /**
     *  Sets the label attribute of the Sequence object
     *
     * @param  name  The new label value
     */
    public void setLabel(String name)
    {
        label = name;
    }


    /**
     *  Adds a feature to the ColumnLabel attribute of the Sequence object
     *
     * @param  label   The feature to be added to the ColumnLabel attribute
     * @param  column  The feature to be added to the ColumnLabel attribute
     */
    public void addColumnLabel(String label, int column)
    {

        columnLabels.add(label);
        if (label.equals("align_bp"))
        {
            align_bpPosition = columnLabels.size() - 1;
        }
    }


    /**
     *  This method has to be called when creating a sequence from a col-file It
     *  must be called when the header has ended, and before reading the
     *  sequencecolumns Lokking for: residue: the actual sequence letter
     *  color2_b: the blue part of the foreground rgb color color2_r: the red
     *  part of the foreground rgb color color2_g: the green part of the
     *  foreground rgb color color_b: the blue part of the background rgb color
     *  color_r: the red part of the background rgb color color_g: the green
     *  part of the background rgb color
     */
    public void prepareForColumns()
    {
        numberOfColumns = columnLabels.size();
        for (int i = 0; i < columnLabels.size(); i++)
        {
            String colLabel = (String)columnLabels.get(i);
            //System.out.println(colLabel);
            //find sequence letter: residue
            if (colLabel.equals("residue"))
            {
                residue = i;
            }
            //find foreground coloring
            else if (colLabel.startsWith("color2"))
            {
                foreground = true;
                char end = colLabel.charAt(colLabel.length() - 1);
                if (end == 'r')
                {
                    color2_r = i;
                }
                else if (end == 'b')
                {
                    color2_b = i;
                }
                else if (end == 'g')
                {
                    color2_g = i;
                }
            }
            //find background coloring
            else if (colLabel.startsWith("color"))
            {
                //System.out.println("true");
                background = true;
                char end = colLabel.charAt(colLabel.length() - 1);
                if (end == 'r')
                {
                    color_r = i;
                }
                else if (end == 'b')
                {
                    color_b = i;
                }
                else if (end == 'g')
                {
                    color_g = i;
                }
            }
            else if (colLabel.equals("align_bp"))
            {
                align_bpPosition = i;
            }
        }
    }


    /**
     * @param  line           The line to be added to the sequence
     * @exception  Exception  Description of the Exception
     */
    public void addColumns(String line) throws Exception
    {
        Matcher wordMatcher = Pattern.compile("\\S+").matcher(line);
        String[] columns = new String[numberOfColumns];
        for (int i = 0; i < numberOfColumns; i++)
        {
            if (wordMatcher.find())
            {
                String word = line.substring(wordMatcher.start(), wordMatcher.end());
                columns[i] = word;
                if (i == residue)
                {
                    tmpSequence.append(word);
                }
            }
            else
            {
                throw new Exception("Invalid columnfile: to few columns");
            }
        }
        if (wordMatcher.find())
        {
            throw new Exception("Invalid columnfile: to many columns");
        }
        sequenceColumns.add(columns);
        if (align_bpPosition != -1)
        {
            String col = columns[align_bpPosition];
            if (col.equals("."))
            {
                getTmpPairings().add(-1);
                //tmpSequence.setCharAt(tmpSequence.length() - 1, Character.toLowerCase(tmpSequence.charAt(tmpSequence.length() - 1)));
            }
            else
            {
                try
                {
                    getTmpPairings().add(Integer.parseInt(col)-1);
                    //tmpSequence.setCharAt(tmpSequence.length() - 1, Character.toUpperCase(tmpSequence.charAt(tmpSequence.length() - 1)));
                }
                catch (Exception e)
                {
                    //if it isn't an integer then use -1 for not pairing
                    getTmpPairings().add(-1);
                }
            }
            //System.out.println(col+" = "+tmpPairings.get(tmpPairings.getLength()-1));
        }
    }


    /**
     *  Used to order the information added to the object from a colreader
     *  Setting colors, structure and deletes unneeded information
     *
     * @return                Description of the Return Value
     */
    public int[] finalProcessing()
    {
        //setting colors
        try
        {
            sequence = tmpSequence.toString().toCharArray();
            if (foreground || background)
            {
                foregroundColors = new Color[getLength()];
                backgroundColors = new Color[getLength()];
                for (int i = 0; i < sequenceColumns.size(); i++)
                {
                    String[] columns = (String[])sequenceColumns.get(i);
                    if (foreground)
                    {
                        foregroundColors[i] = new Color(Float.parseFloat(columns[color2_r]), Float.parseFloat(columns[color2_g]), Float.parseFloat(columns[color2_b]));
                    }
                    if (background)
                    {
                        backgroundColors[i] = new Color(Float.parseFloat(columns[color_r]), Float.parseFloat(columns[color_g]), Float.parseFloat(columns[color_b]));
                        //System.out.println("done: "+1);
                    }
                }
            }
        }
        catch (Exception e)
        {
            e.printStackTrace();
            foreground = false;
            background = false;
        }

        //removing col-info for columns that is automatically created when saved
        //columnlabels for removal
        String[] remColumns = {"label", "residue", "seqpos", "alignpos", "align_bp", "color_r", "color_g", "color_b", "color2_r", "color2_g", "color2_b"};
        IntArray colNumbers = new IntArray();
        Vector newLabels = new Vector();
        for (int i = columnLabels.size() - 1; i >= 0; i--)
        {
            boolean b = true;
            for (int j = 0; j < remColumns.length; j++)
            {
                if (((String)columnLabels.get(i)).equals(remColumns[j]))
                {
                    b = false;
                    break;
                }
            }
            if (b)
            {
                colNumbers.add(i);
                //label transfer
                newLabels.add(columnLabels.get(i));
            }
        }
        //transfer column info
        Vector newSequenceColumns = new Vector();
        for (int i = 0; i < getLength(); i++)
        {
            String[] colInfo = new String[colNumbers.getLength()];
            String[] oldColInfo = getColumnInfo(i);
            for (int j = 0; j < colNumbers.getLength(); j++)
            {
                colInfo[j] = oldColInfo[colNumbers.get(j)];
            }
            newSequenceColumns.add(colInfo);
        }
        columnLabels = newLabels;
        sequenceColumns = newSequenceColumns;
        //transfer pairings from tmpPairings if tmpParings exists
        if (align_bpPosition != -1)
        {
        	pairings = tmpPairings.toArray();
//            if (tmpPairings.getLength() == getLength())
//            {
//                pairings = tmpPairings.toArray();
//            }
        }
        return pairings;
    }


    /**
     *  Description of the Method
     *
     * @param  col1  Description of the Parameter
     * @param  col2  Description of the Parameter
     * @return       Description of the Return Value
     */
    public boolean pairBases(int col1, int col2)
    {
//        if(alphabet.isPairing(getSymbolAt(col1),getSymbolAt(col2)) )
//        {
        sequence[col1] = Character.toUpperCase(sequence[col1]);
        sequence[col2] = Character.toUpperCase(sequence[col2]);
        pairings[col1]= col2;
        pairings[col2]= col1;
        return true;
//        }
//        else
//        {
//            return false;
//        }
        
    }


    /**
     *  Gets the tmpPairings attribute of the Sequence object
     *
     * @return    The tmpPairings value
     */
    public IntArray getTmpPairings()
    {
        if (tmpPairings == null)
        {
            tmpPairings = new IntArray(100);
        }
        return tmpPairings;
    }


    /**
     *  Gets the backgroundColor attribute of the Sequence object
     *
     * @param  index  Description of the Parameter
     * @return        The backgroundColor value
     */
    public Color getBackgroundColor(int index)
    {
        if (background)
        {
            return backgroundColors[index];
        }
        else
        {
            return Color.WHITE;
        }
    }


    /**
     *  Gets the foregroundColor attribute of the Sequence object
     *
     * @param  index  Description of the Parameter
     * @return        The foregroundColor value
     */
    public Color getForegroundColor(int index)
    {
        if (foreground)
        {
            return foregroundColors[index];
        }
        else
        {
            return Color.BLACK;
        }
    }


    /**
     *  Sets the sequenceColumns attribute of the Sequence object
     *
     * @param  seqColumns  The new sequenceColumns value
     */
    public void setSequenceColumns(Vector seqColumns)
    {
        sequenceColumns = seqColumns;
    }


    /**
     *  Sets the columnLabels attribute of the Sequence object
     *
     * @param  labels  The new columnLabels value
     */
    public void setColumnLabels(Vector labels)
    {
        columnLabels = labels;
    }


    /**
     *  Sets the foreground attribute of the Sequence object
     *
     * @param  f  The new foreground value
     */
    public void setForeground(boolean f)
    {
        foreground = f;
    }


    /**
     *  Sets the background attribute of the Sequence object
     *
     * @param  f  The new background value
     */
    public void setBackground(boolean f)
    {
        background = f;
    }


    /**
     *  Sets the backgroundColors attribute of the Sequence object
     *
     * @param  colors  The new backgroundColors value
     */
    public void setBackgroundColors(Color[] colors)
    {
        backgroundColors = colors;
    }


    /**
     *  Sets the foregroundColors attribute of the Sequence object
     *
     * @param  colors  The new foregroundColors value
     */
    public void setForegroundColors(Color[] colors)
    {
        foregroundColors = colors;
    }


    /**
     *  Adds a feature to the Information attribute of the Sequence object
     *
     * @param  info  The feature to be added to the Information attribute
     */
    public void addInformation(String info)
    {
        information.append(info);
        if (!info.endsWith("\n"))
        {
            information.append("\n");
        }
    }


    /**
     *  Gets the information attribute of the Sequence object
     *
     * @return    The information value
     */
    public String getInformation()
    {
        return information.toString();
    }


    /**
     *  Description of the Method
     *
     * @param  start  Description of the Parameter
     * @param  end    Description of the Parameter
     * @return        Description of the Return Value
     */
  public boolean rotateLeft(int start, int end)
    {
        if (start < end)
        {
            int gaps = 0;
            while (getSymbolAt(end - gaps) == '-' && end - gaps >= start)
            {
                gaps++;
            }
            if (gaps > 0)
            {
                for (int i = end - 1; i >= start; i--)
                {
                   swapSymbols(i, i + 1);
                }
            }
            else
            {
                return false;
            }
        }
        else
        {
            return false;
        }
        return true;
    }

    
    /**
     *  Description of the Method
     *
     * @param  start  Description of the Parameter
     * @param  end    Description of the Parameter
     * @return        Description of the Return Value
     */
    public boolean rotateRight(int start, int end)
    {
        if (start < end)
        {
            int gaps = 0;
            while (getSymbolAt(start + gaps) == '-' && start + gaps <= end)
            {
                gaps++;
            }
            if (gaps > 0 )
            {
                for (int i = start + 1; i <= end; i++)
                {
                    swapSymbols(i - 1, i);
                }
            }
            else
            {
                return false;
            }
        }
        else
        {
            return false;
        }
        return true;
    }


    /**
     *  Gets the columnInfo attribute of the Sequence object
     *
     * @param  index  Description of the Parameter
     * @return        The columnInfo value
     */
    public String[] getColumnInfo(int index)
    {
        if (index >= 0 && index < sequenceColumns.size())
        {
            return (String[])sequenceColumns.get(index);
        }
        return null;
    }


    /**
     *  Gets the columnLabels attribute of the Sequence object
     *
     * @return    The columnLabels value
     */
    public String[] getColumnLabels()
    {
        return (String[])columnLabels.toArray(new String[columnLabels.size()]);
    }


    /**
     *  Description of the Method
     */
    public void emptyInfo()
    {
        //columnLabels = new Vector();
        //sequenceColumns = new Vector();
    }


    /**
     *  Gets the backgroundColred attribute of the Sequence object
     *
     * @return    The backgroundColred value
     */
    public boolean isBackgroundColred()
    {
        return background;
    }


    /**
     *  Gets the foregroundColred attribute of the Sequence object
     *
     * @return    The foregroundColred value
     */
    public boolean isForegroundColred()
    {
        return foreground;
    }


    /**
     *  Sets the information attribute of the Sequence object
     *
     * @param  info  The new information value
     */
    public void setInformation(String info)
    {
        information = new StringBuffer(info);
    }


    /**
     *  Sets the number attribute of the Sequence object
     *
     * @param  newNumber  The new number value
     */
    public void setNumber(int newNumber)
    {
        number = newNumber;
    }


    /**
     *  Gets the number attribute of the Sequence object
     *
     * @return    The number value
     */
    public int getNumber()
    {
        return number;
    }



    /**
     *  Sets the pairingmask attribute of the Sequence object
     *
     * @param  pairingmask  The new pairingmask value
     * @return              Description of the Return Value
     */
    public boolean setPairingmask(char[] pairingmask)
    {
        if (sequence.length == pairingmask.length)
        {
            char[] tmpPair = new char[pairingmask.length];
            for (int i = 0; i < sequence.length; i++)
            {
                if (Character.isUpperCase(sequence[i]) && 
                		pairingmask[i] != '-' && pairingmask[i] != '.')
                {
                    tmpPair[i] = pairingmask[i];
                }
                else
                {
                    tmpPair[i] = '-';
                }
            }
            int[] tmpPairings = PairingMask.updatePairings(tmpPair); 
            if(pairings!=null)
            {
            for(int i = 0;i<pairings.length;i++)
            {
            	if(tmpPairings[i]==PairingMask.NO_PAIRING && pairings[i] != PairingMask.NO_PAIRING)
            	{
            		tmpPairings[i]=pairings[i];
            	}
            }
            }
            pairings = tmpPairings;
            return true;
        }
        return false;
    }


    /**
     *  Gets the pairings attribute of the Sequence object
     *
     * @return    The pairings value
     */
    public int[] getPairings()
    {
        return pairings;
    }


    /**
     *  Gets the pairing attribute of the PairingMask object
     *
     * @param  index  Description of the Parameter
     * @return        The pairing value
     */
    public int getPairing(int index)
    {
        if (pairings == null)
        {
            return PairingMask.NO_PAIRINGMASK;
        }
        else if (index >= 0 && index < pairings.length)
        {
            return pairings[index];
        }
        else
        {
            return PairingMask.OUT_OF_SEQUENCE;
        }
    }


    /**
     *  Description of the Method
     *
     * @param  index  Description of the Parameter
     * @return        Description of the Return Value
     */
    public char removeColumn(int index)
    {
        char removed = getSymbolAt(index);
        char[] newSequence = new char[sequence.length - 1];
        for (int i = 0; i < index; i++)
        {
            newSequence[i] = sequence[i];
        }
        for (int i = index + 1; i < sequence.length; i++)
        {
            newSequence[i - 1] = sequence[i];
        }
        sequence = newSequence;
        return removed;
    }


    /**
     *  Adds a feature to the Column attribute of the Sequence object
     *
     * @param  index   The feature to be added to the Column attribute
     * @param  symbol  The feature to be added to the Column attribute
     */
    public void addColumn(int index, char symbol)
    {
        char[] newSequence = new char[sequence.length + 1];
        if (index == 0)
        {
            newSequence[0] = symbol;
            for (int i = 0; i < sequence.length; i++)
            {
                newSequence[i + 1] = sequence[i];
            }
        }
        else if (index >= sequence.length)
        {
            for (int i = 0; i < sequence.length; i++)
            {
                newSequence[i] = sequence[i];
            }
            newSequence[newSequence.length - 1] = symbol;
        }
        else
        {
            for (int i = 0; i < index; i++)
            {
                newSequence[i] = sequence[i];
            }
            newSequence[index] = symbol;
            for (int i = index; i < sequence.length; i++)
            {
                newSequence[i + 1] = sequence[i];
            }
        }
        updateColumns(index);
        if(foreground)
        {
        	insertForeground(index);
        }
        if(background)
        {
        	insertBackground(index);
        }
        sequence = newSequence;
        
    }


    /**
     *  Sets the cases attribute of the Sequence object
     */
    public void setCases()
    {
        for (int i = 0; i < getLength(); i++)
        {
            if (getPairing(i) > -1)
            {
                setSymbolAt(i, Character.toUpperCase(getSymbolAt(i)));
            }
            else
            {
                setSymbolAt(i, Character.toLowerCase(getSymbolAt(i)));
            }
        }
    }


    /**
     *  Description of the Method
     */
    public void removeColors()
    {
        background = false;
        foreground = false;
        backgroundColors = null;
        foregroundColors = null;
    }


    /**
     *  Description of the Method
     *
     * @param  cols  Description of the Parameter
     */
    public int[][] unpair(int[] cols)
    {
        int[][] pairs = new int[cols.length][2];
        for (int i = 0; i < cols.length; i++)
        {
            if(getPairing(cols[i]) >= 0)
            {
                pairs[i][0] = cols[i];
                pairs[i][1] = getPairing(cols[i]);
                toLowerCase(getPairing(cols[i]));
                pairings[getPairing(cols[i])]= -1;
                toLowerCase(cols[i]);
                pairings[cols[i]] = -1;
            }
            else
            {
                pairs[i][0] = -1;
                pairs[i][1] = -1;
            }
        }
        return pairs;
    }
    
    public int compareTo(Object o){
	final int before = -1;
	final int equal = 0;
	final int after = 1;
	
	Sequence s = (Sequence)o;
	
	if (s.getNumber() > this.getNumber()){
	    return before;
	}
	if (s.getNumber() < this.getNumber()){
	    return after;
	}
	else{
	    return equal;
	}
    }
    
    private void updateColumns(int index)
    {
    	//align_bp
    	int[] newPairs= new int[pairings.length+1];
    	if (index == 0)
        {
            newPairs[0] = PairingMask.NO_PAIRING;
            for (int i = 0; i < pairings.length; i++)
            {
                newPairs[i + 1] = pairings[i]+1;
            }
        }
        else if (index >= pairings.length)
        {
            for (int i = 0; i < pairings.length; i++)
            {
                newPairs[i] = pairings[i];
            }
            newPairs[newPairs.length - 1] = PairingMask.NO_PAIRING;
        }
        else
        {
            for (int i = 0; i < index; i++)
            {
                newPairs[i] = pairings[i];
                if(newPairs[i]>=index)
                {
                	newPairs[i]++;
                }
            }
            newPairs[index] = PairingMask.NO_PAIRING;
            for (int i = index; i < pairings.length; i++)
            {
                newPairs[i + 1] = pairings[i];
                if(newPairs[i+1]>=index)
                {
                	newPairs[i+1]++;
                }
            }
        }  
    	pairings=newPairs;
    	//other columns
    	if(columnLabels!= null && columnLabels.size()>0)
    	{
    	String[] cs = new String[columnLabels.size()];
    	for(int i  = 0;i<cs.length;i++)
    	{
    		cs[i]="0";
    	}
    	if(index<sequenceColumns.size())
    	{
    	sequenceColumns.add(index, cs);
    	}
    	else
    	{
    		sequenceColumns.add(cs);
    	}
    	}
    	
    }
	
    private void insertBackground(int index)
    {
    	backgroundColors = insertColor(backgroundColors,index,Color.WHITE);
    }
    
    private void insertForeground(int index)
    {
    	foregroundColors = insertColor(foregroundColors,index,Color.BLACK);
    }
    
    private Color[] insertColor(Color[] colors,int index,Color col)
    {
    	Color[] newColors = new Color[colors.length+1];
    	if (index == 0)
        {
            newColors[0] = col;
            for (int i = 0; i < colors.length; i++)
            {
                newColors[i + 1] = colors[i];
            }
        }
        else if (index >= colors.length)
        {
            for (int i = 0; i < colors.length; i++)
            {
                newColors[i] = colors[i];
            }
            newColors[newColors.length - 1] = col;
        }
        else
        {
            for (int i = 0; i < index; i++)
            {
                newColors[i] = colors[i];
            }
            newColors[index] = col;
            for (int i = index; i < colors.length; i++)
            {
                newColors[i + 1] = colors[i];
            }
        }
    	return newColors;
    }


	public void setPairings(int[] pairings) {
		this.pairings = pairings;
	}
	
	public void deletePairing(int index) {
		this.pairings[index] = PairingMask.NO_PAIRING;
	}
}


